﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/core/client/ClientConfiguration.h>
#include <aws/core/AmazonSerializableWebServiceRequest.h>
#include <aws/core/client/AWSClient.h>
#include <aws/core/client/AWSClientAsyncCRTP.h>
#include <aws/core/utils/xml/XmlSerializer.h>
#include <aws/ec2/EC2ServiceClientModel.h>

namespace Aws
{
namespace EC2
{
  /**
   * <fullname>Amazon Elastic Compute Cloud</fullname> <p>You can access the features
   * of Amazon Elastic Compute Cloud (Amazon EC2) programmatically. For more
   * information, see the <a
   * href="https://docs.aws.amazon.com/ec2/latest/devguide">Amazon EC2 Developer
   * Guide</a>.</p>
   */
  class AWS_EC2_API EC2Client : public Aws::Client::AWSXMLClient, public Aws::Client::ClientWithAsyncTemplateMethods<EC2Client>
  {
    public:
      typedef Aws::Client::AWSXMLClient BASECLASS;
      static const char* GetServiceName();
      static const char* GetAllocationTag();

      typedef EC2ClientConfiguration ClientConfigurationType;
      typedef EC2EndpointProvider EndpointProviderType;

       /**
        * Initializes client to use DefaultCredentialProviderChain, with default http client factory, and optional client config. If client config
        * is not specified, it will be initialized to default values.
        */
        EC2Client(const Aws::EC2::EC2ClientConfiguration& clientConfiguration = Aws::EC2::EC2ClientConfiguration(),
                  std::shared_ptr<EC2EndpointProviderBase> endpointProvider = nullptr);

       /**
        * Initializes client to use SimpleAWSCredentialsProvider, with default http client factory, and optional client config. If client config
        * is not specified, it will be initialized to default values.
        */
        EC2Client(const Aws::Auth::AWSCredentials& credentials,
                  std::shared_ptr<EC2EndpointProviderBase> endpointProvider = nullptr,
                  const Aws::EC2::EC2ClientConfiguration& clientConfiguration = Aws::EC2::EC2ClientConfiguration());

       /**
        * Initializes client to use specified credentials provider with specified client config. If http client factory is not supplied,
        * the default http client factory will be used
        */
        EC2Client(const std::shared_ptr<Aws::Auth::AWSCredentialsProvider>& credentialsProvider,
                  std::shared_ptr<EC2EndpointProviderBase> endpointProvider = nullptr,
                  const Aws::EC2::EC2ClientConfiguration& clientConfiguration = Aws::EC2::EC2ClientConfiguration());


        /* Legacy constructors due deprecation */
       /**
        * Initializes client to use DefaultCredentialProviderChain, with default http client factory, and optional client config. If client config
        * is not specified, it will be initialized to default values.
        */
        EC2Client(const Aws::Client::ClientConfiguration& clientConfiguration);

       /**
        * Initializes client to use SimpleAWSCredentialsProvider, with default http client factory, and optional client config. If client config
        * is not specified, it will be initialized to default values.
        */
        EC2Client(const Aws::Auth::AWSCredentials& credentials,
                  const Aws::Client::ClientConfiguration& clientConfiguration);

       /**
        * Initializes client to use specified credentials provider with specified client config. If http client factory is not supplied,
        * the default http client factory will be used
        */
        EC2Client(const std::shared_ptr<Aws::Auth::AWSCredentialsProvider>& credentialsProvider,
                  const Aws::Client::ClientConfiguration& clientConfiguration);

        /* End of legacy constructors due deprecation */
        virtual ~EC2Client();


        /**
         * <p>Accepts an Elastic IP address transfer. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#using-instance-addressing-eips-transfer-accept">Accept
         * a transferred Elastic IP address</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptAddressTransfer">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptAddressTransferOutcome AcceptAddressTransfer(const Model::AcceptAddressTransferRequest& request) const;

        /**
         * A Callable wrapper for AcceptAddressTransfer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptAddressTransferRequestT = Model::AcceptAddressTransferRequest>
        Model::AcceptAddressTransferOutcomeCallable AcceptAddressTransferCallable(const AcceptAddressTransferRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptAddressTransfer, request);
        }

        /**
         * An Async wrapper for AcceptAddressTransfer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptAddressTransferRequestT = Model::AcceptAddressTransferRequest>
        void AcceptAddressTransferAsync(const AcceptAddressTransferRequestT& request, const AcceptAddressTransferResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptAddressTransfer, request, handler, context);
        }

        /**
         * <p>Accepts a request to assign billing of the available capacity of a shared
         * Capacity Reservation to your account. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html">
         * Billing assignment for shared Amazon EC2 Capacity
         * Reservations</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptCapacityReservationBillingOwnership">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptCapacityReservationBillingOwnershipOutcome AcceptCapacityReservationBillingOwnership(const Model::AcceptCapacityReservationBillingOwnershipRequest& request) const;

        /**
         * A Callable wrapper for AcceptCapacityReservationBillingOwnership that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptCapacityReservationBillingOwnershipRequestT = Model::AcceptCapacityReservationBillingOwnershipRequest>
        Model::AcceptCapacityReservationBillingOwnershipOutcomeCallable AcceptCapacityReservationBillingOwnershipCallable(const AcceptCapacityReservationBillingOwnershipRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptCapacityReservationBillingOwnership, request);
        }

        /**
         * An Async wrapper for AcceptCapacityReservationBillingOwnership that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptCapacityReservationBillingOwnershipRequestT = Model::AcceptCapacityReservationBillingOwnershipRequest>
        void AcceptCapacityReservationBillingOwnershipAsync(const AcceptCapacityReservationBillingOwnershipRequestT& request, const AcceptCapacityReservationBillingOwnershipResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptCapacityReservationBillingOwnership, request, handler, context);
        }

        /**
         * <p>Accepts the Convertible Reserved Instance exchange quote described in the
         * <a>GetReservedInstancesExchangeQuote</a> call.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptReservedInstancesExchangeQuote">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptReservedInstancesExchangeQuoteOutcome AcceptReservedInstancesExchangeQuote(const Model::AcceptReservedInstancesExchangeQuoteRequest& request) const;

        /**
         * A Callable wrapper for AcceptReservedInstancesExchangeQuote that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptReservedInstancesExchangeQuoteRequestT = Model::AcceptReservedInstancesExchangeQuoteRequest>
        Model::AcceptReservedInstancesExchangeQuoteOutcomeCallable AcceptReservedInstancesExchangeQuoteCallable(const AcceptReservedInstancesExchangeQuoteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptReservedInstancesExchangeQuote, request);
        }

        /**
         * An Async wrapper for AcceptReservedInstancesExchangeQuote that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptReservedInstancesExchangeQuoteRequestT = Model::AcceptReservedInstancesExchangeQuoteRequest>
        void AcceptReservedInstancesExchangeQuoteAsync(const AcceptReservedInstancesExchangeQuoteRequestT& request, const AcceptReservedInstancesExchangeQuoteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptReservedInstancesExchangeQuote, request, handler, context);
        }

        /**
         * <p>Accepts a request to associate subnets with a transit gateway multicast
         * domain.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptTransitGatewayMulticastDomainAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptTransitGatewayMulticastDomainAssociationsOutcome AcceptTransitGatewayMulticastDomainAssociations(const Model::AcceptTransitGatewayMulticastDomainAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for AcceptTransitGatewayMulticastDomainAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptTransitGatewayMulticastDomainAssociationsRequestT = Model::AcceptTransitGatewayMulticastDomainAssociationsRequest>
        Model::AcceptTransitGatewayMulticastDomainAssociationsOutcomeCallable AcceptTransitGatewayMulticastDomainAssociationsCallable(const AcceptTransitGatewayMulticastDomainAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::AcceptTransitGatewayMulticastDomainAssociations, request);
        }

        /**
         * An Async wrapper for AcceptTransitGatewayMulticastDomainAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptTransitGatewayMulticastDomainAssociationsRequestT = Model::AcceptTransitGatewayMulticastDomainAssociationsRequest>
        void AcceptTransitGatewayMulticastDomainAssociationsAsync(const AcceptTransitGatewayMulticastDomainAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const AcceptTransitGatewayMulticastDomainAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::AcceptTransitGatewayMulticastDomainAssociations, request, handler, context);
        }

        /**
         * <p>Accepts a transit gateway peering attachment request. The peering attachment
         * must be in the <code>pendingAcceptance</code> state.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptTransitGatewayPeeringAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptTransitGatewayPeeringAttachmentOutcome AcceptTransitGatewayPeeringAttachment(const Model::AcceptTransitGatewayPeeringAttachmentRequest& request) const;

        /**
         * A Callable wrapper for AcceptTransitGatewayPeeringAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptTransitGatewayPeeringAttachmentRequestT = Model::AcceptTransitGatewayPeeringAttachmentRequest>
        Model::AcceptTransitGatewayPeeringAttachmentOutcomeCallable AcceptTransitGatewayPeeringAttachmentCallable(const AcceptTransitGatewayPeeringAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptTransitGatewayPeeringAttachment, request);
        }

        /**
         * An Async wrapper for AcceptTransitGatewayPeeringAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptTransitGatewayPeeringAttachmentRequestT = Model::AcceptTransitGatewayPeeringAttachmentRequest>
        void AcceptTransitGatewayPeeringAttachmentAsync(const AcceptTransitGatewayPeeringAttachmentRequestT& request, const AcceptTransitGatewayPeeringAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptTransitGatewayPeeringAttachment, request, handler, context);
        }

        /**
         * <p>Accepts a request to attach a VPC to a transit gateway.</p> <p>The VPC
         * attachment must be in the <code>pendingAcceptance</code> state. Use
         * <a>DescribeTransitGatewayVpcAttachments</a> to view your pending VPC attachment
         * requests. Use <a>RejectTransitGatewayVpcAttachment</a> to reject a VPC
         * attachment request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptTransitGatewayVpcAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptTransitGatewayVpcAttachmentOutcome AcceptTransitGatewayVpcAttachment(const Model::AcceptTransitGatewayVpcAttachmentRequest& request) const;

        /**
         * A Callable wrapper for AcceptTransitGatewayVpcAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptTransitGatewayVpcAttachmentRequestT = Model::AcceptTransitGatewayVpcAttachmentRequest>
        Model::AcceptTransitGatewayVpcAttachmentOutcomeCallable AcceptTransitGatewayVpcAttachmentCallable(const AcceptTransitGatewayVpcAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptTransitGatewayVpcAttachment, request);
        }

        /**
         * An Async wrapper for AcceptTransitGatewayVpcAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptTransitGatewayVpcAttachmentRequestT = Model::AcceptTransitGatewayVpcAttachmentRequest>
        void AcceptTransitGatewayVpcAttachmentAsync(const AcceptTransitGatewayVpcAttachmentRequestT& request, const AcceptTransitGatewayVpcAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptTransitGatewayVpcAttachment, request, handler, context);
        }

        /**
         * <p>Accepts connection requests to your VPC endpoint service.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptVpcEndpointConnections">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptVpcEndpointConnectionsOutcome AcceptVpcEndpointConnections(const Model::AcceptVpcEndpointConnectionsRequest& request) const;

        /**
         * A Callable wrapper for AcceptVpcEndpointConnections that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptVpcEndpointConnectionsRequestT = Model::AcceptVpcEndpointConnectionsRequest>
        Model::AcceptVpcEndpointConnectionsOutcomeCallable AcceptVpcEndpointConnectionsCallable(const AcceptVpcEndpointConnectionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptVpcEndpointConnections, request);
        }

        /**
         * An Async wrapper for AcceptVpcEndpointConnections that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptVpcEndpointConnectionsRequestT = Model::AcceptVpcEndpointConnectionsRequest>
        void AcceptVpcEndpointConnectionsAsync(const AcceptVpcEndpointConnectionsRequestT& request, const AcceptVpcEndpointConnectionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptVpcEndpointConnections, request, handler, context);
        }

        /**
         * <p>Accept a VPC peering connection request. To accept a request, the VPC peering
         * connection must be in the <code>pending-acceptance</code> state, and you must be
         * the owner of the peer VPC. Use <a>DescribeVpcPeeringConnections</a> to view your
         * outstanding VPC peering connection requests.</p> <p>For an inter-Region VPC
         * peering connection request, you must accept the VPC peering connection in the
         * Region of the accepter VPC.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AcceptVpcPeeringConnection">AWS
         * API Reference</a></p>
         */
        virtual Model::AcceptVpcPeeringConnectionOutcome AcceptVpcPeeringConnection(const Model::AcceptVpcPeeringConnectionRequest& request) const;

        /**
         * A Callable wrapper for AcceptVpcPeeringConnection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AcceptVpcPeeringConnectionRequestT = Model::AcceptVpcPeeringConnectionRequest>
        Model::AcceptVpcPeeringConnectionOutcomeCallable AcceptVpcPeeringConnectionCallable(const AcceptVpcPeeringConnectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AcceptVpcPeeringConnection, request);
        }

        /**
         * An Async wrapper for AcceptVpcPeeringConnection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AcceptVpcPeeringConnectionRequestT = Model::AcceptVpcPeeringConnectionRequest>
        void AcceptVpcPeeringConnectionAsync(const AcceptVpcPeeringConnectionRequestT& request, const AcceptVpcPeeringConnectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AcceptVpcPeeringConnection, request, handler, context);
        }

        /**
         * <p>Advertises an IPv4 or IPv6 address range that is provisioned for use with
         * your Amazon Web Services resources through bring your own IP addresses
         * (BYOIP).</p> <p>You can perform this operation at most once every 10 seconds,
         * even if you specify different address ranges each time.</p> <p>We recommend that
         * you stop advertising the BYOIP CIDR from other locations when you advertise it
         * from Amazon Web Services. To minimize down time, you can configure your Amazon
         * Web Services resources to use an address from a BYOIP CIDR before it is
         * advertised, and then simultaneously stop advertising it from the current
         * location and start advertising it through Amazon Web Services.</p> <p>It can
         * take a few minutes before traffic to the specified addresses starts routing to
         * Amazon Web Services because of BGP propagation delays.</p> <p>To stop
         * advertising the BYOIP CIDR, use <a>WithdrawByoipCidr</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AdvertiseByoipCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::AdvertiseByoipCidrOutcome AdvertiseByoipCidr(const Model::AdvertiseByoipCidrRequest& request) const;

        /**
         * A Callable wrapper for AdvertiseByoipCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AdvertiseByoipCidrRequestT = Model::AdvertiseByoipCidrRequest>
        Model::AdvertiseByoipCidrOutcomeCallable AdvertiseByoipCidrCallable(const AdvertiseByoipCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AdvertiseByoipCidr, request);
        }

        /**
         * An Async wrapper for AdvertiseByoipCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AdvertiseByoipCidrRequestT = Model::AdvertiseByoipCidrRequest>
        void AdvertiseByoipCidrAsync(const AdvertiseByoipCidrRequestT& request, const AdvertiseByoipCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AdvertiseByoipCidr, request, handler, context);
        }

        /**
         * <p>Allocates an Elastic IP address to your Amazon Web Services account. After
         * you allocate the Elastic IP address you can associate it with an instance or
         * network interface. After you release an Elastic IP address, it is released to
         * the IP address pool and can be allocated to a different Amazon Web Services
         * account.</p> <p>You can allocate an Elastic IP address from an address pool
         * owned by Amazon Web Services or from an address pool created from a public IPv4
         * address range that you have brought to Amazon Web Services for use with your
         * Amazon Web Services resources using bring your own IP addresses (BYOIP). For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html">Bring
         * Your Own IP Addresses (BYOIP)</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>If
         * you release an Elastic IP address, you might be able to recover it. You cannot
         * recover an Elastic IP address that you released after it is allocated to another
         * Amazon Web Services account. To attempt to recover an Elastic IP address that
         * you released, specify it in this operation.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic
         * IP Addresses</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>You can allocate a
         * carrier IP address which is a public IP address from a telecommunication
         * carrier, to a network interface which resides in a subnet in a Wavelength Zone
         * (for example an EC2 instance).</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AllocateAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::AllocateAddressOutcome AllocateAddress(const Model::AllocateAddressRequest& request = {}) const;

        /**
         * A Callable wrapper for AllocateAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AllocateAddressRequestT = Model::AllocateAddressRequest>
        Model::AllocateAddressOutcomeCallable AllocateAddressCallable(const AllocateAddressRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::AllocateAddress, request);
        }

        /**
         * An Async wrapper for AllocateAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AllocateAddressRequestT = Model::AllocateAddressRequest>
        void AllocateAddressAsync(const AllocateAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const AllocateAddressRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::AllocateAddress, request, handler, context);
        }

        /**
         * <p>Allocates a Dedicated Host to your account. At a minimum, specify the
         * supported instance type or instance family, the Availability Zone in which to
         * allocate the host, and the number of hosts to allocate.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AllocateHosts">AWS
         * API Reference</a></p>
         */
        virtual Model::AllocateHostsOutcome AllocateHosts(const Model::AllocateHostsRequest& request = {}) const;

        /**
         * A Callable wrapper for AllocateHosts that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AllocateHostsRequestT = Model::AllocateHostsRequest>
        Model::AllocateHostsOutcomeCallable AllocateHostsCallable(const AllocateHostsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::AllocateHosts, request);
        }

        /**
         * An Async wrapper for AllocateHosts that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AllocateHostsRequestT = Model::AllocateHostsRequest>
        void AllocateHostsAsync(const AllocateHostsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const AllocateHostsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::AllocateHosts, request, handler, context);
        }

        /**
         * <p>Allocate a CIDR from an IPAM pool. The Region you use should be the IPAM pool
         * locale. The locale is the Amazon Web Services Region where this IPAM pool is
         * available for allocations.</p> <p>In IPAM, an allocation is a CIDR assignment
         * from an IPAM pool to another IPAM pool or to a resource. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/allocate-cidrs-ipam.html">Allocate
         * CIDRs</a> in the <i>Amazon VPC IPAM User Guide</i>.</p>  <p>This action
         * creates an allocation with strong consistency. The returned CIDR will not
         * overlap with any other allocations from the same pool.</p> <p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AllocateIpamPoolCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::AllocateIpamPoolCidrOutcome AllocateIpamPoolCidr(const Model::AllocateIpamPoolCidrRequest& request) const;

        /**
         * A Callable wrapper for AllocateIpamPoolCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AllocateIpamPoolCidrRequestT = Model::AllocateIpamPoolCidrRequest>
        Model::AllocateIpamPoolCidrOutcomeCallable AllocateIpamPoolCidrCallable(const AllocateIpamPoolCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AllocateIpamPoolCidr, request);
        }

        /**
         * An Async wrapper for AllocateIpamPoolCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AllocateIpamPoolCidrRequestT = Model::AllocateIpamPoolCidrRequest>
        void AllocateIpamPoolCidrAsync(const AllocateIpamPoolCidrRequestT& request, const AllocateIpamPoolCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AllocateIpamPoolCidr, request, handler, context);
        }

        /**
         * <p>Applies a security group to the association between the target network and
         * the Client VPN endpoint. This action replaces the existing security groups with
         * the specified security groups.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ApplySecurityGroupsToClientVpnTargetNetwork">AWS
         * API Reference</a></p>
         */
        virtual Model::ApplySecurityGroupsToClientVpnTargetNetworkOutcome ApplySecurityGroupsToClientVpnTargetNetwork(const Model::ApplySecurityGroupsToClientVpnTargetNetworkRequest& request) const;

        /**
         * A Callable wrapper for ApplySecurityGroupsToClientVpnTargetNetwork that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ApplySecurityGroupsToClientVpnTargetNetworkRequestT = Model::ApplySecurityGroupsToClientVpnTargetNetworkRequest>
        Model::ApplySecurityGroupsToClientVpnTargetNetworkOutcomeCallable ApplySecurityGroupsToClientVpnTargetNetworkCallable(const ApplySecurityGroupsToClientVpnTargetNetworkRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ApplySecurityGroupsToClientVpnTargetNetwork, request);
        }

        /**
         * An Async wrapper for ApplySecurityGroupsToClientVpnTargetNetwork that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ApplySecurityGroupsToClientVpnTargetNetworkRequestT = Model::ApplySecurityGroupsToClientVpnTargetNetworkRequest>
        void ApplySecurityGroupsToClientVpnTargetNetworkAsync(const ApplySecurityGroupsToClientVpnTargetNetworkRequestT& request, const ApplySecurityGroupsToClientVpnTargetNetworkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ApplySecurityGroupsToClientVpnTargetNetwork, request, handler, context);
        }

        /**
         * <p>Assigns the specified IPv6 addresses to the specified network interface. You
         * can specify specific IPv6 addresses, or you can specify the number of IPv6
         * addresses to be automatically assigned from the subnet's IPv6 CIDR block range.
         * You can assign as many IPv6 addresses to a network interface as you can assign
         * private IPv4 addresses, and the limit varies by instance type.</p> <p>You must
         * specify either the IPv6 addresses or the IPv6 address count in the request. </p>
         * <p>You can optionally use Prefix Delegation on the network interface. You must
         * specify either the IPV6 Prefix Delegation prefixes, or the IPv6 Prefix
         * Delegation count. For information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-prefix-eni.html">
         * Assigning prefixes to network interfaces</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssignIpv6Addresses">AWS
         * API Reference</a></p>
         */
        virtual Model::AssignIpv6AddressesOutcome AssignIpv6Addresses(const Model::AssignIpv6AddressesRequest& request) const;

        /**
         * A Callable wrapper for AssignIpv6Addresses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssignIpv6AddressesRequestT = Model::AssignIpv6AddressesRequest>
        Model::AssignIpv6AddressesOutcomeCallable AssignIpv6AddressesCallable(const AssignIpv6AddressesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssignIpv6Addresses, request);
        }

        /**
         * An Async wrapper for AssignIpv6Addresses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssignIpv6AddressesRequestT = Model::AssignIpv6AddressesRequest>
        void AssignIpv6AddressesAsync(const AssignIpv6AddressesRequestT& request, const AssignIpv6AddressesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssignIpv6Addresses, request, handler, context);
        }

        /**
         * <p>Assigns the specified secondary private IP addresses to the specified network
         * interface.</p> <p>You can specify specific secondary IP addresses, or you can
         * specify the number of secondary IP addresses to be automatically assigned from
         * the subnet's CIDR block range. The number of secondary IP addresses that you can
         * assign to an instance varies by instance type. For more information about
         * Elastic IP addresses, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html">Elastic
         * IP Addresses</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>When you move a
         * secondary private IP address to another network interface, any Elastic IP
         * address that is associated with the IP address is also moved.</p> <p>Remapping
         * an IP address is an asynchronous operation. When you move an IP address from one
         * network interface to another, check
         * <code>network/interfaces/macs/mac/local-ipv4s</code> in the instance metadata to
         * confirm that the remapping is complete.</p> <p>You must specify either the IP
         * addresses or the IP address count in the request.</p> <p>You can optionally use
         * Prefix Delegation on the network interface. You must specify either the IPv4
         * Prefix Delegation prefixes, or the IPv4 Prefix Delegation count. For
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-prefix-eni.html">
         * Assigning prefixes to network interfaces</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssignPrivateIpAddresses">AWS
         * API Reference</a></p>
         */
        virtual Model::AssignPrivateIpAddressesOutcome AssignPrivateIpAddresses(const Model::AssignPrivateIpAddressesRequest& request) const;

        /**
         * A Callable wrapper for AssignPrivateIpAddresses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssignPrivateIpAddressesRequestT = Model::AssignPrivateIpAddressesRequest>
        Model::AssignPrivateIpAddressesOutcomeCallable AssignPrivateIpAddressesCallable(const AssignPrivateIpAddressesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssignPrivateIpAddresses, request);
        }

        /**
         * An Async wrapper for AssignPrivateIpAddresses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssignPrivateIpAddressesRequestT = Model::AssignPrivateIpAddressesRequest>
        void AssignPrivateIpAddressesAsync(const AssignPrivateIpAddressesRequestT& request, const AssignPrivateIpAddressesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssignPrivateIpAddresses, request, handler, context);
        }

        /**
         * <p>Assigns private IPv4 addresses to a private NAT gateway. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/nat-gateway-working-with.html">Work
         * with NAT gateways</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssignPrivateNatGatewayAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::AssignPrivateNatGatewayAddressOutcome AssignPrivateNatGatewayAddress(const Model::AssignPrivateNatGatewayAddressRequest& request) const;

        /**
         * A Callable wrapper for AssignPrivateNatGatewayAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssignPrivateNatGatewayAddressRequestT = Model::AssignPrivateNatGatewayAddressRequest>
        Model::AssignPrivateNatGatewayAddressOutcomeCallable AssignPrivateNatGatewayAddressCallable(const AssignPrivateNatGatewayAddressRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssignPrivateNatGatewayAddress, request);
        }

        /**
         * An Async wrapper for AssignPrivateNatGatewayAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssignPrivateNatGatewayAddressRequestT = Model::AssignPrivateNatGatewayAddressRequest>
        void AssignPrivateNatGatewayAddressAsync(const AssignPrivateNatGatewayAddressRequestT& request, const AssignPrivateNatGatewayAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssignPrivateNatGatewayAddress, request, handler, context);
        }

        /**
         * <p>Associates an Elastic IP address, or carrier IP address (for instances that
         * are in subnets in Wavelength Zones) with an instance or a network interface.
         * Before you can use an Elastic IP address, you must allocate it to your
         * account.</p> <p>If the Elastic IP address is already associated with a different
         * instance, it is disassociated from that instance and associated with the
         * specified instance. If you associate an Elastic IP address with an instance that
         * has an existing Elastic IP address, the existing address is disassociated from
         * the instance, but remains allocated to your account.</p> <p>[Subnets in
         * Wavelength Zones] You can associate an IP address from the telecommunication
         * carrier to the instance or network interface. </p> <p>You cannot associate an
         * Elastic IP address with an interface in a different network border group.</p>
         *  <p>This is an idempotent operation. If you perform the operation
         * more than once, Amazon EC2 doesn't return an error, and you may be charged for
         * each time the Elastic IP address is remapped to the same instance. For more
         * information, see the <i>Elastic IP Addresses</i> section of <a
         * href="http://aws.amazon.com/ec2/pricing/">Amazon EC2 Pricing</a>.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateAddressOutcome AssociateAddress(const Model::AssociateAddressRequest& request = {}) const;

        /**
         * A Callable wrapper for AssociateAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateAddressRequestT = Model::AssociateAddressRequest>
        Model::AssociateAddressOutcomeCallable AssociateAddressCallable(const AssociateAddressRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::AssociateAddress, request);
        }

        /**
         * An Async wrapper for AssociateAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateAddressRequestT = Model::AssociateAddressRequest>
        void AssociateAddressAsync(const AssociateAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const AssociateAddressRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::AssociateAddress, request, handler, context);
        }

        /**
         * <p>Initiates a request to assign billing of the unused capacity of a shared
         * Capacity Reservation to a consumer account that is consolidated under the same
         * Amazon Web Services organizations payer account. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html">Billing
         * assignment for shared Amazon EC2 Capacity Reservations</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateCapacityReservationBillingOwner">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateCapacityReservationBillingOwnerOutcome AssociateCapacityReservationBillingOwner(const Model::AssociateCapacityReservationBillingOwnerRequest& request) const;

        /**
         * A Callable wrapper for AssociateCapacityReservationBillingOwner that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateCapacityReservationBillingOwnerRequestT = Model::AssociateCapacityReservationBillingOwnerRequest>
        Model::AssociateCapacityReservationBillingOwnerOutcomeCallable AssociateCapacityReservationBillingOwnerCallable(const AssociateCapacityReservationBillingOwnerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateCapacityReservationBillingOwner, request);
        }

        /**
         * An Async wrapper for AssociateCapacityReservationBillingOwner that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateCapacityReservationBillingOwnerRequestT = Model::AssociateCapacityReservationBillingOwnerRequest>
        void AssociateCapacityReservationBillingOwnerAsync(const AssociateCapacityReservationBillingOwnerRequestT& request, const AssociateCapacityReservationBillingOwnerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateCapacityReservationBillingOwner, request, handler, context);
        }

        /**
         * <p>Associates a target network with a Client VPN endpoint. A target network is a
         * subnet in a VPC. You can associate multiple subnets from the same VPC with a
         * Client VPN endpoint. You can associate only one subnet in each Availability
         * Zone. We recommend that you associate at least two subnets to provide
         * Availability Zone redundancy.</p> <p>If you specified a VPC when you created the
         * Client VPN endpoint or if you have previous subnet associations, the specified
         * subnet must be in the same VPC. To specify a subnet that's in a different VPC,
         * you must first modify the Client VPN endpoint (<a>ModifyClientVpnEndpoint</a>)
         * and change the VPC that's associated with it.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateClientVpnTargetNetwork">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateClientVpnTargetNetworkOutcome AssociateClientVpnTargetNetwork(const Model::AssociateClientVpnTargetNetworkRequest& request) const;

        /**
         * A Callable wrapper for AssociateClientVpnTargetNetwork that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateClientVpnTargetNetworkRequestT = Model::AssociateClientVpnTargetNetworkRequest>
        Model::AssociateClientVpnTargetNetworkOutcomeCallable AssociateClientVpnTargetNetworkCallable(const AssociateClientVpnTargetNetworkRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateClientVpnTargetNetwork, request);
        }

        /**
         * An Async wrapper for AssociateClientVpnTargetNetwork that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateClientVpnTargetNetworkRequestT = Model::AssociateClientVpnTargetNetworkRequest>
        void AssociateClientVpnTargetNetworkAsync(const AssociateClientVpnTargetNetworkRequestT& request, const AssociateClientVpnTargetNetworkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateClientVpnTargetNetwork, request, handler, context);
        }

        /**
         * <p>Associates a set of DHCP options (that you've previously created) with the
         * specified VPC, or associates no DHCP options with the VPC.</p> <p>After you
         * associate the options with the VPC, any existing instances and all new instances
         * that you launch in that VPC use the options. You don't need to restart or
         * relaunch the instances. They automatically pick up the changes within a few
         * hours, depending on how frequently the instance renews its DHCP lease. You can
         * explicitly renew the lease using the operating system on the instance.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html">DHCP
         * option sets</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateDhcpOptions">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateDhcpOptionsOutcome AssociateDhcpOptions(const Model::AssociateDhcpOptionsRequest& request) const;

        /**
         * A Callable wrapper for AssociateDhcpOptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateDhcpOptionsRequestT = Model::AssociateDhcpOptionsRequest>
        Model::AssociateDhcpOptionsOutcomeCallable AssociateDhcpOptionsCallable(const AssociateDhcpOptionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateDhcpOptions, request);
        }

        /**
         * An Async wrapper for AssociateDhcpOptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateDhcpOptionsRequestT = Model::AssociateDhcpOptionsRequest>
        void AssociateDhcpOptionsAsync(const AssociateDhcpOptionsRequestT& request, const AssociateDhcpOptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateDhcpOptions, request, handler, context);
        }

        /**
         * <p>Associates an Identity and Access Management (IAM) role with an Certificate
         * Manager (ACM) certificate. This enables the certificate to be used by the ACM
         * for Nitro Enclaves application inside an enclave. For more information, see <a
         * href="https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html">Certificate
         * Manager for Nitro Enclaves</a> in the <i>Amazon Web Services Nitro Enclaves User
         * Guide</i>.</p> <p>When the IAM role is associated with the ACM certificate, the
         * certificate, certificate chain, and encrypted private key are placed in an
         * Amazon S3 location that only the associated IAM role can access. The private key
         * of the certificate is encrypted with an Amazon Web Services managed key that has
         * an attached attestation-based key policy.</p> <p>To enable the IAM role to
         * access the Amazon S3 object, you must grant it permission to call
         * <code>s3:GetObject</code> on the Amazon S3 bucket returned by the command. To
         * enable the IAM role to access the KMS key, you must grant it permission to call
         * <code>kms:Decrypt</code> on the KMS key returned by the command. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html#add-policy">
         * Grant the role permission to access the certificate and encryption key</a> in
         * the <i>Amazon Web Services Nitro Enclaves User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateEnclaveCertificateIamRole">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateEnclaveCertificateIamRoleOutcome AssociateEnclaveCertificateIamRole(const Model::AssociateEnclaveCertificateIamRoleRequest& request) const;

        /**
         * A Callable wrapper for AssociateEnclaveCertificateIamRole that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateEnclaveCertificateIamRoleRequestT = Model::AssociateEnclaveCertificateIamRoleRequest>
        Model::AssociateEnclaveCertificateIamRoleOutcomeCallable AssociateEnclaveCertificateIamRoleCallable(const AssociateEnclaveCertificateIamRoleRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateEnclaveCertificateIamRole, request);
        }

        /**
         * An Async wrapper for AssociateEnclaveCertificateIamRole that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateEnclaveCertificateIamRoleRequestT = Model::AssociateEnclaveCertificateIamRoleRequest>
        void AssociateEnclaveCertificateIamRoleAsync(const AssociateEnclaveCertificateIamRoleRequestT& request, const AssociateEnclaveCertificateIamRoleResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateEnclaveCertificateIamRole, request, handler, context);
        }

        /**
         * <p>Associates an IAM instance profile with a running or stopped instance. You
         * cannot associate more than one IAM instance profile with an
         * instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateIamInstanceProfile">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateIamInstanceProfileOutcome AssociateIamInstanceProfile(const Model::AssociateIamInstanceProfileRequest& request) const;

        /**
         * A Callable wrapper for AssociateIamInstanceProfile that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateIamInstanceProfileRequestT = Model::AssociateIamInstanceProfileRequest>
        Model::AssociateIamInstanceProfileOutcomeCallable AssociateIamInstanceProfileCallable(const AssociateIamInstanceProfileRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateIamInstanceProfile, request);
        }

        /**
         * An Async wrapper for AssociateIamInstanceProfile that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateIamInstanceProfileRequestT = Model::AssociateIamInstanceProfileRequest>
        void AssociateIamInstanceProfileAsync(const AssociateIamInstanceProfileRequestT& request, const AssociateIamInstanceProfileResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateIamInstanceProfile, request, handler, context);
        }

        /**
         * <p>Associates one or more targets with an event window. Only one type of target
         * (instance IDs, Dedicated Host IDs, or tags) can be specified with an event
         * window.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/event-windows.html">Define
         * event windows for scheduled events</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateInstanceEventWindow">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateInstanceEventWindowOutcome AssociateInstanceEventWindow(const Model::AssociateInstanceEventWindowRequest& request) const;

        /**
         * A Callable wrapper for AssociateInstanceEventWindow that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateInstanceEventWindowRequestT = Model::AssociateInstanceEventWindowRequest>
        Model::AssociateInstanceEventWindowOutcomeCallable AssociateInstanceEventWindowCallable(const AssociateInstanceEventWindowRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateInstanceEventWindow, request);
        }

        /**
         * An Async wrapper for AssociateInstanceEventWindow that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateInstanceEventWindowRequestT = Model::AssociateInstanceEventWindowRequest>
        void AssociateInstanceEventWindowAsync(const AssociateInstanceEventWindowRequestT& request, const AssociateInstanceEventWindowResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateInstanceEventWindow, request, handler, context);
        }

        /**
         * <p>Associates your Autonomous System Number (ASN) with a BYOIP CIDR that you own
         * in the same Amazon Web Services Region. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-byoasn.html">Tutorial:
         * Bring your ASN to IPAM</a> in the <i>Amazon VPC IPAM guide</i>.</p> <p>After the
         * association succeeds, the ASN is eligible for advertisement. You can view the
         * association with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html">DescribeByoipCidrs</a>.
         * You can advertise the CIDR with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html">AdvertiseByoipCidr</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateIpamByoasn">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateIpamByoasnOutcome AssociateIpamByoasn(const Model::AssociateIpamByoasnRequest& request) const;

        /**
         * A Callable wrapper for AssociateIpamByoasn that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateIpamByoasnRequestT = Model::AssociateIpamByoasnRequest>
        Model::AssociateIpamByoasnOutcomeCallable AssociateIpamByoasnCallable(const AssociateIpamByoasnRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateIpamByoasn, request);
        }

        /**
         * An Async wrapper for AssociateIpamByoasn that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateIpamByoasnRequestT = Model::AssociateIpamByoasnRequest>
        void AssociateIpamByoasnAsync(const AssociateIpamByoasnRequestT& request, const AssociateIpamByoasnResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateIpamByoasn, request, handler, context);
        }

        /**
         * <p>Associates an IPAM resource discovery with an Amazon VPC IPAM. A resource
         * discovery is an IPAM component that enables IPAM to manage and monitor resources
         * that belong to the owning account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateIpamResourceDiscovery">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateIpamResourceDiscoveryOutcome AssociateIpamResourceDiscovery(const Model::AssociateIpamResourceDiscoveryRequest& request) const;

        /**
         * A Callable wrapper for AssociateIpamResourceDiscovery that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateIpamResourceDiscoveryRequestT = Model::AssociateIpamResourceDiscoveryRequest>
        Model::AssociateIpamResourceDiscoveryOutcomeCallable AssociateIpamResourceDiscoveryCallable(const AssociateIpamResourceDiscoveryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateIpamResourceDiscovery, request);
        }

        /**
         * An Async wrapper for AssociateIpamResourceDiscovery that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateIpamResourceDiscoveryRequestT = Model::AssociateIpamResourceDiscoveryRequest>
        void AssociateIpamResourceDiscoveryAsync(const AssociateIpamResourceDiscoveryRequestT& request, const AssociateIpamResourceDiscoveryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateIpamResourceDiscovery, request, handler, context);
        }

        /**
         * <p>Associates Elastic IP addresses (EIPs) and private IPv4 addresses with a
         * public NAT gateway. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/nat-gateway-working-with.html">Work
         * with NAT gateways</a> in the <i>Amazon VPC User Guide</i>.</p> <p>By default,
         * you can associate up to 2 Elastic IP addresses per public NAT gateway. You can
         * increase the limit by requesting a quota adjustment. For more information, see
         * <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html#vpc-limits-eips">Elastic
         * IP address quotas</a> in the <i>Amazon VPC User Guide</i>.</p> 
         * <p>When you associate an EIP or secondary EIPs with a public NAT gateway, the
         * network border group of the EIPs must match the network border group of the
         * Availability Zone (AZ) that the public NAT gateway is in. If it's not the same,
         * the EIP will fail to associate. You can see the network border group for the
         * subnet's AZ by viewing the details of the subnet. Similarly, you can view the
         * network border group of an EIP by viewing the details of the EIP address. For
         * more information about network border groups and EIPs, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/WorkWithEIPs.html">Allocate
         * an Elastic IP address</a> in the <i>Amazon VPC User Guide</i>. </p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateNatGatewayAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateNatGatewayAddressOutcome AssociateNatGatewayAddress(const Model::AssociateNatGatewayAddressRequest& request) const;

        /**
         * A Callable wrapper for AssociateNatGatewayAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateNatGatewayAddressRequestT = Model::AssociateNatGatewayAddressRequest>
        Model::AssociateNatGatewayAddressOutcomeCallable AssociateNatGatewayAddressCallable(const AssociateNatGatewayAddressRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateNatGatewayAddress, request);
        }

        /**
         * An Async wrapper for AssociateNatGatewayAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateNatGatewayAddressRequestT = Model::AssociateNatGatewayAddressRequest>
        void AssociateNatGatewayAddressAsync(const AssociateNatGatewayAddressRequestT& request, const AssociateNatGatewayAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateNatGatewayAddress, request, handler, context);
        }

        /**
         * <p>Associates a route server with a VPC to enable dynamic route updates.</p>
         * <p>A route server association is the connection established between a route
         * server and a VPC.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateRouteServer">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateRouteServerOutcome AssociateRouteServer(const Model::AssociateRouteServerRequest& request) const;

        /**
         * A Callable wrapper for AssociateRouteServer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateRouteServerRequestT = Model::AssociateRouteServerRequest>
        Model::AssociateRouteServerOutcomeCallable AssociateRouteServerCallable(const AssociateRouteServerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateRouteServer, request);
        }

        /**
         * An Async wrapper for AssociateRouteServer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateRouteServerRequestT = Model::AssociateRouteServerRequest>
        void AssociateRouteServerAsync(const AssociateRouteServerRequestT& request, const AssociateRouteServerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateRouteServer, request, handler, context);
        }

        /**
         * <p>Associates a subnet in your VPC or an internet gateway or virtual private
         * gateway attached to your VPC with a route table in your VPC. This association
         * causes traffic from the subnet or gateway to be routed according to the routes
         * in the route table. The action returns an association ID, which you need in
         * order to disassociate the route table later. A route table can be associated
         * with multiple subnets.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html">Route
         * tables</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateRouteTableOutcome AssociateRouteTable(const Model::AssociateRouteTableRequest& request) const;

        /**
         * A Callable wrapper for AssociateRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateRouteTableRequestT = Model::AssociateRouteTableRequest>
        Model::AssociateRouteTableOutcomeCallable AssociateRouteTableCallable(const AssociateRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateRouteTable, request);
        }

        /**
         * An Async wrapper for AssociateRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateRouteTableRequestT = Model::AssociateRouteTableRequest>
        void AssociateRouteTableAsync(const AssociateRouteTableRequestT& request, const AssociateRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateRouteTable, request, handler, context);
        }

        /**
         * <p>Associates a security group with another VPC in the same Region. This enables
         * you to use the same security group with network interfaces and instances in the
         * specified VPC.</p>  <ul> <li> <p>The VPC you want to associate the
         * security group with must be in the same Region.</p> </li> <li> <p>You can
         * associate the security group with another VPC if your account owns the VPC or if
         * the VPC was shared with you.</p> </li> <li> <p>You must own the security group
         * and the VPC that it was created in.</p> </li> <li> <p>You cannot use this
         * feature with default security groups.</p> </li> <li> <p>You cannot use this
         * feature with the default VPC.</p> </li> </ul> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateSecurityGroupVpc">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateSecurityGroupVpcOutcome AssociateSecurityGroupVpc(const Model::AssociateSecurityGroupVpcRequest& request) const;

        /**
         * A Callable wrapper for AssociateSecurityGroupVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateSecurityGroupVpcRequestT = Model::AssociateSecurityGroupVpcRequest>
        Model::AssociateSecurityGroupVpcOutcomeCallable AssociateSecurityGroupVpcCallable(const AssociateSecurityGroupVpcRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateSecurityGroupVpc, request);
        }

        /**
         * An Async wrapper for AssociateSecurityGroupVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateSecurityGroupVpcRequestT = Model::AssociateSecurityGroupVpcRequest>
        void AssociateSecurityGroupVpcAsync(const AssociateSecurityGroupVpcRequestT& request, const AssociateSecurityGroupVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateSecurityGroupVpc, request, handler, context);
        }

        /**
         * <p>Associates a CIDR block with your subnet. You can only associate a single
         * IPv6 CIDR block with your subnet.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateSubnetCidrBlock">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateSubnetCidrBlockOutcome AssociateSubnetCidrBlock(const Model::AssociateSubnetCidrBlockRequest& request) const;

        /**
         * A Callable wrapper for AssociateSubnetCidrBlock that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateSubnetCidrBlockRequestT = Model::AssociateSubnetCidrBlockRequest>
        Model::AssociateSubnetCidrBlockOutcomeCallable AssociateSubnetCidrBlockCallable(const AssociateSubnetCidrBlockRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateSubnetCidrBlock, request);
        }

        /**
         * An Async wrapper for AssociateSubnetCidrBlock that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateSubnetCidrBlockRequestT = Model::AssociateSubnetCidrBlockRequest>
        void AssociateSubnetCidrBlockAsync(const AssociateSubnetCidrBlockRequestT& request, const AssociateSubnetCidrBlockResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateSubnetCidrBlock, request, handler, context);
        }

        /**
         * <p>Associates the specified subnets and transit gateway attachments with the
         * specified transit gateway multicast domain.</p> <p>The transit gateway
         * attachment must be in the available state before you can add a resource. Use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html">DescribeTransitGatewayAttachments</a>
         * to see the state of the attachment.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateTransitGatewayMulticastDomain">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateTransitGatewayMulticastDomainOutcome AssociateTransitGatewayMulticastDomain(const Model::AssociateTransitGatewayMulticastDomainRequest& request) const;

        /**
         * A Callable wrapper for AssociateTransitGatewayMulticastDomain that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateTransitGatewayMulticastDomainRequestT = Model::AssociateTransitGatewayMulticastDomainRequest>
        Model::AssociateTransitGatewayMulticastDomainOutcomeCallable AssociateTransitGatewayMulticastDomainCallable(const AssociateTransitGatewayMulticastDomainRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateTransitGatewayMulticastDomain, request);
        }

        /**
         * An Async wrapper for AssociateTransitGatewayMulticastDomain that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateTransitGatewayMulticastDomainRequestT = Model::AssociateTransitGatewayMulticastDomainRequest>
        void AssociateTransitGatewayMulticastDomainAsync(const AssociateTransitGatewayMulticastDomainRequestT& request, const AssociateTransitGatewayMulticastDomainResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateTransitGatewayMulticastDomain, request, handler, context);
        }

        /**
         * <p>Associates the specified transit gateway attachment with a transit gateway
         * policy table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateTransitGatewayPolicyTable">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateTransitGatewayPolicyTableOutcome AssociateTransitGatewayPolicyTable(const Model::AssociateTransitGatewayPolicyTableRequest& request) const;

        /**
         * A Callable wrapper for AssociateTransitGatewayPolicyTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateTransitGatewayPolicyTableRequestT = Model::AssociateTransitGatewayPolicyTableRequest>
        Model::AssociateTransitGatewayPolicyTableOutcomeCallable AssociateTransitGatewayPolicyTableCallable(const AssociateTransitGatewayPolicyTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateTransitGatewayPolicyTable, request);
        }

        /**
         * An Async wrapper for AssociateTransitGatewayPolicyTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateTransitGatewayPolicyTableRequestT = Model::AssociateTransitGatewayPolicyTableRequest>
        void AssociateTransitGatewayPolicyTableAsync(const AssociateTransitGatewayPolicyTableRequestT& request, const AssociateTransitGatewayPolicyTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateTransitGatewayPolicyTable, request, handler, context);
        }

        /**
         * <p>Associates the specified attachment with the specified transit gateway route
         * table. You can associate only one route table with an attachment.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateTransitGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateTransitGatewayRouteTableOutcome AssociateTransitGatewayRouteTable(const Model::AssociateTransitGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for AssociateTransitGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateTransitGatewayRouteTableRequestT = Model::AssociateTransitGatewayRouteTableRequest>
        Model::AssociateTransitGatewayRouteTableOutcomeCallable AssociateTransitGatewayRouteTableCallable(const AssociateTransitGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateTransitGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for AssociateTransitGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateTransitGatewayRouteTableRequestT = Model::AssociateTransitGatewayRouteTableRequest>
        void AssociateTransitGatewayRouteTableAsync(const AssociateTransitGatewayRouteTableRequestT& request, const AssociateTransitGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateTransitGatewayRouteTable, request, handler, context);
        }

        /**
         * <p>Associates a branch network interface with a trunk network interface.</p>
         * <p>Before you create the association, use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html">CreateNetworkInterface</a>
         * command and set the interface type to <code>trunk</code>. You must also create a
         * network interface for each branch network interface that you want to associate
         * with the trunk network interface.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateTrunkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateTrunkInterfaceOutcome AssociateTrunkInterface(const Model::AssociateTrunkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for AssociateTrunkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateTrunkInterfaceRequestT = Model::AssociateTrunkInterfaceRequest>
        Model::AssociateTrunkInterfaceOutcomeCallable AssociateTrunkInterfaceCallable(const AssociateTrunkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateTrunkInterface, request);
        }

        /**
         * An Async wrapper for AssociateTrunkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateTrunkInterfaceRequestT = Model::AssociateTrunkInterfaceRequest>
        void AssociateTrunkInterfaceAsync(const AssociateTrunkInterfaceRequestT& request, const AssociateTrunkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateTrunkInterface, request, handler, context);
        }

        /**
         * <p>Associates a CIDR block with your VPC. You can associate a secondary IPv4
         * CIDR block, an Amazon-provided IPv6 CIDR block, or an IPv6 CIDR block from an
         * IPv6 address pool that you provisioned through bring your own IP addresses (<a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html">BYOIP</a>).</p>
         * <p>You must specify one of the following in the request: an IPv4 CIDR block, an
         * IPv6 pool, or an Amazon-provided IPv6 CIDR block.</p> <p>For more information
         * about associating CIDR blocks with your VPC and applicable restrictions, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html">IP
         * addressing for your VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AssociateVpcCidrBlock">AWS
         * API Reference</a></p>
         */
        virtual Model::AssociateVpcCidrBlockOutcome AssociateVpcCidrBlock(const Model::AssociateVpcCidrBlockRequest& request) const;

        /**
         * A Callable wrapper for AssociateVpcCidrBlock that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AssociateVpcCidrBlockRequestT = Model::AssociateVpcCidrBlockRequest>
        Model::AssociateVpcCidrBlockOutcomeCallable AssociateVpcCidrBlockCallable(const AssociateVpcCidrBlockRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AssociateVpcCidrBlock, request);
        }

        /**
         * An Async wrapper for AssociateVpcCidrBlock that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AssociateVpcCidrBlockRequestT = Model::AssociateVpcCidrBlockRequest>
        void AssociateVpcCidrBlockAsync(const AssociateVpcCidrBlockRequestT& request, const AssociateVpcCidrBlockResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AssociateVpcCidrBlock, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Links an EC2-Classic
         * instance to a ClassicLink-enabled VPC through one or more of the VPC security
         * groups. You cannot link an EC2-Classic instance to more than one VPC at a time.
         * You can only link an instance that's in the <code>running</code> state. An
         * instance is automatically unlinked from a VPC when it's stopped - you can link
         * it to the VPC again when you restart it.</p> <p>After you've linked an instance,
         * you cannot change the VPC security groups that are associated with it. To change
         * the security groups, you must first unlink the instance, and then link it
         * again.</p> <p>Linking your instance to a VPC is sometimes referred to as
         * <i>attaching</i> your instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachClassicLinkVpc">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachClassicLinkVpcOutcome AttachClassicLinkVpc(const Model::AttachClassicLinkVpcRequest& request) const;

        /**
         * A Callable wrapper for AttachClassicLinkVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachClassicLinkVpcRequestT = Model::AttachClassicLinkVpcRequest>
        Model::AttachClassicLinkVpcOutcomeCallable AttachClassicLinkVpcCallable(const AttachClassicLinkVpcRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachClassicLinkVpc, request);
        }

        /**
         * An Async wrapper for AttachClassicLinkVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachClassicLinkVpcRequestT = Model::AttachClassicLinkVpcRequest>
        void AttachClassicLinkVpcAsync(const AttachClassicLinkVpcRequestT& request, const AttachClassicLinkVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachClassicLinkVpc, request, handler, context);
        }

        /**
         * <p>Attaches an internet gateway or a virtual private gateway to a VPC, enabling
         * connectivity between the internet and the VPC. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html">Internet
         * gateways</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachInternetGatewayOutcome AttachInternetGateway(const Model::AttachInternetGatewayRequest& request) const;

        /**
         * A Callable wrapper for AttachInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachInternetGatewayRequestT = Model::AttachInternetGatewayRequest>
        Model::AttachInternetGatewayOutcomeCallable AttachInternetGatewayCallable(const AttachInternetGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachInternetGateway, request);
        }

        /**
         * An Async wrapper for AttachInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachInternetGatewayRequestT = Model::AttachInternetGatewayRequest>
        void AttachInternetGatewayAsync(const AttachInternetGatewayRequestT& request, const AttachInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachInternetGateway, request, handler, context);
        }

        /**
         * <p>Attaches a network interface to an instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachNetworkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachNetworkInterfaceOutcome AttachNetworkInterface(const Model::AttachNetworkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for AttachNetworkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachNetworkInterfaceRequestT = Model::AttachNetworkInterfaceRequest>
        Model::AttachNetworkInterfaceOutcomeCallable AttachNetworkInterfaceCallable(const AttachNetworkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachNetworkInterface, request);
        }

        /**
         * An Async wrapper for AttachNetworkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachNetworkInterfaceRequestT = Model::AttachNetworkInterfaceRequest>
        void AttachNetworkInterfaceAsync(const AttachNetworkInterfaceRequestT& request, const AttachNetworkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachNetworkInterface, request, handler, context);
        }

        /**
         * <p>Attaches the specified Amazon Web Services Verified Access trust provider to
         * the specified Amazon Web Services Verified Access instance.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachVerifiedAccessTrustProvider">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachVerifiedAccessTrustProviderOutcome AttachVerifiedAccessTrustProvider(const Model::AttachVerifiedAccessTrustProviderRequest& request) const;

        /**
         * A Callable wrapper for AttachVerifiedAccessTrustProvider that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachVerifiedAccessTrustProviderRequestT = Model::AttachVerifiedAccessTrustProviderRequest>
        Model::AttachVerifiedAccessTrustProviderOutcomeCallable AttachVerifiedAccessTrustProviderCallable(const AttachVerifiedAccessTrustProviderRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachVerifiedAccessTrustProvider, request);
        }

        /**
         * An Async wrapper for AttachVerifiedAccessTrustProvider that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachVerifiedAccessTrustProviderRequestT = Model::AttachVerifiedAccessTrustProviderRequest>
        void AttachVerifiedAccessTrustProviderAsync(const AttachVerifiedAccessTrustProviderRequestT& request, const AttachVerifiedAccessTrustProviderResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachVerifiedAccessTrustProvider, request, handler, context);
        }

        /**
         * <p>Attaches an EBS volume to a running or stopped instance and exposes it to the
         * instance with the specified device name.</p> <p>Encrypted EBS volumes must be
         * attached to instances that support Amazon EBS encryption. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p> <p>After you attach
         * an EBS volume, you must make it available. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-using-volumes.html">Make
         * an EBS volume available for use</a>.</p> <p>If a volume has an Amazon Web
         * Services Marketplace product code:</p> <ul> <li> <p>The volume can be attached
         * only to a stopped instance.</p> </li> <li> <p>Amazon Web Services Marketplace
         * product codes are copied from the volume to the instance.</p> </li> <li> <p>You
         * must be subscribed to the product.</p> </li> <li> <p>The instance type and
         * operating system of the instance must support the product. For example, you
         * can't detach a volume from a Windows instance and attach it to a Linux
         * instance.</p> </li> </ul> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-attaching-volume.html">Attach
         * an Amazon EBS volume to an instance</a> in the <i>Amazon EBS User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachVolume">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachVolumeOutcome AttachVolume(const Model::AttachVolumeRequest& request) const;

        /**
         * A Callable wrapper for AttachVolume that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachVolumeRequestT = Model::AttachVolumeRequest>
        Model::AttachVolumeOutcomeCallable AttachVolumeCallable(const AttachVolumeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachVolume, request);
        }

        /**
         * An Async wrapper for AttachVolume that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachVolumeRequestT = Model::AttachVolumeRequest>
        void AttachVolumeAsync(const AttachVolumeRequestT& request, const AttachVolumeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachVolume, request, handler, context);
        }

        /**
         * <p>Attaches an available virtual private gateway to a VPC. You can attach one
         * virtual private gateway to one VPC at a time.</p> <p>For more information, see
         * <a href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AttachVpnGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::AttachVpnGatewayOutcome AttachVpnGateway(const Model::AttachVpnGatewayRequest& request) const;

        /**
         * A Callable wrapper for AttachVpnGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AttachVpnGatewayRequestT = Model::AttachVpnGatewayRequest>
        Model::AttachVpnGatewayOutcomeCallable AttachVpnGatewayCallable(const AttachVpnGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AttachVpnGateway, request);
        }

        /**
         * An Async wrapper for AttachVpnGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AttachVpnGatewayRequestT = Model::AttachVpnGatewayRequest>
        void AttachVpnGatewayAsync(const AttachVpnGatewayRequestT& request, const AttachVpnGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AttachVpnGateway, request, handler, context);
        }

        /**
         * <p>Adds an ingress authorization rule to a Client VPN endpoint. Ingress
         * authorization rules act as firewall rules that grant access to networks. You
         * must configure ingress authorization rules to enable clients to access resources
         * in Amazon Web Services or on-premises networks.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AuthorizeClientVpnIngress">AWS
         * API Reference</a></p>
         */
        virtual Model::AuthorizeClientVpnIngressOutcome AuthorizeClientVpnIngress(const Model::AuthorizeClientVpnIngressRequest& request) const;

        /**
         * A Callable wrapper for AuthorizeClientVpnIngress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AuthorizeClientVpnIngressRequestT = Model::AuthorizeClientVpnIngressRequest>
        Model::AuthorizeClientVpnIngressOutcomeCallable AuthorizeClientVpnIngressCallable(const AuthorizeClientVpnIngressRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AuthorizeClientVpnIngress, request);
        }

        /**
         * An Async wrapper for AuthorizeClientVpnIngress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AuthorizeClientVpnIngressRequestT = Model::AuthorizeClientVpnIngressRequest>
        void AuthorizeClientVpnIngressAsync(const AuthorizeClientVpnIngressRequestT& request, const AuthorizeClientVpnIngressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AuthorizeClientVpnIngress, request, handler, context);
        }

        /**
         * <p>Adds the specified outbound (egress) rules to a security group.</p> <p>An
         * outbound rule permits instances to send traffic to the specified IPv4 or IPv6
         * address ranges, the IP address ranges specified by a prefix list, or the
         * instances that are associated with a source security group. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-group-rules.html">Security
         * group rules</a>.</p> <p>You must specify exactly one of the following
         * destinations: an IPv4 or IPv6 address range, a prefix list, or a security group.
         * You must specify a protocol for each rule (for example, TCP). If the protocol is
         * TCP or UDP, you must also specify a port or port range. If the protocol is ICMP
         * or ICMPv6, you must also specify the ICMP type and code.</p> <p>Rule changes are
         * propagated to instances associated with the security group as quickly as
         * possible. However, a small delay might occur.</p> <p>For examples of rules that
         * you can add to security groups for specific access scenarios, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-rules-reference.html">Security
         * group rules for different use cases</a> in the <i>Amazon EC2 User Guide</i>.</p>
         * <p>For information about security group quotas, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon
         * VPC quotas</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AuthorizeSecurityGroupEgress">AWS
         * API Reference</a></p>
         */
        virtual Model::AuthorizeSecurityGroupEgressOutcome AuthorizeSecurityGroupEgress(const Model::AuthorizeSecurityGroupEgressRequest& request) const;

        /**
         * A Callable wrapper for AuthorizeSecurityGroupEgress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AuthorizeSecurityGroupEgressRequestT = Model::AuthorizeSecurityGroupEgressRequest>
        Model::AuthorizeSecurityGroupEgressOutcomeCallable AuthorizeSecurityGroupEgressCallable(const AuthorizeSecurityGroupEgressRequestT& request) const
        {
            return SubmitCallable(&EC2Client::AuthorizeSecurityGroupEgress, request);
        }

        /**
         * An Async wrapper for AuthorizeSecurityGroupEgress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AuthorizeSecurityGroupEgressRequestT = Model::AuthorizeSecurityGroupEgressRequest>
        void AuthorizeSecurityGroupEgressAsync(const AuthorizeSecurityGroupEgressRequestT& request, const AuthorizeSecurityGroupEgressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::AuthorizeSecurityGroupEgress, request, handler, context);
        }

        /**
         * <p>Adds the specified inbound (ingress) rules to a security group.</p> <p>An
         * inbound rule permits instances to receive traffic from the specified IPv4 or
         * IPv6 address range, the IP address ranges that are specified by a prefix list,
         * or the instances that are associated with a destination security group. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-group-rules.html">Security
         * group rules</a>.</p> <p>You must specify exactly one of the following sources:
         * an IPv4 or IPv6 address range, a prefix list, or a security group. You must
         * specify a protocol for each rule (for example, TCP). If the protocol is TCP or
         * UDP, you must also specify a port or port range. If the protocol is ICMP or
         * ICMPv6, you must also specify the ICMP/ICMPv6 type and code.</p> <p>Rule changes
         * are propagated to instances associated with the security group as quickly as
         * possible. However, a small delay might occur.</p> <p>For examples of rules that
         * you can add to security groups for specific access scenarios, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-rules-reference.html">Security
         * group rules for different use cases</a> in the <i>Amazon EC2 User Guide</i>.</p>
         * <p>For more information about security group quotas, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon
         * VPC quotas</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AuthorizeSecurityGroupIngress">AWS
         * API Reference</a></p>
         */
        virtual Model::AuthorizeSecurityGroupIngressOutcome AuthorizeSecurityGroupIngress(const Model::AuthorizeSecurityGroupIngressRequest& request = {}) const;

        /**
         * A Callable wrapper for AuthorizeSecurityGroupIngress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename AuthorizeSecurityGroupIngressRequestT = Model::AuthorizeSecurityGroupIngressRequest>
        Model::AuthorizeSecurityGroupIngressOutcomeCallable AuthorizeSecurityGroupIngressCallable(const AuthorizeSecurityGroupIngressRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::AuthorizeSecurityGroupIngress, request);
        }

        /**
         * An Async wrapper for AuthorizeSecurityGroupIngress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename AuthorizeSecurityGroupIngressRequestT = Model::AuthorizeSecurityGroupIngressRequest>
        void AuthorizeSecurityGroupIngressAsync(const AuthorizeSecurityGroupIngressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const AuthorizeSecurityGroupIngressRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::AuthorizeSecurityGroupIngress, request, handler, context);
        }

        /**
         * <p>Bundles an Amazon instance store-backed Windows instance.</p> <p>During
         * bundling, only the root device volume (C:\) is bundled. Data on other instance
         * store volumes is not preserved.</p>  <p>This action is not applicable for
         * Linux/Unix instances or Windows instances that are backed by Amazon EBS.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/BundleInstance">AWS
         * API Reference</a></p>
         */
        virtual Model::BundleInstanceOutcome BundleInstance(const Model::BundleInstanceRequest& request) const;

        /**
         * A Callable wrapper for BundleInstance that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename BundleInstanceRequestT = Model::BundleInstanceRequest>
        Model::BundleInstanceOutcomeCallable BundleInstanceCallable(const BundleInstanceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::BundleInstance, request);
        }

        /**
         * An Async wrapper for BundleInstance that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename BundleInstanceRequestT = Model::BundleInstanceRequest>
        void BundleInstanceAsync(const BundleInstanceRequestT& request, const BundleInstanceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::BundleInstance, request, handler, context);
        }

        /**
         * <p>Cancels a bundling operation for an instance store-backed Windows
         * instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelBundleTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelBundleTaskOutcome CancelBundleTask(const Model::CancelBundleTaskRequest& request) const;

        /**
         * A Callable wrapper for CancelBundleTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelBundleTaskRequestT = Model::CancelBundleTaskRequest>
        Model::CancelBundleTaskOutcomeCallable CancelBundleTaskCallable(const CancelBundleTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelBundleTask, request);
        }

        /**
         * An Async wrapper for CancelBundleTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelBundleTaskRequestT = Model::CancelBundleTaskRequest>
        void CancelBundleTaskAsync(const CancelBundleTaskRequestT& request, const CancelBundleTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelBundleTask, request, handler, context);
        }

        /**
         * <p>Cancels the specified Capacity Reservation, releases the reserved capacity,
         * and changes the Capacity Reservation's state to <code>cancelled</code>.</p>
         * <p>You can cancel a Capacity Reservation that is in the following states:</p>
         * <ul> <li> <p> <code>assessing</code> </p> </li> <li> <p> <code>active</code> and
         * there is no commitment duration or the commitment duration has elapsed. You
         * can't cancel a future-dated Capacity Reservation during the commitment
         * duration.</p> </li> </ul>  <p>You can't modify or cancel a Capacity Block.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-blocks.html">Capacity
         * Blocks for ML</a>.</p>  <p>If a future-dated Capacity Reservation enters
         * the <code>delayed</code> state, the commitment duration is waived, and you can
         * cancel it as soon as it enters the <code>active</code> state.</p> <p>Instances
         * running in the reserved capacity continue running until you stop them. Stopped
         * instances that target the Capacity Reservation can no longer launch. Modify
         * these instances to either target a different Capacity Reservation, launch
         * On-Demand Instance capacity, or run in any open Capacity Reservation that has
         * matching attributes and sufficient capacity.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelCapacityReservation">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelCapacityReservationOutcome CancelCapacityReservation(const Model::CancelCapacityReservationRequest& request) const;

        /**
         * A Callable wrapper for CancelCapacityReservation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelCapacityReservationRequestT = Model::CancelCapacityReservationRequest>
        Model::CancelCapacityReservationOutcomeCallable CancelCapacityReservationCallable(const CancelCapacityReservationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelCapacityReservation, request);
        }

        /**
         * An Async wrapper for CancelCapacityReservation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelCapacityReservationRequestT = Model::CancelCapacityReservationRequest>
        void CancelCapacityReservationAsync(const CancelCapacityReservationRequestT& request, const CancelCapacityReservationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelCapacityReservation, request, handler, context);
        }

        /**
         * <p>Cancels one or more Capacity Reservation Fleets. When you cancel a Capacity
         * Reservation Fleet, the following happens:</p> <ul> <li> <p>The Capacity
         * Reservation Fleet's status changes to <code>cancelled</code>.</p> </li> <li>
         * <p>The individual Capacity Reservations in the Fleet are cancelled. Instances
         * running in the Capacity Reservations at the time of cancelling the Fleet
         * continue to run in shared capacity.</p> </li> <li> <p>The Fleet stops creating
         * new Capacity Reservations.</p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelCapacityReservationFleets">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelCapacityReservationFleetsOutcome CancelCapacityReservationFleets(const Model::CancelCapacityReservationFleetsRequest& request) const;

        /**
         * A Callable wrapper for CancelCapacityReservationFleets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelCapacityReservationFleetsRequestT = Model::CancelCapacityReservationFleetsRequest>
        Model::CancelCapacityReservationFleetsOutcomeCallable CancelCapacityReservationFleetsCallable(const CancelCapacityReservationFleetsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelCapacityReservationFleets, request);
        }

        /**
         * An Async wrapper for CancelCapacityReservationFleets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelCapacityReservationFleetsRequestT = Model::CancelCapacityReservationFleetsRequest>
        void CancelCapacityReservationFleetsAsync(const CancelCapacityReservationFleetsRequestT& request, const CancelCapacityReservationFleetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelCapacityReservationFleets, request, handler, context);
        }

        /**
         * <p>Cancels an active conversion task. The task can be the import of an instance
         * or volume. The action removes all artifacts of the conversion, including a
         * partially uploaded volume or instance. If the conversion is complete or is in
         * the process of transferring the final disk image, the command fails and returns
         * an exception.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelConversionTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelConversionTaskOutcome CancelConversionTask(const Model::CancelConversionTaskRequest& request) const;

        /**
         * A Callable wrapper for CancelConversionTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelConversionTaskRequestT = Model::CancelConversionTaskRequest>
        Model::CancelConversionTaskOutcomeCallable CancelConversionTaskCallable(const CancelConversionTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelConversionTask, request);
        }

        /**
         * An Async wrapper for CancelConversionTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelConversionTaskRequestT = Model::CancelConversionTaskRequest>
        void CancelConversionTaskAsync(const CancelConversionTaskRequestT& request, const CancelConversionTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelConversionTask, request, handler, context);
        }

        /**
         * <p>Cancels the generation of an account status report.</p> <p>You can only
         * cancel a report while it has the <code>running</code> status. Reports with other
         * statuses (<code>complete</code>, <code>cancelled</code>, or <code>error</code>)
         * can't be canceled.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_declarative_status-report.html">Generating
         * the account status report for declarative policies</a> in the <i>Amazon Web
         * Services Organizations User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelDeclarativePoliciesReport">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelDeclarativePoliciesReportOutcome CancelDeclarativePoliciesReport(const Model::CancelDeclarativePoliciesReportRequest& request) const;

        /**
         * A Callable wrapper for CancelDeclarativePoliciesReport that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelDeclarativePoliciesReportRequestT = Model::CancelDeclarativePoliciesReportRequest>
        Model::CancelDeclarativePoliciesReportOutcomeCallable CancelDeclarativePoliciesReportCallable(const CancelDeclarativePoliciesReportRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelDeclarativePoliciesReport, request);
        }

        /**
         * An Async wrapper for CancelDeclarativePoliciesReport that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelDeclarativePoliciesReportRequestT = Model::CancelDeclarativePoliciesReportRequest>
        void CancelDeclarativePoliciesReportAsync(const CancelDeclarativePoliciesReportRequestT& request, const CancelDeclarativePoliciesReportResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelDeclarativePoliciesReport, request, handler, context);
        }

        /**
         * <p>Cancels an active export task. The request removes all artifacts of the
         * export, including any partially-created Amazon S3 objects. If the export task is
         * complete or is in the process of transferring the final disk image, the command
         * fails and returns an error.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelExportTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelExportTaskOutcome CancelExportTask(const Model::CancelExportTaskRequest& request) const;

        /**
         * A Callable wrapper for CancelExportTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelExportTaskRequestT = Model::CancelExportTaskRequest>
        Model::CancelExportTaskOutcomeCallable CancelExportTaskCallable(const CancelExportTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelExportTask, request);
        }

        /**
         * An Async wrapper for CancelExportTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelExportTaskRequestT = Model::CancelExportTaskRequest>
        void CancelExportTaskAsync(const CancelExportTaskRequestT& request, const CancelExportTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelExportTask, request, handler, context);
        }

        /**
         * <p>Removes your Amazon Web Services account from the launch permissions for the
         * specified AMI. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cancel-sharing-an-AMI.html">Cancel
         * having an AMI shared with your Amazon Web Services account</a> in the <i>Amazon
         * EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelImageLaunchPermission">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelImageLaunchPermissionOutcome CancelImageLaunchPermission(const Model::CancelImageLaunchPermissionRequest& request) const;

        /**
         * A Callable wrapper for CancelImageLaunchPermission that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelImageLaunchPermissionRequestT = Model::CancelImageLaunchPermissionRequest>
        Model::CancelImageLaunchPermissionOutcomeCallable CancelImageLaunchPermissionCallable(const CancelImageLaunchPermissionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelImageLaunchPermission, request);
        }

        /**
         * An Async wrapper for CancelImageLaunchPermission that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelImageLaunchPermissionRequestT = Model::CancelImageLaunchPermissionRequest>
        void CancelImageLaunchPermissionAsync(const CancelImageLaunchPermissionRequestT& request, const CancelImageLaunchPermissionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelImageLaunchPermission, request, handler, context);
        }

        /**
         * <p>Cancels an in-process import virtual machine or import snapshot
         * task.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelImportTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelImportTaskOutcome CancelImportTask(const Model::CancelImportTaskRequest& request = {}) const;

        /**
         * A Callable wrapper for CancelImportTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelImportTaskRequestT = Model::CancelImportTaskRequest>
        Model::CancelImportTaskOutcomeCallable CancelImportTaskCallable(const CancelImportTaskRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CancelImportTask, request);
        }

        /**
         * An Async wrapper for CancelImportTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelImportTaskRequestT = Model::CancelImportTaskRequest>
        void CancelImportTaskAsync(const CancelImportTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CancelImportTaskRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CancelImportTask, request, handler, context);
        }

        /**
         * <p>Cancels the specified Reserved Instance listing in the Reserved Instance
         * Marketplace.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ri-market-general.html">Sell
         * in the Reserved Instance Marketplace</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelReservedInstancesListing">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelReservedInstancesListingOutcome CancelReservedInstancesListing(const Model::CancelReservedInstancesListingRequest& request) const;

        /**
         * A Callable wrapper for CancelReservedInstancesListing that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelReservedInstancesListingRequestT = Model::CancelReservedInstancesListingRequest>
        Model::CancelReservedInstancesListingOutcomeCallable CancelReservedInstancesListingCallable(const CancelReservedInstancesListingRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelReservedInstancesListing, request);
        }

        /**
         * An Async wrapper for CancelReservedInstancesListing that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelReservedInstancesListingRequestT = Model::CancelReservedInstancesListingRequest>
        void CancelReservedInstancesListingAsync(const CancelReservedInstancesListingRequestT& request, const CancelReservedInstancesListingResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelReservedInstancesListing, request, handler, context);
        }

        /**
         * <p>Cancels the specified Spot Fleet requests.</p> <p>After you cancel a Spot
         * Fleet request, the Spot Fleet launches no new instances.</p> <p>You must also
         * specify whether a canceled Spot Fleet request should terminate its instances. If
         * you choose to terminate the instances, the Spot Fleet request enters the
         * <code>cancelled_terminating</code> state. Otherwise, the Spot Fleet request
         * enters the <code>cancelled_running</code> state and the instances continue to
         * run until they are interrupted or you terminate them manually.</p> <p
         * class="title"> <b>Restrictions</b> </p> <ul> <li> <p>You can delete up to 100
         * fleets in a single request. If you exceed the specified number, no fleets are
         * deleted.</p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelSpotFleetRequests">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelSpotFleetRequestsOutcome CancelSpotFleetRequests(const Model::CancelSpotFleetRequestsRequest& request) const;

        /**
         * A Callable wrapper for CancelSpotFleetRequests that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelSpotFleetRequestsRequestT = Model::CancelSpotFleetRequestsRequest>
        Model::CancelSpotFleetRequestsOutcomeCallable CancelSpotFleetRequestsCallable(const CancelSpotFleetRequestsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelSpotFleetRequests, request);
        }

        /**
         * An Async wrapper for CancelSpotFleetRequests that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelSpotFleetRequestsRequestT = Model::CancelSpotFleetRequestsRequest>
        void CancelSpotFleetRequestsAsync(const CancelSpotFleetRequestsRequestT& request, const CancelSpotFleetRequestsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelSpotFleetRequests, request, handler, context);
        }

        /**
         * <p>Cancels one or more Spot Instance requests.</p>  <p>Canceling a
         * Spot Instance request does not terminate running Spot Instances associated with
         * the request.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CancelSpotInstanceRequests">AWS
         * API Reference</a></p>
         */
        virtual Model::CancelSpotInstanceRequestsOutcome CancelSpotInstanceRequests(const Model::CancelSpotInstanceRequestsRequest& request) const;

        /**
         * A Callable wrapper for CancelSpotInstanceRequests that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CancelSpotInstanceRequestsRequestT = Model::CancelSpotInstanceRequestsRequest>
        Model::CancelSpotInstanceRequestsOutcomeCallable CancelSpotInstanceRequestsCallable(const CancelSpotInstanceRequestsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CancelSpotInstanceRequests, request);
        }

        /**
         * An Async wrapper for CancelSpotInstanceRequests that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CancelSpotInstanceRequestsRequestT = Model::CancelSpotInstanceRequestsRequest>
        void CancelSpotInstanceRequestsAsync(const CancelSpotInstanceRequestsRequestT& request, const CancelSpotInstanceRequestsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CancelSpotInstanceRequests, request, handler, context);
        }

        /**
         * <p>Determines whether a product code is associated with an instance. This action
         * can only be used by the owner of the product code. It is useful when a product
         * code owner must verify whether another user's instance is eligible for
         * support.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ConfirmProductInstance">AWS
         * API Reference</a></p>
         */
        virtual Model::ConfirmProductInstanceOutcome ConfirmProductInstance(const Model::ConfirmProductInstanceRequest& request) const;

        /**
         * A Callable wrapper for ConfirmProductInstance that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ConfirmProductInstanceRequestT = Model::ConfirmProductInstanceRequest>
        Model::ConfirmProductInstanceOutcomeCallable ConfirmProductInstanceCallable(const ConfirmProductInstanceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ConfirmProductInstance, request);
        }

        /**
         * An Async wrapper for ConfirmProductInstance that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ConfirmProductInstanceRequestT = Model::ConfirmProductInstanceRequest>
        void ConfirmProductInstanceAsync(const ConfirmProductInstanceRequestT& request, const ConfirmProductInstanceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ConfirmProductInstance, request, handler, context);
        }

        /**
         * <p>Copies the specified Amazon FPGA Image (AFI) to the current
         * Region.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CopyFpgaImage">AWS
         * API Reference</a></p>
         */
        virtual Model::CopyFpgaImageOutcome CopyFpgaImage(const Model::CopyFpgaImageRequest& request) const;

        /**
         * A Callable wrapper for CopyFpgaImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CopyFpgaImageRequestT = Model::CopyFpgaImageRequest>
        Model::CopyFpgaImageOutcomeCallable CopyFpgaImageCallable(const CopyFpgaImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CopyFpgaImage, request);
        }

        /**
         * An Async wrapper for CopyFpgaImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CopyFpgaImageRequestT = Model::CopyFpgaImageRequest>
        void CopyFpgaImageAsync(const CopyFpgaImageRequestT& request, const CopyFpgaImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CopyFpgaImage, request, handler, context);
        }

        /**
         * <p>Initiates an AMI copy operation. You can copy an AMI from one Region to
         * another, or from a Region to an Outpost. You can't copy an AMI from an Outpost
         * to a Region, from one Outpost to another, or within the same Outpost. To copy an
         * AMI to another partition, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateStoreImageTask.html">CreateStoreImageTask</a>.</p>
         * <p>When you copy an AMI from one Region to another, the destination Region is
         * the current Region.</p> <p>When you copy an AMI from a Region to an Outpost,
         * specify the ARN of the Outpost as the destination. Backing snapshots copied to
         * an Outpost are encrypted by default using the default encryption key for the
         * Region or the key that you specify. Outposts do not support unencrypted
         * snapshots.</p> <p>For information about the prerequisites when copying an AMI,
         * see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/CopyingAMIs.html">Copy
         * an Amazon EC2 AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CopyImage">AWS API
         * Reference</a></p>
         */
        virtual Model::CopyImageOutcome CopyImage(const Model::CopyImageRequest& request) const;

        /**
         * A Callable wrapper for CopyImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CopyImageRequestT = Model::CopyImageRequest>
        Model::CopyImageOutcomeCallable CopyImageCallable(const CopyImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CopyImage, request);
        }

        /**
         * An Async wrapper for CopyImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CopyImageRequestT = Model::CopyImageRequest>
        void CopyImageAsync(const CopyImageRequestT& request, const CopyImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CopyImage, request, handler, context);
        }

        /**
         * <p>Copies a point-in-time snapshot of an EBS volume and stores it in Amazon S3.
         * You can copy a snapshot within the same Region, from one Region to another, or
         * from a Region to an Outpost. You can't copy a snapshot from an Outpost to a
         * Region, from one Outpost to another, or within the same Outpost.</p> <p>You can
         * use the snapshot to create EBS volumes or Amazon Machine Images (AMIs).</p>
         * <p>When copying snapshots to a Region, copies of encrypted EBS snapshots remain
         * encrypted. Copies of unencrypted snapshots remain unencrypted, unless you enable
         * encryption for the snapshot copy operation. By default, encrypted snapshot
         * copies use the default KMS key; however, you can specify a different KMS key. To
         * copy an encrypted snapshot that has been shared from another account, you must
         * have permissions for the KMS key used to encrypt the snapshot.</p> <p>Snapshots
         * copied to an Outpost are encrypted by default using the default encryption key
         * for the Region, or a different key that you specify in the request using
         * <b>KmsKeyId</b>. Outposts do not support unencrypted snapshots. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#ami">Amazon
         * EBS local snapshots on Outposts</a> in the <i>Amazon EBS User Guide</i>.</p>
         * <p>Snapshots created by copying another snapshot have an arbitrary volume ID
         * that should not be used for any purpose.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-copy-snapshot.html">Copy
         * an Amazon EBS snapshot</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CopySnapshot">AWS
         * API Reference</a></p>
         */
        virtual Model::CopySnapshotOutcome CopySnapshot(const Model::CopySnapshotRequest& request) const;

        /**
         * A Callable wrapper for CopySnapshot that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CopySnapshotRequestT = Model::CopySnapshotRequest>
        Model::CopySnapshotOutcomeCallable CopySnapshotCallable(const CopySnapshotRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CopySnapshot, request);
        }

        /**
         * An Async wrapper for CopySnapshot that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CopySnapshotRequestT = Model::CopySnapshotRequest>
        void CopySnapshotAsync(const CopySnapshotRequestT& request, const CopySnapshotResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CopySnapshot, request, handler, context);
        }

        /**
         * <p>Creates a new Capacity Reservation with the specified attributes. Capacity
         * Reservations enable you to reserve capacity for your Amazon EC2 instances in a
         * specific Availability Zone for any duration.</p> <p>You can create a Capacity
         * Reservation at any time, and you can choose when it starts. You can create a
         * Capacity Reservation for immediate use or you can request a Capacity Reservation
         * for a future date.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html">
         * Reserve compute capacity with On-Demand Capacity Reservations</a> in the
         * <i>Amazon EC2 User Guide</i>.</p> <p>Your request to create a Capacity
         * Reservation could fail if:</p> <ul> <li> <p>Amazon EC2 does not have sufficient
         * capacity. In this case, try again at a later time, try in a different
         * Availability Zone, or request a smaller Capacity Reservation. If your workload
         * is flexible across instance types and sizes, try with different instance
         * attributes.</p> </li> <li> <p>The requested quantity exceeds your On-Demand
         * Instance quota. In this case, increase your On-Demand Instance quota for the
         * requested instance type and try again. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">
         * Amazon EC2 Service Quotas</a> in the <i>Amazon EC2 User Guide</i>.</p> </li>
         * </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCapacityReservation">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCapacityReservationOutcome CreateCapacityReservation(const Model::CreateCapacityReservationRequest& request) const;

        /**
         * A Callable wrapper for CreateCapacityReservation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCapacityReservationRequestT = Model::CreateCapacityReservationRequest>
        Model::CreateCapacityReservationOutcomeCallable CreateCapacityReservationCallable(const CreateCapacityReservationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCapacityReservation, request);
        }

        /**
         * An Async wrapper for CreateCapacityReservation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCapacityReservationRequestT = Model::CreateCapacityReservationRequest>
        void CreateCapacityReservationAsync(const CreateCapacityReservationRequestT& request, const CreateCapacityReservationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCapacityReservation, request, handler, context);
        }

        /**
         * <p> Create a new Capacity Reservation by splitting the capacity of the source
         * Capacity Reservation. The new Capacity Reservation will have the same attributes
         * as the source Capacity Reservation except for tags. The source Capacity
         * Reservation must be <code>active</code> and owned by your Amazon Web Services
         * account. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCapacityReservationBySplitting">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCapacityReservationBySplittingOutcome CreateCapacityReservationBySplitting(const Model::CreateCapacityReservationBySplittingRequest& request) const;

        /**
         * A Callable wrapper for CreateCapacityReservationBySplitting that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCapacityReservationBySplittingRequestT = Model::CreateCapacityReservationBySplittingRequest>
        Model::CreateCapacityReservationBySplittingOutcomeCallable CreateCapacityReservationBySplittingCallable(const CreateCapacityReservationBySplittingRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCapacityReservationBySplitting, request);
        }

        /**
         * An Async wrapper for CreateCapacityReservationBySplitting that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCapacityReservationBySplittingRequestT = Model::CreateCapacityReservationBySplittingRequest>
        void CreateCapacityReservationBySplittingAsync(const CreateCapacityReservationBySplittingRequestT& request, const CreateCapacityReservationBySplittingResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCapacityReservationBySplitting, request, handler, context);
        }

        /**
         * <p>Creates a Capacity Reservation Fleet. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/work-with-cr-fleets.html#create-crfleet">Create
         * a Capacity Reservation Fleet</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCapacityReservationFleet">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCapacityReservationFleetOutcome CreateCapacityReservationFleet(const Model::CreateCapacityReservationFleetRequest& request) const;

        /**
         * A Callable wrapper for CreateCapacityReservationFleet that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCapacityReservationFleetRequestT = Model::CreateCapacityReservationFleetRequest>
        Model::CreateCapacityReservationFleetOutcomeCallable CreateCapacityReservationFleetCallable(const CreateCapacityReservationFleetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCapacityReservationFleet, request);
        }

        /**
         * An Async wrapper for CreateCapacityReservationFleet that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCapacityReservationFleetRequestT = Model::CreateCapacityReservationFleetRequest>
        void CreateCapacityReservationFleetAsync(const CreateCapacityReservationFleetRequestT& request, const CreateCapacityReservationFleetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCapacityReservationFleet, request, handler, context);
        }

        /**
         * <p>Creates a carrier gateway. For more information about carrier gateways, see
         * <a
         * href="https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#wavelength-carrier-gateway">Carrier
         * gateways</a> in the <i>Amazon Web Services Wavelength Developer
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCarrierGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCarrierGatewayOutcome CreateCarrierGateway(const Model::CreateCarrierGatewayRequest& request) const;

        /**
         * A Callable wrapper for CreateCarrierGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCarrierGatewayRequestT = Model::CreateCarrierGatewayRequest>
        Model::CreateCarrierGatewayOutcomeCallable CreateCarrierGatewayCallable(const CreateCarrierGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCarrierGateway, request);
        }

        /**
         * An Async wrapper for CreateCarrierGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCarrierGatewayRequestT = Model::CreateCarrierGatewayRequest>
        void CreateCarrierGatewayAsync(const CreateCarrierGatewayRequestT& request, const CreateCarrierGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCarrierGateway, request, handler, context);
        }

        /**
         * <p>Creates a Client VPN endpoint. A Client VPN endpoint is the resource you
         * create and configure to enable and manage client VPN sessions. It is the
         * destination endpoint at which all client VPN sessions are
         * terminated.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateClientVpnEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateClientVpnEndpointOutcome CreateClientVpnEndpoint(const Model::CreateClientVpnEndpointRequest& request) const;

        /**
         * A Callable wrapper for CreateClientVpnEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateClientVpnEndpointRequestT = Model::CreateClientVpnEndpointRequest>
        Model::CreateClientVpnEndpointOutcomeCallable CreateClientVpnEndpointCallable(const CreateClientVpnEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateClientVpnEndpoint, request);
        }

        /**
         * An Async wrapper for CreateClientVpnEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateClientVpnEndpointRequestT = Model::CreateClientVpnEndpointRequest>
        void CreateClientVpnEndpointAsync(const CreateClientVpnEndpointRequestT& request, const CreateClientVpnEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateClientVpnEndpoint, request, handler, context);
        }

        /**
         * <p>Adds a route to a network to a Client VPN endpoint. Each Client VPN endpoint
         * has a route table that describes the available destination network routes. Each
         * route in the route table specifies the path for traﬃc to speciﬁc resources or
         * networks.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateClientVpnRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateClientVpnRouteOutcome CreateClientVpnRoute(const Model::CreateClientVpnRouteRequest& request) const;

        /**
         * A Callable wrapper for CreateClientVpnRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateClientVpnRouteRequestT = Model::CreateClientVpnRouteRequest>
        Model::CreateClientVpnRouteOutcomeCallable CreateClientVpnRouteCallable(const CreateClientVpnRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateClientVpnRoute, request);
        }

        /**
         * An Async wrapper for CreateClientVpnRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateClientVpnRouteRequestT = Model::CreateClientVpnRouteRequest>
        void CreateClientVpnRouteAsync(const CreateClientVpnRouteRequestT& request, const CreateClientVpnRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateClientVpnRoute, request, handler, context);
        }

        /**
         * <p> Creates a range of customer-owned IP addresses. </p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCoipCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCoipCidrOutcome CreateCoipCidr(const Model::CreateCoipCidrRequest& request) const;

        /**
         * A Callable wrapper for CreateCoipCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCoipCidrRequestT = Model::CreateCoipCidrRequest>
        Model::CreateCoipCidrOutcomeCallable CreateCoipCidrCallable(const CreateCoipCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCoipCidr, request);
        }

        /**
         * An Async wrapper for CreateCoipCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCoipCidrRequestT = Model::CreateCoipCidrRequest>
        void CreateCoipCidrAsync(const CreateCoipCidrRequestT& request, const CreateCoipCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCoipCidr, request, handler, context);
        }

        /**
         * <p> Creates a pool of customer-owned IP (CoIP) addresses. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCoipPool">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCoipPoolOutcome CreateCoipPool(const Model::CreateCoipPoolRequest& request) const;

        /**
         * A Callable wrapper for CreateCoipPool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCoipPoolRequestT = Model::CreateCoipPoolRequest>
        Model::CreateCoipPoolOutcomeCallable CreateCoipPoolCallable(const CreateCoipPoolRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCoipPool, request);
        }

        /**
         * An Async wrapper for CreateCoipPool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCoipPoolRequestT = Model::CreateCoipPoolRequest>
        void CreateCoipPoolAsync(const CreateCoipPoolRequestT& request, const CreateCoipPoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCoipPool, request, handler, context);
        }

        /**
         * <p>Provides information to Amazon Web Services about your customer gateway
         * device. The customer gateway device is the appliance at your end of the VPN
         * connection. You must provide the IP address of the customer gateway device’s
         * external interface. The IP address must be static and can be behind a device
         * performing network address translation (NAT).</p> <p>For devices that use Border
         * Gateway Protocol (BGP), you can also provide the device's BGP Autonomous System
         * Number (ASN). You can use an existing ASN assigned to your network. If you don't
         * have an ASN already, you can use a private ASN. For more information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/cgw-options.html">Customer
         * gateway options for your Site-to-Site VPN connection</a> in the <i>Amazon Web
         * Services Site-to-Site VPN User Guide</i>.</p> <p>To create more than one
         * customer gateway with the same VPN type, IP address, and BGP ASN, specify a
         * unique device name for each customer gateway. An identical request returns
         * information about the existing customer gateway; it doesn't create a new
         * customer gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateCustomerGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateCustomerGatewayOutcome CreateCustomerGateway(const Model::CreateCustomerGatewayRequest& request) const;

        /**
         * A Callable wrapper for CreateCustomerGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateCustomerGatewayRequestT = Model::CreateCustomerGatewayRequest>
        Model::CreateCustomerGatewayOutcomeCallable CreateCustomerGatewayCallable(const CreateCustomerGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateCustomerGateway, request);
        }

        /**
         * An Async wrapper for CreateCustomerGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateCustomerGatewayRequestT = Model::CreateCustomerGatewayRequest>
        void CreateCustomerGatewayAsync(const CreateCustomerGatewayRequestT& request, const CreateCustomerGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateCustomerGateway, request, handler, context);
        }

        /**
         * <p>Creates a default subnet with a size <code>/20</code> IPv4 CIDR block in the
         * specified Availability Zone in your default VPC. You can have only one default
         * subnet per Availability Zone. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/work-with-default-vpc.html#create-default-subnet">Create
         * a default subnet</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateDefaultSubnet">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateDefaultSubnetOutcome CreateDefaultSubnet(const Model::CreateDefaultSubnetRequest& request) const;

        /**
         * A Callable wrapper for CreateDefaultSubnet that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateDefaultSubnetRequestT = Model::CreateDefaultSubnetRequest>
        Model::CreateDefaultSubnetOutcomeCallable CreateDefaultSubnetCallable(const CreateDefaultSubnetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateDefaultSubnet, request);
        }

        /**
         * An Async wrapper for CreateDefaultSubnet that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateDefaultSubnetRequestT = Model::CreateDefaultSubnetRequest>
        void CreateDefaultSubnetAsync(const CreateDefaultSubnetRequestT& request, const CreateDefaultSubnetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateDefaultSubnet, request, handler, context);
        }

        /**
         * <p>Creates a default VPC with a size <code>/16</code> IPv4 CIDR block and a
         * default subnet in each Availability Zone. For more information about the
         * components of a default VPC, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/default-vpc.html">Default
         * VPCs</a> in the <i>Amazon VPC User Guide</i>. You cannot specify the components
         * of the default VPC yourself.</p> <p>If you deleted your previous default VPC,
         * you can create a default VPC. You cannot have more than one default VPC per
         * Region.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateDefaultVpc">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateDefaultVpcOutcome CreateDefaultVpc(const Model::CreateDefaultVpcRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateDefaultVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateDefaultVpcRequestT = Model::CreateDefaultVpcRequest>
        Model::CreateDefaultVpcOutcomeCallable CreateDefaultVpcCallable(const CreateDefaultVpcRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateDefaultVpc, request);
        }

        /**
         * An Async wrapper for CreateDefaultVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateDefaultVpcRequestT = Model::CreateDefaultVpcRequest>
        void CreateDefaultVpcAsync(const CreateDefaultVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateDefaultVpcRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateDefaultVpc, request, handler, context);
        }

        /**
         * <p>Delegates ownership of the Amazon EBS root volume for an Apple silicon Mac
         * instance to an administrative user.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateDelegateMacVolumeOwnershipTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateDelegateMacVolumeOwnershipTaskOutcome CreateDelegateMacVolumeOwnershipTask(const Model::CreateDelegateMacVolumeOwnershipTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateDelegateMacVolumeOwnershipTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateDelegateMacVolumeOwnershipTaskRequestT = Model::CreateDelegateMacVolumeOwnershipTaskRequest>
        Model::CreateDelegateMacVolumeOwnershipTaskOutcomeCallable CreateDelegateMacVolumeOwnershipTaskCallable(const CreateDelegateMacVolumeOwnershipTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateDelegateMacVolumeOwnershipTask, request);
        }

        /**
         * An Async wrapper for CreateDelegateMacVolumeOwnershipTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateDelegateMacVolumeOwnershipTaskRequestT = Model::CreateDelegateMacVolumeOwnershipTaskRequest>
        void CreateDelegateMacVolumeOwnershipTaskAsync(const CreateDelegateMacVolumeOwnershipTaskRequestT& request, const CreateDelegateMacVolumeOwnershipTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateDelegateMacVolumeOwnershipTask, request, handler, context);
        }

        /**
         * <p>Creates a custom set of DHCP options. After you create a DHCP option set, you
         * associate it with a VPC. After you associate a DHCP option set with a VPC, all
         * existing and newly launched instances in the VPC use this set of DHCP
         * options.</p> <p>The following are the individual DHCP options you can specify.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html">DHCP
         * option sets</a> in the <i>Amazon VPC User Guide</i>.</p> <ul> <li> <p>
         * <code>domain-name</code> - If you're using AmazonProvidedDNS in
         * <code>us-east-1</code>, specify <code>ec2.internal</code>. If you're using
         * AmazonProvidedDNS in any other Region, specify
         * <code>region.compute.internal</code>. Otherwise, specify a custom domain name.
         * This value is used to complete unqualified DNS hostnames.</p> <p>Some Linux
         * operating systems accept multiple domain names separated by spaces. However,
         * Windows and other Linux operating systems treat the value as a single domain,
         * which results in unexpected behavior. If your DHCP option set is associated with
         * a VPC that has instances running operating systems that treat the value as a
         * single domain, specify only one domain name.</p> </li> <li> <p>
         * <code>domain-name-servers</code> - The IP addresses of up to four DNS servers,
         * or AmazonProvidedDNS. To specify multiple domain name servers in a single
         * parameter, separate the IP addresses using commas. To have your instances
         * receive custom DNS hostnames as specified in <code>domain-name</code>, you must
         * specify a custom DNS server.</p> </li> <li> <p> <code>ntp-servers</code> - The
         * IP addresses of up to eight Network Time Protocol (NTP) servers (four IPv4
         * addresses and four IPv6 addresses).</p> </li> <li> <p>
         * <code>netbios-name-servers</code> - The IP addresses of up to four NetBIOS name
         * servers.</p> </li> <li> <p> <code>netbios-node-type</code> - The NetBIOS node
         * type (1, 2, 4, or 8). We recommend that you specify 2. Broadcast and multicast
         * are not supported. For more information about NetBIOS node types, see <a
         * href="https://www.ietf.org/rfc/rfc2132.txt">RFC 2132</a>.</p> </li> <li> <p>
         * <code>ipv6-address-preferred-lease-time</code> - A value (in seconds, minutes,
         * hours, or years) for how frequently a running instance with an IPv6 assigned to
         * it goes through DHCPv6 lease renewal. Acceptable values are between 140 and
         * 2147483647 seconds (approximately 68 years). If no value is entered, the default
         * lease time is 140 seconds. If you use long-term addressing for EC2 instances,
         * you can increase the lease time and avoid frequent lease renewal requests. Lease
         * renewal typically occurs when half of the lease time has elapsed.</p> </li>
         * </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateDhcpOptions">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateDhcpOptionsOutcome CreateDhcpOptions(const Model::CreateDhcpOptionsRequest& request) const;

        /**
         * A Callable wrapper for CreateDhcpOptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateDhcpOptionsRequestT = Model::CreateDhcpOptionsRequest>
        Model::CreateDhcpOptionsOutcomeCallable CreateDhcpOptionsCallable(const CreateDhcpOptionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateDhcpOptions, request);
        }

        /**
         * An Async wrapper for CreateDhcpOptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateDhcpOptionsRequestT = Model::CreateDhcpOptionsRequest>
        void CreateDhcpOptionsAsync(const CreateDhcpOptionsRequestT& request, const CreateDhcpOptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateDhcpOptions, request, handler, context);
        }

        /**
         * <p>[IPv6 only] Creates an egress-only internet gateway for your VPC. An
         * egress-only internet gateway is used to enable outbound communication over IPv6
         * from instances in your VPC to the internet, and prevents hosts outside of your
         * VPC from initiating an IPv6 connection with your instance.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateEgressOnlyInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateEgressOnlyInternetGatewayOutcome CreateEgressOnlyInternetGateway(const Model::CreateEgressOnlyInternetGatewayRequest& request) const;

        /**
         * A Callable wrapper for CreateEgressOnlyInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateEgressOnlyInternetGatewayRequestT = Model::CreateEgressOnlyInternetGatewayRequest>
        Model::CreateEgressOnlyInternetGatewayOutcomeCallable CreateEgressOnlyInternetGatewayCallable(const CreateEgressOnlyInternetGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateEgressOnlyInternetGateway, request);
        }

        /**
         * An Async wrapper for CreateEgressOnlyInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateEgressOnlyInternetGatewayRequestT = Model::CreateEgressOnlyInternetGatewayRequest>
        void CreateEgressOnlyInternetGatewayAsync(const CreateEgressOnlyInternetGatewayRequestT& request, const CreateEgressOnlyInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateEgressOnlyInternetGateway, request, handler, context);
        }

        /**
         * <p>Creates an EC2 Fleet that contains the configuration information for
         * On-Demand Instances and Spot Instances. Instances are launched immediately if
         * there is available capacity.</p> <p>A single EC2 Fleet can include multiple
         * launch specifications that vary by instance type, AMI, Availability Zone, or
         * subnet.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html">EC2
         * Fleet</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateFleet">AWS API
         * Reference</a></p>
         */
        virtual Model::CreateFleetOutcome CreateFleet(const Model::CreateFleetRequest& request) const;

        /**
         * A Callable wrapper for CreateFleet that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateFleetRequestT = Model::CreateFleetRequest>
        Model::CreateFleetOutcomeCallable CreateFleetCallable(const CreateFleetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateFleet, request);
        }

        /**
         * An Async wrapper for CreateFleet that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateFleetRequestT = Model::CreateFleetRequest>
        void CreateFleetAsync(const CreateFleetRequestT& request, const CreateFleetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateFleet, request, handler, context);
        }

        /**
         * <p>Creates one or more flow logs to capture information about IP traffic for a
         * specific network interface, subnet, or VPC. </p> <p>Flow log data for a
         * monitored network interface is recorded as flow log records, which are log
         * events consisting of fields that describe the traffic flow. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-log-records.html">Flow
         * log records</a> in the <i>Amazon VPC User Guide</i>.</p> <p>When publishing to
         * CloudWatch Logs, flow log records are published to a log group, and each network
         * interface has a unique log stream in the log group. When publishing to Amazon
         * S3, flow log records for all of the monitored network interfaces are published
         * to a single log file object that is stored in the specified bucket.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html">VPC Flow
         * Logs</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateFlowLogs">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateFlowLogsOutcome CreateFlowLogs(const Model::CreateFlowLogsRequest& request) const;

        /**
         * A Callable wrapper for CreateFlowLogs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateFlowLogsRequestT = Model::CreateFlowLogsRequest>
        Model::CreateFlowLogsOutcomeCallable CreateFlowLogsCallable(const CreateFlowLogsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateFlowLogs, request);
        }

        /**
         * An Async wrapper for CreateFlowLogs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateFlowLogsRequestT = Model::CreateFlowLogsRequest>
        void CreateFlowLogsAsync(const CreateFlowLogsRequestT& request, const CreateFlowLogsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateFlowLogs, request, handler, context);
        }

        /**
         * <p>Creates an Amazon FPGA Image (AFI) from the specified design checkpoint
         * (DCP).</p> <p>The create operation is asynchronous. To verify that the AFI is
         * ready for use, check the output logs.</p> <p>An AFI contains the FPGA bitstream
         * that is ready to download to an FPGA. You can securely deploy an AFI on multiple
         * FPGA-accelerated instances. For more information, see the <a
         * href="https://github.com/aws/aws-fpga/">Amazon Web Services FPGA Hardware
         * Development Kit</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateFpgaImage">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateFpgaImageOutcome CreateFpgaImage(const Model::CreateFpgaImageRequest& request) const;

        /**
         * A Callable wrapper for CreateFpgaImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateFpgaImageRequestT = Model::CreateFpgaImageRequest>
        Model::CreateFpgaImageOutcomeCallable CreateFpgaImageCallable(const CreateFpgaImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateFpgaImage, request);
        }

        /**
         * An Async wrapper for CreateFpgaImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateFpgaImageRequestT = Model::CreateFpgaImageRequest>
        void CreateFpgaImageAsync(const CreateFpgaImageRequestT& request, const CreateFpgaImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateFpgaImage, request, handler, context);
        }

        /**
         * <p>Creates an Amazon EBS-backed AMI from an Amazon EBS-backed instance that is
         * either running or stopped.</p> <p>If you customized your instance with instance
         * store volumes or Amazon EBS volumes in addition to the root device volume, the
         * new AMI contains block device mapping information for those volumes. When you
         * launch an instance from this new AMI, the instance automatically launches with
         * those additional volumes.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/creating-an-ami-ebs.html">Create
         * an Amazon EBS-backed AMI</a> in the <i>Amazon Elastic Compute Cloud User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateImage">AWS API
         * Reference</a></p>
         */
        virtual Model::CreateImageOutcome CreateImage(const Model::CreateImageRequest& request) const;

        /**
         * A Callable wrapper for CreateImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateImageRequestT = Model::CreateImageRequest>
        Model::CreateImageOutcomeCallable CreateImageCallable(const CreateImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateImage, request);
        }

        /**
         * An Async wrapper for CreateImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateImageRequestT = Model::CreateImageRequest>
        void CreateImageAsync(const CreateImageRequestT& request, const CreateImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateImage, request, handler, context);
        }

        /**
         * <p>Creates an EC2 Instance Connect Endpoint.</p> <p>An EC2 Instance Connect
         * Endpoint allows you to connect to an instance, without requiring the instance to
         * have a public IPv4 address. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Connect-using-EC2-Instance-Connect-Endpoint.html">Connect
         * to your instances without requiring a public IPv4 address using EC2 Instance
         * Connect Endpoint</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateInstanceConnectEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateInstanceConnectEndpointOutcome CreateInstanceConnectEndpoint(const Model::CreateInstanceConnectEndpointRequest& request) const;

        /**
         * A Callable wrapper for CreateInstanceConnectEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateInstanceConnectEndpointRequestT = Model::CreateInstanceConnectEndpointRequest>
        Model::CreateInstanceConnectEndpointOutcomeCallable CreateInstanceConnectEndpointCallable(const CreateInstanceConnectEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateInstanceConnectEndpoint, request);
        }

        /**
         * An Async wrapper for CreateInstanceConnectEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateInstanceConnectEndpointRequestT = Model::CreateInstanceConnectEndpointRequest>
        void CreateInstanceConnectEndpointAsync(const CreateInstanceConnectEndpointRequestT& request, const CreateInstanceConnectEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateInstanceConnectEndpoint, request, handler, context);
        }

        /**
         * <p>Creates an event window in which scheduled events for the associated Amazon
         * EC2 instances can run.</p> <p>You can define either a set of time ranges or a
         * cron expression when creating the event window, but not both. All event window
         * times are in UTC.</p> <p>You can create up to 200 event windows per Amazon Web
         * Services Region.</p> <p>When you create the event window, targets (instance IDs,
         * Dedicated Host IDs, or tags) are not yet associated with it. To ensure that the
         * event window can be used, you must associate one or more targets with it by
         * using the <a>AssociateInstanceEventWindow</a> API.</p>  <p>Event
         * windows are applicable only for scheduled events that stop, reboot, or terminate
         * instances.</p> <p>Event windows are <i>not</i> applicable for:</p> <ul> <li>
         * <p>Expedited scheduled events and network maintenance events. </p> </li> <li>
         * <p>Unscheduled maintenance such as AutoRecovery and unplanned reboots.</p> </li>
         * </ul>  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/event-windows.html">Define
         * event windows for scheduled events</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateInstanceEventWindow">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateInstanceEventWindowOutcome CreateInstanceEventWindow(const Model::CreateInstanceEventWindowRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateInstanceEventWindow that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateInstanceEventWindowRequestT = Model::CreateInstanceEventWindowRequest>
        Model::CreateInstanceEventWindowOutcomeCallable CreateInstanceEventWindowCallable(const CreateInstanceEventWindowRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateInstanceEventWindow, request);
        }

        /**
         * An Async wrapper for CreateInstanceEventWindow that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateInstanceEventWindowRequestT = Model::CreateInstanceEventWindowRequest>
        void CreateInstanceEventWindowAsync(const CreateInstanceEventWindowResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateInstanceEventWindowRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateInstanceEventWindow, request, handler, context);
        }

        /**
         * <p>Exports a running or stopped instance to an Amazon S3 bucket.</p> <p>For
         * information about the prerequisites for your Amazon S3 bucket, supported
         * operating systems, image formats, and known limitations for the types of
         * instances you can export, see <a
         * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html">Exporting
         * an instance as a VM Using VM Import/Export</a> in the <i>VM Import/Export User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateInstanceExportTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateInstanceExportTaskOutcome CreateInstanceExportTask(const Model::CreateInstanceExportTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateInstanceExportTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateInstanceExportTaskRequestT = Model::CreateInstanceExportTaskRequest>
        Model::CreateInstanceExportTaskOutcomeCallable CreateInstanceExportTaskCallable(const CreateInstanceExportTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateInstanceExportTask, request);
        }

        /**
         * An Async wrapper for CreateInstanceExportTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateInstanceExportTaskRequestT = Model::CreateInstanceExportTaskRequest>
        void CreateInstanceExportTaskAsync(const CreateInstanceExportTaskRequestT& request, const CreateInstanceExportTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateInstanceExportTask, request, handler, context);
        }

        /**
         * <p>Creates an internet gateway for use with a VPC. After creating the internet
         * gateway, you attach it to a VPC using <a>AttachInternetGateway</a>.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html">Internet
         * gateways</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateInternetGatewayOutcome CreateInternetGateway(const Model::CreateInternetGatewayRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateInternetGatewayRequestT = Model::CreateInternetGatewayRequest>
        Model::CreateInternetGatewayOutcomeCallable CreateInternetGatewayCallable(const CreateInternetGatewayRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateInternetGateway, request);
        }

        /**
         * An Async wrapper for CreateInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateInternetGatewayRequestT = Model::CreateInternetGatewayRequest>
        void CreateInternetGatewayAsync(const CreateInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateInternetGatewayRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateInternetGateway, request, handler, context);
        }

        /**
         * <p>Create an IPAM. Amazon VPC IP Address Manager (IPAM) is a VPC feature that
         * you can use to automate your IP address management workflows including
         * assigning, tracking, troubleshooting, and auditing IP addresses across Amazon
         * Web Services Regions and accounts throughout your Amazon Web Services
         * Organization.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an
         * IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See Also:</h3>  
         * <a href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateIpam">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateIpamOutcome CreateIpam(const Model::CreateIpamRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateIpam that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateIpamRequestT = Model::CreateIpamRequest>
        Model::CreateIpamOutcomeCallable CreateIpamCallable(const CreateIpamRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateIpam, request);
        }

        /**
         * An Async wrapper for CreateIpam that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateIpamRequestT = Model::CreateIpamRequest>
        void CreateIpamAsync(const CreateIpamResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateIpamRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateIpam, request, handler, context);
        }

        /**
         * <p>Create a verification token. A verification token is an Amazon Web
         * Services-generated random value that you can use to prove ownership of an
         * external resource. For example, you can use a verification token to validate
         * that you control a public IP address range when you bring an IP address range to
         * Amazon Web Services (BYOIP). </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateIpamExternalResourceVerificationToken">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateIpamExternalResourceVerificationTokenOutcome CreateIpamExternalResourceVerificationToken(const Model::CreateIpamExternalResourceVerificationTokenRequest& request) const;

        /**
         * A Callable wrapper for CreateIpamExternalResourceVerificationToken that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateIpamExternalResourceVerificationTokenRequestT = Model::CreateIpamExternalResourceVerificationTokenRequest>
        Model::CreateIpamExternalResourceVerificationTokenOutcomeCallable CreateIpamExternalResourceVerificationTokenCallable(const CreateIpamExternalResourceVerificationTokenRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateIpamExternalResourceVerificationToken, request);
        }

        /**
         * An Async wrapper for CreateIpamExternalResourceVerificationToken that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateIpamExternalResourceVerificationTokenRequestT = Model::CreateIpamExternalResourceVerificationTokenRequest>
        void CreateIpamExternalResourceVerificationTokenAsync(const CreateIpamExternalResourceVerificationTokenRequestT& request, const CreateIpamExternalResourceVerificationTokenResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateIpamExternalResourceVerificationToken, request, handler, context);
        }

        /**
         * <p>Create an IP address pool for Amazon VPC IP Address Manager (IPAM). In IPAM,
         * a pool is a collection of contiguous IP addresses CIDRs. Pools enable you to
         * organize your IP addresses according to your routing and security needs. For
         * example, if you have separate routing and security needs for development and
         * production applications, you can create a pool for each.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-top-ipam.html">Create a
         * top-level pool</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateIpamPool">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateIpamPoolOutcome CreateIpamPool(const Model::CreateIpamPoolRequest& request) const;

        /**
         * A Callable wrapper for CreateIpamPool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateIpamPoolRequestT = Model::CreateIpamPoolRequest>
        Model::CreateIpamPoolOutcomeCallable CreateIpamPoolCallable(const CreateIpamPoolRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateIpamPool, request);
        }

        /**
         * An Async wrapper for CreateIpamPool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateIpamPoolRequestT = Model::CreateIpamPoolRequest>
        void CreateIpamPoolAsync(const CreateIpamPoolRequestT& request, const CreateIpamPoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateIpamPool, request, handler, context);
        }

        /**
         * <p>Creates an IPAM resource discovery. A resource discovery is an IPAM component
         * that enables IPAM to manage and monitor resources that belong to the owning
         * account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateIpamResourceDiscovery">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateIpamResourceDiscoveryOutcome CreateIpamResourceDiscovery(const Model::CreateIpamResourceDiscoveryRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateIpamResourceDiscovery that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateIpamResourceDiscoveryRequestT = Model::CreateIpamResourceDiscoveryRequest>
        Model::CreateIpamResourceDiscoveryOutcomeCallable CreateIpamResourceDiscoveryCallable(const CreateIpamResourceDiscoveryRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateIpamResourceDiscovery, request);
        }

        /**
         * An Async wrapper for CreateIpamResourceDiscovery that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateIpamResourceDiscoveryRequestT = Model::CreateIpamResourceDiscoveryRequest>
        void CreateIpamResourceDiscoveryAsync(const CreateIpamResourceDiscoveryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateIpamResourceDiscoveryRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateIpamResourceDiscovery, request, handler, context);
        }

        /**
         * <p>Create an IPAM scope. In IPAM, a scope is the highest-level container within
         * IPAM. An IPAM contains two default scopes. Each scope represents the IP space
         * for a single network. The private scope is intended for all private IP address
         * space. The public scope is intended for all public IP address space. Scopes
         * enable you to reuse IP addresses across multiple unconnected networks without
         * causing IP address overlap or conflict.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/add-scope-ipam.html">Add a
         * scope</a> in the <i>Amazon VPC IPAM User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateIpamScope">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateIpamScopeOutcome CreateIpamScope(const Model::CreateIpamScopeRequest& request) const;

        /**
         * A Callable wrapper for CreateIpamScope that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateIpamScopeRequestT = Model::CreateIpamScopeRequest>
        Model::CreateIpamScopeOutcomeCallable CreateIpamScopeCallable(const CreateIpamScopeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateIpamScope, request);
        }

        /**
         * An Async wrapper for CreateIpamScope that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateIpamScopeRequestT = Model::CreateIpamScopeRequest>
        void CreateIpamScopeAsync(const CreateIpamScopeRequestT& request, const CreateIpamScopeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateIpamScope, request, handler, context);
        }

        /**
         * <p>Creates an ED25519 or 2048-bit RSA key pair with the specified name and in
         * the specified format. Amazon EC2 stores the public key and displays the private
         * key for you to save to a file. The private key is returned as an unencrypted PEM
         * encoded PKCS#1 private key or an unencrypted PPK formatted private key for use
         * with PuTTY. If a key with the specified name already exists, Amazon EC2 returns
         * an error.</p> <p>The key pair returned to you is available only in the Amazon
         * Web Services Region in which you create it. If you prefer, you can create your
         * own key pair using a third-party tool and upload it to any Region using
         * <a>ImportKeyPair</a>.</p> <p>You can have up to 5,000 key pairs per Amazon Web
         * Services Region.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon
         * EC2 key pairs</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateKeyPair">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateKeyPairOutcome CreateKeyPair(const Model::CreateKeyPairRequest& request) const;

        /**
         * A Callable wrapper for CreateKeyPair that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateKeyPairRequestT = Model::CreateKeyPairRequest>
        Model::CreateKeyPairOutcomeCallable CreateKeyPairCallable(const CreateKeyPairRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateKeyPair, request);
        }

        /**
         * An Async wrapper for CreateKeyPair that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateKeyPairRequestT = Model::CreateKeyPairRequest>
        void CreateKeyPairAsync(const CreateKeyPairRequestT& request, const CreateKeyPairResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateKeyPair, request, handler, context);
        }

        /**
         * <p>Creates a launch template.</p> <p>A launch template contains the parameters
         * to launch an instance. When you launch an instance using <a>RunInstances</a>,
         * you can specify a launch template instead of providing the launch parameters in
         * the request. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store
         * instance launch parameters in Amazon EC2 launch templates</a> in the <i>Amazon
         * EC2 User Guide</i>.</p> <p>To clone an existing launch template as the basis for
         * a new launch template, use the Amazon EC2 console. The API, SDKs, and CLI do not
         * support cloning a template. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/create-launch-template.html#create-launch-template-from-existing-launch-template">Create
         * a launch template from an existing launch template</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLaunchTemplate">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLaunchTemplateOutcome CreateLaunchTemplate(const Model::CreateLaunchTemplateRequest& request) const;

        /**
         * A Callable wrapper for CreateLaunchTemplate that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLaunchTemplateRequestT = Model::CreateLaunchTemplateRequest>
        Model::CreateLaunchTemplateOutcomeCallable CreateLaunchTemplateCallable(const CreateLaunchTemplateRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLaunchTemplate, request);
        }

        /**
         * An Async wrapper for CreateLaunchTemplate that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLaunchTemplateRequestT = Model::CreateLaunchTemplateRequest>
        void CreateLaunchTemplateAsync(const CreateLaunchTemplateRequestT& request, const CreateLaunchTemplateResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLaunchTemplate, request, handler, context);
        }

        /**
         * <p>Creates a new version of a launch template. You must specify an existing
         * launch template, either by name or ID. You can determine whether the new version
         * inherits parameters from a source version, and add or overwrite parameters as
         * needed.</p> <p>Launch template versions are numbered in the order in which they
         * are created. You can't specify, change, or replace the numbering of launch
         * template versions.</p> <p>Launch templates are immutable; after you create a
         * launch template, you can't modify it. Instead, you can create a new version of
         * the launch template that includes the changes that you require.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-launch-template-versions.html">Modify
         * a launch template (manage launch template versions)</a> in the <i>Amazon EC2
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLaunchTemplateVersion">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLaunchTemplateVersionOutcome CreateLaunchTemplateVersion(const Model::CreateLaunchTemplateVersionRequest& request) const;

        /**
         * A Callable wrapper for CreateLaunchTemplateVersion that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLaunchTemplateVersionRequestT = Model::CreateLaunchTemplateVersionRequest>
        Model::CreateLaunchTemplateVersionOutcomeCallable CreateLaunchTemplateVersionCallable(const CreateLaunchTemplateVersionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLaunchTemplateVersion, request);
        }

        /**
         * An Async wrapper for CreateLaunchTemplateVersion that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLaunchTemplateVersionRequestT = Model::CreateLaunchTemplateVersionRequest>
        void CreateLaunchTemplateVersionAsync(const CreateLaunchTemplateVersionRequestT& request, const CreateLaunchTemplateVersionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLaunchTemplateVersion, request, handler, context);
        }

        /**
         * <p>Creates a static route for the specified local gateway route table. You must
         * specify one of the following targets: </p> <ul> <li> <p>
         * <code>LocalGatewayVirtualInterfaceGroupId</code> </p> </li> <li> <p>
         * <code>NetworkInterfaceId</code> </p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayRouteOutcome CreateLocalGatewayRoute(const Model::CreateLocalGatewayRouteRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayRouteRequestT = Model::CreateLocalGatewayRouteRequest>
        Model::CreateLocalGatewayRouteOutcomeCallable CreateLocalGatewayRouteCallable(const CreateLocalGatewayRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayRoute, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayRouteRequestT = Model::CreateLocalGatewayRouteRequest>
        void CreateLocalGatewayRouteAsync(const CreateLocalGatewayRouteRequestT& request, const CreateLocalGatewayRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayRoute, request, handler, context);
        }

        /**
         * <p> Creates a local gateway route table. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayRouteTableOutcome CreateLocalGatewayRouteTable(const Model::CreateLocalGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayRouteTableRequestT = Model::CreateLocalGatewayRouteTableRequest>
        Model::CreateLocalGatewayRouteTableOutcomeCallable CreateLocalGatewayRouteTableCallable(const CreateLocalGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayRouteTableRequestT = Model::CreateLocalGatewayRouteTableRequest>
        void CreateLocalGatewayRouteTableAsync(const CreateLocalGatewayRouteTableRequestT& request, const CreateLocalGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayRouteTable, request, handler, context);
        }

        /**
         * <p> Creates a local gateway route table virtual interface group association.
         * </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationOutcome CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation(const Model::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT = Model::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest>
        Model::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationOutcomeCallable CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationCallable(const CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT = Model::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest>
        void CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationAsync(const CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT& request, const CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayRouteTableVirtualInterfaceGroupAssociation, request, handler, context);
        }

        /**
         * <p>Associates the specified VPC with the specified local gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayRouteTableVpcAssociation">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayRouteTableVpcAssociationOutcome CreateLocalGatewayRouteTableVpcAssociation(const Model::CreateLocalGatewayRouteTableVpcAssociationRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayRouteTableVpcAssociation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayRouteTableVpcAssociationRequestT = Model::CreateLocalGatewayRouteTableVpcAssociationRequest>
        Model::CreateLocalGatewayRouteTableVpcAssociationOutcomeCallable CreateLocalGatewayRouteTableVpcAssociationCallable(const CreateLocalGatewayRouteTableVpcAssociationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayRouteTableVpcAssociation, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayRouteTableVpcAssociation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayRouteTableVpcAssociationRequestT = Model::CreateLocalGatewayRouteTableVpcAssociationRequest>
        void CreateLocalGatewayRouteTableVpcAssociationAsync(const CreateLocalGatewayRouteTableVpcAssociationRequestT& request, const CreateLocalGatewayRouteTableVpcAssociationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayRouteTableVpcAssociation, request, handler, context);
        }

        /**
         * <p>Create a virtual interface for a local gateway. </p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayVirtualInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayVirtualInterfaceOutcome CreateLocalGatewayVirtualInterface(const Model::CreateLocalGatewayVirtualInterfaceRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayVirtualInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayVirtualInterfaceRequestT = Model::CreateLocalGatewayVirtualInterfaceRequest>
        Model::CreateLocalGatewayVirtualInterfaceOutcomeCallable CreateLocalGatewayVirtualInterfaceCallable(const CreateLocalGatewayVirtualInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayVirtualInterface, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayVirtualInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayVirtualInterfaceRequestT = Model::CreateLocalGatewayVirtualInterfaceRequest>
        void CreateLocalGatewayVirtualInterfaceAsync(const CreateLocalGatewayVirtualInterfaceRequestT& request, const CreateLocalGatewayVirtualInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayVirtualInterface, request, handler, context);
        }

        /**
         * <p>Create a local gateway virtual interface group.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateLocalGatewayVirtualInterfaceGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateLocalGatewayVirtualInterfaceGroupOutcome CreateLocalGatewayVirtualInterfaceGroup(const Model::CreateLocalGatewayVirtualInterfaceGroupRequest& request) const;

        /**
         * A Callable wrapper for CreateLocalGatewayVirtualInterfaceGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateLocalGatewayVirtualInterfaceGroupRequestT = Model::CreateLocalGatewayVirtualInterfaceGroupRequest>
        Model::CreateLocalGatewayVirtualInterfaceGroupOutcomeCallable CreateLocalGatewayVirtualInterfaceGroupCallable(const CreateLocalGatewayVirtualInterfaceGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateLocalGatewayVirtualInterfaceGroup, request);
        }

        /**
         * An Async wrapper for CreateLocalGatewayVirtualInterfaceGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateLocalGatewayVirtualInterfaceGroupRequestT = Model::CreateLocalGatewayVirtualInterfaceGroupRequest>
        void CreateLocalGatewayVirtualInterfaceGroupAsync(const CreateLocalGatewayVirtualInterfaceGroupRequestT& request, const CreateLocalGatewayVirtualInterfaceGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateLocalGatewayVirtualInterfaceGroup, request, handler, context);
        }

        /**
         * <p>Creates a System Integrity Protection (SIP) modification task to configure
         * the SIP settings for an x86 Mac instance or Apple silicon Mac instance. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/mac-sip-settings.html#mac-sip-configure">
         * Configure SIP for Amazon EC2 instances</a> in the <i>Amazon EC2 User
         * Guide</i>.</p> <p>When you configure the SIP settings for your instance, you can
         * either enable or disable all SIP settings, or you can specify a custom SIP
         * configuration that selectively enables or disables specific SIP settings.</p>
         *  <p>If you implement a custom configuration, <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/mac-sip-settings.html#mac-sip-check-settings">
         * connect to the instance and verify the settings</a> to ensure that your
         * requirements are properly implemented and functioning as intended.</p> <p>SIP
         * configurations might change with macOS updates. We recommend that you review
         * custom SIP settings after any macOS version upgrade to ensure continued
         * compatibility and proper functionality of your security configurations.</p>
         *  <p>To enable or disable all SIP settings, use the
         * <b>MacSystemIntegrityProtectionStatus</b> parameter only. For example, to enable
         * all SIP settings, specify the following:</p> <ul> <li> <p>
         * <code>MacSystemIntegrityProtectionStatus=enabled</code> </p> </li> </ul> <p>To
         * specify a custom configuration that selectively enables or disables specific SIP
         * settings, use the <b>MacSystemIntegrityProtectionStatus</b> parameter to enable
         * or disable all SIP settings, and then use the
         * <b>MacSystemIntegrityProtectionConfiguration</b> parameter to specify
         * exceptions. In this case, the exceptions you specify for
         * <b>MacSystemIntegrityProtectionConfiguration</b> override the value you specify
         * for <b>MacSystemIntegrityProtectionStatus</b>. For example, to enable all SIP
         * settings, except <code>NvramProtections</code>, specify the following:</p> <ul>
         * <li> <p> <code>MacSystemIntegrityProtectionStatus=enabled</code> </p> </li> <li>
         * <p> <code>MacSystemIntegrityProtectionConfigurationRequest
         * "NvramProtections=disabled"</code> </p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateMacSystemIntegrityProtectionModificationTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateMacSystemIntegrityProtectionModificationTaskOutcome CreateMacSystemIntegrityProtectionModificationTask(const Model::CreateMacSystemIntegrityProtectionModificationTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateMacSystemIntegrityProtectionModificationTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateMacSystemIntegrityProtectionModificationTaskRequestT = Model::CreateMacSystemIntegrityProtectionModificationTaskRequest>
        Model::CreateMacSystemIntegrityProtectionModificationTaskOutcomeCallable CreateMacSystemIntegrityProtectionModificationTaskCallable(const CreateMacSystemIntegrityProtectionModificationTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateMacSystemIntegrityProtectionModificationTask, request);
        }

        /**
         * An Async wrapper for CreateMacSystemIntegrityProtectionModificationTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateMacSystemIntegrityProtectionModificationTaskRequestT = Model::CreateMacSystemIntegrityProtectionModificationTaskRequest>
        void CreateMacSystemIntegrityProtectionModificationTaskAsync(const CreateMacSystemIntegrityProtectionModificationTaskRequestT& request, const CreateMacSystemIntegrityProtectionModificationTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateMacSystemIntegrityProtectionModificationTask, request, handler, context);
        }

        /**
         * <p>Creates a managed prefix list. You can specify entries for the prefix list.
         * Each entry consists of a CIDR block and an optional description.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateManagedPrefixList">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateManagedPrefixListOutcome CreateManagedPrefixList(const Model::CreateManagedPrefixListRequest& request) const;

        /**
         * A Callable wrapper for CreateManagedPrefixList that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateManagedPrefixListRequestT = Model::CreateManagedPrefixListRequest>
        Model::CreateManagedPrefixListOutcomeCallable CreateManagedPrefixListCallable(const CreateManagedPrefixListRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateManagedPrefixList, request);
        }

        /**
         * An Async wrapper for CreateManagedPrefixList that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateManagedPrefixListRequestT = Model::CreateManagedPrefixListRequest>
        void CreateManagedPrefixListAsync(const CreateManagedPrefixListRequestT& request, const CreateManagedPrefixListResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateManagedPrefixList, request, handler, context);
        }

        /**
         * <p>Creates a NAT gateway in the specified subnet. This action creates a network
         * interface in the specified subnet with a private IP address from the IP address
         * range of the subnet. You can create either a public NAT gateway or a private NAT
         * gateway.</p> <p>With a public NAT gateway, internet-bound traffic from a private
         * subnet can be routed to the NAT gateway, so that instances in a private subnet
         * can connect to the internet.</p> <p>With a private NAT gateway, private
         * communication is routed across VPCs and on-premises networks through a transit
         * gateway or virtual private gateway. Common use cases include running large
         * workloads behind a small pool of allowlisted IPv4 addresses, preserving private
         * IPv4 addresses, and communicating between overlapping networks.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html">NAT
         * gateways</a> in the <i>Amazon VPC User Guide</i>.</p>  <p>When you
         * create a public NAT gateway and assign it an EIP or secondary EIPs, the network
         * border group of the EIPs must match the network border group of the Availability
         * Zone (AZ) that the public NAT gateway is in. If it's not the same, the NAT
         * gateway will fail to launch. You can see the network border group for the
         * subnet's AZ by viewing the details of the subnet. Similarly, you can view the
         * network border group of an EIP by viewing the details of the EIP address. For
         * more information about network border groups and EIPs, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/WorkWithEIPs.html">Allocate
         * an Elastic IP address</a> in the <i>Amazon VPC User Guide</i>. </p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNatGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNatGatewayOutcome CreateNatGateway(const Model::CreateNatGatewayRequest& request) const;

        /**
         * A Callable wrapper for CreateNatGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNatGatewayRequestT = Model::CreateNatGatewayRequest>
        Model::CreateNatGatewayOutcomeCallable CreateNatGatewayCallable(const CreateNatGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNatGateway, request);
        }

        /**
         * An Async wrapper for CreateNatGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNatGatewayRequestT = Model::CreateNatGatewayRequest>
        void CreateNatGatewayAsync(const CreateNatGatewayRequestT& request, const CreateNatGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNatGateway, request, handler, context);
        }

        /**
         * <p>Creates a network ACL in a VPC. Network ACLs provide an optional layer of
         * security (in addition to security groups) for the instances in your VPC.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html">Network
         * ACLs</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkAcl">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkAclOutcome CreateNetworkAcl(const Model::CreateNetworkAclRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkAcl that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkAclRequestT = Model::CreateNetworkAclRequest>
        Model::CreateNetworkAclOutcomeCallable CreateNetworkAclCallable(const CreateNetworkAclRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkAcl, request);
        }

        /**
         * An Async wrapper for CreateNetworkAcl that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkAclRequestT = Model::CreateNetworkAclRequest>
        void CreateNetworkAclAsync(const CreateNetworkAclRequestT& request, const CreateNetworkAclResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkAcl, request, handler, context);
        }

        /**
         * <p>Creates an entry (a rule) in a network ACL with the specified rule number.
         * Each network ACL has a set of numbered ingress rules and a separate set of
         * numbered egress rules. When determining whether a packet should be allowed in or
         * out of a subnet associated with the ACL, we process the entries in the ACL
         * according to the rule numbers, in ascending order. Each network ACL has a set of
         * ingress rules and a separate set of egress rules.</p> <p>We recommend that you
         * leave room between the rule numbers (for example, 100, 110, 120, ...), and not
         * number them one right after the other (for example, 101, 102, 103, ...). This
         * makes it easier to add a rule between existing ones without having to renumber
         * the rules.</p> <p>After you add an entry, you can't modify it; you must either
         * replace it, or create an entry and delete the old one.</p> <p>For more
         * information about network ACLs, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html">Network
         * ACLs</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkAclEntry">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkAclEntryOutcome CreateNetworkAclEntry(const Model::CreateNetworkAclEntryRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkAclEntry that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkAclEntryRequestT = Model::CreateNetworkAclEntryRequest>
        Model::CreateNetworkAclEntryOutcomeCallable CreateNetworkAclEntryCallable(const CreateNetworkAclEntryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkAclEntry, request);
        }

        /**
         * An Async wrapper for CreateNetworkAclEntry that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkAclEntryRequestT = Model::CreateNetworkAclEntryRequest>
        void CreateNetworkAclEntryAsync(const CreateNetworkAclEntryRequestT& request, const CreateNetworkAclEntryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkAclEntry, request, handler, context);
        }

        /**
         * <p>Creates a Network Access Scope.</p> <p>Amazon Web Services Network Access
         * Analyzer enables cloud networking and cloud operations teams to verify that
         * their networks on Amazon Web Services conform to their network security and
         * governance objectives. For more information, see the <a
         * href="https://docs.aws.amazon.com/vpc/latest/network-access-analyzer/">Amazon
         * Web Services Network Access Analyzer Guide</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkInsightsAccessScope">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkInsightsAccessScopeOutcome CreateNetworkInsightsAccessScope(const Model::CreateNetworkInsightsAccessScopeRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkInsightsAccessScope that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkInsightsAccessScopeRequestT = Model::CreateNetworkInsightsAccessScopeRequest>
        Model::CreateNetworkInsightsAccessScopeOutcomeCallable CreateNetworkInsightsAccessScopeCallable(const CreateNetworkInsightsAccessScopeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkInsightsAccessScope, request);
        }

        /**
         * An Async wrapper for CreateNetworkInsightsAccessScope that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkInsightsAccessScopeRequestT = Model::CreateNetworkInsightsAccessScopeRequest>
        void CreateNetworkInsightsAccessScopeAsync(const CreateNetworkInsightsAccessScopeRequestT& request, const CreateNetworkInsightsAccessScopeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkInsightsAccessScope, request, handler, context);
        }

        /**
         * <p>Creates a path to analyze for reachability.</p> <p>Reachability Analyzer
         * enables you to analyze and debug network reachability between two resources in
         * your virtual private cloud (VPC). For more information, see the <a
         * href="https://docs.aws.amazon.com/vpc/latest/reachability/">Reachability
         * Analyzer Guide</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkInsightsPath">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkInsightsPathOutcome CreateNetworkInsightsPath(const Model::CreateNetworkInsightsPathRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkInsightsPath that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkInsightsPathRequestT = Model::CreateNetworkInsightsPathRequest>
        Model::CreateNetworkInsightsPathOutcomeCallable CreateNetworkInsightsPathCallable(const CreateNetworkInsightsPathRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkInsightsPath, request);
        }

        /**
         * An Async wrapper for CreateNetworkInsightsPath that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkInsightsPathRequestT = Model::CreateNetworkInsightsPathRequest>
        void CreateNetworkInsightsPathAsync(const CreateNetworkInsightsPathRequestT& request, const CreateNetworkInsightsPathResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkInsightsPath, request, handler, context);
        }

        /**
         * <p>Creates a network interface in the specified subnet.</p> <p>The number of IP
         * addresses you can assign to a network interface varies by instance type.</p>
         * <p>For more information about network interfaces, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic
         * network interfaces</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkInterfaceOutcome CreateNetworkInterface(const Model::CreateNetworkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkInterfaceRequestT = Model::CreateNetworkInterfaceRequest>
        Model::CreateNetworkInterfaceOutcomeCallable CreateNetworkInterfaceCallable(const CreateNetworkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkInterface, request);
        }

        /**
         * An Async wrapper for CreateNetworkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkInterfaceRequestT = Model::CreateNetworkInterfaceRequest>
        void CreateNetworkInterfaceAsync(const CreateNetworkInterfaceRequestT& request, const CreateNetworkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkInterface, request, handler, context);
        }

        /**
         * <p>Grants an Amazon Web Services-authorized account permission to attach the
         * specified network interface to an instance in their account.</p> <p>You can
         * grant permission to a single Amazon Web Services account only, and only one
         * account at a time.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateNetworkInterfacePermission">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateNetworkInterfacePermissionOutcome CreateNetworkInterfacePermission(const Model::CreateNetworkInterfacePermissionRequest& request) const;

        /**
         * A Callable wrapper for CreateNetworkInterfacePermission that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateNetworkInterfacePermissionRequestT = Model::CreateNetworkInterfacePermissionRequest>
        Model::CreateNetworkInterfacePermissionOutcomeCallable CreateNetworkInterfacePermissionCallable(const CreateNetworkInterfacePermissionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateNetworkInterfacePermission, request);
        }

        /**
         * An Async wrapper for CreateNetworkInterfacePermission that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateNetworkInterfacePermissionRequestT = Model::CreateNetworkInterfacePermissionRequest>
        void CreateNetworkInterfacePermissionAsync(const CreateNetworkInterfacePermissionRequestT& request, const CreateNetworkInterfacePermissionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateNetworkInterfacePermission, request, handler, context);
        }

        /**
         * <p>Creates a placement group in which to launch instances. The strategy of the
         * placement group determines how the instances are organized within the group.
         * </p> <p>A <code>cluster</code> placement group is a logical grouping of
         * instances within a single Availability Zone that benefit from low network
         * latency, high network throughput. A <code>spread</code> placement group places
         * instances on distinct hardware. A <code>partition</code> placement group places
         * groups of instances in different partitions, where instances in one partition do
         * not share the same hardware with instances in another partition.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement
         * groups</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreatePlacementGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::CreatePlacementGroupOutcome CreatePlacementGroup(const Model::CreatePlacementGroupRequest& request = {}) const;

        /**
         * A Callable wrapper for CreatePlacementGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreatePlacementGroupRequestT = Model::CreatePlacementGroupRequest>
        Model::CreatePlacementGroupOutcomeCallable CreatePlacementGroupCallable(const CreatePlacementGroupRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreatePlacementGroup, request);
        }

        /**
         * An Async wrapper for CreatePlacementGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreatePlacementGroupRequestT = Model::CreatePlacementGroupRequest>
        void CreatePlacementGroupAsync(const CreatePlacementGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreatePlacementGroupRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreatePlacementGroup, request, handler, context);
        }

        /**
         * <p>Creates a public IPv4 address pool. A public IPv4 pool is an EC2 IP address
         * pool required for the public IPv4 CIDRs that you own and bring to Amazon Web
         * Services to manage with IPAM. IPv6 addresses you bring to Amazon Web Services,
         * however, use IPAM pools only. To monitor the status of pool creation, use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html">DescribePublicIpv4Pools</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreatePublicIpv4Pool">AWS
         * API Reference</a></p>
         */
        virtual Model::CreatePublicIpv4PoolOutcome CreatePublicIpv4Pool(const Model::CreatePublicIpv4PoolRequest& request = {}) const;

        /**
         * A Callable wrapper for CreatePublicIpv4Pool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreatePublicIpv4PoolRequestT = Model::CreatePublicIpv4PoolRequest>
        Model::CreatePublicIpv4PoolOutcomeCallable CreatePublicIpv4PoolCallable(const CreatePublicIpv4PoolRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreatePublicIpv4Pool, request);
        }

        /**
         * An Async wrapper for CreatePublicIpv4Pool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreatePublicIpv4PoolRequestT = Model::CreatePublicIpv4PoolRequest>
        void CreatePublicIpv4PoolAsync(const CreatePublicIpv4PoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreatePublicIpv4PoolRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreatePublicIpv4Pool, request, handler, context);
        }

        /**
         * <p>Replaces the EBS-backed root volume for a <code>running</code> instance with
         * a new volume that is restored to the original root volume's launch state, that
         * is restored to a specific snapshot taken from the original root volume, or that
         * is restored from an AMI that has the same key characteristics as that of the
         * instance.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/replace-root.html">Replace
         * a root volume</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateReplaceRootVolumeTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateReplaceRootVolumeTaskOutcome CreateReplaceRootVolumeTask(const Model::CreateReplaceRootVolumeTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateReplaceRootVolumeTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateReplaceRootVolumeTaskRequestT = Model::CreateReplaceRootVolumeTaskRequest>
        Model::CreateReplaceRootVolumeTaskOutcomeCallable CreateReplaceRootVolumeTaskCallable(const CreateReplaceRootVolumeTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateReplaceRootVolumeTask, request);
        }

        /**
         * An Async wrapper for CreateReplaceRootVolumeTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateReplaceRootVolumeTaskRequestT = Model::CreateReplaceRootVolumeTaskRequest>
        void CreateReplaceRootVolumeTaskAsync(const CreateReplaceRootVolumeTaskRequestT& request, const CreateReplaceRootVolumeTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateReplaceRootVolumeTask, request, handler, context);
        }

        /**
         * <p>Creates a listing for Amazon EC2 Standard Reserved Instances to be sold in
         * the Reserved Instance Marketplace. You can submit one Standard Reserved Instance
         * listing at a time. To get a list of your Standard Reserved Instances, you can
         * use the <a>DescribeReservedInstances</a> operation.</p>  <p>Only Standard
         * Reserved Instances can be sold in the Reserved Instance Marketplace. Convertible
         * Reserved Instances cannot be sold.</p>  <p>The Reserved Instance
         * Marketplace matches sellers who want to resell Standard Reserved Instance
         * capacity that they no longer need with buyers who want to purchase additional
         * capacity. Reserved Instances bought and sold through the Reserved Instance
         * Marketplace work like any other Reserved Instances.</p> <p>To sell your Standard
         * Reserved Instances, you must first register as a seller in the Reserved Instance
         * Marketplace. After completing the registration process, you can create a
         * Reserved Instance Marketplace listing of some or all of your Standard Reserved
         * Instances, and specify the upfront price to receive for them. Your Standard
         * Reserved Instance listings then become available for purchase. To view the
         * details of your Standard Reserved Instance listing, you can use the
         * <a>DescribeReservedInstancesListings</a> operation.</p> <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ri-market-general.html">Sell
         * in the Reserved Instance Marketplace</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateReservedInstancesListing">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateReservedInstancesListingOutcome CreateReservedInstancesListing(const Model::CreateReservedInstancesListingRequest& request) const;

        /**
         * A Callable wrapper for CreateReservedInstancesListing that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateReservedInstancesListingRequestT = Model::CreateReservedInstancesListingRequest>
        Model::CreateReservedInstancesListingOutcomeCallable CreateReservedInstancesListingCallable(const CreateReservedInstancesListingRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateReservedInstancesListing, request);
        }

        /**
         * An Async wrapper for CreateReservedInstancesListing that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateReservedInstancesListingRequestT = Model::CreateReservedInstancesListingRequest>
        void CreateReservedInstancesListingAsync(const CreateReservedInstancesListingRequestT& request, const CreateReservedInstancesListingResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateReservedInstancesListing, request, handler, context);
        }

        /**
         * <p>Starts a task that restores an AMI from an Amazon S3 object that was
         * previously created by using <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateStoreImageTask.html">CreateStoreImageTask</a>.</p>
         * <p>To use this API, you must have the required permissions. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/work-with-ami-store-restore.html#ami-s3-permissions">Permissions
         * for storing and restoring AMIs using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-store-restore.html">Store
         * and restore an AMI using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRestoreImageTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateRestoreImageTaskOutcome CreateRestoreImageTask(const Model::CreateRestoreImageTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateRestoreImageTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRestoreImageTaskRequestT = Model::CreateRestoreImageTaskRequest>
        Model::CreateRestoreImageTaskOutcomeCallable CreateRestoreImageTaskCallable(const CreateRestoreImageTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRestoreImageTask, request);
        }

        /**
         * An Async wrapper for CreateRestoreImageTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRestoreImageTaskRequestT = Model::CreateRestoreImageTaskRequest>
        void CreateRestoreImageTaskAsync(const CreateRestoreImageTaskRequestT& request, const CreateRestoreImageTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRestoreImageTask, request, handler, context);
        }

        /**
         * <p>Creates a route in a route table within a VPC.</p> <p>You must specify either
         * a destination CIDR block or a prefix list ID. You must also specify exactly one
         * of the resources from the parameter list.</p> <p>When determining how to route
         * traffic, we use the route with the most specific match. For example, traffic is
         * destined for the IPv4 address <code>192.0.2.3</code>, and the route table
         * includes the following two IPv4 routes:</p> <ul> <li> <p>
         * <code>192.0.2.0/24</code> (goes to some target A)</p> </li> <li> <p>
         * <code>192.0.2.0/28</code> (goes to some target B)</p> </li> </ul> <p>Both routes
         * apply to the traffic destined for <code>192.0.2.3</code>. However, the second
         * route in the list covers a smaller number of IP addresses and is therefore more
         * specific, so we use that route to determine where to target the traffic.</p>
         * <p>For more information about route tables, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html">Route
         * tables</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRoute">AWS API
         * Reference</a></p>
         */
        virtual Model::CreateRouteOutcome CreateRoute(const Model::CreateRouteRequest& request) const;

        /**
         * A Callable wrapper for CreateRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRouteRequestT = Model::CreateRouteRequest>
        Model::CreateRouteOutcomeCallable CreateRouteCallable(const CreateRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRoute, request);
        }

        /**
         * An Async wrapper for CreateRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRouteRequestT = Model::CreateRouteRequest>
        void CreateRouteAsync(const CreateRouteRequestT& request, const CreateRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRoute, request, handler, context);
        }

        /**
         * <p>Creates a new route server to manage dynamic routing in a VPC.</p> <p>Amazon
         * VPC Route Server simplifies routing for traffic between workloads that are
         * deployed within a VPC and its internet gateways. With this feature, VPC Route
         * Server dynamically updates VPC and internet gateway route tables with your
         * preferred IPv4 or IPv6 routes to achieve routing fault tolerance for those
         * workloads. This enables you to automatically reroute traffic within a VPC, which
         * increases the manageability of VPC routing and interoperability with third-party
         * workloads.</p> <p>Route server supports the follow route table types:</p> <ul>
         * <li> <p>VPC route tables not associated with subnets</p> </li> <li> <p>Subnet
         * route tables</p> </li> <li> <p>Internet gateway route tables</p> </li> </ul>
         * <p>Route server does not support route tables associated with virtual private
         * gateways. To propagate routes into a transit gateway route table, use <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html">Transit
         * Gateway Connect</a>.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRouteServer">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateRouteServerOutcome CreateRouteServer(const Model::CreateRouteServerRequest& request) const;

        /**
         * A Callable wrapper for CreateRouteServer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRouteServerRequestT = Model::CreateRouteServerRequest>
        Model::CreateRouteServerOutcomeCallable CreateRouteServerCallable(const CreateRouteServerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRouteServer, request);
        }

        /**
         * An Async wrapper for CreateRouteServer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRouteServerRequestT = Model::CreateRouteServerRequest>
        void CreateRouteServerAsync(const CreateRouteServerRequestT& request, const CreateRouteServerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRouteServer, request, handler, context);
        }

        /**
         * <p>Creates a new endpoint for a route server in a specified subnet.</p> <p>A
         * route server endpoint is an Amazon Web Services-managed component inside a
         * subnet that facilitates <a
         * href="https://en.wikipedia.org/wiki/Border_Gateway_Protocol">BGP (Border Gateway
         * Protocol)</a> connections between your route server and your BGP peers.</p>
         * <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRouteServerEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateRouteServerEndpointOutcome CreateRouteServerEndpoint(const Model::CreateRouteServerEndpointRequest& request) const;

        /**
         * A Callable wrapper for CreateRouteServerEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRouteServerEndpointRequestT = Model::CreateRouteServerEndpointRequest>
        Model::CreateRouteServerEndpointOutcomeCallable CreateRouteServerEndpointCallable(const CreateRouteServerEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRouteServerEndpoint, request);
        }

        /**
         * An Async wrapper for CreateRouteServerEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRouteServerEndpointRequestT = Model::CreateRouteServerEndpointRequest>
        void CreateRouteServerEndpointAsync(const CreateRouteServerEndpointRequestT& request, const CreateRouteServerEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRouteServerEndpoint, request, handler, context);
        }

        /**
         * <p>Creates a new BGP peer for a specified route server endpoint.</p> <p>A route
         * server peer is a session between a route server endpoint and the device deployed
         * in Amazon Web Services (such as a firewall appliance or other network security
         * function running on an EC2 instance). The device must meet these
         * requirements:</p> <ul> <li> <p>Have an elastic network interface in the VPC</p>
         * </li> <li> <p>Support BGP (Border Gateway Protocol)</p> </li> <li> <p>Can
         * initiate BGP sessions</p> </li> </ul> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRouteServerPeer">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateRouteServerPeerOutcome CreateRouteServerPeer(const Model::CreateRouteServerPeerRequest& request) const;

        /**
         * A Callable wrapper for CreateRouteServerPeer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRouteServerPeerRequestT = Model::CreateRouteServerPeerRequest>
        Model::CreateRouteServerPeerOutcomeCallable CreateRouteServerPeerCallable(const CreateRouteServerPeerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRouteServerPeer, request);
        }

        /**
         * An Async wrapper for CreateRouteServerPeer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRouteServerPeerRequestT = Model::CreateRouteServerPeerRequest>
        void CreateRouteServerPeerAsync(const CreateRouteServerPeerRequestT& request, const CreateRouteServerPeerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRouteServerPeer, request, handler, context);
        }

        /**
         * <p>Creates a route table for the specified VPC. After you create a route table,
         * you can add routes and associate the table with a subnet.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html">Route
         * tables</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateRouteTableOutcome CreateRouteTable(const Model::CreateRouteTableRequest& request) const;

        /**
         * A Callable wrapper for CreateRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateRouteTableRequestT = Model::CreateRouteTableRequest>
        Model::CreateRouteTableOutcomeCallable CreateRouteTableCallable(const CreateRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateRouteTable, request);
        }

        /**
         * An Async wrapper for CreateRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateRouteTableRequestT = Model::CreateRouteTableRequest>
        void CreateRouteTableAsync(const CreateRouteTableRequestT& request, const CreateRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateRouteTable, request, handler, context);
        }

        /**
         * <p>Creates a security group.</p> <p>A security group acts as a virtual firewall
         * for your instance to control inbound and outbound traffic. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-network-security.html">Amazon
         * EC2 security groups</a> in the <i>Amazon EC2 User Guide</i> and <a
         * href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_SecurityGroups.html">Security
         * groups for your VPC</a> in the <i>Amazon VPC User Guide</i>.</p> <p>When you
         * create a security group, you specify a friendly name of your choice. You can't
         * have two security groups for the same VPC with the same name.</p> <p>You have a
         * default security group for use in your VPC. If you don't specify a security
         * group when you launch an instance, the instance is launched into the appropriate
         * default security group. A default security group includes a default rule that
         * grants instances unrestricted network access to each other.</p> <p>You can add
         * or remove rules from your security groups using
         * <a>AuthorizeSecurityGroupIngress</a>, <a>AuthorizeSecurityGroupEgress</a>,
         * <a>RevokeSecurityGroupIngress</a>, and <a>RevokeSecurityGroupEgress</a>.</p>
         * <p>For more information about VPC security group limits, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/amazon-vpc-limits.html">Amazon
         * VPC Limits</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSecurityGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSecurityGroupOutcome CreateSecurityGroup(const Model::CreateSecurityGroupRequest& request) const;

        /**
         * A Callable wrapper for CreateSecurityGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSecurityGroupRequestT = Model::CreateSecurityGroupRequest>
        Model::CreateSecurityGroupOutcomeCallable CreateSecurityGroupCallable(const CreateSecurityGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSecurityGroup, request);
        }

        /**
         * An Async wrapper for CreateSecurityGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSecurityGroupRequestT = Model::CreateSecurityGroupRequest>
        void CreateSecurityGroupAsync(const CreateSecurityGroupRequestT& request, const CreateSecurityGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSecurityGroup, request, handler, context);
        }

        /**
         * <p>Creates a snapshot of an EBS volume and stores it in Amazon S3. You can use
         * snapshots for backups, to make copies of EBS volumes, and to save data before
         * shutting down an instance.</p> <p>The location of the source EBS volume
         * determines where you can create the snapshot.</p> <ul> <li> <p>If the source
         * volume is in a Region, you must create the snapshot in the same Region as the
         * volume.</p> </li> <li> <p>If the source volume is in a Local Zone, you can
         * create the snapshot in the same Local Zone or in its parent Amazon Web Services
         * Region.</p> </li> <li> <p>If the source volume is on an Outpost, you can create
         * the snapshot on the same Outpost or in its parent Amazon Web Services
         * Region.</p> </li> </ul> <p>When a snapshot is created, any Amazon Web Services
         * Marketplace product codes that are associated with the source volume are
         * propagated to the snapshot.</p> <p>You can take a snapshot of an attached volume
         * that is in use. However, snapshots only capture data that has been written to
         * your Amazon EBS volume at the time the snapshot command is issued; this might
         * exclude any data that has been cached by any applications or the operating
         * system. If you can pause any file systems on the volume long enough to take a
         * snapshot, your snapshot should be complete. However, if you cannot pause all
         * file writes to the volume, you should unmount the volume from within the
         * instance, issue the snapshot command, and then remount the volume to ensure a
         * consistent and complete snapshot. You may remount and use your volume while the
         * snapshot status is <code>pending</code>.</p> <p>When you create a snapshot for
         * an EBS volume that serves as a root device, we recommend that you stop the
         * instance before taking the snapshot.</p> <p>Snapshots that are taken from
         * encrypted volumes are automatically encrypted. Volumes that are created from
         * encrypted snapshots are also automatically encrypted. Your encrypted volumes and
         * any associated snapshots always remain protected. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSnapshot">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSnapshotOutcome CreateSnapshot(const Model::CreateSnapshotRequest& request) const;

        /**
         * A Callable wrapper for CreateSnapshot that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSnapshotRequestT = Model::CreateSnapshotRequest>
        Model::CreateSnapshotOutcomeCallable CreateSnapshotCallable(const CreateSnapshotRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSnapshot, request);
        }

        /**
         * An Async wrapper for CreateSnapshot that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSnapshotRequestT = Model::CreateSnapshotRequest>
        void CreateSnapshotAsync(const CreateSnapshotRequestT& request, const CreateSnapshotResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSnapshot, request, handler, context);
        }

        /**
         * <p>Creates crash-consistent snapshots of multiple EBS volumes attached to an
         * Amazon EC2 instance. Volumes are chosen by specifying an instance. Each volume
         * attached to the specified instance will produce one snapshot that is
         * crash-consistent across the instance. You can include all of the volumes
         * currently attached to the instance, or you can exclude the root volume or
         * specific data (non-root) volumes from the multi-volume snapshot set.</p> <p>The
         * location of the source instance determines where you can create the
         * snapshots.</p> <ul> <li> <p>If the source instance is in a Region, you must
         * create the snapshots in the same Region as the instance.</p> </li> <li> <p>If
         * the source instance is in a Local Zone, you can create the snapshots in the same
         * Local Zone or in its parent Amazon Web Services Region.</p> </li> <li> <p>If the
         * source instance is on an Outpost, you can create the snapshots on the same
         * Outpost or in its parent Amazon Web Services Region.</p> </li> </ul><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSnapshots">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSnapshotsOutcome CreateSnapshots(const Model::CreateSnapshotsRequest& request) const;

        /**
         * A Callable wrapper for CreateSnapshots that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSnapshotsRequestT = Model::CreateSnapshotsRequest>
        Model::CreateSnapshotsOutcomeCallable CreateSnapshotsCallable(const CreateSnapshotsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSnapshots, request);
        }

        /**
         * An Async wrapper for CreateSnapshots that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSnapshotsRequestT = Model::CreateSnapshotsRequest>
        void CreateSnapshotsAsync(const CreateSnapshotsRequestT& request, const CreateSnapshotsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSnapshots, request, handler, context);
        }

        /**
         * <p>Creates a data feed for Spot Instances, enabling you to view Spot Instance
         * usage logs. You can create one data feed per Amazon Web Services account. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-data-feeds.html">Spot
         * Instance data feed</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSpotDatafeedSubscription">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSpotDatafeedSubscriptionOutcome CreateSpotDatafeedSubscription(const Model::CreateSpotDatafeedSubscriptionRequest& request) const;

        /**
         * A Callable wrapper for CreateSpotDatafeedSubscription that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSpotDatafeedSubscriptionRequestT = Model::CreateSpotDatafeedSubscriptionRequest>
        Model::CreateSpotDatafeedSubscriptionOutcomeCallable CreateSpotDatafeedSubscriptionCallable(const CreateSpotDatafeedSubscriptionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSpotDatafeedSubscription, request);
        }

        /**
         * An Async wrapper for CreateSpotDatafeedSubscription that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSpotDatafeedSubscriptionRequestT = Model::CreateSpotDatafeedSubscriptionRequest>
        void CreateSpotDatafeedSubscriptionAsync(const CreateSpotDatafeedSubscriptionRequestT& request, const CreateSpotDatafeedSubscriptionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSpotDatafeedSubscription, request, handler, context);
        }

        /**
         * <p>Stores an AMI as a single object in an Amazon S3 bucket.</p> <p>To use this
         * API, you must have the required permissions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/work-with-ami-store-restore.html#ami-s3-permissions">Permissions
         * for storing and restoring AMIs using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-store-restore.html">Store
         * and restore an AMI using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateStoreImageTask">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateStoreImageTaskOutcome CreateStoreImageTask(const Model::CreateStoreImageTaskRequest& request) const;

        /**
         * A Callable wrapper for CreateStoreImageTask that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateStoreImageTaskRequestT = Model::CreateStoreImageTaskRequest>
        Model::CreateStoreImageTaskOutcomeCallable CreateStoreImageTaskCallable(const CreateStoreImageTaskRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateStoreImageTask, request);
        }

        /**
         * An Async wrapper for CreateStoreImageTask that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateStoreImageTaskRequestT = Model::CreateStoreImageTaskRequest>
        void CreateStoreImageTaskAsync(const CreateStoreImageTaskRequestT& request, const CreateStoreImageTaskResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateStoreImageTask, request, handler, context);
        }

        /**
         * <p>Creates a subnet in the specified VPC. For an IPv4 only subnet, specify an
         * IPv4 CIDR block. If the VPC has an IPv6 CIDR block, you can create an IPv6 only
         * subnet or a dual stack subnet instead. For an IPv6 only subnet, specify an IPv6
         * CIDR block. For a dual stack subnet, specify both an IPv4 CIDR block and an IPv6
         * CIDR block.</p> <p>A subnet CIDR block must not overlap the CIDR block of an
         * existing subnet in the VPC. After you create a subnet, you can't change its CIDR
         * block.</p> <p>The allowed size for an IPv4 subnet is between a /28 netmask (16
         * IP addresses) and a /16 netmask (65,536 IP addresses). Amazon Web Services
         * reserves both the first four and the last IPv4 address in each subnet's CIDR
         * block. They're not available for your use.</p> <p>If you've associated an IPv6
         * CIDR block with your VPC, you can associate an IPv6 CIDR block with a subnet
         * when you create it. </p> <p>If you add more than one subnet to a VPC, they're
         * set up in a star topology with a logical router in the middle.</p> <p>When you
         * stop an instance in a subnet, it retains its private IPv4 address. It's
         * therefore possible to have a subnet with no running instances (they're all
         * stopped), but no remaining IP addresses available.</p> <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/configure-subnets.html">Subnets</a>
         * in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSubnet">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSubnetOutcome CreateSubnet(const Model::CreateSubnetRequest& request) const;

        /**
         * A Callable wrapper for CreateSubnet that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSubnetRequestT = Model::CreateSubnetRequest>
        Model::CreateSubnetOutcomeCallable CreateSubnetCallable(const CreateSubnetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSubnet, request);
        }

        /**
         * An Async wrapper for CreateSubnet that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSubnetRequestT = Model::CreateSubnetRequest>
        void CreateSubnetAsync(const CreateSubnetRequestT& request, const CreateSubnetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSubnet, request, handler, context);
        }

        /**
         * <p>Creates a subnet CIDR reservation. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/subnet-cidr-reservation.html">Subnet
         * CIDR reservations</a> in the <i>Amazon VPC User Guide</i> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/work-with-prefixes.html">Manage
         * prefixes for your network interfaces</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateSubnetCidrReservation">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateSubnetCidrReservationOutcome CreateSubnetCidrReservation(const Model::CreateSubnetCidrReservationRequest& request) const;

        /**
         * A Callable wrapper for CreateSubnetCidrReservation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateSubnetCidrReservationRequestT = Model::CreateSubnetCidrReservationRequest>
        Model::CreateSubnetCidrReservationOutcomeCallable CreateSubnetCidrReservationCallable(const CreateSubnetCidrReservationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateSubnetCidrReservation, request);
        }

        /**
         * An Async wrapper for CreateSubnetCidrReservation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateSubnetCidrReservationRequestT = Model::CreateSubnetCidrReservationRequest>
        void CreateSubnetCidrReservationAsync(const CreateSubnetCidrReservationRequestT& request, const CreateSubnetCidrReservationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateSubnetCidrReservation, request, handler, context);
        }

        /**
         * <p>Adds or overwrites only the specified tags for the specified Amazon EC2
         * resource or resources. When you specify an existing tag key, the value is
         * overwritten with the new value. Each resource can have a maximum of 50 tags.
         * Each tag consists of a key and optional value. Tag keys must be unique per
         * resource.</p> <p>For more information about tags, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tag
         * your Amazon EC2 resources</a> in the <i>Amazon Elastic Compute Cloud User
         * Guide</i>. For more information about creating IAM policies that control users'
         * access to resources based on tags, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-supported-iam-actions-resources.html">Supported
         * resource-level permissions for Amazon EC2 API actions</a> in the <i>Amazon
         * Elastic Compute Cloud User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTags">AWS API
         * Reference</a></p>
         */
        virtual Model::CreateTagsOutcome CreateTags(const Model::CreateTagsRequest& request) const;

        /**
         * A Callable wrapper for CreateTags that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTagsRequestT = Model::CreateTagsRequest>
        Model::CreateTagsOutcomeCallable CreateTagsCallable(const CreateTagsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTags, request);
        }

        /**
         * An Async wrapper for CreateTags that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTagsRequestT = Model::CreateTagsRequest>
        void CreateTagsAsync(const CreateTagsRequestT& request, const CreateTagsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTags, request, handler, context);
        }

        /**
         * <p>Creates a Traffic Mirror filter.</p> <p>A Traffic Mirror filter is a set of
         * rules that defines the traffic to mirror.</p> <p>By default, no traffic is
         * mirrored. To mirror traffic, use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.htm">CreateTrafficMirrorFilterRule</a>
         * to add Traffic Mirror rules to the filter. The rules you add define what traffic
         * gets mirrored. You can also use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html">ModifyTrafficMirrorFilterNetworkServices</a>
         * to mirror supported network services.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTrafficMirrorFilter">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTrafficMirrorFilterOutcome CreateTrafficMirrorFilter(const Model::CreateTrafficMirrorFilterRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateTrafficMirrorFilter that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTrafficMirrorFilterRequestT = Model::CreateTrafficMirrorFilterRequest>
        Model::CreateTrafficMirrorFilterOutcomeCallable CreateTrafficMirrorFilterCallable(const CreateTrafficMirrorFilterRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateTrafficMirrorFilter, request);
        }

        /**
         * An Async wrapper for CreateTrafficMirrorFilter that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTrafficMirrorFilterRequestT = Model::CreateTrafficMirrorFilterRequest>
        void CreateTrafficMirrorFilterAsync(const CreateTrafficMirrorFilterResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateTrafficMirrorFilterRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateTrafficMirrorFilter, request, handler, context);
        }

        /**
         * <p>Creates a Traffic Mirror filter rule.</p> <p>A Traffic Mirror rule defines
         * the Traffic Mirror source traffic to mirror.</p> <p>You need the Traffic Mirror
         * filter ID when you create the rule.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTrafficMirrorFilterRule">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTrafficMirrorFilterRuleOutcome CreateTrafficMirrorFilterRule(const Model::CreateTrafficMirrorFilterRuleRequest& request) const;

        /**
         * A Callable wrapper for CreateTrafficMirrorFilterRule that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTrafficMirrorFilterRuleRequestT = Model::CreateTrafficMirrorFilterRuleRequest>
        Model::CreateTrafficMirrorFilterRuleOutcomeCallable CreateTrafficMirrorFilterRuleCallable(const CreateTrafficMirrorFilterRuleRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTrafficMirrorFilterRule, request);
        }

        /**
         * An Async wrapper for CreateTrafficMirrorFilterRule that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTrafficMirrorFilterRuleRequestT = Model::CreateTrafficMirrorFilterRuleRequest>
        void CreateTrafficMirrorFilterRuleAsync(const CreateTrafficMirrorFilterRuleRequestT& request, const CreateTrafficMirrorFilterRuleResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTrafficMirrorFilterRule, request, handler, context);
        }

        /**
         * <p>Creates a Traffic Mirror session.</p> <p>A Traffic Mirror session actively
         * copies packets from a Traffic Mirror source to a Traffic Mirror target. Create a
         * filter, and then assign it to the session to define a subset of the traffic to
         * mirror, for example all TCP traffic.</p> <p>The Traffic Mirror source and the
         * Traffic Mirror target (monitoring appliances) can be in the same VPC, or in a
         * different VPC connected via VPC peering or a transit gateway. </p> <p>By
         * default, no traffic is mirrored. Use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html">CreateTrafficMirrorFilter</a>
         * to create filter rules that specify the traffic to mirror.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTrafficMirrorSession">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTrafficMirrorSessionOutcome CreateTrafficMirrorSession(const Model::CreateTrafficMirrorSessionRequest& request) const;

        /**
         * A Callable wrapper for CreateTrafficMirrorSession that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTrafficMirrorSessionRequestT = Model::CreateTrafficMirrorSessionRequest>
        Model::CreateTrafficMirrorSessionOutcomeCallable CreateTrafficMirrorSessionCallable(const CreateTrafficMirrorSessionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTrafficMirrorSession, request);
        }

        /**
         * An Async wrapper for CreateTrafficMirrorSession that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTrafficMirrorSessionRequestT = Model::CreateTrafficMirrorSessionRequest>
        void CreateTrafficMirrorSessionAsync(const CreateTrafficMirrorSessionRequestT& request, const CreateTrafficMirrorSessionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTrafficMirrorSession, request, handler, context);
        }

        /**
         * <p>Creates a target for your Traffic Mirror session.</p> <p>A Traffic Mirror
         * target is the destination for mirrored traffic. The Traffic Mirror source and
         * the Traffic Mirror target (monitoring appliances) can be in the same VPC, or in
         * different VPCs connected via VPC peering or a transit gateway.</p> <p>A Traffic
         * Mirror target can be a network interface, a Network Load Balancer, or a Gateway
         * Load Balancer endpoint.</p> <p>To use the target in a Traffic Mirror session,
         * use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.htm">CreateTrafficMirrorSession</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTrafficMirrorTarget">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTrafficMirrorTargetOutcome CreateTrafficMirrorTarget(const Model::CreateTrafficMirrorTargetRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateTrafficMirrorTarget that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTrafficMirrorTargetRequestT = Model::CreateTrafficMirrorTargetRequest>
        Model::CreateTrafficMirrorTargetOutcomeCallable CreateTrafficMirrorTargetCallable(const CreateTrafficMirrorTargetRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateTrafficMirrorTarget, request);
        }

        /**
         * An Async wrapper for CreateTrafficMirrorTarget that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTrafficMirrorTargetRequestT = Model::CreateTrafficMirrorTargetRequest>
        void CreateTrafficMirrorTargetAsync(const CreateTrafficMirrorTargetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateTrafficMirrorTargetRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateTrafficMirrorTarget, request, handler, context);
        }

        /**
         * <p>Creates a transit gateway.</p> <p>You can use a transit gateway to
         * interconnect your virtual private clouds (VPC) and on-premises networks. After
         * the transit gateway enters the <code>available</code> state, you can attach your
         * VPCs and VPN connections to the transit gateway.</p> <p>To attach your VPCs, use
         * <a>CreateTransitGatewayVpcAttachment</a>.</p> <p>To attach a VPN connection, use
         * <a>CreateCustomerGateway</a> to create a customer gateway and specify the ID of
         * the customer gateway and the ID of the transit gateway in a call to
         * <a>CreateVpnConnection</a>.</p> <p>When you create a transit gateway, we create
         * a default transit gateway route table and use it as the default association
         * route table and the default propagation route table. You can use
         * <a>CreateTransitGatewayRouteTable</a> to create additional transit gateway route
         * tables. If you disable automatic route propagation, we do not create a default
         * transit gateway route table. You can use
         * <a>EnableTransitGatewayRouteTablePropagation</a> to propagate routes from a
         * resource attachment to a transit gateway route table. If you disable automatic
         * associations, you can use <a>AssociateTransitGatewayRouteTable</a> to associate
         * a resource attachment with a transit gateway route table.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayOutcome CreateTransitGateway(const Model::CreateTransitGatewayRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateTransitGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayRequestT = Model::CreateTransitGatewayRequest>
        Model::CreateTransitGatewayOutcomeCallable CreateTransitGatewayCallable(const CreateTransitGatewayRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGateway, request);
        }

        /**
         * An Async wrapper for CreateTransitGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayRequestT = Model::CreateTransitGatewayRequest>
        void CreateTransitGatewayAsync(const CreateTransitGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateTransitGatewayRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGateway, request, handler, context);
        }

        /**
         * <p>Creates a Connect attachment from a specified transit gateway attachment. A
         * Connect attachment is a GRE-based tunnel attachment that you can use to
         * establish a connection between a transit gateway and an appliance.</p> <p>A
         * Connect attachment uses an existing VPC or Amazon Web Services Direct Connect
         * attachment as the underlying transport mechanism.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayConnect">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayConnectOutcome CreateTransitGatewayConnect(const Model::CreateTransitGatewayConnectRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayConnect that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayConnectRequestT = Model::CreateTransitGatewayConnectRequest>
        Model::CreateTransitGatewayConnectOutcomeCallable CreateTransitGatewayConnectCallable(const CreateTransitGatewayConnectRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayConnect, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayConnect that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayConnectRequestT = Model::CreateTransitGatewayConnectRequest>
        void CreateTransitGatewayConnectAsync(const CreateTransitGatewayConnectRequestT& request, const CreateTransitGatewayConnectResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayConnect, request, handler, context);
        }

        /**
         * <p>Creates a Connect peer for a specified transit gateway Connect attachment
         * between a transit gateway and an appliance.</p> <p>The peer address and transit
         * gateway address must be the same IP address family (IPv4 or IPv6).</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html#tgw-connect-peer">Connect
         * peers</a> in the <i>Amazon Web Services Transit Gateways
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayConnectPeer">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayConnectPeerOutcome CreateTransitGatewayConnectPeer(const Model::CreateTransitGatewayConnectPeerRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayConnectPeer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayConnectPeerRequestT = Model::CreateTransitGatewayConnectPeerRequest>
        Model::CreateTransitGatewayConnectPeerOutcomeCallable CreateTransitGatewayConnectPeerCallable(const CreateTransitGatewayConnectPeerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayConnectPeer, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayConnectPeer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayConnectPeerRequestT = Model::CreateTransitGatewayConnectPeerRequest>
        void CreateTransitGatewayConnectPeerAsync(const CreateTransitGatewayConnectPeerRequestT& request, const CreateTransitGatewayConnectPeerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayConnectPeer, request, handler, context);
        }

        /**
         * <p>Creates a multicast domain using the specified transit gateway.</p> <p>The
         * transit gateway must be in the available state before you create a domain. Use
         * <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html">DescribeTransitGateways</a>
         * to see the state of transit gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayMulticastDomain">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayMulticastDomainOutcome CreateTransitGatewayMulticastDomain(const Model::CreateTransitGatewayMulticastDomainRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayMulticastDomain that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayMulticastDomainRequestT = Model::CreateTransitGatewayMulticastDomainRequest>
        Model::CreateTransitGatewayMulticastDomainOutcomeCallable CreateTransitGatewayMulticastDomainCallable(const CreateTransitGatewayMulticastDomainRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayMulticastDomain, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayMulticastDomain that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayMulticastDomainRequestT = Model::CreateTransitGatewayMulticastDomainRequest>
        void CreateTransitGatewayMulticastDomainAsync(const CreateTransitGatewayMulticastDomainRequestT& request, const CreateTransitGatewayMulticastDomainResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayMulticastDomain, request, handler, context);
        }

        /**
         * <p>Requests a transit gateway peering attachment between the specified transit
         * gateway (requester) and a peer transit gateway (accepter). The peer transit
         * gateway can be in your account or a different Amazon Web Services account.</p>
         * <p>After you create the peering attachment, the owner of the accepter transit
         * gateway must accept the attachment request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayPeeringAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayPeeringAttachmentOutcome CreateTransitGatewayPeeringAttachment(const Model::CreateTransitGatewayPeeringAttachmentRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayPeeringAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayPeeringAttachmentRequestT = Model::CreateTransitGatewayPeeringAttachmentRequest>
        Model::CreateTransitGatewayPeeringAttachmentOutcomeCallable CreateTransitGatewayPeeringAttachmentCallable(const CreateTransitGatewayPeeringAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayPeeringAttachment, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayPeeringAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayPeeringAttachmentRequestT = Model::CreateTransitGatewayPeeringAttachmentRequest>
        void CreateTransitGatewayPeeringAttachmentAsync(const CreateTransitGatewayPeeringAttachmentRequestT& request, const CreateTransitGatewayPeeringAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayPeeringAttachment, request, handler, context);
        }

        /**
         * <p>Creates a transit gateway policy table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayPolicyTable">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayPolicyTableOutcome CreateTransitGatewayPolicyTable(const Model::CreateTransitGatewayPolicyTableRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayPolicyTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayPolicyTableRequestT = Model::CreateTransitGatewayPolicyTableRequest>
        Model::CreateTransitGatewayPolicyTableOutcomeCallable CreateTransitGatewayPolicyTableCallable(const CreateTransitGatewayPolicyTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayPolicyTable, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayPolicyTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayPolicyTableRequestT = Model::CreateTransitGatewayPolicyTableRequest>
        void CreateTransitGatewayPolicyTableAsync(const CreateTransitGatewayPolicyTableRequestT& request, const CreateTransitGatewayPolicyTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayPolicyTable, request, handler, context);
        }

        /**
         * <p>Creates a reference (route) to a prefix list in a specified transit gateway
         * route table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayPrefixListReference">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayPrefixListReferenceOutcome CreateTransitGatewayPrefixListReference(const Model::CreateTransitGatewayPrefixListReferenceRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayPrefixListReference that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayPrefixListReferenceRequestT = Model::CreateTransitGatewayPrefixListReferenceRequest>
        Model::CreateTransitGatewayPrefixListReferenceOutcomeCallable CreateTransitGatewayPrefixListReferenceCallable(const CreateTransitGatewayPrefixListReferenceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayPrefixListReference, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayPrefixListReference that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayPrefixListReferenceRequestT = Model::CreateTransitGatewayPrefixListReferenceRequest>
        void CreateTransitGatewayPrefixListReferenceAsync(const CreateTransitGatewayPrefixListReferenceRequestT& request, const CreateTransitGatewayPrefixListReferenceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayPrefixListReference, request, handler, context);
        }

        /**
         * <p>Creates a static route for the specified transit gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayRouteOutcome CreateTransitGatewayRoute(const Model::CreateTransitGatewayRouteRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayRouteRequestT = Model::CreateTransitGatewayRouteRequest>
        Model::CreateTransitGatewayRouteOutcomeCallable CreateTransitGatewayRouteCallable(const CreateTransitGatewayRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayRoute, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayRouteRequestT = Model::CreateTransitGatewayRouteRequest>
        void CreateTransitGatewayRouteAsync(const CreateTransitGatewayRouteRequestT& request, const CreateTransitGatewayRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayRoute, request, handler, context);
        }

        /**
         * <p>Creates a route table for the specified transit gateway.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayRouteTableOutcome CreateTransitGatewayRouteTable(const Model::CreateTransitGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayRouteTableRequestT = Model::CreateTransitGatewayRouteTableRequest>
        Model::CreateTransitGatewayRouteTableOutcomeCallable CreateTransitGatewayRouteTableCallable(const CreateTransitGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayRouteTableRequestT = Model::CreateTransitGatewayRouteTableRequest>
        void CreateTransitGatewayRouteTableAsync(const CreateTransitGatewayRouteTableRequestT& request, const CreateTransitGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayRouteTable, request, handler, context);
        }

        /**
         * <p>Advertises a new transit gateway route table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayRouteTableAnnouncement">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayRouteTableAnnouncementOutcome CreateTransitGatewayRouteTableAnnouncement(const Model::CreateTransitGatewayRouteTableAnnouncementRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayRouteTableAnnouncement that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayRouteTableAnnouncementRequestT = Model::CreateTransitGatewayRouteTableAnnouncementRequest>
        Model::CreateTransitGatewayRouteTableAnnouncementOutcomeCallable CreateTransitGatewayRouteTableAnnouncementCallable(const CreateTransitGatewayRouteTableAnnouncementRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayRouteTableAnnouncement, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayRouteTableAnnouncement that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayRouteTableAnnouncementRequestT = Model::CreateTransitGatewayRouteTableAnnouncementRequest>
        void CreateTransitGatewayRouteTableAnnouncementAsync(const CreateTransitGatewayRouteTableAnnouncementRequestT& request, const CreateTransitGatewayRouteTableAnnouncementResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayRouteTableAnnouncement, request, handler, context);
        }

        /**
         * <p>Attaches the specified VPC to the specified transit gateway.</p> <p>If you
         * attach a VPC with a CIDR range that overlaps the CIDR range of a VPC that is
         * already attached, the new VPC CIDR range is not propagated to the default
         * propagation route table.</p> <p>To send VPC traffic to an attached transit
         * gateway, add a route to the VPC route table using
         * <a>CreateRoute</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateTransitGatewayVpcAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateTransitGatewayVpcAttachmentOutcome CreateTransitGatewayVpcAttachment(const Model::CreateTransitGatewayVpcAttachmentRequest& request) const;

        /**
         * A Callable wrapper for CreateTransitGatewayVpcAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateTransitGatewayVpcAttachmentRequestT = Model::CreateTransitGatewayVpcAttachmentRequest>
        Model::CreateTransitGatewayVpcAttachmentOutcomeCallable CreateTransitGatewayVpcAttachmentCallable(const CreateTransitGatewayVpcAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateTransitGatewayVpcAttachment, request);
        }

        /**
         * An Async wrapper for CreateTransitGatewayVpcAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateTransitGatewayVpcAttachmentRequestT = Model::CreateTransitGatewayVpcAttachmentRequest>
        void CreateTransitGatewayVpcAttachmentAsync(const CreateTransitGatewayVpcAttachmentRequestT& request, const CreateTransitGatewayVpcAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateTransitGatewayVpcAttachment, request, handler, context);
        }

        /**
         * <p>An Amazon Web Services Verified Access endpoint is where you define your
         * application along with an optional endpoint-level access policy.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVerifiedAccessEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVerifiedAccessEndpointOutcome CreateVerifiedAccessEndpoint(const Model::CreateVerifiedAccessEndpointRequest& request) const;

        /**
         * A Callable wrapper for CreateVerifiedAccessEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVerifiedAccessEndpointRequestT = Model::CreateVerifiedAccessEndpointRequest>
        Model::CreateVerifiedAccessEndpointOutcomeCallable CreateVerifiedAccessEndpointCallable(const CreateVerifiedAccessEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVerifiedAccessEndpoint, request);
        }

        /**
         * An Async wrapper for CreateVerifiedAccessEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVerifiedAccessEndpointRequestT = Model::CreateVerifiedAccessEndpointRequest>
        void CreateVerifiedAccessEndpointAsync(const CreateVerifiedAccessEndpointRequestT& request, const CreateVerifiedAccessEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVerifiedAccessEndpoint, request, handler, context);
        }

        /**
         * <p>An Amazon Web Services Verified Access group is a collection of Amazon Web
         * Services Verified Access endpoints who's associated applications have similar
         * security requirements. Each instance within a Verified Access group shares an
         * Verified Access policy. For example, you can group all Verified Access instances
         * associated with "sales" applications together and use one common Verified Access
         * policy.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVerifiedAccessGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVerifiedAccessGroupOutcome CreateVerifiedAccessGroup(const Model::CreateVerifiedAccessGroupRequest& request) const;

        /**
         * A Callable wrapper for CreateVerifiedAccessGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVerifiedAccessGroupRequestT = Model::CreateVerifiedAccessGroupRequest>
        Model::CreateVerifiedAccessGroupOutcomeCallable CreateVerifiedAccessGroupCallable(const CreateVerifiedAccessGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVerifiedAccessGroup, request);
        }

        /**
         * An Async wrapper for CreateVerifiedAccessGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVerifiedAccessGroupRequestT = Model::CreateVerifiedAccessGroupRequest>
        void CreateVerifiedAccessGroupAsync(const CreateVerifiedAccessGroupRequestT& request, const CreateVerifiedAccessGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVerifiedAccessGroup, request, handler, context);
        }

        /**
         * <p>An Amazon Web Services Verified Access instance is a regional entity that
         * evaluates application requests and grants access only when your security
         * requirements are met.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVerifiedAccessInstance">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVerifiedAccessInstanceOutcome CreateVerifiedAccessInstance(const Model::CreateVerifiedAccessInstanceRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateVerifiedAccessInstance that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVerifiedAccessInstanceRequestT = Model::CreateVerifiedAccessInstanceRequest>
        Model::CreateVerifiedAccessInstanceOutcomeCallable CreateVerifiedAccessInstanceCallable(const CreateVerifiedAccessInstanceRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateVerifiedAccessInstance, request);
        }

        /**
         * An Async wrapper for CreateVerifiedAccessInstance that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVerifiedAccessInstanceRequestT = Model::CreateVerifiedAccessInstanceRequest>
        void CreateVerifiedAccessInstanceAsync(const CreateVerifiedAccessInstanceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateVerifiedAccessInstanceRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateVerifiedAccessInstance, request, handler, context);
        }

        /**
         * <p>A trust provider is a third-party entity that creates, maintains, and manages
         * identity information for users and devices. When an application request is made,
         * the identity information sent by the trust provider is evaluated by Verified
         * Access before allowing or denying the application request.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVerifiedAccessTrustProvider">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVerifiedAccessTrustProviderOutcome CreateVerifiedAccessTrustProvider(const Model::CreateVerifiedAccessTrustProviderRequest& request) const;

        /**
         * A Callable wrapper for CreateVerifiedAccessTrustProvider that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVerifiedAccessTrustProviderRequestT = Model::CreateVerifiedAccessTrustProviderRequest>
        Model::CreateVerifiedAccessTrustProviderOutcomeCallable CreateVerifiedAccessTrustProviderCallable(const CreateVerifiedAccessTrustProviderRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVerifiedAccessTrustProvider, request);
        }

        /**
         * An Async wrapper for CreateVerifiedAccessTrustProvider that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVerifiedAccessTrustProviderRequestT = Model::CreateVerifiedAccessTrustProviderRequest>
        void CreateVerifiedAccessTrustProviderAsync(const CreateVerifiedAccessTrustProviderRequestT& request, const CreateVerifiedAccessTrustProviderResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVerifiedAccessTrustProvider, request, handler, context);
        }

        /**
         * <p>Creates an EBS volume that can be attached to an instance in the same
         * Availability Zone.</p> <p>You can create a new empty volume or restore a volume
         * from an EBS snapshot. Any Amazon Web Services Marketplace product codes from the
         * snapshot are propagated to the volume.</p> <p>You can create encrypted volumes.
         * Encrypted volumes must be attached to instances that support Amazon EBS
         * encryption. Volumes that are created from encrypted snapshots are also
         * automatically encrypted. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p> <p>You can tag your
         * volumes during creation. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tag
         * your Amazon EC2 resources</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-creating-volume.html">Create
         * an Amazon EBS volume</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVolume">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVolumeOutcome CreateVolume(const Model::CreateVolumeRequest& request) const;

        /**
         * A Callable wrapper for CreateVolume that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVolumeRequestT = Model::CreateVolumeRequest>
        Model::CreateVolumeOutcomeCallable CreateVolumeCallable(const CreateVolumeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVolume, request);
        }

        /**
         * An Async wrapper for CreateVolume that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVolumeRequestT = Model::CreateVolumeRequest>
        void CreateVolumeAsync(const CreateVolumeRequestT& request, const CreateVolumeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVolume, request, handler, context);
        }

        /**
         * <p>Creates a VPC with the specified CIDR blocks. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html">IP
         * addressing for your VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p> <p>You can optionally request an IPv6 CIDR block for the VPC. You
         * can request an Amazon-provided IPv6 CIDR block from Amazon's pool of IPv6
         * addresses or an IPv6 CIDR block from an IPv6 address pool that you provisioned
         * through bring your own IP addresses (<a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html">BYOIP</a>).</p>
         * <p>By default, each instance that you launch in the VPC has the default DHCP
         * options, which include only a default DNS server that we provide
         * (AmazonProvidedDNS). For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html">DHCP
         * option sets</a> in the <i>Amazon VPC User Guide</i>.</p> <p>You can specify the
         * instance tenancy value for the VPC when you create it. You can't change this
         * value for the VPC after you create it. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-instance.html">Dedicated
         * Instances</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpc">AWS API
         * Reference</a></p>
         */
        virtual Model::CreateVpcOutcome CreateVpc(const Model::CreateVpcRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcRequestT = Model::CreateVpcRequest>
        Model::CreateVpcOutcomeCallable CreateVpcCallable(const CreateVpcRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateVpc, request);
        }

        /**
         * An Async wrapper for CreateVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcRequestT = Model::CreateVpcRequest>
        void CreateVpcAsync(const CreateVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateVpcRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateVpc, request, handler, context);
        }

        /**
         * <p>Create a VPC Block Public Access (BPA) exclusion. A VPC BPA exclusion is a
         * mode that can be applied to a single VPC or subnet that exempts it from the
         * account’s BPA mode and will allow bidirectional or egress-only access. You can
         * create BPA exclusions for VPCs and subnets even when BPA is not enabled on the
         * account to ensure that there is no traffic disruption to the exclusions when VPC
         * BPA is turned on. To learn more about VPC BPA, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-vpc-bpa.html">Block
         * public access to VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpcBlockPublicAccessExclusion">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpcBlockPublicAccessExclusionOutcome CreateVpcBlockPublicAccessExclusion(const Model::CreateVpcBlockPublicAccessExclusionRequest& request) const;

        /**
         * A Callable wrapper for CreateVpcBlockPublicAccessExclusion that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcBlockPublicAccessExclusionRequestT = Model::CreateVpcBlockPublicAccessExclusionRequest>
        Model::CreateVpcBlockPublicAccessExclusionOutcomeCallable CreateVpcBlockPublicAccessExclusionCallable(const CreateVpcBlockPublicAccessExclusionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpcBlockPublicAccessExclusion, request);
        }

        /**
         * An Async wrapper for CreateVpcBlockPublicAccessExclusion that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcBlockPublicAccessExclusionRequestT = Model::CreateVpcBlockPublicAccessExclusionRequest>
        void CreateVpcBlockPublicAccessExclusionAsync(const CreateVpcBlockPublicAccessExclusionRequestT& request, const CreateVpcBlockPublicAccessExclusionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpcBlockPublicAccessExclusion, request, handler, context);
        }

        /**
         * <p>Creates a VPC endpoint. A VPC endpoint provides a private connection between
         * the specified VPC and the specified endpoint service. You can use an endpoint
         * service provided by Amazon Web Services, an Amazon Web Services Marketplace
         * Partner, or another Amazon Web Services account. For more information, see the
         * <a href="https://docs.aws.amazon.com/vpc/latest/privatelink/">Amazon Web
         * Services PrivateLink User Guide</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpcEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpcEndpointOutcome CreateVpcEndpoint(const Model::CreateVpcEndpointRequest& request) const;

        /**
         * A Callable wrapper for CreateVpcEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcEndpointRequestT = Model::CreateVpcEndpointRequest>
        Model::CreateVpcEndpointOutcomeCallable CreateVpcEndpointCallable(const CreateVpcEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpcEndpoint, request);
        }

        /**
         * An Async wrapper for CreateVpcEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcEndpointRequestT = Model::CreateVpcEndpointRequest>
        void CreateVpcEndpointAsync(const CreateVpcEndpointRequestT& request, const CreateVpcEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpcEndpoint, request, handler, context);
        }

        /**
         * <p>Creates a connection notification for a specified VPC endpoint or VPC
         * endpoint service. A connection notification notifies you of specific endpoint
         * events. You must create an SNS topic to receive notifications. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Creating an
         * Amazon SNS topic</a> in the <i>Amazon SNS Developer Guide</i>.</p> <p>You can
         * create a connection notification for interface endpoints only.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpcEndpointConnectionNotification">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpcEndpointConnectionNotificationOutcome CreateVpcEndpointConnectionNotification(const Model::CreateVpcEndpointConnectionNotificationRequest& request) const;

        /**
         * A Callable wrapper for CreateVpcEndpointConnectionNotification that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcEndpointConnectionNotificationRequestT = Model::CreateVpcEndpointConnectionNotificationRequest>
        Model::CreateVpcEndpointConnectionNotificationOutcomeCallable CreateVpcEndpointConnectionNotificationCallable(const CreateVpcEndpointConnectionNotificationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpcEndpointConnectionNotification, request);
        }

        /**
         * An Async wrapper for CreateVpcEndpointConnectionNotification that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcEndpointConnectionNotificationRequestT = Model::CreateVpcEndpointConnectionNotificationRequest>
        void CreateVpcEndpointConnectionNotificationAsync(const CreateVpcEndpointConnectionNotificationRequestT& request, const CreateVpcEndpointConnectionNotificationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpcEndpointConnectionNotification, request, handler, context);
        }

        /**
         * <p>Creates a VPC endpoint service to which service consumers (Amazon Web
         * Services accounts, users, and IAM roles) can connect.</p> <p>Before you create
         * an endpoint service, you must create one of the following for your service:</p>
         * <ul> <li> <p>A <a
         * href="https://docs.aws.amazon.com/elasticloadbalancing/latest/network/">Network
         * Load Balancer</a>. Service consumers connect to your service using an interface
         * endpoint.</p> </li> <li> <p>A <a
         * href="https://docs.aws.amazon.com/elasticloadbalancing/latest/gateway/">Gateway
         * Load Balancer</a>. Service consumers connect to your service using a Gateway
         * Load Balancer endpoint.</p> </li> </ul> <p>If you set the private DNS name, you
         * must prove that you own the private DNS domain name.</p> <p>For more
         * information, see the <a
         * href="https://docs.aws.amazon.com/vpc/latest/privatelink/">Amazon Web Services
         * PrivateLink Guide</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpcEndpointServiceConfiguration">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpcEndpointServiceConfigurationOutcome CreateVpcEndpointServiceConfiguration(const Model::CreateVpcEndpointServiceConfigurationRequest& request = {}) const;

        /**
         * A Callable wrapper for CreateVpcEndpointServiceConfiguration that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcEndpointServiceConfigurationRequestT = Model::CreateVpcEndpointServiceConfigurationRequest>
        Model::CreateVpcEndpointServiceConfigurationOutcomeCallable CreateVpcEndpointServiceConfigurationCallable(const CreateVpcEndpointServiceConfigurationRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::CreateVpcEndpointServiceConfiguration, request);
        }

        /**
         * An Async wrapper for CreateVpcEndpointServiceConfiguration that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcEndpointServiceConfigurationRequestT = Model::CreateVpcEndpointServiceConfigurationRequest>
        void CreateVpcEndpointServiceConfigurationAsync(const CreateVpcEndpointServiceConfigurationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const CreateVpcEndpointServiceConfigurationRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::CreateVpcEndpointServiceConfiguration, request, handler, context);
        }

        /**
         * <p>Requests a VPC peering connection between two VPCs: a requester VPC that you
         * own and an accepter VPC with which to create the connection. The accepter VPC
         * can belong to another Amazon Web Services account and can be in a different
         * Region to the requester VPC. The requester VPC and accepter VPC cannot have
         * overlapping CIDR blocks.</p>  <p>Limitations and rules apply to a VPC
         * peering connection. For more information, see the <a
         * href="https://docs.aws.amazon.com/vpc/latest/peering/vpc-peering-basics.html#vpc-peering-limitations">VPC
         * peering limitations</a> in the <i>VPC Peering Guide</i>.</p>  <p>The
         * owner of the accepter VPC must accept the peering request to activate the
         * peering connection. The VPC peering connection request expires after 7 days,
         * after which it cannot be accepted or rejected.</p> <p>If you create a VPC
         * peering connection request between VPCs with overlapping CIDR blocks, the VPC
         * peering connection has a status of <code>failed</code>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpcPeeringConnection">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpcPeeringConnectionOutcome CreateVpcPeeringConnection(const Model::CreateVpcPeeringConnectionRequest& request) const;

        /**
         * A Callable wrapper for CreateVpcPeeringConnection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpcPeeringConnectionRequestT = Model::CreateVpcPeeringConnectionRequest>
        Model::CreateVpcPeeringConnectionOutcomeCallable CreateVpcPeeringConnectionCallable(const CreateVpcPeeringConnectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpcPeeringConnection, request);
        }

        /**
         * An Async wrapper for CreateVpcPeeringConnection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpcPeeringConnectionRequestT = Model::CreateVpcPeeringConnectionRequest>
        void CreateVpcPeeringConnectionAsync(const CreateVpcPeeringConnectionRequestT& request, const CreateVpcPeeringConnectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpcPeeringConnection, request, handler, context);
        }

        /**
         * <p>Creates a VPN connection between an existing virtual private gateway or
         * transit gateway and a customer gateway. The supported connection type is
         * <code>ipsec.1</code>.</p> <p>The response includes information that you need to
         * give to your network administrator to configure your customer gateway.</p>
         *  <p>We strongly recommend that you use HTTPS when calling this
         * operation because the response contains sensitive cryptographic information for
         * configuring your customer gateway device.</p>  <p>If you decide to
         * shut down your VPN connection for any reason and later create a new VPN
         * connection, you must reconfigure your customer gateway with the new information
         * returned from this call.</p> <p>This is an idempotent operation. If you perform
         * the operation more than once, Amazon EC2 doesn't return an error.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpnConnection">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpnConnectionOutcome CreateVpnConnection(const Model::CreateVpnConnectionRequest& request) const;

        /**
         * A Callable wrapper for CreateVpnConnection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpnConnectionRequestT = Model::CreateVpnConnectionRequest>
        Model::CreateVpnConnectionOutcomeCallable CreateVpnConnectionCallable(const CreateVpnConnectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpnConnection, request);
        }

        /**
         * An Async wrapper for CreateVpnConnection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpnConnectionRequestT = Model::CreateVpnConnectionRequest>
        void CreateVpnConnectionAsync(const CreateVpnConnectionRequestT& request, const CreateVpnConnectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpnConnection, request, handler, context);
        }

        /**
         * <p>Creates a static route associated with a VPN connection between an existing
         * virtual private gateway and a VPN customer gateway. The static route allows
         * traffic to be routed from the virtual private gateway to the VPN customer
         * gateway.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpnConnectionRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpnConnectionRouteOutcome CreateVpnConnectionRoute(const Model::CreateVpnConnectionRouteRequest& request) const;

        /**
         * A Callable wrapper for CreateVpnConnectionRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpnConnectionRouteRequestT = Model::CreateVpnConnectionRouteRequest>
        Model::CreateVpnConnectionRouteOutcomeCallable CreateVpnConnectionRouteCallable(const CreateVpnConnectionRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpnConnectionRoute, request);
        }

        /**
         * An Async wrapper for CreateVpnConnectionRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpnConnectionRouteRequestT = Model::CreateVpnConnectionRouteRequest>
        void CreateVpnConnectionRouteAsync(const CreateVpnConnectionRouteRequestT& request, const CreateVpnConnectionRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpnConnectionRoute, request, handler, context);
        }

        /**
         * <p>Creates a virtual private gateway. A virtual private gateway is the endpoint
         * on the VPC side of your VPN connection. You can create a virtual private gateway
         * before creating the VPC itself.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/CreateVpnGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::CreateVpnGatewayOutcome CreateVpnGateway(const Model::CreateVpnGatewayRequest& request) const;

        /**
         * A Callable wrapper for CreateVpnGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename CreateVpnGatewayRequestT = Model::CreateVpnGatewayRequest>
        Model::CreateVpnGatewayOutcomeCallable CreateVpnGatewayCallable(const CreateVpnGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::CreateVpnGateway, request);
        }

        /**
         * An Async wrapper for CreateVpnGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename CreateVpnGatewayRequestT = Model::CreateVpnGatewayRequest>
        void CreateVpnGatewayAsync(const CreateVpnGatewayRequestT& request, const CreateVpnGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::CreateVpnGateway, request, handler, context);
        }

        /**
         * <p>Deletes a carrier gateway.</p>  <p>If you do not delete the route
         * that contains the carrier gateway as the Target, the route is a blackhole route.
         * For information about how to delete a route, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html">DeleteRoute</a>.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteCarrierGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteCarrierGatewayOutcome DeleteCarrierGateway(const Model::DeleteCarrierGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteCarrierGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteCarrierGatewayRequestT = Model::DeleteCarrierGatewayRequest>
        Model::DeleteCarrierGatewayOutcomeCallable DeleteCarrierGatewayCallable(const DeleteCarrierGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteCarrierGateway, request);
        }

        /**
         * An Async wrapper for DeleteCarrierGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteCarrierGatewayRequestT = Model::DeleteCarrierGatewayRequest>
        void DeleteCarrierGatewayAsync(const DeleteCarrierGatewayRequestT& request, const DeleteCarrierGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteCarrierGateway, request, handler, context);
        }

        /**
         * <p>Deletes the specified Client VPN endpoint. You must disassociate all target
         * networks before you can delete a Client VPN endpoint.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteClientVpnEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteClientVpnEndpointOutcome DeleteClientVpnEndpoint(const Model::DeleteClientVpnEndpointRequest& request) const;

        /**
         * A Callable wrapper for DeleteClientVpnEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteClientVpnEndpointRequestT = Model::DeleteClientVpnEndpointRequest>
        Model::DeleteClientVpnEndpointOutcomeCallable DeleteClientVpnEndpointCallable(const DeleteClientVpnEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteClientVpnEndpoint, request);
        }

        /**
         * An Async wrapper for DeleteClientVpnEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteClientVpnEndpointRequestT = Model::DeleteClientVpnEndpointRequest>
        void DeleteClientVpnEndpointAsync(const DeleteClientVpnEndpointRequestT& request, const DeleteClientVpnEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteClientVpnEndpoint, request, handler, context);
        }

        /**
         * <p>Deletes a route from a Client VPN endpoint. You can only delete routes that
         * you manually added using the <b>CreateClientVpnRoute</b> action. You cannot
         * delete routes that were automatically added when associating a subnet. To remove
         * routes that have been automatically added, disassociate the target subnet from
         * the Client VPN endpoint.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteClientVpnRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteClientVpnRouteOutcome DeleteClientVpnRoute(const Model::DeleteClientVpnRouteRequest& request) const;

        /**
         * A Callable wrapper for DeleteClientVpnRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteClientVpnRouteRequestT = Model::DeleteClientVpnRouteRequest>
        Model::DeleteClientVpnRouteOutcomeCallable DeleteClientVpnRouteCallable(const DeleteClientVpnRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteClientVpnRoute, request);
        }

        /**
         * An Async wrapper for DeleteClientVpnRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteClientVpnRouteRequestT = Model::DeleteClientVpnRouteRequest>
        void DeleteClientVpnRouteAsync(const DeleteClientVpnRouteRequestT& request, const DeleteClientVpnRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteClientVpnRoute, request, handler, context);
        }

        /**
         * <p> Deletes a range of customer-owned IP addresses. </p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteCoipCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteCoipCidrOutcome DeleteCoipCidr(const Model::DeleteCoipCidrRequest& request) const;

        /**
         * A Callable wrapper for DeleteCoipCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteCoipCidrRequestT = Model::DeleteCoipCidrRequest>
        Model::DeleteCoipCidrOutcomeCallable DeleteCoipCidrCallable(const DeleteCoipCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteCoipCidr, request);
        }

        /**
         * An Async wrapper for DeleteCoipCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteCoipCidrRequestT = Model::DeleteCoipCidrRequest>
        void DeleteCoipCidrAsync(const DeleteCoipCidrRequestT& request, const DeleteCoipCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteCoipCidr, request, handler, context);
        }

        /**
         * <p>Deletes a pool of customer-owned IP (CoIP) addresses. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteCoipPool">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteCoipPoolOutcome DeleteCoipPool(const Model::DeleteCoipPoolRequest& request) const;

        /**
         * A Callable wrapper for DeleteCoipPool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteCoipPoolRequestT = Model::DeleteCoipPoolRequest>
        Model::DeleteCoipPoolOutcomeCallable DeleteCoipPoolCallable(const DeleteCoipPoolRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteCoipPool, request);
        }

        /**
         * An Async wrapper for DeleteCoipPool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteCoipPoolRequestT = Model::DeleteCoipPoolRequest>
        void DeleteCoipPoolAsync(const DeleteCoipPoolRequestT& request, const DeleteCoipPoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteCoipPool, request, handler, context);
        }

        /**
         * <p>Deletes the specified customer gateway. You must delete the VPN connection
         * before you can delete the customer gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteCustomerGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteCustomerGatewayOutcome DeleteCustomerGateway(const Model::DeleteCustomerGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteCustomerGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteCustomerGatewayRequestT = Model::DeleteCustomerGatewayRequest>
        Model::DeleteCustomerGatewayOutcomeCallable DeleteCustomerGatewayCallable(const DeleteCustomerGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteCustomerGateway, request);
        }

        /**
         * An Async wrapper for DeleteCustomerGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteCustomerGatewayRequestT = Model::DeleteCustomerGatewayRequest>
        void DeleteCustomerGatewayAsync(const DeleteCustomerGatewayRequestT& request, const DeleteCustomerGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteCustomerGateway, request, handler, context);
        }

        /**
         * <p>Deletes the specified set of DHCP options. You must disassociate the set of
         * DHCP options before you can delete it. You can disassociate the set of DHCP
         * options by associating either a new set of options or the default set of options
         * with the VPC.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteDhcpOptions">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteDhcpOptionsOutcome DeleteDhcpOptions(const Model::DeleteDhcpOptionsRequest& request) const;

        /**
         * A Callable wrapper for DeleteDhcpOptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteDhcpOptionsRequestT = Model::DeleteDhcpOptionsRequest>
        Model::DeleteDhcpOptionsOutcomeCallable DeleteDhcpOptionsCallable(const DeleteDhcpOptionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteDhcpOptions, request);
        }

        /**
         * An Async wrapper for DeleteDhcpOptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteDhcpOptionsRequestT = Model::DeleteDhcpOptionsRequest>
        void DeleteDhcpOptionsAsync(const DeleteDhcpOptionsRequestT& request, const DeleteDhcpOptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteDhcpOptions, request, handler, context);
        }

        /**
         * <p>Deletes an egress-only internet gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteEgressOnlyInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteEgressOnlyInternetGatewayOutcome DeleteEgressOnlyInternetGateway(const Model::DeleteEgressOnlyInternetGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteEgressOnlyInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteEgressOnlyInternetGatewayRequestT = Model::DeleteEgressOnlyInternetGatewayRequest>
        Model::DeleteEgressOnlyInternetGatewayOutcomeCallable DeleteEgressOnlyInternetGatewayCallable(const DeleteEgressOnlyInternetGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteEgressOnlyInternetGateway, request);
        }

        /**
         * An Async wrapper for DeleteEgressOnlyInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteEgressOnlyInternetGatewayRequestT = Model::DeleteEgressOnlyInternetGatewayRequest>
        void DeleteEgressOnlyInternetGatewayAsync(const DeleteEgressOnlyInternetGatewayRequestT& request, const DeleteEgressOnlyInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteEgressOnlyInternetGateway, request, handler, context);
        }

        /**
         * <p>Deletes the specified EC2 Fleet request.</p> <p>After you delete an EC2 Fleet
         * request, it launches no new instances.</p> <p>You must also specify whether a
         * deleted EC2 Fleet request should terminate its instances. If you choose to
         * terminate the instances, the EC2 Fleet request enters the
         * <code>deleted_terminating</code> state. Otherwise, it enters the
         * <code>deleted_running</code> state, and the instances continue to run until they
         * are interrupted or you terminate them manually.</p> <p>A deleted
         * <code>instant</code> fleet with running instances is not supported. When you
         * delete an <code>instant</code> fleet, Amazon EC2 automatically terminates all
         * its instances. For fleets with more than 1000 instances, the deletion request
         * might fail. If your fleet has more than 1000 instances, first terminate most of
         * the instances manually, leaving 1000 or fewer. Then delete the fleet, and the
         * remaining instances will be terminated automatically.</p> <p class="title">
         * <b>Restrictions</b> </p> <ul> <li> <p>You can delete up to 25 fleets of type
         * <code>instant</code> in a single request.</p> </li> <li> <p>You can delete up to
         * 100 fleets of type <code>maintain</code> or <code>request</code> in a single
         * request.</p> </li> <li> <p>You can delete up to 125 fleets in a single request,
         * provided you do not exceed the quota for each fleet type, as specified
         * above.</p> </li> <li> <p>If you exceed the specified number of fleets to delete,
         * no fleets are deleted.</p> </li> </ul> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/delete-fleet.html">Delete
         * an EC2 Fleet request and the instances in the fleet</a> in the <i>Amazon EC2
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteFleets">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteFleetsOutcome DeleteFleets(const Model::DeleteFleetsRequest& request) const;

        /**
         * A Callable wrapper for DeleteFleets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteFleetsRequestT = Model::DeleteFleetsRequest>
        Model::DeleteFleetsOutcomeCallable DeleteFleetsCallable(const DeleteFleetsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteFleets, request);
        }

        /**
         * An Async wrapper for DeleteFleets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteFleetsRequestT = Model::DeleteFleetsRequest>
        void DeleteFleetsAsync(const DeleteFleetsRequestT& request, const DeleteFleetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteFleets, request, handler, context);
        }

        /**
         * <p>Deletes one or more flow logs.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteFlowLogs">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteFlowLogsOutcome DeleteFlowLogs(const Model::DeleteFlowLogsRequest& request) const;

        /**
         * A Callable wrapper for DeleteFlowLogs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteFlowLogsRequestT = Model::DeleteFlowLogsRequest>
        Model::DeleteFlowLogsOutcomeCallable DeleteFlowLogsCallable(const DeleteFlowLogsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteFlowLogs, request);
        }

        /**
         * An Async wrapper for DeleteFlowLogs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteFlowLogsRequestT = Model::DeleteFlowLogsRequest>
        void DeleteFlowLogsAsync(const DeleteFlowLogsRequestT& request, const DeleteFlowLogsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteFlowLogs, request, handler, context);
        }

        /**
         * <p>Deletes the specified Amazon FPGA Image (AFI).</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteFpgaImage">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteFpgaImageOutcome DeleteFpgaImage(const Model::DeleteFpgaImageRequest& request) const;

        /**
         * A Callable wrapper for DeleteFpgaImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteFpgaImageRequestT = Model::DeleteFpgaImageRequest>
        Model::DeleteFpgaImageOutcomeCallable DeleteFpgaImageCallable(const DeleteFpgaImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteFpgaImage, request);
        }

        /**
         * An Async wrapper for DeleteFpgaImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteFpgaImageRequestT = Model::DeleteFpgaImageRequest>
        void DeleteFpgaImageAsync(const DeleteFpgaImageRequestT& request, const DeleteFpgaImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteFpgaImage, request, handler, context);
        }

        /**
         * <p>Deletes the specified EC2 Instance Connect Endpoint.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteInstanceConnectEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteInstanceConnectEndpointOutcome DeleteInstanceConnectEndpoint(const Model::DeleteInstanceConnectEndpointRequest& request) const;

        /**
         * A Callable wrapper for DeleteInstanceConnectEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteInstanceConnectEndpointRequestT = Model::DeleteInstanceConnectEndpointRequest>
        Model::DeleteInstanceConnectEndpointOutcomeCallable DeleteInstanceConnectEndpointCallable(const DeleteInstanceConnectEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteInstanceConnectEndpoint, request);
        }

        /**
         * An Async wrapper for DeleteInstanceConnectEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteInstanceConnectEndpointRequestT = Model::DeleteInstanceConnectEndpointRequest>
        void DeleteInstanceConnectEndpointAsync(const DeleteInstanceConnectEndpointRequestT& request, const DeleteInstanceConnectEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteInstanceConnectEndpoint, request, handler, context);
        }

        /**
         * <p>Deletes the specified event window.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/event-windows.html">Define
         * event windows for scheduled events</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteInstanceEventWindow">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteInstanceEventWindowOutcome DeleteInstanceEventWindow(const Model::DeleteInstanceEventWindowRequest& request) const;

        /**
         * A Callable wrapper for DeleteInstanceEventWindow that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteInstanceEventWindowRequestT = Model::DeleteInstanceEventWindowRequest>
        Model::DeleteInstanceEventWindowOutcomeCallable DeleteInstanceEventWindowCallable(const DeleteInstanceEventWindowRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteInstanceEventWindow, request);
        }

        /**
         * An Async wrapper for DeleteInstanceEventWindow that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteInstanceEventWindowRequestT = Model::DeleteInstanceEventWindowRequest>
        void DeleteInstanceEventWindowAsync(const DeleteInstanceEventWindowRequestT& request, const DeleteInstanceEventWindowResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteInstanceEventWindow, request, handler, context);
        }

        /**
         * <p>Deletes the specified internet gateway. You must detach the internet gateway
         * from the VPC before you can delete it.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteInternetGatewayOutcome DeleteInternetGateway(const Model::DeleteInternetGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteInternetGatewayRequestT = Model::DeleteInternetGatewayRequest>
        Model::DeleteInternetGatewayOutcomeCallable DeleteInternetGatewayCallable(const DeleteInternetGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteInternetGateway, request);
        }

        /**
         * An Async wrapper for DeleteInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteInternetGatewayRequestT = Model::DeleteInternetGatewayRequest>
        void DeleteInternetGatewayAsync(const DeleteInternetGatewayRequestT& request, const DeleteInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteInternetGateway, request, handler, context);
        }

        /**
         * <p>Delete an IPAM. Deleting an IPAM removes all monitored data associated with
         * the IPAM including the historical data for CIDRs.</p> <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/delete-ipam.html">Delete an
         * IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See Also:</h3>  
         * <a href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteIpam">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteIpamOutcome DeleteIpam(const Model::DeleteIpamRequest& request) const;

        /**
         * A Callable wrapper for DeleteIpam that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteIpamRequestT = Model::DeleteIpamRequest>
        Model::DeleteIpamOutcomeCallable DeleteIpamCallable(const DeleteIpamRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteIpam, request);
        }

        /**
         * An Async wrapper for DeleteIpam that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteIpamRequestT = Model::DeleteIpamRequest>
        void DeleteIpamAsync(const DeleteIpamRequestT& request, const DeleteIpamResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteIpam, request, handler, context);
        }

        /**
         * <p>Delete a verification token. A verification token is an Amazon Web
         * Services-generated random value that you can use to prove ownership of an
         * external resource. For example, you can use a verification token to validate
         * that you control a public IP address range when you bring an IP address range to
         * Amazon Web Services (BYOIP). </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteIpamExternalResourceVerificationToken">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteIpamExternalResourceVerificationTokenOutcome DeleteIpamExternalResourceVerificationToken(const Model::DeleteIpamExternalResourceVerificationTokenRequest& request) const;

        /**
         * A Callable wrapper for DeleteIpamExternalResourceVerificationToken that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteIpamExternalResourceVerificationTokenRequestT = Model::DeleteIpamExternalResourceVerificationTokenRequest>
        Model::DeleteIpamExternalResourceVerificationTokenOutcomeCallable DeleteIpamExternalResourceVerificationTokenCallable(const DeleteIpamExternalResourceVerificationTokenRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteIpamExternalResourceVerificationToken, request);
        }

        /**
         * An Async wrapper for DeleteIpamExternalResourceVerificationToken that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteIpamExternalResourceVerificationTokenRequestT = Model::DeleteIpamExternalResourceVerificationTokenRequest>
        void DeleteIpamExternalResourceVerificationTokenAsync(const DeleteIpamExternalResourceVerificationTokenRequestT& request, const DeleteIpamExternalResourceVerificationTokenResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteIpamExternalResourceVerificationToken, request, handler, context);
        }

        /**
         * <p>Delete an IPAM pool.</p>  <p>You cannot delete an IPAM pool if there
         * are allocations in it or CIDRs provisioned to it. To release allocations, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseIpamPoolAllocation.html">ReleaseIpamPoolAllocation</a>.
         * To deprovision pool CIDRs, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionIpamPoolCidr.html">DeprovisionIpamPoolCidr</a>.</p>
         *  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/delete-pool-ipam.html">Delete
         * a pool</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteIpamPool">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteIpamPoolOutcome DeleteIpamPool(const Model::DeleteIpamPoolRequest& request) const;

        /**
         * A Callable wrapper for DeleteIpamPool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteIpamPoolRequestT = Model::DeleteIpamPoolRequest>
        Model::DeleteIpamPoolOutcomeCallable DeleteIpamPoolCallable(const DeleteIpamPoolRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteIpamPool, request);
        }

        /**
         * An Async wrapper for DeleteIpamPool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteIpamPoolRequestT = Model::DeleteIpamPoolRequest>
        void DeleteIpamPoolAsync(const DeleteIpamPoolRequestT& request, const DeleteIpamPoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteIpamPool, request, handler, context);
        }

        /**
         * <p>Deletes an IPAM resource discovery. A resource discovery is an IPAM component
         * that enables IPAM to manage and monitor resources that belong to the owning
         * account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteIpamResourceDiscovery">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteIpamResourceDiscoveryOutcome DeleteIpamResourceDiscovery(const Model::DeleteIpamResourceDiscoveryRequest& request) const;

        /**
         * A Callable wrapper for DeleteIpamResourceDiscovery that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteIpamResourceDiscoveryRequestT = Model::DeleteIpamResourceDiscoveryRequest>
        Model::DeleteIpamResourceDiscoveryOutcomeCallable DeleteIpamResourceDiscoveryCallable(const DeleteIpamResourceDiscoveryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteIpamResourceDiscovery, request);
        }

        /**
         * An Async wrapper for DeleteIpamResourceDiscovery that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteIpamResourceDiscoveryRequestT = Model::DeleteIpamResourceDiscoveryRequest>
        void DeleteIpamResourceDiscoveryAsync(const DeleteIpamResourceDiscoveryRequestT& request, const DeleteIpamResourceDiscoveryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteIpamResourceDiscovery, request, handler, context);
        }

        /**
         * <p>Delete the scope for an IPAM. You cannot delete the default scopes.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/delete-scope-ipam.html">Delete
         * a scope</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteIpamScope">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteIpamScopeOutcome DeleteIpamScope(const Model::DeleteIpamScopeRequest& request) const;

        /**
         * A Callable wrapper for DeleteIpamScope that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteIpamScopeRequestT = Model::DeleteIpamScopeRequest>
        Model::DeleteIpamScopeOutcomeCallable DeleteIpamScopeCallable(const DeleteIpamScopeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteIpamScope, request);
        }

        /**
         * An Async wrapper for DeleteIpamScope that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteIpamScopeRequestT = Model::DeleteIpamScopeRequest>
        void DeleteIpamScopeAsync(const DeleteIpamScopeRequestT& request, const DeleteIpamScopeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteIpamScope, request, handler, context);
        }

        /**
         * <p>Deletes the specified key pair, by removing the public key from Amazon
         * EC2.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteKeyPair">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteKeyPairOutcome DeleteKeyPair(const Model::DeleteKeyPairRequest& request = {}) const;

        /**
         * A Callable wrapper for DeleteKeyPair that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteKeyPairRequestT = Model::DeleteKeyPairRequest>
        Model::DeleteKeyPairOutcomeCallable DeleteKeyPairCallable(const DeleteKeyPairRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeleteKeyPair, request);
        }

        /**
         * An Async wrapper for DeleteKeyPair that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteKeyPairRequestT = Model::DeleteKeyPairRequest>
        void DeleteKeyPairAsync(const DeleteKeyPairResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeleteKeyPairRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeleteKeyPair, request, handler, context);
        }

        /**
         * <p>Deletes a launch template. Deleting a launch template deletes all of its
         * versions.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLaunchTemplate">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLaunchTemplateOutcome DeleteLaunchTemplate(const Model::DeleteLaunchTemplateRequest& request = {}) const;

        /**
         * A Callable wrapper for DeleteLaunchTemplate that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLaunchTemplateRequestT = Model::DeleteLaunchTemplateRequest>
        Model::DeleteLaunchTemplateOutcomeCallable DeleteLaunchTemplateCallable(const DeleteLaunchTemplateRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeleteLaunchTemplate, request);
        }

        /**
         * An Async wrapper for DeleteLaunchTemplate that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLaunchTemplateRequestT = Model::DeleteLaunchTemplateRequest>
        void DeleteLaunchTemplateAsync(const DeleteLaunchTemplateResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeleteLaunchTemplateRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeleteLaunchTemplate, request, handler, context);
        }

        /**
         * <p>Deletes one or more versions of a launch template.</p> <p>You can't delete
         * the default version of a launch template; you must first assign a different
         * version as the default. If the default version is the only version for the
         * launch template, you must delete the entire launch template using
         * <a>DeleteLaunchTemplate</a>.</p> <p>You can delete up to 200 launch template
         * versions in a single request. To delete more than 200 versions in a single
         * request, use <a>DeleteLaunchTemplate</a>, which deletes the launch template and
         * all of its versions.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/delete-launch-template.html#delete-launch-template-version">Delete
         * a launch template version</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLaunchTemplateVersions">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLaunchTemplateVersionsOutcome DeleteLaunchTemplateVersions(const Model::DeleteLaunchTemplateVersionsRequest& request) const;

        /**
         * A Callable wrapper for DeleteLaunchTemplateVersions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLaunchTemplateVersionsRequestT = Model::DeleteLaunchTemplateVersionsRequest>
        Model::DeleteLaunchTemplateVersionsOutcomeCallable DeleteLaunchTemplateVersionsCallable(const DeleteLaunchTemplateVersionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLaunchTemplateVersions, request);
        }

        /**
         * An Async wrapper for DeleteLaunchTemplateVersions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLaunchTemplateVersionsRequestT = Model::DeleteLaunchTemplateVersionsRequest>
        void DeleteLaunchTemplateVersionsAsync(const DeleteLaunchTemplateVersionsRequestT& request, const DeleteLaunchTemplateVersionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLaunchTemplateVersions, request, handler, context);
        }

        /**
         * <p>Deletes the specified route from the specified local gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayRouteOutcome DeleteLocalGatewayRoute(const Model::DeleteLocalGatewayRouteRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayRouteRequestT = Model::DeleteLocalGatewayRouteRequest>
        Model::DeleteLocalGatewayRouteOutcomeCallable DeleteLocalGatewayRouteCallable(const DeleteLocalGatewayRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayRoute, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayRouteRequestT = Model::DeleteLocalGatewayRouteRequest>
        void DeleteLocalGatewayRouteAsync(const DeleteLocalGatewayRouteRequestT& request, const DeleteLocalGatewayRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayRoute, request, handler, context);
        }

        /**
         * <p> Deletes a local gateway route table. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayRouteTableOutcome DeleteLocalGatewayRouteTable(const Model::DeleteLocalGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayRouteTableRequestT = Model::DeleteLocalGatewayRouteTableRequest>
        Model::DeleteLocalGatewayRouteTableOutcomeCallable DeleteLocalGatewayRouteTableCallable(const DeleteLocalGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayRouteTableRequestT = Model::DeleteLocalGatewayRouteTableRequest>
        void DeleteLocalGatewayRouteTableAsync(const DeleteLocalGatewayRouteTableRequestT& request, const DeleteLocalGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayRouteTable, request, handler, context);
        }

        /**
         * <p> Deletes a local gateway route table virtual interface group association.
         * </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationOutcome DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation(const Model::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT = Model::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest>
        Model::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationOutcomeCallable DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationCallable(const DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT = Model::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequest>
        void DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationAsync(const DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationRequestT& request, const DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayRouteTableVirtualInterfaceGroupAssociation, request, handler, context);
        }

        /**
         * <p>Deletes the specified association between a VPC and local gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayRouteTableVpcAssociation">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayRouteTableVpcAssociationOutcome DeleteLocalGatewayRouteTableVpcAssociation(const Model::DeleteLocalGatewayRouteTableVpcAssociationRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayRouteTableVpcAssociation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayRouteTableVpcAssociationRequestT = Model::DeleteLocalGatewayRouteTableVpcAssociationRequest>
        Model::DeleteLocalGatewayRouteTableVpcAssociationOutcomeCallable DeleteLocalGatewayRouteTableVpcAssociationCallable(const DeleteLocalGatewayRouteTableVpcAssociationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayRouteTableVpcAssociation, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayRouteTableVpcAssociation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayRouteTableVpcAssociationRequestT = Model::DeleteLocalGatewayRouteTableVpcAssociationRequest>
        void DeleteLocalGatewayRouteTableVpcAssociationAsync(const DeleteLocalGatewayRouteTableVpcAssociationRequestT& request, const DeleteLocalGatewayRouteTableVpcAssociationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayRouteTableVpcAssociation, request, handler, context);
        }

        /**
         * <p>Deletes the specified local gateway virtual interface.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayVirtualInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayVirtualInterfaceOutcome DeleteLocalGatewayVirtualInterface(const Model::DeleteLocalGatewayVirtualInterfaceRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayVirtualInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayVirtualInterfaceRequestT = Model::DeleteLocalGatewayVirtualInterfaceRequest>
        Model::DeleteLocalGatewayVirtualInterfaceOutcomeCallable DeleteLocalGatewayVirtualInterfaceCallable(const DeleteLocalGatewayVirtualInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayVirtualInterface, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayVirtualInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayVirtualInterfaceRequestT = Model::DeleteLocalGatewayVirtualInterfaceRequest>
        void DeleteLocalGatewayVirtualInterfaceAsync(const DeleteLocalGatewayVirtualInterfaceRequestT& request, const DeleteLocalGatewayVirtualInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayVirtualInterface, request, handler, context);
        }

        /**
         * <p>Delete the specified local gateway interface group.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteLocalGatewayVirtualInterfaceGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteLocalGatewayVirtualInterfaceGroupOutcome DeleteLocalGatewayVirtualInterfaceGroup(const Model::DeleteLocalGatewayVirtualInterfaceGroupRequest& request) const;

        /**
         * A Callable wrapper for DeleteLocalGatewayVirtualInterfaceGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteLocalGatewayVirtualInterfaceGroupRequestT = Model::DeleteLocalGatewayVirtualInterfaceGroupRequest>
        Model::DeleteLocalGatewayVirtualInterfaceGroupOutcomeCallable DeleteLocalGatewayVirtualInterfaceGroupCallable(const DeleteLocalGatewayVirtualInterfaceGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteLocalGatewayVirtualInterfaceGroup, request);
        }

        /**
         * An Async wrapper for DeleteLocalGatewayVirtualInterfaceGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteLocalGatewayVirtualInterfaceGroupRequestT = Model::DeleteLocalGatewayVirtualInterfaceGroupRequest>
        void DeleteLocalGatewayVirtualInterfaceGroupAsync(const DeleteLocalGatewayVirtualInterfaceGroupRequestT& request, const DeleteLocalGatewayVirtualInterfaceGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteLocalGatewayVirtualInterfaceGroup, request, handler, context);
        }

        /**
         * <p>Deletes the specified managed prefix list. You must first remove all
         * references to the prefix list in your resources.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteManagedPrefixList">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteManagedPrefixListOutcome DeleteManagedPrefixList(const Model::DeleteManagedPrefixListRequest& request) const;

        /**
         * A Callable wrapper for DeleteManagedPrefixList that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteManagedPrefixListRequestT = Model::DeleteManagedPrefixListRequest>
        Model::DeleteManagedPrefixListOutcomeCallable DeleteManagedPrefixListCallable(const DeleteManagedPrefixListRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteManagedPrefixList, request);
        }

        /**
         * An Async wrapper for DeleteManagedPrefixList that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteManagedPrefixListRequestT = Model::DeleteManagedPrefixListRequest>
        void DeleteManagedPrefixListAsync(const DeleteManagedPrefixListRequestT& request, const DeleteManagedPrefixListResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteManagedPrefixList, request, handler, context);
        }

        /**
         * <p>Deletes the specified NAT gateway. Deleting a public NAT gateway
         * disassociates its Elastic IP address, but does not release the address from your
         * account. Deleting a NAT gateway does not delete any NAT gateway routes in your
         * route tables.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNatGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNatGatewayOutcome DeleteNatGateway(const Model::DeleteNatGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteNatGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNatGatewayRequestT = Model::DeleteNatGatewayRequest>
        Model::DeleteNatGatewayOutcomeCallable DeleteNatGatewayCallable(const DeleteNatGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNatGateway, request);
        }

        /**
         * An Async wrapper for DeleteNatGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNatGatewayRequestT = Model::DeleteNatGatewayRequest>
        void DeleteNatGatewayAsync(const DeleteNatGatewayRequestT& request, const DeleteNatGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNatGateway, request, handler, context);
        }

        /**
         * <p>Deletes the specified network ACL. You can't delete the ACL if it's
         * associated with any subnets. You can't delete the default network
         * ACL.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkAcl">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkAclOutcome DeleteNetworkAcl(const Model::DeleteNetworkAclRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkAcl that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkAclRequestT = Model::DeleteNetworkAclRequest>
        Model::DeleteNetworkAclOutcomeCallable DeleteNetworkAclCallable(const DeleteNetworkAclRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkAcl, request);
        }

        /**
         * An Async wrapper for DeleteNetworkAcl that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkAclRequestT = Model::DeleteNetworkAclRequest>
        void DeleteNetworkAclAsync(const DeleteNetworkAclRequestT& request, const DeleteNetworkAclResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkAcl, request, handler, context);
        }

        /**
         * <p>Deletes the specified ingress or egress entry (rule) from the specified
         * network ACL.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkAclEntry">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkAclEntryOutcome DeleteNetworkAclEntry(const Model::DeleteNetworkAclEntryRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkAclEntry that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkAclEntryRequestT = Model::DeleteNetworkAclEntryRequest>
        Model::DeleteNetworkAclEntryOutcomeCallable DeleteNetworkAclEntryCallable(const DeleteNetworkAclEntryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkAclEntry, request);
        }

        /**
         * An Async wrapper for DeleteNetworkAclEntry that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkAclEntryRequestT = Model::DeleteNetworkAclEntryRequest>
        void DeleteNetworkAclEntryAsync(const DeleteNetworkAclEntryRequestT& request, const DeleteNetworkAclEntryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkAclEntry, request, handler, context);
        }

        /**
         * <p>Deletes the specified Network Access Scope.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInsightsAccessScope">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInsightsAccessScopeOutcome DeleteNetworkInsightsAccessScope(const Model::DeleteNetworkInsightsAccessScopeRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInsightsAccessScope that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInsightsAccessScopeRequestT = Model::DeleteNetworkInsightsAccessScopeRequest>
        Model::DeleteNetworkInsightsAccessScopeOutcomeCallable DeleteNetworkInsightsAccessScopeCallable(const DeleteNetworkInsightsAccessScopeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInsightsAccessScope, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInsightsAccessScope that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInsightsAccessScopeRequestT = Model::DeleteNetworkInsightsAccessScopeRequest>
        void DeleteNetworkInsightsAccessScopeAsync(const DeleteNetworkInsightsAccessScopeRequestT& request, const DeleteNetworkInsightsAccessScopeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInsightsAccessScope, request, handler, context);
        }

        /**
         * <p>Deletes the specified Network Access Scope analysis.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInsightsAccessScopeAnalysis">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInsightsAccessScopeAnalysisOutcome DeleteNetworkInsightsAccessScopeAnalysis(const Model::DeleteNetworkInsightsAccessScopeAnalysisRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInsightsAccessScopeAnalysis that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInsightsAccessScopeAnalysisRequestT = Model::DeleteNetworkInsightsAccessScopeAnalysisRequest>
        Model::DeleteNetworkInsightsAccessScopeAnalysisOutcomeCallable DeleteNetworkInsightsAccessScopeAnalysisCallable(const DeleteNetworkInsightsAccessScopeAnalysisRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInsightsAccessScopeAnalysis, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInsightsAccessScopeAnalysis that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInsightsAccessScopeAnalysisRequestT = Model::DeleteNetworkInsightsAccessScopeAnalysisRequest>
        void DeleteNetworkInsightsAccessScopeAnalysisAsync(const DeleteNetworkInsightsAccessScopeAnalysisRequestT& request, const DeleteNetworkInsightsAccessScopeAnalysisResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInsightsAccessScopeAnalysis, request, handler, context);
        }

        /**
         * <p>Deletes the specified network insights analysis.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInsightsAnalysis">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInsightsAnalysisOutcome DeleteNetworkInsightsAnalysis(const Model::DeleteNetworkInsightsAnalysisRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInsightsAnalysis that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInsightsAnalysisRequestT = Model::DeleteNetworkInsightsAnalysisRequest>
        Model::DeleteNetworkInsightsAnalysisOutcomeCallable DeleteNetworkInsightsAnalysisCallable(const DeleteNetworkInsightsAnalysisRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInsightsAnalysis, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInsightsAnalysis that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInsightsAnalysisRequestT = Model::DeleteNetworkInsightsAnalysisRequest>
        void DeleteNetworkInsightsAnalysisAsync(const DeleteNetworkInsightsAnalysisRequestT& request, const DeleteNetworkInsightsAnalysisResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInsightsAnalysis, request, handler, context);
        }

        /**
         * <p>Deletes the specified path.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInsightsPath">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInsightsPathOutcome DeleteNetworkInsightsPath(const Model::DeleteNetworkInsightsPathRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInsightsPath that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInsightsPathRequestT = Model::DeleteNetworkInsightsPathRequest>
        Model::DeleteNetworkInsightsPathOutcomeCallable DeleteNetworkInsightsPathCallable(const DeleteNetworkInsightsPathRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInsightsPath, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInsightsPath that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInsightsPathRequestT = Model::DeleteNetworkInsightsPathRequest>
        void DeleteNetworkInsightsPathAsync(const DeleteNetworkInsightsPathRequestT& request, const DeleteNetworkInsightsPathResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInsightsPath, request, handler, context);
        }

        /**
         * <p>Deletes the specified network interface. You must detach the network
         * interface before you can delete it.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInterfaceOutcome DeleteNetworkInterface(const Model::DeleteNetworkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInterfaceRequestT = Model::DeleteNetworkInterfaceRequest>
        Model::DeleteNetworkInterfaceOutcomeCallable DeleteNetworkInterfaceCallable(const DeleteNetworkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInterface, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInterfaceRequestT = Model::DeleteNetworkInterfaceRequest>
        void DeleteNetworkInterfaceAsync(const DeleteNetworkInterfaceRequestT& request, const DeleteNetworkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInterface, request, handler, context);
        }

        /**
         * <p>Deletes a permission for a network interface. By default, you cannot delete
         * the permission if the account for which you're removing the permission has
         * attached the network interface to an instance. However, you can force delete the
         * permission, regardless of any attachment.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteNetworkInterfacePermission">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteNetworkInterfacePermissionOutcome DeleteNetworkInterfacePermission(const Model::DeleteNetworkInterfacePermissionRequest& request) const;

        /**
         * A Callable wrapper for DeleteNetworkInterfacePermission that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteNetworkInterfacePermissionRequestT = Model::DeleteNetworkInterfacePermissionRequest>
        Model::DeleteNetworkInterfacePermissionOutcomeCallable DeleteNetworkInterfacePermissionCallable(const DeleteNetworkInterfacePermissionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteNetworkInterfacePermission, request);
        }

        /**
         * An Async wrapper for DeleteNetworkInterfacePermission that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteNetworkInterfacePermissionRequestT = Model::DeleteNetworkInterfacePermissionRequest>
        void DeleteNetworkInterfacePermissionAsync(const DeleteNetworkInterfacePermissionRequestT& request, const DeleteNetworkInterfacePermissionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteNetworkInterfacePermission, request, handler, context);
        }

        /**
         * <p>Deletes the specified placement group. You must terminate all instances in
         * the placement group before you can delete the placement group. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement
         * groups</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeletePlacementGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::DeletePlacementGroupOutcome DeletePlacementGroup(const Model::DeletePlacementGroupRequest& request) const;

        /**
         * A Callable wrapper for DeletePlacementGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeletePlacementGroupRequestT = Model::DeletePlacementGroupRequest>
        Model::DeletePlacementGroupOutcomeCallable DeletePlacementGroupCallable(const DeletePlacementGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeletePlacementGroup, request);
        }

        /**
         * An Async wrapper for DeletePlacementGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeletePlacementGroupRequestT = Model::DeletePlacementGroupRequest>
        void DeletePlacementGroupAsync(const DeletePlacementGroupRequestT& request, const DeletePlacementGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeletePlacementGroup, request, handler, context);
        }

        /**
         * <p>Delete a public IPv4 pool. A public IPv4 pool is an EC2 IP address pool
         * required for the public IPv4 CIDRs that you own and bring to Amazon Web Services
         * to manage with IPAM. IPv6 addresses you bring to Amazon Web Services, however,
         * use IPAM pools only.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeletePublicIpv4Pool">AWS
         * API Reference</a></p>
         */
        virtual Model::DeletePublicIpv4PoolOutcome DeletePublicIpv4Pool(const Model::DeletePublicIpv4PoolRequest& request) const;

        /**
         * A Callable wrapper for DeletePublicIpv4Pool that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeletePublicIpv4PoolRequestT = Model::DeletePublicIpv4PoolRequest>
        Model::DeletePublicIpv4PoolOutcomeCallable DeletePublicIpv4PoolCallable(const DeletePublicIpv4PoolRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeletePublicIpv4Pool, request);
        }

        /**
         * An Async wrapper for DeletePublicIpv4Pool that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeletePublicIpv4PoolRequestT = Model::DeletePublicIpv4PoolRequest>
        void DeletePublicIpv4PoolAsync(const DeletePublicIpv4PoolRequestT& request, const DeletePublicIpv4PoolResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeletePublicIpv4Pool, request, handler, context);
        }

        /**
         * <p>Deletes the queued purchases for the specified Reserved
         * Instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteQueuedReservedInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteQueuedReservedInstancesOutcome DeleteQueuedReservedInstances(const Model::DeleteQueuedReservedInstancesRequest& request) const;

        /**
         * A Callable wrapper for DeleteQueuedReservedInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteQueuedReservedInstancesRequestT = Model::DeleteQueuedReservedInstancesRequest>
        Model::DeleteQueuedReservedInstancesOutcomeCallable DeleteQueuedReservedInstancesCallable(const DeleteQueuedReservedInstancesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteQueuedReservedInstances, request);
        }

        /**
         * An Async wrapper for DeleteQueuedReservedInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteQueuedReservedInstancesRequestT = Model::DeleteQueuedReservedInstancesRequest>
        void DeleteQueuedReservedInstancesAsync(const DeleteQueuedReservedInstancesRequestT& request, const DeleteQueuedReservedInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteQueuedReservedInstances, request, handler, context);
        }

        /**
         * <p>Deletes the specified route from the specified route table.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteRoute">AWS API
         * Reference</a></p>
         */
        virtual Model::DeleteRouteOutcome DeleteRoute(const Model::DeleteRouteRequest& request) const;

        /**
         * A Callable wrapper for DeleteRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteRouteRequestT = Model::DeleteRouteRequest>
        Model::DeleteRouteOutcomeCallable DeleteRouteCallable(const DeleteRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteRoute, request);
        }

        /**
         * An Async wrapper for DeleteRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteRouteRequestT = Model::DeleteRouteRequest>
        void DeleteRouteAsync(const DeleteRouteRequestT& request, const DeleteRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteRoute, request, handler, context);
        }

        /**
         * <p>Deletes the specified route server.</p> <p>Amazon VPC Route Server simplifies
         * routing for traffic between workloads that are deployed within a VPC and its
         * internet gateways. With this feature, VPC Route Server dynamically updates VPC
         * and internet gateway route tables with your preferred IPv4 or IPv6 routes to
         * achieve routing fault tolerance for those workloads. This enables you to
         * automatically reroute traffic within a VPC, which increases the manageability of
         * VPC routing and interoperability with third-party workloads.</p> <p>Route server
         * supports the follow route table types:</p> <ul> <li> <p>VPC route tables not
         * associated with subnets</p> </li> <li> <p>Subnet route tables</p> </li> <li>
         * <p>Internet gateway route tables</p> </li> </ul> <p>Route server does not
         * support route tables associated with virtual private gateways. To propagate
         * routes into a transit gateway route table, use <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html">Transit
         * Gateway Connect</a>.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteRouteServer">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteRouteServerOutcome DeleteRouteServer(const Model::DeleteRouteServerRequest& request) const;

        /**
         * A Callable wrapper for DeleteRouteServer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteRouteServerRequestT = Model::DeleteRouteServerRequest>
        Model::DeleteRouteServerOutcomeCallable DeleteRouteServerCallable(const DeleteRouteServerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteRouteServer, request);
        }

        /**
         * An Async wrapper for DeleteRouteServer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteRouteServerRequestT = Model::DeleteRouteServerRequest>
        void DeleteRouteServerAsync(const DeleteRouteServerRequestT& request, const DeleteRouteServerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteRouteServer, request, handler, context);
        }

        /**
         * <p>Deletes the specified route server endpoint.</p> <p>A route server endpoint
         * is an Amazon Web Services-managed component inside a subnet that facilitates <a
         * href="https://en.wikipedia.org/wiki/Border_Gateway_Protocol">BGP (Border Gateway
         * Protocol)</a> connections between your route server and your BGP
         * peers.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteRouteServerEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteRouteServerEndpointOutcome DeleteRouteServerEndpoint(const Model::DeleteRouteServerEndpointRequest& request) const;

        /**
         * A Callable wrapper for DeleteRouteServerEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteRouteServerEndpointRequestT = Model::DeleteRouteServerEndpointRequest>
        Model::DeleteRouteServerEndpointOutcomeCallable DeleteRouteServerEndpointCallable(const DeleteRouteServerEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteRouteServerEndpoint, request);
        }

        /**
         * An Async wrapper for DeleteRouteServerEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteRouteServerEndpointRequestT = Model::DeleteRouteServerEndpointRequest>
        void DeleteRouteServerEndpointAsync(const DeleteRouteServerEndpointRequestT& request, const DeleteRouteServerEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteRouteServerEndpoint, request, handler, context);
        }

        /**
         * <p>Deletes the specified BGP peer from a route server.</p> <p>A route server
         * peer is a session between a route server endpoint and the device deployed in
         * Amazon Web Services (such as a firewall appliance or other network security
         * function running on an EC2 instance). The device must meet these
         * requirements:</p> <ul> <li> <p>Have an elastic network interface in the VPC</p>
         * </li> <li> <p>Support BGP (Border Gateway Protocol)</p> </li> <li> <p>Can
         * initiate BGP sessions</p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteRouteServerPeer">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteRouteServerPeerOutcome DeleteRouteServerPeer(const Model::DeleteRouteServerPeerRequest& request) const;

        /**
         * A Callable wrapper for DeleteRouteServerPeer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteRouteServerPeerRequestT = Model::DeleteRouteServerPeerRequest>
        Model::DeleteRouteServerPeerOutcomeCallable DeleteRouteServerPeerCallable(const DeleteRouteServerPeerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteRouteServerPeer, request);
        }

        /**
         * An Async wrapper for DeleteRouteServerPeer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteRouteServerPeerRequestT = Model::DeleteRouteServerPeerRequest>
        void DeleteRouteServerPeerAsync(const DeleteRouteServerPeerRequestT& request, const DeleteRouteServerPeerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteRouteServerPeer, request, handler, context);
        }

        /**
         * <p>Deletes the specified route table. You must disassociate the route table from
         * any subnets before you can delete it. You can't delete the main route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteRouteTableOutcome DeleteRouteTable(const Model::DeleteRouteTableRequest& request) const;

        /**
         * A Callable wrapper for DeleteRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteRouteTableRequestT = Model::DeleteRouteTableRequest>
        Model::DeleteRouteTableOutcomeCallable DeleteRouteTableCallable(const DeleteRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteRouteTable, request);
        }

        /**
         * An Async wrapper for DeleteRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteRouteTableRequestT = Model::DeleteRouteTableRequest>
        void DeleteRouteTableAsync(const DeleteRouteTableRequestT& request, const DeleteRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteRouteTable, request, handler, context);
        }

        /**
         * <p>Deletes a security group.</p> <p>If you attempt to delete a security group
         * that is associated with an instance or network interface, is referenced by
         * another security group in the same VPC, or has a VPC association, the operation
         * fails with <code>DependencyViolation</code>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteSecurityGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteSecurityGroupOutcome DeleteSecurityGroup(const Model::DeleteSecurityGroupRequest& request = {}) const;

        /**
         * A Callable wrapper for DeleteSecurityGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteSecurityGroupRequestT = Model::DeleteSecurityGroupRequest>
        Model::DeleteSecurityGroupOutcomeCallable DeleteSecurityGroupCallable(const DeleteSecurityGroupRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeleteSecurityGroup, request);
        }

        /**
         * An Async wrapper for DeleteSecurityGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteSecurityGroupRequestT = Model::DeleteSecurityGroupRequest>
        void DeleteSecurityGroupAsync(const DeleteSecurityGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeleteSecurityGroupRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeleteSecurityGroup, request, handler, context);
        }

        /**
         * <p>Deletes the specified snapshot.</p> <p>When you make periodic snapshots of a
         * volume, the snapshots are incremental, and only the blocks on the device that
         * have changed since your last snapshot are saved in the new snapshot. When you
         * delete a snapshot, only the data not needed for any other snapshot is removed.
         * So regardless of which prior snapshots have been deleted, all active snapshots
         * will have access to all the information needed to restore the volume.</p> <p>You
         * cannot delete a snapshot of the root device of an EBS volume used by a
         * registered AMI. You must first deregister the AMI before you can delete the
         * snapshot.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-deleting-snapshot.html">Delete
         * an Amazon EBS snapshot</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteSnapshot">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteSnapshotOutcome DeleteSnapshot(const Model::DeleteSnapshotRequest& request) const;

        /**
         * A Callable wrapper for DeleteSnapshot that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteSnapshotRequestT = Model::DeleteSnapshotRequest>
        Model::DeleteSnapshotOutcomeCallable DeleteSnapshotCallable(const DeleteSnapshotRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteSnapshot, request);
        }

        /**
         * An Async wrapper for DeleteSnapshot that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteSnapshotRequestT = Model::DeleteSnapshotRequest>
        void DeleteSnapshotAsync(const DeleteSnapshotRequestT& request, const DeleteSnapshotResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteSnapshot, request, handler, context);
        }

        /**
         * <p>Deletes the data feed for Spot Instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteSpotDatafeedSubscription">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteSpotDatafeedSubscriptionOutcome DeleteSpotDatafeedSubscription(const Model::DeleteSpotDatafeedSubscriptionRequest& request = {}) const;

        /**
         * A Callable wrapper for DeleteSpotDatafeedSubscription that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteSpotDatafeedSubscriptionRequestT = Model::DeleteSpotDatafeedSubscriptionRequest>
        Model::DeleteSpotDatafeedSubscriptionOutcomeCallable DeleteSpotDatafeedSubscriptionCallable(const DeleteSpotDatafeedSubscriptionRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeleteSpotDatafeedSubscription, request);
        }

        /**
         * An Async wrapper for DeleteSpotDatafeedSubscription that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteSpotDatafeedSubscriptionRequestT = Model::DeleteSpotDatafeedSubscriptionRequest>
        void DeleteSpotDatafeedSubscriptionAsync(const DeleteSpotDatafeedSubscriptionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeleteSpotDatafeedSubscriptionRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeleteSpotDatafeedSubscription, request, handler, context);
        }

        /**
         * <p>Deletes the specified subnet. You must terminate all running instances in the
         * subnet before you can delete the subnet.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteSubnet">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteSubnetOutcome DeleteSubnet(const Model::DeleteSubnetRequest& request) const;

        /**
         * A Callable wrapper for DeleteSubnet that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteSubnetRequestT = Model::DeleteSubnetRequest>
        Model::DeleteSubnetOutcomeCallable DeleteSubnetCallable(const DeleteSubnetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteSubnet, request);
        }

        /**
         * An Async wrapper for DeleteSubnet that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteSubnetRequestT = Model::DeleteSubnetRequest>
        void DeleteSubnetAsync(const DeleteSubnetRequestT& request, const DeleteSubnetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteSubnet, request, handler, context);
        }

        /**
         * <p>Deletes a subnet CIDR reservation.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteSubnetCidrReservation">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteSubnetCidrReservationOutcome DeleteSubnetCidrReservation(const Model::DeleteSubnetCidrReservationRequest& request) const;

        /**
         * A Callable wrapper for DeleteSubnetCidrReservation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteSubnetCidrReservationRequestT = Model::DeleteSubnetCidrReservationRequest>
        Model::DeleteSubnetCidrReservationOutcomeCallable DeleteSubnetCidrReservationCallable(const DeleteSubnetCidrReservationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteSubnetCidrReservation, request);
        }

        /**
         * An Async wrapper for DeleteSubnetCidrReservation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteSubnetCidrReservationRequestT = Model::DeleteSubnetCidrReservationRequest>
        void DeleteSubnetCidrReservationAsync(const DeleteSubnetCidrReservationRequestT& request, const DeleteSubnetCidrReservationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteSubnetCidrReservation, request, handler, context);
        }

        /**
         * <p>Deletes the specified set of tags from the specified set of resources.</p>
         * <p>To list the current tags, use <a>DescribeTags</a>. For more information about
         * tags, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tag
         * your Amazon EC2 resources</a> in the <i>Amazon Elastic Compute Cloud User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTags">AWS API
         * Reference</a></p>
         */
        virtual Model::DeleteTagsOutcome DeleteTags(const Model::DeleteTagsRequest& request) const;

        /**
         * A Callable wrapper for DeleteTags that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTagsRequestT = Model::DeleteTagsRequest>
        Model::DeleteTagsOutcomeCallable DeleteTagsCallable(const DeleteTagsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTags, request);
        }

        /**
         * An Async wrapper for DeleteTags that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTagsRequestT = Model::DeleteTagsRequest>
        void DeleteTagsAsync(const DeleteTagsRequestT& request, const DeleteTagsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTags, request, handler, context);
        }

        /**
         * <p>Deletes the specified Traffic Mirror filter.</p> <p>You cannot delete a
         * Traffic Mirror filter that is in use by a Traffic Mirror session.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTrafficMirrorFilter">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTrafficMirrorFilterOutcome DeleteTrafficMirrorFilter(const Model::DeleteTrafficMirrorFilterRequest& request) const;

        /**
         * A Callable wrapper for DeleteTrafficMirrorFilter that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTrafficMirrorFilterRequestT = Model::DeleteTrafficMirrorFilterRequest>
        Model::DeleteTrafficMirrorFilterOutcomeCallable DeleteTrafficMirrorFilterCallable(const DeleteTrafficMirrorFilterRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTrafficMirrorFilter, request);
        }

        /**
         * An Async wrapper for DeleteTrafficMirrorFilter that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTrafficMirrorFilterRequestT = Model::DeleteTrafficMirrorFilterRequest>
        void DeleteTrafficMirrorFilterAsync(const DeleteTrafficMirrorFilterRequestT& request, const DeleteTrafficMirrorFilterResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTrafficMirrorFilter, request, handler, context);
        }

        /**
         * <p>Deletes the specified Traffic Mirror rule.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTrafficMirrorFilterRule">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTrafficMirrorFilterRuleOutcome DeleteTrafficMirrorFilterRule(const Model::DeleteTrafficMirrorFilterRuleRequest& request) const;

        /**
         * A Callable wrapper for DeleteTrafficMirrorFilterRule that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTrafficMirrorFilterRuleRequestT = Model::DeleteTrafficMirrorFilterRuleRequest>
        Model::DeleteTrafficMirrorFilterRuleOutcomeCallable DeleteTrafficMirrorFilterRuleCallable(const DeleteTrafficMirrorFilterRuleRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTrafficMirrorFilterRule, request);
        }

        /**
         * An Async wrapper for DeleteTrafficMirrorFilterRule that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTrafficMirrorFilterRuleRequestT = Model::DeleteTrafficMirrorFilterRuleRequest>
        void DeleteTrafficMirrorFilterRuleAsync(const DeleteTrafficMirrorFilterRuleRequestT& request, const DeleteTrafficMirrorFilterRuleResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTrafficMirrorFilterRule, request, handler, context);
        }

        /**
         * <p>Deletes the specified Traffic Mirror session.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTrafficMirrorSession">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTrafficMirrorSessionOutcome DeleteTrafficMirrorSession(const Model::DeleteTrafficMirrorSessionRequest& request) const;

        /**
         * A Callable wrapper for DeleteTrafficMirrorSession that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTrafficMirrorSessionRequestT = Model::DeleteTrafficMirrorSessionRequest>
        Model::DeleteTrafficMirrorSessionOutcomeCallable DeleteTrafficMirrorSessionCallable(const DeleteTrafficMirrorSessionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTrafficMirrorSession, request);
        }

        /**
         * An Async wrapper for DeleteTrafficMirrorSession that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTrafficMirrorSessionRequestT = Model::DeleteTrafficMirrorSessionRequest>
        void DeleteTrafficMirrorSessionAsync(const DeleteTrafficMirrorSessionRequestT& request, const DeleteTrafficMirrorSessionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTrafficMirrorSession, request, handler, context);
        }

        /**
         * <p>Deletes the specified Traffic Mirror target.</p> <p>You cannot delete a
         * Traffic Mirror target that is in use by a Traffic Mirror session.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTrafficMirrorTarget">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTrafficMirrorTargetOutcome DeleteTrafficMirrorTarget(const Model::DeleteTrafficMirrorTargetRequest& request) const;

        /**
         * A Callable wrapper for DeleteTrafficMirrorTarget that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTrafficMirrorTargetRequestT = Model::DeleteTrafficMirrorTargetRequest>
        Model::DeleteTrafficMirrorTargetOutcomeCallable DeleteTrafficMirrorTargetCallable(const DeleteTrafficMirrorTargetRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTrafficMirrorTarget, request);
        }

        /**
         * An Async wrapper for DeleteTrafficMirrorTarget that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTrafficMirrorTargetRequestT = Model::DeleteTrafficMirrorTargetRequest>
        void DeleteTrafficMirrorTargetAsync(const DeleteTrafficMirrorTargetRequestT& request, const DeleteTrafficMirrorTargetResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTrafficMirrorTarget, request, handler, context);
        }

        /**
         * <p>Deletes the specified transit gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayOutcome DeleteTransitGateway(const Model::DeleteTransitGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayRequestT = Model::DeleteTransitGatewayRequest>
        Model::DeleteTransitGatewayOutcomeCallable DeleteTransitGatewayCallable(const DeleteTransitGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGateway, request);
        }

        /**
         * An Async wrapper for DeleteTransitGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayRequestT = Model::DeleteTransitGatewayRequest>
        void DeleteTransitGatewayAsync(const DeleteTransitGatewayRequestT& request, const DeleteTransitGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGateway, request, handler, context);
        }

        /**
         * <p>Deletes the specified Connect attachment. You must first delete any Connect
         * peers for the attachment.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayConnect">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayConnectOutcome DeleteTransitGatewayConnect(const Model::DeleteTransitGatewayConnectRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayConnect that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayConnectRequestT = Model::DeleteTransitGatewayConnectRequest>
        Model::DeleteTransitGatewayConnectOutcomeCallable DeleteTransitGatewayConnectCallable(const DeleteTransitGatewayConnectRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayConnect, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayConnect that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayConnectRequestT = Model::DeleteTransitGatewayConnectRequest>
        void DeleteTransitGatewayConnectAsync(const DeleteTransitGatewayConnectRequestT& request, const DeleteTransitGatewayConnectResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayConnect, request, handler, context);
        }

        /**
         * <p>Deletes the specified Connect peer.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayConnectPeer">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayConnectPeerOutcome DeleteTransitGatewayConnectPeer(const Model::DeleteTransitGatewayConnectPeerRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayConnectPeer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayConnectPeerRequestT = Model::DeleteTransitGatewayConnectPeerRequest>
        Model::DeleteTransitGatewayConnectPeerOutcomeCallable DeleteTransitGatewayConnectPeerCallable(const DeleteTransitGatewayConnectPeerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayConnectPeer, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayConnectPeer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayConnectPeerRequestT = Model::DeleteTransitGatewayConnectPeerRequest>
        void DeleteTransitGatewayConnectPeerAsync(const DeleteTransitGatewayConnectPeerRequestT& request, const DeleteTransitGatewayConnectPeerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayConnectPeer, request, handler, context);
        }

        /**
         * <p>Deletes the specified transit gateway multicast domain.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayMulticastDomain">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayMulticastDomainOutcome DeleteTransitGatewayMulticastDomain(const Model::DeleteTransitGatewayMulticastDomainRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayMulticastDomain that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayMulticastDomainRequestT = Model::DeleteTransitGatewayMulticastDomainRequest>
        Model::DeleteTransitGatewayMulticastDomainOutcomeCallable DeleteTransitGatewayMulticastDomainCallable(const DeleteTransitGatewayMulticastDomainRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayMulticastDomain, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayMulticastDomain that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayMulticastDomainRequestT = Model::DeleteTransitGatewayMulticastDomainRequest>
        void DeleteTransitGatewayMulticastDomainAsync(const DeleteTransitGatewayMulticastDomainRequestT& request, const DeleteTransitGatewayMulticastDomainResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayMulticastDomain, request, handler, context);
        }

        /**
         * <p>Deletes a transit gateway peering attachment.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayPeeringAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayPeeringAttachmentOutcome DeleteTransitGatewayPeeringAttachment(const Model::DeleteTransitGatewayPeeringAttachmentRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayPeeringAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayPeeringAttachmentRequestT = Model::DeleteTransitGatewayPeeringAttachmentRequest>
        Model::DeleteTransitGatewayPeeringAttachmentOutcomeCallable DeleteTransitGatewayPeeringAttachmentCallable(const DeleteTransitGatewayPeeringAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayPeeringAttachment, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayPeeringAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayPeeringAttachmentRequestT = Model::DeleteTransitGatewayPeeringAttachmentRequest>
        void DeleteTransitGatewayPeeringAttachmentAsync(const DeleteTransitGatewayPeeringAttachmentRequestT& request, const DeleteTransitGatewayPeeringAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayPeeringAttachment, request, handler, context);
        }

        /**
         * <p>Deletes the specified transit gateway policy table.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayPolicyTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayPolicyTableOutcome DeleteTransitGatewayPolicyTable(const Model::DeleteTransitGatewayPolicyTableRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayPolicyTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayPolicyTableRequestT = Model::DeleteTransitGatewayPolicyTableRequest>
        Model::DeleteTransitGatewayPolicyTableOutcomeCallable DeleteTransitGatewayPolicyTableCallable(const DeleteTransitGatewayPolicyTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayPolicyTable, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayPolicyTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayPolicyTableRequestT = Model::DeleteTransitGatewayPolicyTableRequest>
        void DeleteTransitGatewayPolicyTableAsync(const DeleteTransitGatewayPolicyTableRequestT& request, const DeleteTransitGatewayPolicyTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayPolicyTable, request, handler, context);
        }

        /**
         * <p>Deletes a reference (route) to a prefix list in a specified transit gateway
         * route table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayPrefixListReference">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayPrefixListReferenceOutcome DeleteTransitGatewayPrefixListReference(const Model::DeleteTransitGatewayPrefixListReferenceRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayPrefixListReference that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayPrefixListReferenceRequestT = Model::DeleteTransitGatewayPrefixListReferenceRequest>
        Model::DeleteTransitGatewayPrefixListReferenceOutcomeCallable DeleteTransitGatewayPrefixListReferenceCallable(const DeleteTransitGatewayPrefixListReferenceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayPrefixListReference, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayPrefixListReference that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayPrefixListReferenceRequestT = Model::DeleteTransitGatewayPrefixListReferenceRequest>
        void DeleteTransitGatewayPrefixListReferenceAsync(const DeleteTransitGatewayPrefixListReferenceRequestT& request, const DeleteTransitGatewayPrefixListReferenceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayPrefixListReference, request, handler, context);
        }

        /**
         * <p>Deletes the specified route from the specified transit gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayRouteOutcome DeleteTransitGatewayRoute(const Model::DeleteTransitGatewayRouteRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayRouteRequestT = Model::DeleteTransitGatewayRouteRequest>
        Model::DeleteTransitGatewayRouteOutcomeCallable DeleteTransitGatewayRouteCallable(const DeleteTransitGatewayRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayRoute, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayRouteRequestT = Model::DeleteTransitGatewayRouteRequest>
        void DeleteTransitGatewayRouteAsync(const DeleteTransitGatewayRouteRequestT& request, const DeleteTransitGatewayRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayRoute, request, handler, context);
        }

        /**
         * <p>Deletes the specified transit gateway route table. If there are any route
         * tables associated with the transit gateway route table, you must first run
         * <a>DisassociateRouteTable</a> before you can delete the transit gateway route
         * table. This removes any route tables associated with the transit gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayRouteTableOutcome DeleteTransitGatewayRouteTable(const Model::DeleteTransitGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayRouteTableRequestT = Model::DeleteTransitGatewayRouteTableRequest>
        Model::DeleteTransitGatewayRouteTableOutcomeCallable DeleteTransitGatewayRouteTableCallable(const DeleteTransitGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayRouteTableRequestT = Model::DeleteTransitGatewayRouteTableRequest>
        void DeleteTransitGatewayRouteTableAsync(const DeleteTransitGatewayRouteTableRequestT& request, const DeleteTransitGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayRouteTable, request, handler, context);
        }

        /**
         * <p>Advertises to the transit gateway that a transit gateway route table is
         * deleted.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayRouteTableAnnouncement">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayRouteTableAnnouncementOutcome DeleteTransitGatewayRouteTableAnnouncement(const Model::DeleteTransitGatewayRouteTableAnnouncementRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayRouteTableAnnouncement that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayRouteTableAnnouncementRequestT = Model::DeleteTransitGatewayRouteTableAnnouncementRequest>
        Model::DeleteTransitGatewayRouteTableAnnouncementOutcomeCallable DeleteTransitGatewayRouteTableAnnouncementCallable(const DeleteTransitGatewayRouteTableAnnouncementRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayRouteTableAnnouncement, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayRouteTableAnnouncement that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayRouteTableAnnouncementRequestT = Model::DeleteTransitGatewayRouteTableAnnouncementRequest>
        void DeleteTransitGatewayRouteTableAnnouncementAsync(const DeleteTransitGatewayRouteTableAnnouncementRequestT& request, const DeleteTransitGatewayRouteTableAnnouncementResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayRouteTableAnnouncement, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPC attachment.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteTransitGatewayVpcAttachment">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteTransitGatewayVpcAttachmentOutcome DeleteTransitGatewayVpcAttachment(const Model::DeleteTransitGatewayVpcAttachmentRequest& request) const;

        /**
         * A Callable wrapper for DeleteTransitGatewayVpcAttachment that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteTransitGatewayVpcAttachmentRequestT = Model::DeleteTransitGatewayVpcAttachmentRequest>
        Model::DeleteTransitGatewayVpcAttachmentOutcomeCallable DeleteTransitGatewayVpcAttachmentCallable(const DeleteTransitGatewayVpcAttachmentRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteTransitGatewayVpcAttachment, request);
        }

        /**
         * An Async wrapper for DeleteTransitGatewayVpcAttachment that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteTransitGatewayVpcAttachmentRequestT = Model::DeleteTransitGatewayVpcAttachmentRequest>
        void DeleteTransitGatewayVpcAttachmentAsync(const DeleteTransitGatewayVpcAttachmentRequestT& request, const DeleteTransitGatewayVpcAttachmentResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteTransitGatewayVpcAttachment, request, handler, context);
        }

        /**
         * <p>Delete an Amazon Web Services Verified Access endpoint.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVerifiedAccessEndpoint">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVerifiedAccessEndpointOutcome DeleteVerifiedAccessEndpoint(const Model::DeleteVerifiedAccessEndpointRequest& request) const;

        /**
         * A Callable wrapper for DeleteVerifiedAccessEndpoint that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVerifiedAccessEndpointRequestT = Model::DeleteVerifiedAccessEndpointRequest>
        Model::DeleteVerifiedAccessEndpointOutcomeCallable DeleteVerifiedAccessEndpointCallable(const DeleteVerifiedAccessEndpointRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVerifiedAccessEndpoint, request);
        }

        /**
         * An Async wrapper for DeleteVerifiedAccessEndpoint that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVerifiedAccessEndpointRequestT = Model::DeleteVerifiedAccessEndpointRequest>
        void DeleteVerifiedAccessEndpointAsync(const DeleteVerifiedAccessEndpointRequestT& request, const DeleteVerifiedAccessEndpointResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVerifiedAccessEndpoint, request, handler, context);
        }

        /**
         * <p>Delete an Amazon Web Services Verified Access group.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVerifiedAccessGroup">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVerifiedAccessGroupOutcome DeleteVerifiedAccessGroup(const Model::DeleteVerifiedAccessGroupRequest& request) const;

        /**
         * A Callable wrapper for DeleteVerifiedAccessGroup that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVerifiedAccessGroupRequestT = Model::DeleteVerifiedAccessGroupRequest>
        Model::DeleteVerifiedAccessGroupOutcomeCallable DeleteVerifiedAccessGroupCallable(const DeleteVerifiedAccessGroupRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVerifiedAccessGroup, request);
        }

        /**
         * An Async wrapper for DeleteVerifiedAccessGroup that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVerifiedAccessGroupRequestT = Model::DeleteVerifiedAccessGroupRequest>
        void DeleteVerifiedAccessGroupAsync(const DeleteVerifiedAccessGroupRequestT& request, const DeleteVerifiedAccessGroupResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVerifiedAccessGroup, request, handler, context);
        }

        /**
         * <p>Delete an Amazon Web Services Verified Access instance.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVerifiedAccessInstance">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVerifiedAccessInstanceOutcome DeleteVerifiedAccessInstance(const Model::DeleteVerifiedAccessInstanceRequest& request) const;

        /**
         * A Callable wrapper for DeleteVerifiedAccessInstance that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVerifiedAccessInstanceRequestT = Model::DeleteVerifiedAccessInstanceRequest>
        Model::DeleteVerifiedAccessInstanceOutcomeCallable DeleteVerifiedAccessInstanceCallable(const DeleteVerifiedAccessInstanceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVerifiedAccessInstance, request);
        }

        /**
         * An Async wrapper for DeleteVerifiedAccessInstance that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVerifiedAccessInstanceRequestT = Model::DeleteVerifiedAccessInstanceRequest>
        void DeleteVerifiedAccessInstanceAsync(const DeleteVerifiedAccessInstanceRequestT& request, const DeleteVerifiedAccessInstanceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVerifiedAccessInstance, request, handler, context);
        }

        /**
         * <p>Delete an Amazon Web Services Verified Access trust provider.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVerifiedAccessTrustProvider">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVerifiedAccessTrustProviderOutcome DeleteVerifiedAccessTrustProvider(const Model::DeleteVerifiedAccessTrustProviderRequest& request) const;

        /**
         * A Callable wrapper for DeleteVerifiedAccessTrustProvider that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVerifiedAccessTrustProviderRequestT = Model::DeleteVerifiedAccessTrustProviderRequest>
        Model::DeleteVerifiedAccessTrustProviderOutcomeCallable DeleteVerifiedAccessTrustProviderCallable(const DeleteVerifiedAccessTrustProviderRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVerifiedAccessTrustProvider, request);
        }

        /**
         * An Async wrapper for DeleteVerifiedAccessTrustProvider that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVerifiedAccessTrustProviderRequestT = Model::DeleteVerifiedAccessTrustProviderRequest>
        void DeleteVerifiedAccessTrustProviderAsync(const DeleteVerifiedAccessTrustProviderRequestT& request, const DeleteVerifiedAccessTrustProviderResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVerifiedAccessTrustProvider, request, handler, context);
        }

        /**
         * <p>Deletes the specified EBS volume. The volume must be in the
         * <code>available</code> state (not attached to an instance).</p> <p>The volume
         * can remain in the <code>deleting</code> state for several minutes.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-deleting-volume.html">Delete
         * an Amazon EBS volume</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVolume">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVolumeOutcome DeleteVolume(const Model::DeleteVolumeRequest& request) const;

        /**
         * A Callable wrapper for DeleteVolume that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVolumeRequestT = Model::DeleteVolumeRequest>
        Model::DeleteVolumeOutcomeCallable DeleteVolumeCallable(const DeleteVolumeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVolume, request);
        }

        /**
         * An Async wrapper for DeleteVolume that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVolumeRequestT = Model::DeleteVolumeRequest>
        void DeleteVolumeAsync(const DeleteVolumeRequestT& request, const DeleteVolumeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVolume, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPC. You must detach or delete all gateways and
         * resources that are associated with the VPC before you can delete it. For
         * example, you must terminate all instances running in the VPC, delete all
         * security groups associated with the VPC (except the default one), delete all
         * route tables associated with the VPC (except the default one), and so on. When
         * you delete the VPC, it deletes the default security group, network ACL, and
         * route table for the VPC.</p> <p>If you created a flow log for the VPC that you
         * are deleting, note that flow logs for deleted VPCs are eventually automatically
         * removed.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpc">AWS API
         * Reference</a></p>
         */
        virtual Model::DeleteVpcOutcome DeleteVpc(const Model::DeleteVpcRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcRequestT = Model::DeleteVpcRequest>
        Model::DeleteVpcOutcomeCallable DeleteVpcCallable(const DeleteVpcRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpc, request);
        }

        /**
         * An Async wrapper for DeleteVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcRequestT = Model::DeleteVpcRequest>
        void DeleteVpcAsync(const DeleteVpcRequestT& request, const DeleteVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpc, request, handler, context);
        }

        /**
         * <p>Delete a VPC Block Public Access (BPA) exclusion. A VPC BPA exclusion is a
         * mode that can be applied to a single VPC or subnet that exempts it from the
         * account’s BPA mode and will allow bidirectional or egress-only access. You can
         * create BPA exclusions for VPCs and subnets even when BPA is not enabled on the
         * account to ensure that there is no traffic disruption to the exclusions when VPC
         * BPA is turned on. To learn more about VPC BPA, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-vpc-bpa.html">Block
         * public access to VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpcBlockPublicAccessExclusion">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpcBlockPublicAccessExclusionOutcome DeleteVpcBlockPublicAccessExclusion(const Model::DeleteVpcBlockPublicAccessExclusionRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpcBlockPublicAccessExclusion that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcBlockPublicAccessExclusionRequestT = Model::DeleteVpcBlockPublicAccessExclusionRequest>
        Model::DeleteVpcBlockPublicAccessExclusionOutcomeCallable DeleteVpcBlockPublicAccessExclusionCallable(const DeleteVpcBlockPublicAccessExclusionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpcBlockPublicAccessExclusion, request);
        }

        /**
         * An Async wrapper for DeleteVpcBlockPublicAccessExclusion that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcBlockPublicAccessExclusionRequestT = Model::DeleteVpcBlockPublicAccessExclusionRequest>
        void DeleteVpcBlockPublicAccessExclusionAsync(const DeleteVpcBlockPublicAccessExclusionRequestT& request, const DeleteVpcBlockPublicAccessExclusionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpcBlockPublicAccessExclusion, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPC endpoint connection notifications.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpcEndpointConnectionNotifications">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpcEndpointConnectionNotificationsOutcome DeleteVpcEndpointConnectionNotifications(const Model::DeleteVpcEndpointConnectionNotificationsRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpcEndpointConnectionNotifications that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcEndpointConnectionNotificationsRequestT = Model::DeleteVpcEndpointConnectionNotificationsRequest>
        Model::DeleteVpcEndpointConnectionNotificationsOutcomeCallable DeleteVpcEndpointConnectionNotificationsCallable(const DeleteVpcEndpointConnectionNotificationsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpcEndpointConnectionNotifications, request);
        }

        /**
         * An Async wrapper for DeleteVpcEndpointConnectionNotifications that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcEndpointConnectionNotificationsRequestT = Model::DeleteVpcEndpointConnectionNotificationsRequest>
        void DeleteVpcEndpointConnectionNotificationsAsync(const DeleteVpcEndpointConnectionNotificationsRequestT& request, const DeleteVpcEndpointConnectionNotificationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpcEndpointConnectionNotifications, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPC endpoint service configurations. Before you can
         * delete an endpoint service configuration, you must reject any
         * <code>Available</code> or <code>PendingAcceptance</code> interface endpoint
         * connections that are attached to the service.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpcEndpointServiceConfigurations">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpcEndpointServiceConfigurationsOutcome DeleteVpcEndpointServiceConfigurations(const Model::DeleteVpcEndpointServiceConfigurationsRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpcEndpointServiceConfigurations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcEndpointServiceConfigurationsRequestT = Model::DeleteVpcEndpointServiceConfigurationsRequest>
        Model::DeleteVpcEndpointServiceConfigurationsOutcomeCallable DeleteVpcEndpointServiceConfigurationsCallable(const DeleteVpcEndpointServiceConfigurationsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpcEndpointServiceConfigurations, request);
        }

        /**
         * An Async wrapper for DeleteVpcEndpointServiceConfigurations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcEndpointServiceConfigurationsRequestT = Model::DeleteVpcEndpointServiceConfigurationsRequest>
        void DeleteVpcEndpointServiceConfigurationsAsync(const DeleteVpcEndpointServiceConfigurationsRequestT& request, const DeleteVpcEndpointServiceConfigurationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpcEndpointServiceConfigurations, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPC endpoints.</p> <p>When you delete a gateway
         * endpoint, we delete the endpoint routes in the route tables for the
         * endpoint.</p> <p>When you delete a Gateway Load Balancer endpoint, we delete its
         * endpoint network interfaces. You can only delete Gateway Load Balancer endpoints
         * when the routes that are associated with the endpoint are deleted.</p> <p>When
         * you delete an interface endpoint, we delete its endpoint network
         * interfaces.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpcEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpcEndpointsOutcome DeleteVpcEndpoints(const Model::DeleteVpcEndpointsRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpcEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcEndpointsRequestT = Model::DeleteVpcEndpointsRequest>
        Model::DeleteVpcEndpointsOutcomeCallable DeleteVpcEndpointsCallable(const DeleteVpcEndpointsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpcEndpoints, request);
        }

        /**
         * An Async wrapper for DeleteVpcEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcEndpointsRequestT = Model::DeleteVpcEndpointsRequest>
        void DeleteVpcEndpointsAsync(const DeleteVpcEndpointsRequestT& request, const DeleteVpcEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpcEndpoints, request, handler, context);
        }

        /**
         * <p>Deletes a VPC peering connection. Either the owner of the requester VPC or
         * the owner of the accepter VPC can delete the VPC peering connection if it's in
         * the <code>active</code> state. The owner of the requester VPC can delete a VPC
         * peering connection in the <code>pending-acceptance</code> state. You cannot
         * delete a VPC peering connection that's in the <code>failed</code> or
         * <code>rejected</code> state.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpcPeeringConnection">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpcPeeringConnectionOutcome DeleteVpcPeeringConnection(const Model::DeleteVpcPeeringConnectionRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpcPeeringConnection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpcPeeringConnectionRequestT = Model::DeleteVpcPeeringConnectionRequest>
        Model::DeleteVpcPeeringConnectionOutcomeCallable DeleteVpcPeeringConnectionCallable(const DeleteVpcPeeringConnectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpcPeeringConnection, request);
        }

        /**
         * An Async wrapper for DeleteVpcPeeringConnection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpcPeeringConnectionRequestT = Model::DeleteVpcPeeringConnectionRequest>
        void DeleteVpcPeeringConnectionAsync(const DeleteVpcPeeringConnectionRequestT& request, const DeleteVpcPeeringConnectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpcPeeringConnection, request, handler, context);
        }

        /**
         * <p>Deletes the specified VPN connection.</p> <p>If you're deleting the VPC and
         * its associated components, we recommend that you detach the virtual private
         * gateway from the VPC and delete the VPC before deleting the VPN connection. If
         * you believe that the tunnel credentials for your VPN connection have been
         * compromised, you can delete the VPN connection and create a new one that has new
         * keys, without needing to delete the VPC or virtual private gateway. If you
         * create a new VPN connection, you must reconfigure the customer gateway device
         * using the new configuration information returned with the new VPN connection
         * ID.</p> <p>For certificate-based authentication, delete all Certificate Manager
         * (ACM) private certificates used for the Amazon Web Services-side tunnel
         * endpoints for the VPN connection before deleting the VPN
         * connection.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpnConnection">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpnConnectionOutcome DeleteVpnConnection(const Model::DeleteVpnConnectionRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpnConnection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpnConnectionRequestT = Model::DeleteVpnConnectionRequest>
        Model::DeleteVpnConnectionOutcomeCallable DeleteVpnConnectionCallable(const DeleteVpnConnectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpnConnection, request);
        }

        /**
         * An Async wrapper for DeleteVpnConnection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpnConnectionRequestT = Model::DeleteVpnConnectionRequest>
        void DeleteVpnConnectionAsync(const DeleteVpnConnectionRequestT& request, const DeleteVpnConnectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpnConnection, request, handler, context);
        }

        /**
         * <p>Deletes the specified static route associated with a VPN connection between
         * an existing virtual private gateway and a VPN customer gateway. The static route
         * allows traffic to be routed from the virtual private gateway to the VPN customer
         * gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpnConnectionRoute">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpnConnectionRouteOutcome DeleteVpnConnectionRoute(const Model::DeleteVpnConnectionRouteRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpnConnectionRoute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpnConnectionRouteRequestT = Model::DeleteVpnConnectionRouteRequest>
        Model::DeleteVpnConnectionRouteOutcomeCallable DeleteVpnConnectionRouteCallable(const DeleteVpnConnectionRouteRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpnConnectionRoute, request);
        }

        /**
         * An Async wrapper for DeleteVpnConnectionRoute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpnConnectionRouteRequestT = Model::DeleteVpnConnectionRouteRequest>
        void DeleteVpnConnectionRouteAsync(const DeleteVpnConnectionRouteRequestT& request, const DeleteVpnConnectionRouteResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpnConnectionRoute, request, handler, context);
        }

        /**
         * <p>Deletes the specified virtual private gateway. You must first detach the
         * virtual private gateway from the VPC. Note that you don't need to delete the
         * virtual private gateway if you plan to delete and recreate the VPN connection
         * between your VPC and your network.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeleteVpnGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DeleteVpnGatewayOutcome DeleteVpnGateway(const Model::DeleteVpnGatewayRequest& request) const;

        /**
         * A Callable wrapper for DeleteVpnGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeleteVpnGatewayRequestT = Model::DeleteVpnGatewayRequest>
        Model::DeleteVpnGatewayOutcomeCallable DeleteVpnGatewayCallable(const DeleteVpnGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeleteVpnGateway, request);
        }

        /**
         * An Async wrapper for DeleteVpnGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeleteVpnGatewayRequestT = Model::DeleteVpnGatewayRequest>
        void DeleteVpnGatewayAsync(const DeleteVpnGatewayRequestT& request, const DeleteVpnGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeleteVpnGateway, request, handler, context);
        }

        /**
         * <p>Releases the specified address range that you provisioned for use with your
         * Amazon Web Services resources through bring your own IP addresses (BYOIP) and
         * deletes the corresponding address pool.</p> <p>Before you can release an address
         * range, you must stop advertising it using <a>WithdrawByoipCidr</a> and you must
         * not have any IP addresses allocated from its address range.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeprovisionByoipCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::DeprovisionByoipCidrOutcome DeprovisionByoipCidr(const Model::DeprovisionByoipCidrRequest& request) const;

        /**
         * A Callable wrapper for DeprovisionByoipCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeprovisionByoipCidrRequestT = Model::DeprovisionByoipCidrRequest>
        Model::DeprovisionByoipCidrOutcomeCallable DeprovisionByoipCidrCallable(const DeprovisionByoipCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeprovisionByoipCidr, request);
        }

        /**
         * An Async wrapper for DeprovisionByoipCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeprovisionByoipCidrRequestT = Model::DeprovisionByoipCidrRequest>
        void DeprovisionByoipCidrAsync(const DeprovisionByoipCidrRequestT& request, const DeprovisionByoipCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeprovisionByoipCidr, request, handler, context);
        }

        /**
         * <p>Deprovisions your Autonomous System Number (ASN) from your Amazon Web
         * Services account. This action can only be called after any BYOIP CIDR
         * associations are removed from your Amazon Web Services account with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIpamByoasn.html">DisassociateIpamByoasn</a>.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-byoasn.html">Tutorial:
         * Bring your ASN to IPAM</a> in the <i>Amazon VPC IPAM guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeprovisionIpamByoasn">AWS
         * API Reference</a></p>
         */
        virtual Model::DeprovisionIpamByoasnOutcome DeprovisionIpamByoasn(const Model::DeprovisionIpamByoasnRequest& request) const;

        /**
         * A Callable wrapper for DeprovisionIpamByoasn that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeprovisionIpamByoasnRequestT = Model::DeprovisionIpamByoasnRequest>
        Model::DeprovisionIpamByoasnOutcomeCallable DeprovisionIpamByoasnCallable(const DeprovisionIpamByoasnRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeprovisionIpamByoasn, request);
        }

        /**
         * An Async wrapper for DeprovisionIpamByoasn that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeprovisionIpamByoasnRequestT = Model::DeprovisionIpamByoasnRequest>
        void DeprovisionIpamByoasnAsync(const DeprovisionIpamByoasnRequestT& request, const DeprovisionIpamByoasnResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeprovisionIpamByoasn, request, handler, context);
        }

        /**
         * <p>Deprovision a CIDR provisioned from an IPAM pool. If you deprovision a CIDR
         * from a pool that has a source pool, the CIDR is recycled back into the source
         * pool. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/depro-pool-cidr-ipam.html">Deprovision
         * pool CIDRs</a> in the <i>Amazon VPC IPAM User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeprovisionIpamPoolCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::DeprovisionIpamPoolCidrOutcome DeprovisionIpamPoolCidr(const Model::DeprovisionIpamPoolCidrRequest& request) const;

        /**
         * A Callable wrapper for DeprovisionIpamPoolCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeprovisionIpamPoolCidrRequestT = Model::DeprovisionIpamPoolCidrRequest>
        Model::DeprovisionIpamPoolCidrOutcomeCallable DeprovisionIpamPoolCidrCallable(const DeprovisionIpamPoolCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeprovisionIpamPoolCidr, request);
        }

        /**
         * An Async wrapper for DeprovisionIpamPoolCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeprovisionIpamPoolCidrRequestT = Model::DeprovisionIpamPoolCidrRequest>
        void DeprovisionIpamPoolCidrAsync(const DeprovisionIpamPoolCidrRequestT& request, const DeprovisionIpamPoolCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeprovisionIpamPoolCidr, request, handler, context);
        }

        /**
         * <p>Deprovision a CIDR from a public IPv4 pool.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeprovisionPublicIpv4PoolCidr">AWS
         * API Reference</a></p>
         */
        virtual Model::DeprovisionPublicIpv4PoolCidrOutcome DeprovisionPublicIpv4PoolCidr(const Model::DeprovisionPublicIpv4PoolCidrRequest& request) const;

        /**
         * A Callable wrapper for DeprovisionPublicIpv4PoolCidr that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeprovisionPublicIpv4PoolCidrRequestT = Model::DeprovisionPublicIpv4PoolCidrRequest>
        Model::DeprovisionPublicIpv4PoolCidrOutcomeCallable DeprovisionPublicIpv4PoolCidrCallable(const DeprovisionPublicIpv4PoolCidrRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeprovisionPublicIpv4PoolCidr, request);
        }

        /**
         * An Async wrapper for DeprovisionPublicIpv4PoolCidr that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeprovisionPublicIpv4PoolCidrRequestT = Model::DeprovisionPublicIpv4PoolCidrRequest>
        void DeprovisionPublicIpv4PoolCidrAsync(const DeprovisionPublicIpv4PoolCidrRequestT& request, const DeprovisionPublicIpv4PoolCidrResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeprovisionPublicIpv4PoolCidr, request, handler, context);
        }

        /**
         * <p>Deregisters the specified AMI. A deregistered AMI can't be used to launch new
         * instances.</p> <p>If a deregistered EBS-backed AMI matches a Recycle Bin
         * retention rule, it moves to the Recycle Bin for the specified retention period.
         * It can be restored before its retention period expires, after which it is
         * permanently deleted. If the deregistered AMI doesn't match a retention rule, it
         * is permanently deleted immediately. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/recycle-bin.html">Recover
         * deleted Amazon EBS snapshots and EBS-backed AMIs with Recycle Bin</a> in the
         * <i>Amazon EBS User Guide</i>.</p> <p>When deregistering an EBS-backed AMI, you
         * can optionally delete its associated snapshots at the same time. However, if a
         * snapshot is associated with multiple AMIs, it won't be deleted even if specified
         * for deletion, although the AMI will still be deregistered.</p> <p>Deregistering
         * an AMI does not delete the following:</p> <ul> <li> <p>Instances already
         * launched from the AMI. You'll continue to incur usage costs for the instances
         * until you terminate them.</p> </li> <li> <p>For EBS-backed AMIs: Snapshots that
         * are associated with multiple AMIs. You'll continue to incur snapshot storage
         * costs.</p> </li> <li> <p>For instance store-backed AMIs: The files uploaded to
         * Amazon S3 during AMI creation. You'll continue to incur S3 storage costs.</p>
         * </li> </ul> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/deregister-ami.html">Deregister
         * an Amazon EC2 AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeregisterImage">AWS
         * API Reference</a></p>
         */
        virtual Model::DeregisterImageOutcome DeregisterImage(const Model::DeregisterImageRequest& request) const;

        /**
         * A Callable wrapper for DeregisterImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeregisterImageRequestT = Model::DeregisterImageRequest>
        Model::DeregisterImageOutcomeCallable DeregisterImageCallable(const DeregisterImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeregisterImage, request);
        }

        /**
         * An Async wrapper for DeregisterImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeregisterImageRequestT = Model::DeregisterImageRequest>
        void DeregisterImageAsync(const DeregisterImageRequestT& request, const DeregisterImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeregisterImage, request, handler, context);
        }

        /**
         * <p>Deregisters tag keys to prevent tags that have the specified tag keys from
         * being included in scheduled event notifications for resources in the
         * Region.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeregisterInstanceEventNotificationAttributes">AWS
         * API Reference</a></p>
         */
        virtual Model::DeregisterInstanceEventNotificationAttributesOutcome DeregisterInstanceEventNotificationAttributes(const Model::DeregisterInstanceEventNotificationAttributesRequest& request) const;

        /**
         * A Callable wrapper for DeregisterInstanceEventNotificationAttributes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeregisterInstanceEventNotificationAttributesRequestT = Model::DeregisterInstanceEventNotificationAttributesRequest>
        Model::DeregisterInstanceEventNotificationAttributesOutcomeCallable DeregisterInstanceEventNotificationAttributesCallable(const DeregisterInstanceEventNotificationAttributesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DeregisterInstanceEventNotificationAttributes, request);
        }

        /**
         * An Async wrapper for DeregisterInstanceEventNotificationAttributes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeregisterInstanceEventNotificationAttributesRequestT = Model::DeregisterInstanceEventNotificationAttributesRequest>
        void DeregisterInstanceEventNotificationAttributesAsync(const DeregisterInstanceEventNotificationAttributesRequestT& request, const DeregisterInstanceEventNotificationAttributesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DeregisterInstanceEventNotificationAttributes, request, handler, context);
        }

        /**
         * <p>Deregisters the specified members (network interfaces) from the transit
         * gateway multicast group.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeregisterTransitGatewayMulticastGroupMembers">AWS
         * API Reference</a></p>
         */
        virtual Model::DeregisterTransitGatewayMulticastGroupMembersOutcome DeregisterTransitGatewayMulticastGroupMembers(const Model::DeregisterTransitGatewayMulticastGroupMembersRequest& request = {}) const;

        /**
         * A Callable wrapper for DeregisterTransitGatewayMulticastGroupMembers that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeregisterTransitGatewayMulticastGroupMembersRequestT = Model::DeregisterTransitGatewayMulticastGroupMembersRequest>
        Model::DeregisterTransitGatewayMulticastGroupMembersOutcomeCallable DeregisterTransitGatewayMulticastGroupMembersCallable(const DeregisterTransitGatewayMulticastGroupMembersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeregisterTransitGatewayMulticastGroupMembers, request);
        }

        /**
         * An Async wrapper for DeregisterTransitGatewayMulticastGroupMembers that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeregisterTransitGatewayMulticastGroupMembersRequestT = Model::DeregisterTransitGatewayMulticastGroupMembersRequest>
        void DeregisterTransitGatewayMulticastGroupMembersAsync(const DeregisterTransitGatewayMulticastGroupMembersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeregisterTransitGatewayMulticastGroupMembersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeregisterTransitGatewayMulticastGroupMembers, request, handler, context);
        }

        /**
         * <p>Deregisters the specified sources (network interfaces) from the transit
         * gateway multicast group.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DeregisterTransitGatewayMulticastGroupSources">AWS
         * API Reference</a></p>
         */
        virtual Model::DeregisterTransitGatewayMulticastGroupSourcesOutcome DeregisterTransitGatewayMulticastGroupSources(const Model::DeregisterTransitGatewayMulticastGroupSourcesRequest& request = {}) const;

        /**
         * A Callable wrapper for DeregisterTransitGatewayMulticastGroupSources that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DeregisterTransitGatewayMulticastGroupSourcesRequestT = Model::DeregisterTransitGatewayMulticastGroupSourcesRequest>
        Model::DeregisterTransitGatewayMulticastGroupSourcesOutcomeCallable DeregisterTransitGatewayMulticastGroupSourcesCallable(const DeregisterTransitGatewayMulticastGroupSourcesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DeregisterTransitGatewayMulticastGroupSources, request);
        }

        /**
         * An Async wrapper for DeregisterTransitGatewayMulticastGroupSources that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DeregisterTransitGatewayMulticastGroupSourcesRequestT = Model::DeregisterTransitGatewayMulticastGroupSourcesRequest>
        void DeregisterTransitGatewayMulticastGroupSourcesAsync(const DeregisterTransitGatewayMulticastGroupSourcesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DeregisterTransitGatewayMulticastGroupSourcesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DeregisterTransitGatewayMulticastGroupSources, request, handler, context);
        }

        /**
         * <p>Describes attributes of your Amazon Web Services account. The following are
         * the supported account attributes:</p> <ul> <li> <p> <code>default-vpc</code>:
         * The ID of the default VPC for your account, or <code>none</code>.</p> </li> <li>
         * <p> <code>max-instances</code>: This attribute is no longer supported. The
         * returned value does not reflect your actual vCPU limit for running On-Demand
         * Instances. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-on-demand-instances.html#ec2-on-demand-instances-limits">On-Demand
         * Instance Limits</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         * </li> <li> <p> <code>max-elastic-ips</code>: The maximum number of Elastic IP
         * addresses that you can allocate.</p> </li> <li> <p>
         * <code>supported-platforms</code>: This attribute is deprecated.</p> </li> <li>
         * <p> <code>vpc-max-elastic-ips</code>: The maximum number of Elastic IP addresses
         * that you can allocate.</p> </li> <li> <p>
         * <code>vpc-max-security-groups-per-interface</code>: The maximum number of
         * security groups that you can assign to a network interface.</p> </li> </ul>
         *  <p>The order of the elements in the response, including those within
         * nested structures, might vary. Applications should not assume the elements
         * appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAccountAttributes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAccountAttributesOutcome DescribeAccountAttributes(const Model::DescribeAccountAttributesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAccountAttributes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAccountAttributesRequestT = Model::DescribeAccountAttributesRequest>
        Model::DescribeAccountAttributesOutcomeCallable DescribeAccountAttributesCallable(const DescribeAccountAttributesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAccountAttributes, request);
        }

        /**
         * An Async wrapper for DescribeAccountAttributes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAccountAttributesRequestT = Model::DescribeAccountAttributesRequest>
        void DescribeAccountAttributesAsync(const DescribeAccountAttributesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAccountAttributesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAccountAttributes, request, handler, context);
        }

        /**
         * <p>Describes an Elastic IP address transfer. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro">Transfer
         * Elastic IP addresses</a> in the <i>Amazon VPC User Guide</i>.</p> <p>When you
         * transfer an Elastic IP address, there is a two-step handshake between the source
         * and transfer Amazon Web Services accounts. When the source account starts the
         * transfer, the transfer account has seven days to accept the Elastic IP address
         * transfer. During those seven days, the source account can view the pending
         * transfer by using this action. After seven days, the transfer expires and
         * ownership of the Elastic IP address returns to the source account. Accepted
         * transfers are visible to the source account for 14 days after the transfers have
         * been accepted.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAddressTransfers">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAddressTransfersOutcome DescribeAddressTransfers(const Model::DescribeAddressTransfersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAddressTransfers that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAddressTransfersRequestT = Model::DescribeAddressTransfersRequest>
        Model::DescribeAddressTransfersOutcomeCallable DescribeAddressTransfersCallable(const DescribeAddressTransfersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAddressTransfers, request);
        }

        /**
         * An Async wrapper for DescribeAddressTransfers that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAddressTransfersRequestT = Model::DescribeAddressTransfersRequest>
        void DescribeAddressTransfersAsync(const DescribeAddressTransfersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAddressTransfersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAddressTransfers, request, handler, context);
        }

        /**
         * <p>Describes the specified Elastic IP addresses or all of your Elastic IP
         * addresses.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAddresses">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAddressesOutcome DescribeAddresses(const Model::DescribeAddressesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAddresses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAddressesRequestT = Model::DescribeAddressesRequest>
        Model::DescribeAddressesOutcomeCallable DescribeAddressesCallable(const DescribeAddressesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAddresses, request);
        }

        /**
         * An Async wrapper for DescribeAddresses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAddressesRequestT = Model::DescribeAddressesRequest>
        void DescribeAddressesAsync(const DescribeAddressesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAddressesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAddresses, request, handler, context);
        }

        /**
         * <p>Describes the attributes of the specified Elastic IP addresses. For
         * requirements, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html#Using_Elastic_Addressing_Reverse_DNS">Using
         * reverse DNS for email applications</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAddressesAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAddressesAttributeOutcome DescribeAddressesAttribute(const Model::DescribeAddressesAttributeRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAddressesAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAddressesAttributeRequestT = Model::DescribeAddressesAttributeRequest>
        Model::DescribeAddressesAttributeOutcomeCallable DescribeAddressesAttributeCallable(const DescribeAddressesAttributeRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAddressesAttribute, request);
        }

        /**
         * An Async wrapper for DescribeAddressesAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAddressesAttributeRequestT = Model::DescribeAddressesAttributeRequest>
        void DescribeAddressesAttributeAsync(const DescribeAddressesAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAddressesAttributeRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAddressesAttribute, request, handler, context);
        }

        /**
         * <p>Describes the longer ID format settings for all resource types in a specific
         * Region. This request is useful for performing a quick audit to determine whether
         * a specific Region is fully opted in for longer IDs (17-character IDs).</p>
         * <p>This request only returns information about resource types that support
         * longer IDs.</p> <p>The following resource types support longer IDs:
         * <code>bundle</code> | <code>conversion-task</code> |
         * <code>customer-gateway</code> | <code>dhcp-options</code> |
         * <code>elastic-ip-allocation</code> | <code>elastic-ip-association</code> |
         * <code>export-task</code> | <code>flow-log</code> | <code>image</code> |
         * <code>import-task</code> | <code>instance</code> | <code>internet-gateway</code>
         * | <code>network-acl</code> | <code>network-acl-association</code> |
         * <code>network-interface</code> | <code>network-interface-attachment</code> |
         * <code>prefix-list</code> | <code>reservation</code> | <code>route-table</code> |
         * <code>route-table-association</code> | <code>security-group</code> |
         * <code>snapshot</code> | <code>subnet</code> |
         * <code>subnet-cidr-block-association</code> | <code>volume</code> |
         * <code>vpc</code> | <code>vpc-cidr-block-association</code> |
         * <code>vpc-endpoint</code> | <code>vpc-peering-connection</code> |
         * <code>vpn-connection</code> | <code>vpn-gateway</code>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAggregateIdFormat">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAggregateIdFormatOutcome DescribeAggregateIdFormat(const Model::DescribeAggregateIdFormatRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAggregateIdFormat that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAggregateIdFormatRequestT = Model::DescribeAggregateIdFormatRequest>
        Model::DescribeAggregateIdFormatOutcomeCallable DescribeAggregateIdFormatCallable(const DescribeAggregateIdFormatRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAggregateIdFormat, request);
        }

        /**
         * An Async wrapper for DescribeAggregateIdFormat that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAggregateIdFormatRequestT = Model::DescribeAggregateIdFormatRequest>
        void DescribeAggregateIdFormatAsync(const DescribeAggregateIdFormatResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAggregateIdFormatRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAggregateIdFormat, request, handler, context);
        }

        /**
         * <p>Describes the Availability Zones, Local Zones, and Wavelength Zones that are
         * available to you.</p> <p>For more information about Availability Zones, Local
         * Zones, and Wavelength Zones, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html">Regions
         * and zones</a> in the <i>Amazon EC2 User Guide</i>.</p>  <p>The order of
         * the elements in the response, including those within nested structures, might
         * vary. Applications should not assume the elements appear in a particular
         * order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAvailabilityZones">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAvailabilityZonesOutcome DescribeAvailabilityZones(const Model::DescribeAvailabilityZonesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAvailabilityZones that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAvailabilityZonesRequestT = Model::DescribeAvailabilityZonesRequest>
        Model::DescribeAvailabilityZonesOutcomeCallable DescribeAvailabilityZonesCallable(const DescribeAvailabilityZonesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAvailabilityZones, request);
        }

        /**
         * An Async wrapper for DescribeAvailabilityZones that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAvailabilityZonesRequestT = Model::DescribeAvailabilityZonesRequest>
        void DescribeAvailabilityZonesAsync(const DescribeAvailabilityZonesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAvailabilityZonesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAvailabilityZones, request, handler, context);
        }

        /**
         * <p>Describes the current Infrastructure Performance metric
         * subscriptions.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeAwsNetworkPerformanceMetricSubscriptions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeAwsNetworkPerformanceMetricSubscriptionsOutcome DescribeAwsNetworkPerformanceMetricSubscriptions(const Model::DescribeAwsNetworkPerformanceMetricSubscriptionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeAwsNetworkPerformanceMetricSubscriptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeAwsNetworkPerformanceMetricSubscriptionsRequestT = Model::DescribeAwsNetworkPerformanceMetricSubscriptionsRequest>
        Model::DescribeAwsNetworkPerformanceMetricSubscriptionsOutcomeCallable DescribeAwsNetworkPerformanceMetricSubscriptionsCallable(const DescribeAwsNetworkPerformanceMetricSubscriptionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeAwsNetworkPerformanceMetricSubscriptions, request);
        }

        /**
         * An Async wrapper for DescribeAwsNetworkPerformanceMetricSubscriptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeAwsNetworkPerformanceMetricSubscriptionsRequestT = Model::DescribeAwsNetworkPerformanceMetricSubscriptionsRequest>
        void DescribeAwsNetworkPerformanceMetricSubscriptionsAsync(const DescribeAwsNetworkPerformanceMetricSubscriptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeAwsNetworkPerformanceMetricSubscriptionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeAwsNetworkPerformanceMetricSubscriptions, request, handler, context);
        }

        /**
         * <p>Describes the specified bundle tasks or all of your bundle tasks.</p> 
         * <p>Completed bundle tasks are listed for only a limited time. If your bundle
         * task is no longer in the list, you can still register an AMI from it. Just use
         * <code>RegisterImage</code> with the Amazon S3 bucket name and image manifest
         * name you provided to the bundle task.</p>   <p>The order of the
         * elements in the response, including those within nested structures, might vary.
         * Applications should not assume the elements appear in a particular order.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeBundleTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeBundleTasksOutcome DescribeBundleTasks(const Model::DescribeBundleTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeBundleTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeBundleTasksRequestT = Model::DescribeBundleTasksRequest>
        Model::DescribeBundleTasksOutcomeCallable DescribeBundleTasksCallable(const DescribeBundleTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeBundleTasks, request);
        }

        /**
         * An Async wrapper for DescribeBundleTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeBundleTasksRequestT = Model::DescribeBundleTasksRequest>
        void DescribeBundleTasksAsync(const DescribeBundleTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeBundleTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeBundleTasks, request, handler, context);
        }

        /**
         * <p>Describes the IP address ranges that were specified in calls to
         * <a>ProvisionByoipCidr</a>.</p> <p>To describe the address pools that were
         * created when you provisioned the address ranges, use
         * <a>DescribePublicIpv4Pools</a> or <a>DescribeIpv6Pools</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeByoipCidrs">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeByoipCidrsOutcome DescribeByoipCidrs(const Model::DescribeByoipCidrsRequest& request) const;

        /**
         * A Callable wrapper for DescribeByoipCidrs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeByoipCidrsRequestT = Model::DescribeByoipCidrsRequest>
        Model::DescribeByoipCidrsOutcomeCallable DescribeByoipCidrsCallable(const DescribeByoipCidrsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeByoipCidrs, request);
        }

        /**
         * An Async wrapper for DescribeByoipCidrs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeByoipCidrsRequestT = Model::DescribeByoipCidrsRequest>
        void DescribeByoipCidrsAsync(const DescribeByoipCidrsRequestT& request, const DescribeByoipCidrsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeByoipCidrs, request, handler, context);
        }

        /**
         * <p>Describes the events for the specified Capacity Block extension during the
         * specified time.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityBlockExtensionHistory">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityBlockExtensionHistoryOutcome DescribeCapacityBlockExtensionHistory(const Model::DescribeCapacityBlockExtensionHistoryRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCapacityBlockExtensionHistory that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityBlockExtensionHistoryRequestT = Model::DescribeCapacityBlockExtensionHistoryRequest>
        Model::DescribeCapacityBlockExtensionHistoryOutcomeCallable DescribeCapacityBlockExtensionHistoryCallable(const DescribeCapacityBlockExtensionHistoryRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityBlockExtensionHistory, request);
        }

        /**
         * An Async wrapper for DescribeCapacityBlockExtensionHistory that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityBlockExtensionHistoryRequestT = Model::DescribeCapacityBlockExtensionHistoryRequest>
        void DescribeCapacityBlockExtensionHistoryAsync(const DescribeCapacityBlockExtensionHistoryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCapacityBlockExtensionHistoryRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityBlockExtensionHistory, request, handler, context);
        }

        /**
         * <p>Describes Capacity Block extension offerings available for purchase in the
         * Amazon Web Services Region that you're currently using.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityBlockExtensionOfferings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityBlockExtensionOfferingsOutcome DescribeCapacityBlockExtensionOfferings(const Model::DescribeCapacityBlockExtensionOfferingsRequest& request) const;

        /**
         * A Callable wrapper for DescribeCapacityBlockExtensionOfferings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityBlockExtensionOfferingsRequestT = Model::DescribeCapacityBlockExtensionOfferingsRequest>
        Model::DescribeCapacityBlockExtensionOfferingsOutcomeCallable DescribeCapacityBlockExtensionOfferingsCallable(const DescribeCapacityBlockExtensionOfferingsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityBlockExtensionOfferings, request);
        }

        /**
         * An Async wrapper for DescribeCapacityBlockExtensionOfferings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityBlockExtensionOfferingsRequestT = Model::DescribeCapacityBlockExtensionOfferingsRequest>
        void DescribeCapacityBlockExtensionOfferingsAsync(const DescribeCapacityBlockExtensionOfferingsRequestT& request, const DescribeCapacityBlockExtensionOfferingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityBlockExtensionOfferings, request, handler, context);
        }

        /**
         * <p>Describes Capacity Block offerings available for purchase in the Amazon Web
         * Services Region that you're currently using. With Capacity Blocks, you purchase
         * a specific instance type for a period of time.</p> <p>To search for an available
         * Capacity Block offering, you specify a reservation duration and instance
         * count.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityBlockOfferings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityBlockOfferingsOutcome DescribeCapacityBlockOfferings(const Model::DescribeCapacityBlockOfferingsRequest& request) const;

        /**
         * A Callable wrapper for DescribeCapacityBlockOfferings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityBlockOfferingsRequestT = Model::DescribeCapacityBlockOfferingsRequest>
        Model::DescribeCapacityBlockOfferingsOutcomeCallable DescribeCapacityBlockOfferingsCallable(const DescribeCapacityBlockOfferingsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityBlockOfferings, request);
        }

        /**
         * An Async wrapper for DescribeCapacityBlockOfferings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityBlockOfferingsRequestT = Model::DescribeCapacityBlockOfferingsRequest>
        void DescribeCapacityBlockOfferingsAsync(const DescribeCapacityBlockOfferingsRequestT& request, const DescribeCapacityBlockOfferingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityBlockOfferings, request, handler, context);
        }

        /**
         * <p>Describes a request to assign the billing of the unused capacity of a
         * Capacity Reservation. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/transfer-billing.html">
         * Billing assignment for shared Amazon EC2 Capacity
         * Reservations</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityReservationBillingRequests">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityReservationBillingRequestsOutcome DescribeCapacityReservationBillingRequests(const Model::DescribeCapacityReservationBillingRequestsRequest& request) const;

        /**
         * A Callable wrapper for DescribeCapacityReservationBillingRequests that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityReservationBillingRequestsRequestT = Model::DescribeCapacityReservationBillingRequestsRequest>
        Model::DescribeCapacityReservationBillingRequestsOutcomeCallable DescribeCapacityReservationBillingRequestsCallable(const DescribeCapacityReservationBillingRequestsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityReservationBillingRequests, request);
        }

        /**
         * An Async wrapper for DescribeCapacityReservationBillingRequests that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityReservationBillingRequestsRequestT = Model::DescribeCapacityReservationBillingRequestsRequest>
        void DescribeCapacityReservationBillingRequestsAsync(const DescribeCapacityReservationBillingRequestsRequestT& request, const DescribeCapacityReservationBillingRequestsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityReservationBillingRequests, request, handler, context);
        }

        /**
         * <p>Describes one or more Capacity Reservation Fleets.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityReservationFleets">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityReservationFleetsOutcome DescribeCapacityReservationFleets(const Model::DescribeCapacityReservationFleetsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCapacityReservationFleets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityReservationFleetsRequestT = Model::DescribeCapacityReservationFleetsRequest>
        Model::DescribeCapacityReservationFleetsOutcomeCallable DescribeCapacityReservationFleetsCallable(const DescribeCapacityReservationFleetsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityReservationFleets, request);
        }

        /**
         * An Async wrapper for DescribeCapacityReservationFleets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityReservationFleetsRequestT = Model::DescribeCapacityReservationFleetsRequest>
        void DescribeCapacityReservationFleetsAsync(const DescribeCapacityReservationFleetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCapacityReservationFleetsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityReservationFleets, request, handler, context);
        }

        /**
         * <p>Describes one or more of your Capacity Reservations. The results describe
         * only the Capacity Reservations in the Amazon Web Services Region that you're
         * currently using.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCapacityReservations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCapacityReservationsOutcome DescribeCapacityReservations(const Model::DescribeCapacityReservationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCapacityReservations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCapacityReservationsRequestT = Model::DescribeCapacityReservationsRequest>
        Model::DescribeCapacityReservationsOutcomeCallable DescribeCapacityReservationsCallable(const DescribeCapacityReservationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCapacityReservations, request);
        }

        /**
         * An Async wrapper for DescribeCapacityReservations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCapacityReservationsRequestT = Model::DescribeCapacityReservationsRequest>
        void DescribeCapacityReservationsAsync(const DescribeCapacityReservationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCapacityReservationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCapacityReservations, request, handler, context);
        }

        /**
         * <p>Describes one or more of your carrier gateways.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCarrierGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCarrierGatewaysOutcome DescribeCarrierGateways(const Model::DescribeCarrierGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCarrierGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCarrierGatewaysRequestT = Model::DescribeCarrierGatewaysRequest>
        Model::DescribeCarrierGatewaysOutcomeCallable DescribeCarrierGatewaysCallable(const DescribeCarrierGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCarrierGateways, request);
        }

        /**
         * An Async wrapper for DescribeCarrierGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCarrierGatewaysRequestT = Model::DescribeCarrierGatewaysRequest>
        void DescribeCarrierGatewaysAsync(const DescribeCarrierGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCarrierGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCarrierGateways, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Describes your linked
         * EC2-Classic instances. This request only returns information about EC2-Classic
         * instances linked to a VPC through ClassicLink. You cannot use this request to
         * return information about other instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClassicLinkInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClassicLinkInstancesOutcome DescribeClassicLinkInstances(const Model::DescribeClassicLinkInstancesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeClassicLinkInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClassicLinkInstancesRequestT = Model::DescribeClassicLinkInstancesRequest>
        Model::DescribeClassicLinkInstancesOutcomeCallable DescribeClassicLinkInstancesCallable(const DescribeClassicLinkInstancesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeClassicLinkInstances, request);
        }

        /**
         * An Async wrapper for DescribeClassicLinkInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClassicLinkInstancesRequestT = Model::DescribeClassicLinkInstancesRequest>
        void DescribeClassicLinkInstancesAsync(const DescribeClassicLinkInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeClassicLinkInstancesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeClassicLinkInstances, request, handler, context);
        }

        /**
         * <p>Describes the authorization rules for a specified Client VPN
         * endpoint.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClientVpnAuthorizationRules">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClientVpnAuthorizationRulesOutcome DescribeClientVpnAuthorizationRules(const Model::DescribeClientVpnAuthorizationRulesRequest& request) const;

        /**
         * A Callable wrapper for DescribeClientVpnAuthorizationRules that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClientVpnAuthorizationRulesRequestT = Model::DescribeClientVpnAuthorizationRulesRequest>
        Model::DescribeClientVpnAuthorizationRulesOutcomeCallable DescribeClientVpnAuthorizationRulesCallable(const DescribeClientVpnAuthorizationRulesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeClientVpnAuthorizationRules, request);
        }

        /**
         * An Async wrapper for DescribeClientVpnAuthorizationRules that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClientVpnAuthorizationRulesRequestT = Model::DescribeClientVpnAuthorizationRulesRequest>
        void DescribeClientVpnAuthorizationRulesAsync(const DescribeClientVpnAuthorizationRulesRequestT& request, const DescribeClientVpnAuthorizationRulesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeClientVpnAuthorizationRules, request, handler, context);
        }

        /**
         * <p>Describes active client connections and connections that have been terminated
         * within the last 60 minutes for the specified Client VPN endpoint.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClientVpnConnections">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClientVpnConnectionsOutcome DescribeClientVpnConnections(const Model::DescribeClientVpnConnectionsRequest& request) const;

        /**
         * A Callable wrapper for DescribeClientVpnConnections that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClientVpnConnectionsRequestT = Model::DescribeClientVpnConnectionsRequest>
        Model::DescribeClientVpnConnectionsOutcomeCallable DescribeClientVpnConnectionsCallable(const DescribeClientVpnConnectionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeClientVpnConnections, request);
        }

        /**
         * An Async wrapper for DescribeClientVpnConnections that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClientVpnConnectionsRequestT = Model::DescribeClientVpnConnectionsRequest>
        void DescribeClientVpnConnectionsAsync(const DescribeClientVpnConnectionsRequestT& request, const DescribeClientVpnConnectionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeClientVpnConnections, request, handler, context);
        }

        /**
         * <p>Describes one or more Client VPN endpoints in the account.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClientVpnEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClientVpnEndpointsOutcome DescribeClientVpnEndpoints(const Model::DescribeClientVpnEndpointsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeClientVpnEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClientVpnEndpointsRequestT = Model::DescribeClientVpnEndpointsRequest>
        Model::DescribeClientVpnEndpointsOutcomeCallable DescribeClientVpnEndpointsCallable(const DescribeClientVpnEndpointsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeClientVpnEndpoints, request);
        }

        /**
         * An Async wrapper for DescribeClientVpnEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClientVpnEndpointsRequestT = Model::DescribeClientVpnEndpointsRequest>
        void DescribeClientVpnEndpointsAsync(const DescribeClientVpnEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeClientVpnEndpointsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeClientVpnEndpoints, request, handler, context);
        }

        /**
         * <p>Describes the routes for the specified Client VPN endpoint.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClientVpnRoutes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClientVpnRoutesOutcome DescribeClientVpnRoutes(const Model::DescribeClientVpnRoutesRequest& request) const;

        /**
         * A Callable wrapper for DescribeClientVpnRoutes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClientVpnRoutesRequestT = Model::DescribeClientVpnRoutesRequest>
        Model::DescribeClientVpnRoutesOutcomeCallable DescribeClientVpnRoutesCallable(const DescribeClientVpnRoutesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeClientVpnRoutes, request);
        }

        /**
         * An Async wrapper for DescribeClientVpnRoutes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClientVpnRoutesRequestT = Model::DescribeClientVpnRoutesRequest>
        void DescribeClientVpnRoutesAsync(const DescribeClientVpnRoutesRequestT& request, const DescribeClientVpnRoutesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeClientVpnRoutes, request, handler, context);
        }

        /**
         * <p>Describes the target networks associated with the specified Client VPN
         * endpoint.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeClientVpnTargetNetworks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeClientVpnTargetNetworksOutcome DescribeClientVpnTargetNetworks(const Model::DescribeClientVpnTargetNetworksRequest& request) const;

        /**
         * A Callable wrapper for DescribeClientVpnTargetNetworks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeClientVpnTargetNetworksRequestT = Model::DescribeClientVpnTargetNetworksRequest>
        Model::DescribeClientVpnTargetNetworksOutcomeCallable DescribeClientVpnTargetNetworksCallable(const DescribeClientVpnTargetNetworksRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeClientVpnTargetNetworks, request);
        }

        /**
         * An Async wrapper for DescribeClientVpnTargetNetworks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeClientVpnTargetNetworksRequestT = Model::DescribeClientVpnTargetNetworksRequest>
        void DescribeClientVpnTargetNetworksAsync(const DescribeClientVpnTargetNetworksRequestT& request, const DescribeClientVpnTargetNetworksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeClientVpnTargetNetworks, request, handler, context);
        }

        /**
         * <p>Describes the specified customer-owned address pools or all of your
         * customer-owned address pools.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCoipPools">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCoipPoolsOutcome DescribeCoipPools(const Model::DescribeCoipPoolsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCoipPools that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCoipPoolsRequestT = Model::DescribeCoipPoolsRequest>
        Model::DescribeCoipPoolsOutcomeCallable DescribeCoipPoolsCallable(const DescribeCoipPoolsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCoipPools, request);
        }

        /**
         * An Async wrapper for DescribeCoipPools that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCoipPoolsRequestT = Model::DescribeCoipPoolsRequest>
        void DescribeCoipPoolsAsync(const DescribeCoipPoolsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCoipPoolsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCoipPools, request, handler, context);
        }

        /**
         * <p>Describes the specified conversion tasks or all your conversion tasks. For
         * more information, see the <a
         * href="https://docs.aws.amazon.com/vm-import/latest/userguide/">VM Import/Export
         * User Guide</a>.</p> <p>For information about the import manifest referenced by
         * this API action, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/manifest.html">VM
         * Import Manifest</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeConversionTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeConversionTasksOutcome DescribeConversionTasks(const Model::DescribeConversionTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeConversionTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeConversionTasksRequestT = Model::DescribeConversionTasksRequest>
        Model::DescribeConversionTasksOutcomeCallable DescribeConversionTasksCallable(const DescribeConversionTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeConversionTasks, request);
        }

        /**
         * An Async wrapper for DescribeConversionTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeConversionTasksRequestT = Model::DescribeConversionTasksRequest>
        void DescribeConversionTasksAsync(const DescribeConversionTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeConversionTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeConversionTasks, request, handler, context);
        }

        /**
         * <p>Describes one or more of your VPN customer gateways.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeCustomerGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeCustomerGatewaysOutcome DescribeCustomerGateways(const Model::DescribeCustomerGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeCustomerGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeCustomerGatewaysRequestT = Model::DescribeCustomerGatewaysRequest>
        Model::DescribeCustomerGatewaysOutcomeCallable DescribeCustomerGatewaysCallable(const DescribeCustomerGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeCustomerGateways, request);
        }

        /**
         * An Async wrapper for DescribeCustomerGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeCustomerGatewaysRequestT = Model::DescribeCustomerGatewaysRequest>
        void DescribeCustomerGatewaysAsync(const DescribeCustomerGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeCustomerGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeCustomerGateways, request, handler, context);
        }

        /**
         * <p>Describes the metadata of an account status report, including the status of
         * the report.</p> <p>To view the full report, download it from the Amazon S3
         * bucket where it was saved. Reports are accessible only when they have the
         * <code>complete</code> status. Reports with other statuses (<code>running</code>,
         * <code>cancelled</code>, or <code>error</code>) are not available in the S3
         * bucket. For more information about downloading objects from an S3 bucket, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/download-objects.html">Downloading
         * objects</a> in the <i>Amazon Simple Storage Service User Guide</i>.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_declarative_status-report.html">Generating
         * the account status report for declarative policies</a> in the <i>Amazon Web
         * Services Organizations User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeDeclarativePoliciesReports">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeDeclarativePoliciesReportsOutcome DescribeDeclarativePoliciesReports(const Model::DescribeDeclarativePoliciesReportsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeDeclarativePoliciesReports that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeDeclarativePoliciesReportsRequestT = Model::DescribeDeclarativePoliciesReportsRequest>
        Model::DescribeDeclarativePoliciesReportsOutcomeCallable DescribeDeclarativePoliciesReportsCallable(const DescribeDeclarativePoliciesReportsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeDeclarativePoliciesReports, request);
        }

        /**
         * An Async wrapper for DescribeDeclarativePoliciesReports that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeDeclarativePoliciesReportsRequestT = Model::DescribeDeclarativePoliciesReportsRequest>
        void DescribeDeclarativePoliciesReportsAsync(const DescribeDeclarativePoliciesReportsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeDeclarativePoliciesReportsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeDeclarativePoliciesReports, request, handler, context);
        }

        /**
         * <p>Describes your DHCP option sets. The default is to describe all your DHCP
         * option sets. Alternatively, you can specify specific DHCP option set IDs or
         * filter the results to include only the DHCP option sets that match specific
         * criteria.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html">DHCP
         * option sets</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeDhcpOptions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeDhcpOptionsOutcome DescribeDhcpOptions(const Model::DescribeDhcpOptionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeDhcpOptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeDhcpOptionsRequestT = Model::DescribeDhcpOptionsRequest>
        Model::DescribeDhcpOptionsOutcomeCallable DescribeDhcpOptionsCallable(const DescribeDhcpOptionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeDhcpOptions, request);
        }

        /**
         * An Async wrapper for DescribeDhcpOptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeDhcpOptionsRequestT = Model::DescribeDhcpOptionsRequest>
        void DescribeDhcpOptionsAsync(const DescribeDhcpOptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeDhcpOptionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeDhcpOptions, request, handler, context);
        }

        /**
         * <p>Describes your egress-only internet gateways. The default is to describe all
         * your egress-only internet gateways. Alternatively, you can specify specific
         * egress-only internet gateway IDs or filter the results to include only the
         * egress-only internet gateways that match specific criteria.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeEgressOnlyInternetGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeEgressOnlyInternetGatewaysOutcome DescribeEgressOnlyInternetGateways(const Model::DescribeEgressOnlyInternetGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeEgressOnlyInternetGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeEgressOnlyInternetGatewaysRequestT = Model::DescribeEgressOnlyInternetGatewaysRequest>
        Model::DescribeEgressOnlyInternetGatewaysOutcomeCallable DescribeEgressOnlyInternetGatewaysCallable(const DescribeEgressOnlyInternetGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeEgressOnlyInternetGateways, request);
        }

        /**
         * An Async wrapper for DescribeEgressOnlyInternetGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeEgressOnlyInternetGatewaysRequestT = Model::DescribeEgressOnlyInternetGatewaysRequest>
        void DescribeEgressOnlyInternetGatewaysAsync(const DescribeEgressOnlyInternetGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeEgressOnlyInternetGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeEgressOnlyInternetGateways, request, handler, context);
        }

        /**
         *  <p>Amazon Elastic Graphics reached end of life on January 8, 2024.</p>
         *  <p>Describes the Elastic Graphics accelerator associated with your
         * instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeElasticGpus">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeElasticGpusOutcome DescribeElasticGpus(const Model::DescribeElasticGpusRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeElasticGpus that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeElasticGpusRequestT = Model::DescribeElasticGpusRequest>
        Model::DescribeElasticGpusOutcomeCallable DescribeElasticGpusCallable(const DescribeElasticGpusRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeElasticGpus, request);
        }

        /**
         * An Async wrapper for DescribeElasticGpus that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeElasticGpusRequestT = Model::DescribeElasticGpusRequest>
        void DescribeElasticGpusAsync(const DescribeElasticGpusResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeElasticGpusRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeElasticGpus, request, handler, context);
        }

        /**
         * <p>Describes the specified export image tasks or all of your export image
         * tasks.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeExportImageTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeExportImageTasksOutcome DescribeExportImageTasks(const Model::DescribeExportImageTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeExportImageTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeExportImageTasksRequestT = Model::DescribeExportImageTasksRequest>
        Model::DescribeExportImageTasksOutcomeCallable DescribeExportImageTasksCallable(const DescribeExportImageTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeExportImageTasks, request);
        }

        /**
         * An Async wrapper for DescribeExportImageTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeExportImageTasksRequestT = Model::DescribeExportImageTasksRequest>
        void DescribeExportImageTasksAsync(const DescribeExportImageTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeExportImageTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeExportImageTasks, request, handler, context);
        }

        /**
         * <p>Describes the specified export instance tasks or all of your export instance
         * tasks.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeExportTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeExportTasksOutcome DescribeExportTasks(const Model::DescribeExportTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeExportTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeExportTasksRequestT = Model::DescribeExportTasksRequest>
        Model::DescribeExportTasksOutcomeCallable DescribeExportTasksCallable(const DescribeExportTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeExportTasks, request);
        }

        /**
         * An Async wrapper for DescribeExportTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeExportTasksRequestT = Model::DescribeExportTasksRequest>
        void DescribeExportTasksAsync(const DescribeExportTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeExportTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeExportTasks, request, handler, context);
        }

        /**
         * <p>Describe details for Windows AMIs that are configured for Windows fast
         * launch.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFastLaunchImages">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFastLaunchImagesOutcome DescribeFastLaunchImages(const Model::DescribeFastLaunchImagesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeFastLaunchImages that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFastLaunchImagesRequestT = Model::DescribeFastLaunchImagesRequest>
        Model::DescribeFastLaunchImagesOutcomeCallable DescribeFastLaunchImagesCallable(const DescribeFastLaunchImagesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeFastLaunchImages, request);
        }

        /**
         * An Async wrapper for DescribeFastLaunchImages that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFastLaunchImagesRequestT = Model::DescribeFastLaunchImagesRequest>
        void DescribeFastLaunchImagesAsync(const DescribeFastLaunchImagesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeFastLaunchImagesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeFastLaunchImages, request, handler, context);
        }

        /**
         * <p>Describes the state of fast snapshot restores for your
         * snapshots.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFastSnapshotRestores">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFastSnapshotRestoresOutcome DescribeFastSnapshotRestores(const Model::DescribeFastSnapshotRestoresRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeFastSnapshotRestores that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFastSnapshotRestoresRequestT = Model::DescribeFastSnapshotRestoresRequest>
        Model::DescribeFastSnapshotRestoresOutcomeCallable DescribeFastSnapshotRestoresCallable(const DescribeFastSnapshotRestoresRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeFastSnapshotRestores, request);
        }

        /**
         * An Async wrapper for DescribeFastSnapshotRestores that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFastSnapshotRestoresRequestT = Model::DescribeFastSnapshotRestoresRequest>
        void DescribeFastSnapshotRestoresAsync(const DescribeFastSnapshotRestoresResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeFastSnapshotRestoresRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeFastSnapshotRestores, request, handler, context);
        }

        /**
         * <p>Describes the events for the specified EC2 Fleet during the specified
         * time.</p> <p>EC2 Fleet events are delayed by up to 30 seconds before they can be
         * described. This ensures that you can query by the last evaluated time and not
         * miss a recorded event. EC2 Fleet events are available for 48 hours.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/fleet-monitor.html">Monitor
         * fleet events using Amazon EventBridge</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFleetHistory">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFleetHistoryOutcome DescribeFleetHistory(const Model::DescribeFleetHistoryRequest& request) const;

        /**
         * A Callable wrapper for DescribeFleetHistory that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFleetHistoryRequestT = Model::DescribeFleetHistoryRequest>
        Model::DescribeFleetHistoryOutcomeCallable DescribeFleetHistoryCallable(const DescribeFleetHistoryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeFleetHistory, request);
        }

        /**
         * An Async wrapper for DescribeFleetHistory that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFleetHistoryRequestT = Model::DescribeFleetHistoryRequest>
        void DescribeFleetHistoryAsync(const DescribeFleetHistoryRequestT& request, const DescribeFleetHistoryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeFleetHistory, request, handler, context);
        }

        /**
         * <p>Describes the running instances for the specified EC2 Fleet.</p> 
         * <p>Currently, <code>DescribeFleetInstances</code> does not support fleets of
         * type <code>instant</code>. Instead, use <code>DescribeFleets</code>, specifying
         * the <code>instant</code> fleet ID in the request.</p>  <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#monitor-ec2-fleet">Describe
         * your EC2 Fleet</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFleetInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFleetInstancesOutcome DescribeFleetInstances(const Model::DescribeFleetInstancesRequest& request) const;

        /**
         * A Callable wrapper for DescribeFleetInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFleetInstancesRequestT = Model::DescribeFleetInstancesRequest>
        Model::DescribeFleetInstancesOutcomeCallable DescribeFleetInstancesCallable(const DescribeFleetInstancesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeFleetInstances, request);
        }

        /**
         * An Async wrapper for DescribeFleetInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFleetInstancesRequestT = Model::DescribeFleetInstancesRequest>
        void DescribeFleetInstancesAsync(const DescribeFleetInstancesRequestT& request, const DescribeFleetInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeFleetInstances, request, handler, context);
        }

        /**
         * <p>Describes the specified EC2 Fleet or all of your EC2 Fleets.</p> 
         * <p>If a fleet is of type <code>instant</code>, you must specify the fleet ID in
         * the request, otherwise the fleet does not appear in the response.</p>
         *  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#monitor-ec2-fleet">Describe
         * your EC2 Fleet</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFleets">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFleetsOutcome DescribeFleets(const Model::DescribeFleetsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeFleets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFleetsRequestT = Model::DescribeFleetsRequest>
        Model::DescribeFleetsOutcomeCallable DescribeFleetsCallable(const DescribeFleetsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeFleets, request);
        }

        /**
         * An Async wrapper for DescribeFleets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFleetsRequestT = Model::DescribeFleetsRequest>
        void DescribeFleetsAsync(const DescribeFleetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeFleetsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeFleets, request, handler, context);
        }

        /**
         * <p>Describes one or more flow logs.</p> <p>To view the published flow log
         * records, you must view the log destination. For example, the CloudWatch Logs log
         * group, the Amazon S3 bucket, or the Kinesis Data Firehose delivery
         * stream.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFlowLogs">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFlowLogsOutcome DescribeFlowLogs(const Model::DescribeFlowLogsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeFlowLogs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFlowLogsRequestT = Model::DescribeFlowLogsRequest>
        Model::DescribeFlowLogsOutcomeCallable DescribeFlowLogsCallable(const DescribeFlowLogsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeFlowLogs, request);
        }

        /**
         * An Async wrapper for DescribeFlowLogs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFlowLogsRequestT = Model::DescribeFlowLogsRequest>
        void DescribeFlowLogsAsync(const DescribeFlowLogsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeFlowLogsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeFlowLogs, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified Amazon FPGA Image
         * (AFI).</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFpgaImageAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFpgaImageAttributeOutcome DescribeFpgaImageAttribute(const Model::DescribeFpgaImageAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeFpgaImageAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFpgaImageAttributeRequestT = Model::DescribeFpgaImageAttributeRequest>
        Model::DescribeFpgaImageAttributeOutcomeCallable DescribeFpgaImageAttributeCallable(const DescribeFpgaImageAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeFpgaImageAttribute, request);
        }

        /**
         * An Async wrapper for DescribeFpgaImageAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFpgaImageAttributeRequestT = Model::DescribeFpgaImageAttributeRequest>
        void DescribeFpgaImageAttributeAsync(const DescribeFpgaImageAttributeRequestT& request, const DescribeFpgaImageAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeFpgaImageAttribute, request, handler, context);
        }

        /**
         * <p>Describes the Amazon FPGA Images (AFIs) available to you. These include
         * public AFIs, private AFIs that you own, and AFIs owned by other Amazon Web
         * Services accounts for which you have load permissions.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeFpgaImages">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeFpgaImagesOutcome DescribeFpgaImages(const Model::DescribeFpgaImagesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeFpgaImages that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeFpgaImagesRequestT = Model::DescribeFpgaImagesRequest>
        Model::DescribeFpgaImagesOutcomeCallable DescribeFpgaImagesCallable(const DescribeFpgaImagesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeFpgaImages, request);
        }

        /**
         * An Async wrapper for DescribeFpgaImages that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeFpgaImagesRequestT = Model::DescribeFpgaImagesRequest>
        void DescribeFpgaImagesAsync(const DescribeFpgaImagesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeFpgaImagesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeFpgaImages, request, handler, context);
        }

        /**
         * <p>Describes the Dedicated Host reservations that are available to purchase.</p>
         * <p>The results describe all of the Dedicated Host reservation offerings,
         * including offerings that might not match the instance family and Region of your
         * Dedicated Hosts. When purchasing an offering, ensure that the instance family
         * and Region of the offering matches that of the Dedicated Hosts with which it is
         * to be associated. For more information about supported instance types, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html">Dedicated
         * Hosts</a> in the <i>Amazon EC2 User Guide</i>. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeHostReservationOfferings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeHostReservationOfferingsOutcome DescribeHostReservationOfferings(const Model::DescribeHostReservationOfferingsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeHostReservationOfferings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeHostReservationOfferingsRequestT = Model::DescribeHostReservationOfferingsRequest>
        Model::DescribeHostReservationOfferingsOutcomeCallable DescribeHostReservationOfferingsCallable(const DescribeHostReservationOfferingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeHostReservationOfferings, request);
        }

        /**
         * An Async wrapper for DescribeHostReservationOfferings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeHostReservationOfferingsRequestT = Model::DescribeHostReservationOfferingsRequest>
        void DescribeHostReservationOfferingsAsync(const DescribeHostReservationOfferingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeHostReservationOfferingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeHostReservationOfferings, request, handler, context);
        }

        /**
         * <p>Describes reservations that are associated with Dedicated Hosts in your
         * account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeHostReservations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeHostReservationsOutcome DescribeHostReservations(const Model::DescribeHostReservationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeHostReservations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeHostReservationsRequestT = Model::DescribeHostReservationsRequest>
        Model::DescribeHostReservationsOutcomeCallable DescribeHostReservationsCallable(const DescribeHostReservationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeHostReservations, request);
        }

        /**
         * An Async wrapper for DescribeHostReservations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeHostReservationsRequestT = Model::DescribeHostReservationsRequest>
        void DescribeHostReservationsAsync(const DescribeHostReservationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeHostReservationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeHostReservations, request, handler, context);
        }

        /**
         * <p>Describes the specified Dedicated Hosts or all your Dedicated Hosts.</p>
         * <p>The results describe only the Dedicated Hosts in the Region you're currently
         * using. All listed instances consume capacity on your Dedicated Host. Dedicated
         * Hosts that have recently been released are listed with the state
         * <code>released</code>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeHosts">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeHostsOutcome DescribeHosts(const Model::DescribeHostsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeHosts that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeHostsRequestT = Model::DescribeHostsRequest>
        Model::DescribeHostsOutcomeCallable DescribeHostsCallable(const DescribeHostsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeHosts, request);
        }

        /**
         * An Async wrapper for DescribeHosts that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeHostsRequestT = Model::DescribeHostsRequest>
        void DescribeHostsAsync(const DescribeHostsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeHostsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeHosts, request, handler, context);
        }

        /**
         * <p>Describes your IAM instance profile associations.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIamInstanceProfileAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIamInstanceProfileAssociationsOutcome DescribeIamInstanceProfileAssociations(const Model::DescribeIamInstanceProfileAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIamInstanceProfileAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIamInstanceProfileAssociationsRequestT = Model::DescribeIamInstanceProfileAssociationsRequest>
        Model::DescribeIamInstanceProfileAssociationsOutcomeCallable DescribeIamInstanceProfileAssociationsCallable(const DescribeIamInstanceProfileAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIamInstanceProfileAssociations, request);
        }

        /**
         * An Async wrapper for DescribeIamInstanceProfileAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIamInstanceProfileAssociationsRequestT = Model::DescribeIamInstanceProfileAssociationsRequest>
        void DescribeIamInstanceProfileAssociationsAsync(const DescribeIamInstanceProfileAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIamInstanceProfileAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIamInstanceProfileAssociations, request, handler, context);
        }

        /**
         * <p>Describes the ID format settings for your resources on a per-Region basis,
         * for example, to view which resource types are enabled for longer IDs. This
         * request only returns information about resource types whose ID formats can be
         * modified; it does not return information about other resource types.</p> <p>The
         * following resource types support longer IDs: <code>bundle</code> |
         * <code>conversion-task</code> | <code>customer-gateway</code> |
         * <code>dhcp-options</code> | <code>elastic-ip-allocation</code> |
         * <code>elastic-ip-association</code> | <code>export-task</code> |
         * <code>flow-log</code> | <code>image</code> | <code>import-task</code> |
         * <code>instance</code> | <code>internet-gateway</code> | <code>network-acl</code>
         * | <code>network-acl-association</code> | <code>network-interface</code> |
         * <code>network-interface-attachment</code> | <code>prefix-list</code> |
         * <code>reservation</code> | <code>route-table</code> |
         * <code>route-table-association</code> | <code>security-group</code> |
         * <code>snapshot</code> | <code>subnet</code> |
         * <code>subnet-cidr-block-association</code> | <code>volume</code> |
         * <code>vpc</code> | <code>vpc-cidr-block-association</code> |
         * <code>vpc-endpoint</code> | <code>vpc-peering-connection</code> |
         * <code>vpn-connection</code> | <code>vpn-gateway</code>. </p> <p>These settings
         * apply to the IAM user who makes the request; they do not apply to the entire
         * Amazon Web Services account. By default, an IAM user defaults to the same
         * settings as the root user, unless they explicitly override the settings by
         * running the <a>ModifyIdFormat</a> command. Resources created with longer IDs are
         * visible to all IAM users, regardless of these settings and provided that they
         * have permission to use the relevant <code>Describe</code> command for the
         * resource type.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIdFormat">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIdFormatOutcome DescribeIdFormat(const Model::DescribeIdFormatRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIdFormat that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIdFormatRequestT = Model::DescribeIdFormatRequest>
        Model::DescribeIdFormatOutcomeCallable DescribeIdFormatCallable(const DescribeIdFormatRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIdFormat, request);
        }

        /**
         * An Async wrapper for DescribeIdFormat that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIdFormatRequestT = Model::DescribeIdFormatRequest>
        void DescribeIdFormatAsync(const DescribeIdFormatResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIdFormatRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIdFormat, request, handler, context);
        }

        /**
         * <p>Describes the ID format settings for resources for the specified IAM user,
         * IAM role, or root user. For example, you can view the resource types that are
         * enabled for longer IDs. This request only returns information about resource
         * types whose ID formats can be modified; it does not return information about
         * other resource types. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/resource-ids.html">Resource
         * IDs</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. </p> <p>The
         * following resource types support longer IDs: <code>bundle</code> |
         * <code>conversion-task</code> | <code>customer-gateway</code> |
         * <code>dhcp-options</code> | <code>elastic-ip-allocation</code> |
         * <code>elastic-ip-association</code> | <code>export-task</code> |
         * <code>flow-log</code> | <code>image</code> | <code>import-task</code> |
         * <code>instance</code> | <code>internet-gateway</code> | <code>network-acl</code>
         * | <code>network-acl-association</code> | <code>network-interface</code> |
         * <code>network-interface-attachment</code> | <code>prefix-list</code> |
         * <code>reservation</code> | <code>route-table</code> |
         * <code>route-table-association</code> | <code>security-group</code> |
         * <code>snapshot</code> | <code>subnet</code> |
         * <code>subnet-cidr-block-association</code> | <code>volume</code> |
         * <code>vpc</code> | <code>vpc-cidr-block-association</code> |
         * <code>vpc-endpoint</code> | <code>vpc-peering-connection</code> |
         * <code>vpn-connection</code> | <code>vpn-gateway</code>. </p> <p>These settings
         * apply to the principal specified in the request. They do not apply to the
         * principal that makes the request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIdentityIdFormat">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIdentityIdFormatOutcome DescribeIdentityIdFormat(const Model::DescribeIdentityIdFormatRequest& request) const;

        /**
         * A Callable wrapper for DescribeIdentityIdFormat that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIdentityIdFormatRequestT = Model::DescribeIdentityIdFormatRequest>
        Model::DescribeIdentityIdFormatOutcomeCallable DescribeIdentityIdFormatCallable(const DescribeIdentityIdFormatRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeIdentityIdFormat, request);
        }

        /**
         * An Async wrapper for DescribeIdentityIdFormat that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIdentityIdFormatRequestT = Model::DescribeIdentityIdFormatRequest>
        void DescribeIdentityIdFormatAsync(const DescribeIdentityIdFormatRequestT& request, const DescribeIdentityIdFormatResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeIdentityIdFormat, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified AMI. You can specify only
         * one attribute at a time.</p>  <p>The order of the elements in the
         * response, including those within nested structures, might vary. Applications
         * should not assume the elements appear in a particular order.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeImageAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeImageAttributeOutcome DescribeImageAttribute(const Model::DescribeImageAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeImageAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeImageAttributeRequestT = Model::DescribeImageAttributeRequest>
        Model::DescribeImageAttributeOutcomeCallable DescribeImageAttributeCallable(const DescribeImageAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeImageAttribute, request);
        }

        /**
         * An Async wrapper for DescribeImageAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeImageAttributeRequestT = Model::DescribeImageAttributeRequest>
        void DescribeImageAttributeAsync(const DescribeImageAttributeRequestT& request, const DescribeImageAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeImageAttribute, request, handler, context);
        }

        /**
         * <p>Describes the specified images (AMIs, AKIs, and ARIs) available to you or all
         * of the images available to you.</p> <p>The images available to you include
         * public images, private images that you own, and private images owned by other
         * Amazon Web Services accounts for which you have explicit launch permissions.</p>
         * <p>Recently deregistered images appear in the returned results for a short
         * interval and then return empty results. After all instances that reference a
         * deregistered AMI are terminated, specifying the ID of the image will eventually
         * return an error indicating that the AMI ID cannot be found.</p> <p>When Allowed
         * AMIs is set to <code>enabled</code>, only allowed images are returned in the
         * results, with the <code>imageAllowed</code> field set to <code>true</code> for
         * each image. In <code>audit-mode</code>, the <code>imageAllowed</code> field is
         * set to <code>true</code> for images that meet the account's Allowed AMIs
         * criteria, and <code>false</code> for images that don't meet the criteria. For
         * more information, see <a>EnableAllowedImagesSettings</a>.</p> <p>The Amazon EC2
         * API follows an eventual consistency model. This means that the result of an API
         * command you run that creates or modifies resources might not be immediately
         * available to all subsequent commands you run. For guidance on how to manage
         * eventual consistency, see <a
         * href="https://docs.aws.amazon.com/ec2/latest/devguide/eventual-consistency.html">Eventual
         * consistency in the Amazon EC2 API</a> in the <i>Amazon EC2 Developer
         * Guide</i>.</p>  <p>We strongly recommend using only paginated
         * requests. Unpaginated requests are susceptible to throttling and timeouts.</p>
         *   <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeImages">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeImagesOutcome DescribeImages(const Model::DescribeImagesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeImages that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeImagesRequestT = Model::DescribeImagesRequest>
        Model::DescribeImagesOutcomeCallable DescribeImagesCallable(const DescribeImagesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeImages, request);
        }

        /**
         * An Async wrapper for DescribeImages that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeImagesRequestT = Model::DescribeImagesRequest>
        void DescribeImagesAsync(const DescribeImagesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeImagesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeImages, request, handler, context);
        }

        /**
         * <p>Displays details about an import virtual machine or import snapshot tasks
         * that are already created.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeImportImageTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeImportImageTasksOutcome DescribeImportImageTasks(const Model::DescribeImportImageTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeImportImageTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeImportImageTasksRequestT = Model::DescribeImportImageTasksRequest>
        Model::DescribeImportImageTasksOutcomeCallable DescribeImportImageTasksCallable(const DescribeImportImageTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeImportImageTasks, request);
        }

        /**
         * An Async wrapper for DescribeImportImageTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeImportImageTasksRequestT = Model::DescribeImportImageTasksRequest>
        void DescribeImportImageTasksAsync(const DescribeImportImageTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeImportImageTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeImportImageTasks, request, handler, context);
        }

        /**
         * <p>Describes your import snapshot tasks.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeImportSnapshotTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeImportSnapshotTasksOutcome DescribeImportSnapshotTasks(const Model::DescribeImportSnapshotTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeImportSnapshotTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeImportSnapshotTasksRequestT = Model::DescribeImportSnapshotTasksRequest>
        Model::DescribeImportSnapshotTasksOutcomeCallable DescribeImportSnapshotTasksCallable(const DescribeImportSnapshotTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeImportSnapshotTasks, request);
        }

        /**
         * An Async wrapper for DescribeImportSnapshotTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeImportSnapshotTasksRequestT = Model::DescribeImportSnapshotTasksRequest>
        void DescribeImportSnapshotTasksAsync(const DescribeImportSnapshotTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeImportSnapshotTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeImportSnapshotTasks, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified instance. You can specify
         * only one attribute at a time. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceAttributeOutcome DescribeInstanceAttribute(const Model::DescribeInstanceAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeInstanceAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceAttributeRequestT = Model::DescribeInstanceAttributeRequest>
        Model::DescribeInstanceAttributeOutcomeCallable DescribeInstanceAttributeCallable(const DescribeInstanceAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceAttribute, request);
        }

        /**
         * An Async wrapper for DescribeInstanceAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceAttributeRequestT = Model::DescribeInstanceAttributeRequest>
        void DescribeInstanceAttributeAsync(const DescribeInstanceAttributeRequestT& request, const DescribeInstanceAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceAttribute, request, handler, context);
        }

        /**
         * <p>Describes the specified EC2 Instance Connect Endpoints or all EC2 Instance
         * Connect Endpoints.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceConnectEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceConnectEndpointsOutcome DescribeInstanceConnectEndpoints(const Model::DescribeInstanceConnectEndpointsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceConnectEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceConnectEndpointsRequestT = Model::DescribeInstanceConnectEndpointsRequest>
        Model::DescribeInstanceConnectEndpointsOutcomeCallable DescribeInstanceConnectEndpointsCallable(const DescribeInstanceConnectEndpointsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceConnectEndpoints, request);
        }

        /**
         * An Async wrapper for DescribeInstanceConnectEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceConnectEndpointsRequestT = Model::DescribeInstanceConnectEndpointsRequest>
        void DescribeInstanceConnectEndpointsAsync(const DescribeInstanceConnectEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceConnectEndpointsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceConnectEndpoints, request, handler, context);
        }

        /**
         * <p>Describes the credit option for CPU usage of the specified burstable
         * performance instances. The credit options are <code>standard</code> and
         * <code>unlimited</code>.</p> <p>If you do not specify an instance ID, Amazon EC2
         * returns burstable performance instances with the <code>unlimited</code> credit
         * option, as well as instances that were previously configured as T2, T3, and T3a
         * with the <code>unlimited</code> credit option. For example, if you resize a T2
         * instance, while it is configured as <code>unlimited</code>, to an M4 instance,
         * Amazon EC2 returns the M4 instance.</p> <p>If you specify one or more instance
         * IDs, Amazon EC2 returns the credit option (<code>standard</code> or
         * <code>unlimited</code>) of those instances. If you specify an instance ID that
         * is not valid, such as an instance that is not a burstable performance instance,
         * an error is returned.</p> <p>Recently terminated instances might appear in the
         * returned results. This interval is usually less than one hour.</p> <p>If an
         * Availability Zone is experiencing a service disruption and you specify instance
         * IDs in the affected zone, or do not specify any instance IDs at all, the call
         * fails. If you specify only instance IDs in an unaffected zone, the call works
         * normally.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
         * performance instances</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceCreditSpecifications">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceCreditSpecificationsOutcome DescribeInstanceCreditSpecifications(const Model::DescribeInstanceCreditSpecificationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceCreditSpecifications that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceCreditSpecificationsRequestT = Model::DescribeInstanceCreditSpecificationsRequest>
        Model::DescribeInstanceCreditSpecificationsOutcomeCallable DescribeInstanceCreditSpecificationsCallable(const DescribeInstanceCreditSpecificationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceCreditSpecifications, request);
        }

        /**
         * An Async wrapper for DescribeInstanceCreditSpecifications that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceCreditSpecificationsRequestT = Model::DescribeInstanceCreditSpecificationsRequest>
        void DescribeInstanceCreditSpecificationsAsync(const DescribeInstanceCreditSpecificationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceCreditSpecificationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceCreditSpecifications, request, handler, context);
        }

        /**
         * <p>Describes the tag keys that are registered to appear in scheduled event
         * notifications for resources in the current Region.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceEventNotificationAttributes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceEventNotificationAttributesOutcome DescribeInstanceEventNotificationAttributes(const Model::DescribeInstanceEventNotificationAttributesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceEventNotificationAttributes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceEventNotificationAttributesRequestT = Model::DescribeInstanceEventNotificationAttributesRequest>
        Model::DescribeInstanceEventNotificationAttributesOutcomeCallable DescribeInstanceEventNotificationAttributesCallable(const DescribeInstanceEventNotificationAttributesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceEventNotificationAttributes, request);
        }

        /**
         * An Async wrapper for DescribeInstanceEventNotificationAttributes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceEventNotificationAttributesRequestT = Model::DescribeInstanceEventNotificationAttributesRequest>
        void DescribeInstanceEventNotificationAttributesAsync(const DescribeInstanceEventNotificationAttributesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceEventNotificationAttributesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceEventNotificationAttributes, request, handler, context);
        }

        /**
         * <p>Describes the specified event windows or all event windows.</p> <p>If you
         * specify event window IDs, the output includes information for only the specified
         * event windows. If you specify filters, the output includes information for only
         * those event windows that meet the filter criteria. If you do not specify event
         * windows IDs or filters, the output includes information for all event windows,
         * which can affect performance. We recommend that you use pagination to ensure
         * that the operation returns quickly and successfully. </p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/event-windows.html">Define
         * event windows for scheduled events</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceEventWindows">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceEventWindowsOutcome DescribeInstanceEventWindows(const Model::DescribeInstanceEventWindowsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceEventWindows that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceEventWindowsRequestT = Model::DescribeInstanceEventWindowsRequest>
        Model::DescribeInstanceEventWindowsOutcomeCallable DescribeInstanceEventWindowsCallable(const DescribeInstanceEventWindowsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceEventWindows, request);
        }

        /**
         * An Async wrapper for DescribeInstanceEventWindows that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceEventWindowsRequestT = Model::DescribeInstanceEventWindowsRequest>
        void DescribeInstanceEventWindowsAsync(const DescribeInstanceEventWindowsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceEventWindowsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceEventWindows, request, handler, context);
        }

        /**
         * <p>Describes the AMI that was used to launch an instance, even if the AMI is
         * deprecated, deregistered, made private (no longer public or shared with your
         * account), or not allowed.</p> <p>If you specify instance IDs, the output
         * includes information for only the specified instances. If you specify filters,
         * the output includes information for only those instances that meet the filter
         * criteria. If you do not specify instance IDs or filters, the output includes
         * information for all instances, which can affect performance.</p> <p>If you
         * specify an instance ID that is not valid, an instance that doesn't exist, or an
         * instance that you do not own, an error (<code>InvalidInstanceID.NotFound</code>)
         * is returned.</p> <p>Recently terminated instances might appear in the returned
         * results. This interval is usually less than one hour.</p> <p>In the rare case
         * where an Availability Zone is experiencing a service disruption and you specify
         * instance IDs that are in the affected Availability Zone, or do not specify any
         * instance IDs at all, the call fails. If you specify only instance IDs that are
         * in an unaffected Availability Zone, the call works normally.</p>  <p>The
         * order of the elements in the response, including those within nested structures,
         * might vary. Applications should not assume the elements appear in a particular
         * order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceImageMetadata">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceImageMetadataOutcome DescribeInstanceImageMetadata(const Model::DescribeInstanceImageMetadataRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceImageMetadata that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceImageMetadataRequestT = Model::DescribeInstanceImageMetadataRequest>
        Model::DescribeInstanceImageMetadataOutcomeCallable DescribeInstanceImageMetadataCallable(const DescribeInstanceImageMetadataRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceImageMetadata, request);
        }

        /**
         * An Async wrapper for DescribeInstanceImageMetadata that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceImageMetadataRequestT = Model::DescribeInstanceImageMetadataRequest>
        void DescribeInstanceImageMetadataAsync(const DescribeInstanceImageMetadataResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceImageMetadataRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceImageMetadata, request, handler, context);
        }

        /**
         * <p>Describes the status of the specified instances or all of your instances. By
         * default, only running instances are described, unless you specifically indicate
         * to return the status of all instances.</p> <p>Instance status includes the
         * following components:</p> <ul> <li> <p> <b>Status checks</b> - Amazon EC2
         * performs status checks on running EC2 instances to identify hardware and
         * software issues. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/monitoring-system-instance-status-check.html">Status
         * checks for your instances</a> and <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/TroubleshootingInstances.html">Troubleshoot
         * instances with failed status checks</a> in the <i>Amazon EC2 User Guide</i>.</p>
         * </li> <li> <p> <b>Scheduled events</b> - Amazon EC2 can schedule events (such as
         * reboot, stop, or terminate) for your instances related to hardware issues,
         * software updates, or system maintenance. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/monitoring-instances-status-check_sched.html">Scheduled
         * events for your instances</a> in the <i>Amazon EC2 User Guide</i>.</p> </li>
         * <li> <p> <b>Instance state</b> - You can manage your instances from the moment
         * you launch them through their termination. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-lifecycle.html">Instance
         * lifecycle</a> in the <i>Amazon EC2 User Guide</i>.</p> </li> </ul> <p>The Amazon
         * EC2 API follows an eventual consistency model. This means that the result of an
         * API command you run that creates or modifies resources might not be immediately
         * available to all subsequent commands you run. For guidance on how to manage
         * eventual consistency, see <a
         * href="https://docs.aws.amazon.com/ec2/latest/devguide/eventual-consistency.html">Eventual
         * consistency in the Amazon EC2 API</a> in the <i>Amazon EC2 Developer
         * Guide</i>.</p>  <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceStatus">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceStatusOutcome DescribeInstanceStatus(const Model::DescribeInstanceStatusRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceStatus that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceStatusRequestT = Model::DescribeInstanceStatusRequest>
        Model::DescribeInstanceStatusOutcomeCallable DescribeInstanceStatusCallable(const DescribeInstanceStatusRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceStatus, request);
        }

        /**
         * An Async wrapper for DescribeInstanceStatus that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceStatusRequestT = Model::DescribeInstanceStatusRequest>
        void DescribeInstanceStatusAsync(const DescribeInstanceStatusResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceStatusRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceStatus, request, handler, context);
        }

        /**
         * <p>Describes a tree-based hierarchy that represents the physical host placement
         * of your EC2 instances within an Availability Zone or Local Zone. You can use
         * this information to determine the relative proximity of your EC2 instances
         * within the Amazon Web Services network to support your tightly coupled
         * workloads.</p> <p class="title"> <b>Limitations</b> </p> <ul> <li> <p>Supported
         * zones</p> <ul> <li> <p>Availability Zone</p> </li> <li> <p>Local Zone</p> </li>
         * </ul> </li> <li> <p>Supported instance types</p> <ul> <li> <p>Returns 3 network
         * nodes in the response</p> <ul> <li> <p> <code>hpc6a.48xlarge</code> |
         * <code>hpc6id.32xlarge</code> | <code>hpc7a.12xlarge</code> |
         * <code>hpc7a.24xlarge</code> | <code>hpc7a.48xlarge</code> |
         * <code>hpc7a.96xlarge</code> | <code>hpc7g.4xlarge</code> |
         * <code>hpc7g.8xlarge</code> | <code>hpc7g.16xlarge</code> </p> </li> <li> <p>
         * <code>p3dn.24xlarge</code> | <code>p4d.24xlarge</code> |
         * <code>p4de.24xlarge</code> | <code>p5.48xlarge</code> |
         * <code>p5e.48xlarge</code> | <code>p5en.48xlarge</code> </p> </li> <li> <p>
         * <code>trn1.2xlarge</code> | <code>trn1.32xlarge</code> |
         * <code>trn1n.32xlarge</code> | <code>trn2.48xlarge</code> |
         * <code>trn2u.48xlarge</code> </p> </li> </ul> </li> <li> <p>Returns 4 network
         * nodes in the response</p> <ul> <li> <p> <code>p6-b200.48xlarge</code> </p> </li>
         * </ul> </li> </ul> </li> </ul> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-topology.html">Amazon
         * EC2 instance topology</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceTopology">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceTopologyOutcome DescribeInstanceTopology(const Model::DescribeInstanceTopologyRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceTopology that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceTopologyRequestT = Model::DescribeInstanceTopologyRequest>
        Model::DescribeInstanceTopologyOutcomeCallable DescribeInstanceTopologyCallable(const DescribeInstanceTopologyRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceTopology, request);
        }

        /**
         * An Async wrapper for DescribeInstanceTopology that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceTopologyRequestT = Model::DescribeInstanceTopologyRequest>
        void DescribeInstanceTopologyAsync(const DescribeInstanceTopologyResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceTopologyRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceTopology, request, handler, context);
        }

        /**
         * <p>Lists the instance types that are offered for the specified location. If no
         * location is specified, the default is to list the instance types that are
         * offered in the current Region.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceTypeOfferings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceTypeOfferingsOutcome DescribeInstanceTypeOfferings(const Model::DescribeInstanceTypeOfferingsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceTypeOfferings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceTypeOfferingsRequestT = Model::DescribeInstanceTypeOfferingsRequest>
        Model::DescribeInstanceTypeOfferingsOutcomeCallable DescribeInstanceTypeOfferingsCallable(const DescribeInstanceTypeOfferingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceTypeOfferings, request);
        }

        /**
         * An Async wrapper for DescribeInstanceTypeOfferings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceTypeOfferingsRequestT = Model::DescribeInstanceTypeOfferingsRequest>
        void DescribeInstanceTypeOfferingsAsync(const DescribeInstanceTypeOfferingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceTypeOfferingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceTypeOfferings, request, handler, context);
        }

        /**
         * <p>Describes the specified instance types. By default, all instance types for
         * the current Region are described. Alternatively, you can filter the
         * results.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstanceTypes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstanceTypesOutcome DescribeInstanceTypes(const Model::DescribeInstanceTypesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstanceTypes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstanceTypesRequestT = Model::DescribeInstanceTypesRequest>
        Model::DescribeInstanceTypesOutcomeCallable DescribeInstanceTypesCallable(const DescribeInstanceTypesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstanceTypes, request);
        }

        /**
         * An Async wrapper for DescribeInstanceTypes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstanceTypesRequestT = Model::DescribeInstanceTypesRequest>
        void DescribeInstanceTypesAsync(const DescribeInstanceTypesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstanceTypesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstanceTypes, request, handler, context);
        }

        /**
         * <p>Describes the specified instances or all instances.</p> <p>If you specify
         * instance IDs, the output includes information for only the specified instances.
         * If you specify filters, the output includes information for only those instances
         * that meet the filter criteria. If you do not specify instance IDs or filters,
         * the output includes information for all instances, which can affect performance.
         * We recommend that you use pagination to ensure that the operation returns
         * quickly and successfully.</p> <p>If you specify an instance ID that is not
         * valid, an error is returned. If you specify an instance that you do not own, it
         * is not included in the output.</p> <p>Recently terminated instances might appear
         * in the returned results. This interval is usually less than one hour.</p> <p>If
         * you describe instances in the rare case where an Availability Zone is
         * experiencing a service disruption and you specify instance IDs that are in the
         * affected zone, or do not specify any instance IDs at all, the call fails. If you
         * describe instances and specify only instance IDs that are in an unaffected zone,
         * the call works normally.</p> <p>The Amazon EC2 API follows an eventual
         * consistency model. This means that the result of an API command you run that
         * creates or modifies resources might not be immediately available to all
         * subsequent commands you run. For guidance on how to manage eventual consistency,
         * see <a
         * href="https://docs.aws.amazon.com/ec2/latest/devguide/eventual-consistency.html">Eventual
         * consistency in the Amazon EC2 API</a> in the <i>Amazon EC2 Developer
         * Guide</i>.</p>  <p>We strongly recommend using only paginated
         * requests. Unpaginated requests are susceptible to throttling and timeouts.</p>
         *   <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInstancesOutcome DescribeInstances(const Model::DescribeInstancesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInstancesRequestT = Model::DescribeInstancesRequest>
        Model::DescribeInstancesOutcomeCallable DescribeInstancesCallable(const DescribeInstancesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInstances, request);
        }

        /**
         * An Async wrapper for DescribeInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInstancesRequestT = Model::DescribeInstancesRequest>
        void DescribeInstancesAsync(const DescribeInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInstancesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInstances, request, handler, context);
        }

        /**
         * <p>Describes your internet gateways. The default is to describe all your
         * internet gateways. Alternatively, you can specify specific internet gateway IDs
         * or filter the results to include only the internet gateways that match specific
         * criteria.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeInternetGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeInternetGatewaysOutcome DescribeInternetGateways(const Model::DescribeInternetGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeInternetGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeInternetGatewaysRequestT = Model::DescribeInternetGatewaysRequest>
        Model::DescribeInternetGatewaysOutcomeCallable DescribeInternetGatewaysCallable(const DescribeInternetGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeInternetGateways, request);
        }

        /**
         * An Async wrapper for DescribeInternetGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeInternetGatewaysRequestT = Model::DescribeInternetGatewaysRequest>
        void DescribeInternetGatewaysAsync(const DescribeInternetGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeInternetGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeInternetGateways, request, handler, context);
        }

        /**
         * <p>Describes your Autonomous System Numbers (ASNs), their provisioning statuses,
         * and the BYOIP CIDRs with which they are associated. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-byoasn.html">Tutorial:
         * Bring your ASN to IPAM</a> in the <i>Amazon VPC IPAM guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamByoasn">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamByoasnOutcome DescribeIpamByoasn(const Model::DescribeIpamByoasnRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamByoasn that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamByoasnRequestT = Model::DescribeIpamByoasnRequest>
        Model::DescribeIpamByoasnOutcomeCallable DescribeIpamByoasnCallable(const DescribeIpamByoasnRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamByoasn, request);
        }

        /**
         * An Async wrapper for DescribeIpamByoasn that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamByoasnRequestT = Model::DescribeIpamByoasnRequest>
        void DescribeIpamByoasnAsync(const DescribeIpamByoasnResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamByoasnRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamByoasn, request, handler, context);
        }

        /**
         * <p>Describe verification tokens. A verification token is an Amazon Web
         * Services-generated random value that you can use to prove ownership of an
         * external resource. For example, you can use a verification token to validate
         * that you control a public IP address range when you bring an IP address range to
         * Amazon Web Services (BYOIP). </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamExternalResourceVerificationTokens">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamExternalResourceVerificationTokensOutcome DescribeIpamExternalResourceVerificationTokens(const Model::DescribeIpamExternalResourceVerificationTokensRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamExternalResourceVerificationTokens that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamExternalResourceVerificationTokensRequestT = Model::DescribeIpamExternalResourceVerificationTokensRequest>
        Model::DescribeIpamExternalResourceVerificationTokensOutcomeCallable DescribeIpamExternalResourceVerificationTokensCallable(const DescribeIpamExternalResourceVerificationTokensRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamExternalResourceVerificationTokens, request);
        }

        /**
         * An Async wrapper for DescribeIpamExternalResourceVerificationTokens that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamExternalResourceVerificationTokensRequestT = Model::DescribeIpamExternalResourceVerificationTokensRequest>
        void DescribeIpamExternalResourceVerificationTokensAsync(const DescribeIpamExternalResourceVerificationTokensResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamExternalResourceVerificationTokensRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamExternalResourceVerificationTokens, request, handler, context);
        }

        /**
         * <p>Get information about your IPAM pools.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamPools">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamPoolsOutcome DescribeIpamPools(const Model::DescribeIpamPoolsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamPools that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamPoolsRequestT = Model::DescribeIpamPoolsRequest>
        Model::DescribeIpamPoolsOutcomeCallable DescribeIpamPoolsCallable(const DescribeIpamPoolsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamPools, request);
        }

        /**
         * An Async wrapper for DescribeIpamPools that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamPoolsRequestT = Model::DescribeIpamPoolsRequest>
        void DescribeIpamPoolsAsync(const DescribeIpamPoolsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamPoolsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamPools, request, handler, context);
        }

        /**
         * <p>Describes IPAM resource discoveries. A resource discovery is an IPAM
         * component that enables IPAM to manage and monitor resources that belong to the
         * owning account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamResourceDiscoveries">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamResourceDiscoveriesOutcome DescribeIpamResourceDiscoveries(const Model::DescribeIpamResourceDiscoveriesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamResourceDiscoveries that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamResourceDiscoveriesRequestT = Model::DescribeIpamResourceDiscoveriesRequest>
        Model::DescribeIpamResourceDiscoveriesOutcomeCallable DescribeIpamResourceDiscoveriesCallable(const DescribeIpamResourceDiscoveriesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamResourceDiscoveries, request);
        }

        /**
         * An Async wrapper for DescribeIpamResourceDiscoveries that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamResourceDiscoveriesRequestT = Model::DescribeIpamResourceDiscoveriesRequest>
        void DescribeIpamResourceDiscoveriesAsync(const DescribeIpamResourceDiscoveriesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamResourceDiscoveriesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamResourceDiscoveries, request, handler, context);
        }

        /**
         * <p>Describes resource discovery association with an Amazon VPC IPAM. An
         * associated resource discovery is a resource discovery that has been associated
         * with an IPAM..</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamResourceDiscoveryAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamResourceDiscoveryAssociationsOutcome DescribeIpamResourceDiscoveryAssociations(const Model::DescribeIpamResourceDiscoveryAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamResourceDiscoveryAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamResourceDiscoveryAssociationsRequestT = Model::DescribeIpamResourceDiscoveryAssociationsRequest>
        Model::DescribeIpamResourceDiscoveryAssociationsOutcomeCallable DescribeIpamResourceDiscoveryAssociationsCallable(const DescribeIpamResourceDiscoveryAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamResourceDiscoveryAssociations, request);
        }

        /**
         * An Async wrapper for DescribeIpamResourceDiscoveryAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamResourceDiscoveryAssociationsRequestT = Model::DescribeIpamResourceDiscoveryAssociationsRequest>
        void DescribeIpamResourceDiscoveryAssociationsAsync(const DescribeIpamResourceDiscoveryAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamResourceDiscoveryAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamResourceDiscoveryAssociations, request, handler, context);
        }

        /**
         * <p>Get information about your IPAM scopes.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpamScopes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamScopesOutcome DescribeIpamScopes(const Model::DescribeIpamScopesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpamScopes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamScopesRequestT = Model::DescribeIpamScopesRequest>
        Model::DescribeIpamScopesOutcomeCallable DescribeIpamScopesCallable(const DescribeIpamScopesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpamScopes, request);
        }

        /**
         * An Async wrapper for DescribeIpamScopes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamScopesRequestT = Model::DescribeIpamScopesRequest>
        void DescribeIpamScopesAsync(const DescribeIpamScopesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamScopesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpamScopes, request, handler, context);
        }

        /**
         * <p>Get information about your IPAM pools.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is
         * IPAM?</a> in the <i>Amazon VPC IPAM User Guide</i>. </p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpams">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpamsOutcome DescribeIpams(const Model::DescribeIpamsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpams that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpamsRequestT = Model::DescribeIpamsRequest>
        Model::DescribeIpamsOutcomeCallable DescribeIpamsCallable(const DescribeIpamsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpams, request);
        }

        /**
         * An Async wrapper for DescribeIpams that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpamsRequestT = Model::DescribeIpamsRequest>
        void DescribeIpamsAsync(const DescribeIpamsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpamsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpams, request, handler, context);
        }

        /**
         * <p>Describes your IPv6 address pools.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeIpv6Pools">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeIpv6PoolsOutcome DescribeIpv6Pools(const Model::DescribeIpv6PoolsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeIpv6Pools that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeIpv6PoolsRequestT = Model::DescribeIpv6PoolsRequest>
        Model::DescribeIpv6PoolsOutcomeCallable DescribeIpv6PoolsCallable(const DescribeIpv6PoolsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeIpv6Pools, request);
        }

        /**
         * An Async wrapper for DescribeIpv6Pools that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeIpv6PoolsRequestT = Model::DescribeIpv6PoolsRequest>
        void DescribeIpv6PoolsAsync(const DescribeIpv6PoolsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeIpv6PoolsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeIpv6Pools, request, handler, context);
        }

        /**
         * <p>Describes the specified key pairs or all of your key pairs.</p> <p>For more
         * information about key pairs, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html">Amazon
         * EC2 key pairs</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeKeyPairs">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeKeyPairsOutcome DescribeKeyPairs(const Model::DescribeKeyPairsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeKeyPairs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeKeyPairsRequestT = Model::DescribeKeyPairsRequest>
        Model::DescribeKeyPairsOutcomeCallable DescribeKeyPairsCallable(const DescribeKeyPairsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeKeyPairs, request);
        }

        /**
         * An Async wrapper for DescribeKeyPairs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeKeyPairsRequestT = Model::DescribeKeyPairsRequest>
        void DescribeKeyPairsAsync(const DescribeKeyPairsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeKeyPairsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeKeyPairs, request, handler, context);
        }

        /**
         * <p>Describes one or more versions of a specified launch template. You can
         * describe all versions, individual versions, or a range of versions. You can also
         * describe all the latest versions or all the default versions of all the launch
         * templates in your account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLaunchTemplateVersions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLaunchTemplateVersionsOutcome DescribeLaunchTemplateVersions(const Model::DescribeLaunchTemplateVersionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLaunchTemplateVersions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLaunchTemplateVersionsRequestT = Model::DescribeLaunchTemplateVersionsRequest>
        Model::DescribeLaunchTemplateVersionsOutcomeCallable DescribeLaunchTemplateVersionsCallable(const DescribeLaunchTemplateVersionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLaunchTemplateVersions, request);
        }

        /**
         * An Async wrapper for DescribeLaunchTemplateVersions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLaunchTemplateVersionsRequestT = Model::DescribeLaunchTemplateVersionsRequest>
        void DescribeLaunchTemplateVersionsAsync(const DescribeLaunchTemplateVersionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLaunchTemplateVersionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLaunchTemplateVersions, request, handler, context);
        }

        /**
         * <p>Describes one or more launch templates.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLaunchTemplates">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLaunchTemplatesOutcome DescribeLaunchTemplates(const Model::DescribeLaunchTemplatesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLaunchTemplates that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLaunchTemplatesRequestT = Model::DescribeLaunchTemplatesRequest>
        Model::DescribeLaunchTemplatesOutcomeCallable DescribeLaunchTemplatesCallable(const DescribeLaunchTemplatesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLaunchTemplates, request);
        }

        /**
         * An Async wrapper for DescribeLaunchTemplates that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLaunchTemplatesRequestT = Model::DescribeLaunchTemplatesRequest>
        void DescribeLaunchTemplatesAsync(const DescribeLaunchTemplatesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLaunchTemplatesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLaunchTemplates, request, handler, context);
        }

        /**
         * <p>Describes the associations between virtual interface groups and local gateway
         * route tables.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsOutcome DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations(const Model::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequestT = Model::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequest>
        Model::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsOutcomeCallable DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsCallable(const DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations, request);
        }

        /**
         * An Async wrapper for DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequestT = Model::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequest>
        void DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsAsync(const DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations, request, handler, context);
        }

        /**
         * <p>Describes the specified associations between VPCs and local gateway route
         * tables.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGatewayRouteTableVpcAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewayRouteTableVpcAssociationsOutcome DescribeLocalGatewayRouteTableVpcAssociations(const Model::DescribeLocalGatewayRouteTableVpcAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGatewayRouteTableVpcAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewayRouteTableVpcAssociationsRequestT = Model::DescribeLocalGatewayRouteTableVpcAssociationsRequest>
        Model::DescribeLocalGatewayRouteTableVpcAssociationsOutcomeCallable DescribeLocalGatewayRouteTableVpcAssociationsCallable(const DescribeLocalGatewayRouteTableVpcAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGatewayRouteTableVpcAssociations, request);
        }

        /**
         * An Async wrapper for DescribeLocalGatewayRouteTableVpcAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewayRouteTableVpcAssociationsRequestT = Model::DescribeLocalGatewayRouteTableVpcAssociationsRequest>
        void DescribeLocalGatewayRouteTableVpcAssociationsAsync(const DescribeLocalGatewayRouteTableVpcAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewayRouteTableVpcAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGatewayRouteTableVpcAssociations, request, handler, context);
        }

        /**
         * <p>Describes one or more local gateway route tables. By default, all local
         * gateway route tables are described. Alternatively, you can filter the
         * results.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGatewayRouteTables">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewayRouteTablesOutcome DescribeLocalGatewayRouteTables(const Model::DescribeLocalGatewayRouteTablesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGatewayRouteTables that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewayRouteTablesRequestT = Model::DescribeLocalGatewayRouteTablesRequest>
        Model::DescribeLocalGatewayRouteTablesOutcomeCallable DescribeLocalGatewayRouteTablesCallable(const DescribeLocalGatewayRouteTablesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGatewayRouteTables, request);
        }

        /**
         * An Async wrapper for DescribeLocalGatewayRouteTables that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewayRouteTablesRequestT = Model::DescribeLocalGatewayRouteTablesRequest>
        void DescribeLocalGatewayRouteTablesAsync(const DescribeLocalGatewayRouteTablesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewayRouteTablesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGatewayRouteTables, request, handler, context);
        }

        /**
         * <p>Describes the specified local gateway virtual interface groups.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGatewayVirtualInterfaceGroups">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewayVirtualInterfaceGroupsOutcome DescribeLocalGatewayVirtualInterfaceGroups(const Model::DescribeLocalGatewayVirtualInterfaceGroupsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGatewayVirtualInterfaceGroups that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewayVirtualInterfaceGroupsRequestT = Model::DescribeLocalGatewayVirtualInterfaceGroupsRequest>
        Model::DescribeLocalGatewayVirtualInterfaceGroupsOutcomeCallable DescribeLocalGatewayVirtualInterfaceGroupsCallable(const DescribeLocalGatewayVirtualInterfaceGroupsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGatewayVirtualInterfaceGroups, request);
        }

        /**
         * An Async wrapper for DescribeLocalGatewayVirtualInterfaceGroups that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewayVirtualInterfaceGroupsRequestT = Model::DescribeLocalGatewayVirtualInterfaceGroupsRequest>
        void DescribeLocalGatewayVirtualInterfaceGroupsAsync(const DescribeLocalGatewayVirtualInterfaceGroupsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewayVirtualInterfaceGroupsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGatewayVirtualInterfaceGroups, request, handler, context);
        }

        /**
         * <p>Describes the specified local gateway virtual interfaces.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGatewayVirtualInterfaces">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewayVirtualInterfacesOutcome DescribeLocalGatewayVirtualInterfaces(const Model::DescribeLocalGatewayVirtualInterfacesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGatewayVirtualInterfaces that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewayVirtualInterfacesRequestT = Model::DescribeLocalGatewayVirtualInterfacesRequest>
        Model::DescribeLocalGatewayVirtualInterfacesOutcomeCallable DescribeLocalGatewayVirtualInterfacesCallable(const DescribeLocalGatewayVirtualInterfacesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGatewayVirtualInterfaces, request);
        }

        /**
         * An Async wrapper for DescribeLocalGatewayVirtualInterfaces that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewayVirtualInterfacesRequestT = Model::DescribeLocalGatewayVirtualInterfacesRequest>
        void DescribeLocalGatewayVirtualInterfacesAsync(const DescribeLocalGatewayVirtualInterfacesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewayVirtualInterfacesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGatewayVirtualInterfaces, request, handler, context);
        }

        /**
         * <p>Describes one or more local gateways. By default, all local gateways are
         * described. Alternatively, you can filter the results.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLocalGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLocalGatewaysOutcome DescribeLocalGateways(const Model::DescribeLocalGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLocalGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLocalGatewaysRequestT = Model::DescribeLocalGatewaysRequest>
        Model::DescribeLocalGatewaysOutcomeCallable DescribeLocalGatewaysCallable(const DescribeLocalGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLocalGateways, request);
        }

        /**
         * An Async wrapper for DescribeLocalGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLocalGatewaysRequestT = Model::DescribeLocalGatewaysRequest>
        void DescribeLocalGatewaysAsync(const DescribeLocalGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLocalGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLocalGateways, request, handler, context);
        }

        /**
         * <p>Describes the lock status for a snapshot.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeLockedSnapshots">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeLockedSnapshotsOutcome DescribeLockedSnapshots(const Model::DescribeLockedSnapshotsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeLockedSnapshots that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeLockedSnapshotsRequestT = Model::DescribeLockedSnapshotsRequest>
        Model::DescribeLockedSnapshotsOutcomeCallable DescribeLockedSnapshotsCallable(const DescribeLockedSnapshotsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeLockedSnapshots, request);
        }

        /**
         * An Async wrapper for DescribeLockedSnapshots that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeLockedSnapshotsRequestT = Model::DescribeLockedSnapshotsRequest>
        void DescribeLockedSnapshotsAsync(const DescribeLockedSnapshotsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeLockedSnapshotsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeLockedSnapshots, request, handler, context);
        }

        /**
         * <p>Describes the specified EC2 Mac Dedicated Host or all of your EC2 Mac
         * Dedicated Hosts.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeMacHosts">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeMacHostsOutcome DescribeMacHosts(const Model::DescribeMacHostsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeMacHosts that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeMacHostsRequestT = Model::DescribeMacHostsRequest>
        Model::DescribeMacHostsOutcomeCallable DescribeMacHostsCallable(const DescribeMacHostsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeMacHosts, request);
        }

        /**
         * An Async wrapper for DescribeMacHosts that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeMacHostsRequestT = Model::DescribeMacHostsRequest>
        void DescribeMacHostsAsync(const DescribeMacHostsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeMacHostsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeMacHosts, request, handler, context);
        }

        /**
         * <p>Describes a System Integrity Protection (SIP) modification task or volume
         * ownership delegation task for an Amazon EC2 Mac instance. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/mac-sip-settings.html#mac-sip-configure">Configure
         * SIP for Amazon EC2 instances</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeMacModificationTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeMacModificationTasksOutcome DescribeMacModificationTasks(const Model::DescribeMacModificationTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeMacModificationTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeMacModificationTasksRequestT = Model::DescribeMacModificationTasksRequest>
        Model::DescribeMacModificationTasksOutcomeCallable DescribeMacModificationTasksCallable(const DescribeMacModificationTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeMacModificationTasks, request);
        }

        /**
         * An Async wrapper for DescribeMacModificationTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeMacModificationTasksRequestT = Model::DescribeMacModificationTasksRequest>
        void DescribeMacModificationTasksAsync(const DescribeMacModificationTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeMacModificationTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeMacModificationTasks, request, handler, context);
        }

        /**
         * <p>Describes your managed prefix lists and any Amazon Web Services-managed
         * prefix lists.</p> <p>To view the entries for your prefix list, use
         * <a>GetManagedPrefixListEntries</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeManagedPrefixLists">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeManagedPrefixListsOutcome DescribeManagedPrefixLists(const Model::DescribeManagedPrefixListsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeManagedPrefixLists that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeManagedPrefixListsRequestT = Model::DescribeManagedPrefixListsRequest>
        Model::DescribeManagedPrefixListsOutcomeCallable DescribeManagedPrefixListsCallable(const DescribeManagedPrefixListsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeManagedPrefixLists, request);
        }

        /**
         * An Async wrapper for DescribeManagedPrefixLists that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeManagedPrefixListsRequestT = Model::DescribeManagedPrefixListsRequest>
        void DescribeManagedPrefixListsAsync(const DescribeManagedPrefixListsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeManagedPrefixListsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeManagedPrefixLists, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Describes your Elastic IP
         * addresses that are being moved from or being restored to the EC2-Classic
         * platform. This request does not return information about any other Elastic IP
         * addresses in your account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeMovingAddresses">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeMovingAddressesOutcome DescribeMovingAddresses(const Model::DescribeMovingAddressesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeMovingAddresses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeMovingAddressesRequestT = Model::DescribeMovingAddressesRequest>
        Model::DescribeMovingAddressesOutcomeCallable DescribeMovingAddressesCallable(const DescribeMovingAddressesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeMovingAddresses, request);
        }

        /**
         * An Async wrapper for DescribeMovingAddresses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeMovingAddressesRequestT = Model::DescribeMovingAddressesRequest>
        void DescribeMovingAddressesAsync(const DescribeMovingAddressesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeMovingAddressesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeMovingAddresses, request, handler, context);
        }

        /**
         * <p>Describes your NAT gateways. The default is to describe all your NAT
         * gateways. Alternatively, you can specify specific NAT gateway IDs or filter the
         * results to include only the NAT gateways that match specific
         * criteria.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNatGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNatGatewaysOutcome DescribeNatGateways(const Model::DescribeNatGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNatGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNatGatewaysRequestT = Model::DescribeNatGatewaysRequest>
        Model::DescribeNatGatewaysOutcomeCallable DescribeNatGatewaysCallable(const DescribeNatGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNatGateways, request);
        }

        /**
         * An Async wrapper for DescribeNatGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNatGatewaysRequestT = Model::DescribeNatGatewaysRequest>
        void DescribeNatGatewaysAsync(const DescribeNatGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNatGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNatGateways, request, handler, context);
        }

        /**
         * <p>Describes your network ACLs. The default is to describe all your network
         * ACLs. Alternatively, you can specify specific network ACL IDs or filter the
         * results to include only the network ACLs that match specific criteria.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html">Network
         * ACLs</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkAcls">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkAclsOutcome DescribeNetworkAcls(const Model::DescribeNetworkAclsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkAcls that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkAclsRequestT = Model::DescribeNetworkAclsRequest>
        Model::DescribeNetworkAclsOutcomeCallable DescribeNetworkAclsCallable(const DescribeNetworkAclsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkAcls, request);
        }

        /**
         * An Async wrapper for DescribeNetworkAcls that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkAclsRequestT = Model::DescribeNetworkAclsRequest>
        void DescribeNetworkAclsAsync(const DescribeNetworkAclsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkAclsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkAcls, request, handler, context);
        }

        /**
         * <p>Describes the specified Network Access Scope analyses.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInsightsAccessScopeAnalyses">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInsightsAccessScopeAnalysesOutcome DescribeNetworkInsightsAccessScopeAnalyses(const Model::DescribeNetworkInsightsAccessScopeAnalysesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInsightsAccessScopeAnalyses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInsightsAccessScopeAnalysesRequestT = Model::DescribeNetworkInsightsAccessScopeAnalysesRequest>
        Model::DescribeNetworkInsightsAccessScopeAnalysesOutcomeCallable DescribeNetworkInsightsAccessScopeAnalysesCallable(const DescribeNetworkInsightsAccessScopeAnalysesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInsightsAccessScopeAnalyses, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInsightsAccessScopeAnalyses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInsightsAccessScopeAnalysesRequestT = Model::DescribeNetworkInsightsAccessScopeAnalysesRequest>
        void DescribeNetworkInsightsAccessScopeAnalysesAsync(const DescribeNetworkInsightsAccessScopeAnalysesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInsightsAccessScopeAnalysesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInsightsAccessScopeAnalyses, request, handler, context);
        }

        /**
         * <p>Describes the specified Network Access Scopes.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInsightsAccessScopes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInsightsAccessScopesOutcome DescribeNetworkInsightsAccessScopes(const Model::DescribeNetworkInsightsAccessScopesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInsightsAccessScopes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInsightsAccessScopesRequestT = Model::DescribeNetworkInsightsAccessScopesRequest>
        Model::DescribeNetworkInsightsAccessScopesOutcomeCallable DescribeNetworkInsightsAccessScopesCallable(const DescribeNetworkInsightsAccessScopesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInsightsAccessScopes, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInsightsAccessScopes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInsightsAccessScopesRequestT = Model::DescribeNetworkInsightsAccessScopesRequest>
        void DescribeNetworkInsightsAccessScopesAsync(const DescribeNetworkInsightsAccessScopesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInsightsAccessScopesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInsightsAccessScopes, request, handler, context);
        }

        /**
         * <p>Describes one or more of your network insights analyses.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInsightsAnalyses">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInsightsAnalysesOutcome DescribeNetworkInsightsAnalyses(const Model::DescribeNetworkInsightsAnalysesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInsightsAnalyses that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInsightsAnalysesRequestT = Model::DescribeNetworkInsightsAnalysesRequest>
        Model::DescribeNetworkInsightsAnalysesOutcomeCallable DescribeNetworkInsightsAnalysesCallable(const DescribeNetworkInsightsAnalysesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInsightsAnalyses, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInsightsAnalyses that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInsightsAnalysesRequestT = Model::DescribeNetworkInsightsAnalysesRequest>
        void DescribeNetworkInsightsAnalysesAsync(const DescribeNetworkInsightsAnalysesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInsightsAnalysesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInsightsAnalyses, request, handler, context);
        }

        /**
         * <p>Describes one or more of your paths.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInsightsPaths">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInsightsPathsOutcome DescribeNetworkInsightsPaths(const Model::DescribeNetworkInsightsPathsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInsightsPaths that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInsightsPathsRequestT = Model::DescribeNetworkInsightsPathsRequest>
        Model::DescribeNetworkInsightsPathsOutcomeCallable DescribeNetworkInsightsPathsCallable(const DescribeNetworkInsightsPathsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInsightsPaths, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInsightsPaths that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInsightsPathsRequestT = Model::DescribeNetworkInsightsPathsRequest>
        void DescribeNetworkInsightsPathsAsync(const DescribeNetworkInsightsPathsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInsightsPathsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInsightsPaths, request, handler, context);
        }

        /**
         * <p>Describes a network interface attribute. You can specify only one attribute
         * at a time.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInterfaceAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInterfaceAttributeOutcome DescribeNetworkInterfaceAttribute(const Model::DescribeNetworkInterfaceAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeNetworkInterfaceAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInterfaceAttributeRequestT = Model::DescribeNetworkInterfaceAttributeRequest>
        Model::DescribeNetworkInterfaceAttributeOutcomeCallable DescribeNetworkInterfaceAttributeCallable(const DescribeNetworkInterfaceAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInterfaceAttribute, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInterfaceAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInterfaceAttributeRequestT = Model::DescribeNetworkInterfaceAttributeRequest>
        void DescribeNetworkInterfaceAttributeAsync(const DescribeNetworkInterfaceAttributeRequestT& request, const DescribeNetworkInterfaceAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInterfaceAttribute, request, handler, context);
        }

        /**
         * <p>Describes the permissions for your network interfaces. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInterfacePermissions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInterfacePermissionsOutcome DescribeNetworkInterfacePermissions(const Model::DescribeNetworkInterfacePermissionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInterfacePermissions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInterfacePermissionsRequestT = Model::DescribeNetworkInterfacePermissionsRequest>
        Model::DescribeNetworkInterfacePermissionsOutcomeCallable DescribeNetworkInterfacePermissionsCallable(const DescribeNetworkInterfacePermissionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInterfacePermissions, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInterfacePermissions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInterfacePermissionsRequestT = Model::DescribeNetworkInterfacePermissionsRequest>
        void DescribeNetworkInterfacePermissionsAsync(const DescribeNetworkInterfacePermissionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInterfacePermissionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInterfacePermissions, request, handler, context);
        }

        /**
         * <p>Describes the specified network interfaces or all your network
         * interfaces.</p> <p>If you have a large number of network interfaces, the
         * operation fails unless you use pagination or one of the following filters:
         * <code>group-id</code>, <code>mac-address</code>, <code>private-dns-name</code>,
         * <code>private-ip-address</code>, <code>subnet-id</code>, or
         * <code>vpc-id</code>.</p>  <p>We strongly recommend using only
         * paginated requests. Unpaginated requests are susceptible to throttling and
         * timeouts.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeNetworkInterfaces">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeNetworkInterfacesOutcome DescribeNetworkInterfaces(const Model::DescribeNetworkInterfacesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeNetworkInterfaces that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeNetworkInterfacesRequestT = Model::DescribeNetworkInterfacesRequest>
        Model::DescribeNetworkInterfacesOutcomeCallable DescribeNetworkInterfacesCallable(const DescribeNetworkInterfacesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeNetworkInterfaces, request);
        }

        /**
         * An Async wrapper for DescribeNetworkInterfaces that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeNetworkInterfacesRequestT = Model::DescribeNetworkInterfacesRequest>
        void DescribeNetworkInterfacesAsync(const DescribeNetworkInterfacesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeNetworkInterfacesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeNetworkInterfaces, request, handler, context);
        }

        /**
         * <p>Describes the Outposts link aggregation groups (LAGs).</p>  <p>LAGs are
         * only available for second-generation Outposts racks at this time.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeOutpostLags">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeOutpostLagsOutcome DescribeOutpostLags(const Model::DescribeOutpostLagsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeOutpostLags that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeOutpostLagsRequestT = Model::DescribeOutpostLagsRequest>
        Model::DescribeOutpostLagsOutcomeCallable DescribeOutpostLagsCallable(const DescribeOutpostLagsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeOutpostLags, request);
        }

        /**
         * An Async wrapper for DescribeOutpostLags that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeOutpostLagsRequestT = Model::DescribeOutpostLagsRequest>
        void DescribeOutpostLagsAsync(const DescribeOutpostLagsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeOutpostLagsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeOutpostLags, request, handler, context);
        }

        /**
         * <p>Describes the specified placement groups or all of your placement groups.</p>
         *  <p>To describe a specific placement group that is <i>shared</i> with your
         * account, you must specify the ID of the placement group using the
         * <code>GroupId</code> parameter. Specifying the name of a <i>shared</i> placement
         * group using the <code>GroupNames</code> parameter will result in an error.</p>
         *  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement
         * groups</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribePlacementGroups">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribePlacementGroupsOutcome DescribePlacementGroups(const Model::DescribePlacementGroupsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribePlacementGroups that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribePlacementGroupsRequestT = Model::DescribePlacementGroupsRequest>
        Model::DescribePlacementGroupsOutcomeCallable DescribePlacementGroupsCallable(const DescribePlacementGroupsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribePlacementGroups, request);
        }

        /**
         * An Async wrapper for DescribePlacementGroups that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribePlacementGroupsRequestT = Model::DescribePlacementGroupsRequest>
        void DescribePlacementGroupsAsync(const DescribePlacementGroupsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribePlacementGroupsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribePlacementGroups, request, handler, context);
        }

        /**
         * <p>Describes available Amazon Web Services services in a prefix list format,
         * which includes the prefix list name and prefix list ID of the service and the IP
         * address range for the service.</p> <p>We recommend that you use
         * <a>DescribeManagedPrefixLists</a> instead.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribePrefixLists">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribePrefixListsOutcome DescribePrefixLists(const Model::DescribePrefixListsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribePrefixLists that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribePrefixListsRequestT = Model::DescribePrefixListsRequest>
        Model::DescribePrefixListsOutcomeCallable DescribePrefixListsCallable(const DescribePrefixListsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribePrefixLists, request);
        }

        /**
         * An Async wrapper for DescribePrefixLists that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribePrefixListsRequestT = Model::DescribePrefixListsRequest>
        void DescribePrefixListsAsync(const DescribePrefixListsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribePrefixListsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribePrefixLists, request, handler, context);
        }

        /**
         * <p>Describes the ID format settings for the root user and all IAM roles and IAM
         * users that have explicitly specified a longer ID (17-character ID) preference.
         * </p> <p>By default, all IAM roles and IAM users default to the same ID settings
         * as the root user, unless they explicitly override the settings. This request is
         * useful for identifying those IAM users and IAM roles that have overridden the
         * default ID settings.</p> <p>The following resource types support longer IDs:
         * <code>bundle</code> | <code>conversion-task</code> |
         * <code>customer-gateway</code> | <code>dhcp-options</code> |
         * <code>elastic-ip-allocation</code> | <code>elastic-ip-association</code> |
         * <code>export-task</code> | <code>flow-log</code> | <code>image</code> |
         * <code>import-task</code> | <code>instance</code> | <code>internet-gateway</code>
         * | <code>network-acl</code> | <code>network-acl-association</code> |
         * <code>network-interface</code> | <code>network-interface-attachment</code> |
         * <code>prefix-list</code> | <code>reservation</code> | <code>route-table</code> |
         * <code>route-table-association</code> | <code>security-group</code> |
         * <code>snapshot</code> | <code>subnet</code> |
         * <code>subnet-cidr-block-association</code> | <code>volume</code> |
         * <code>vpc</code> | <code>vpc-cidr-block-association</code> |
         * <code>vpc-endpoint</code> | <code>vpc-peering-connection</code> |
         * <code>vpn-connection</code> | <code>vpn-gateway</code>. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribePrincipalIdFormat">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribePrincipalIdFormatOutcome DescribePrincipalIdFormat(const Model::DescribePrincipalIdFormatRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribePrincipalIdFormat that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribePrincipalIdFormatRequestT = Model::DescribePrincipalIdFormatRequest>
        Model::DescribePrincipalIdFormatOutcomeCallable DescribePrincipalIdFormatCallable(const DescribePrincipalIdFormatRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribePrincipalIdFormat, request);
        }

        /**
         * An Async wrapper for DescribePrincipalIdFormat that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribePrincipalIdFormatRequestT = Model::DescribePrincipalIdFormatRequest>
        void DescribePrincipalIdFormatAsync(const DescribePrincipalIdFormatResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribePrincipalIdFormatRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribePrincipalIdFormat, request, handler, context);
        }

        /**
         * <p>Describes the specified IPv4 address pools.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribePublicIpv4Pools">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribePublicIpv4PoolsOutcome DescribePublicIpv4Pools(const Model::DescribePublicIpv4PoolsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribePublicIpv4Pools that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribePublicIpv4PoolsRequestT = Model::DescribePublicIpv4PoolsRequest>
        Model::DescribePublicIpv4PoolsOutcomeCallable DescribePublicIpv4PoolsCallable(const DescribePublicIpv4PoolsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribePublicIpv4Pools, request);
        }

        /**
         * An Async wrapper for DescribePublicIpv4Pools that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribePublicIpv4PoolsRequestT = Model::DescribePublicIpv4PoolsRequest>
        void DescribePublicIpv4PoolsAsync(const DescribePublicIpv4PoolsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribePublicIpv4PoolsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribePublicIpv4Pools, request, handler, context);
        }

        /**
         * <p>Describes the Regions that are enabled for your account, or all Regions.</p>
         * <p>For a list of the Regions supported by Amazon EC2, see <a
         * href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-endpoints.html">Amazon
         * EC2 service endpoints</a>.</p> <p>For information about enabling and disabling
         * Regions for your account, see <a
         * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-regions.html">Specify
         * which Amazon Web Services Regions your account can use</a> in the <i>Amazon Web
         * Services Account Management Reference Guide</i>.</p>  <p>The order of the
         * elements in the response, including those within nested structures, might vary.
         * Applications should not assume the elements appear in a particular order.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeRegions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeRegionsOutcome DescribeRegions(const Model::DescribeRegionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeRegions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeRegionsRequestT = Model::DescribeRegionsRequest>
        Model::DescribeRegionsOutcomeCallable DescribeRegionsCallable(const DescribeRegionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeRegions, request);
        }

        /**
         * An Async wrapper for DescribeRegions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeRegionsRequestT = Model::DescribeRegionsRequest>
        void DescribeRegionsAsync(const DescribeRegionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeRegionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeRegions, request, handler, context);
        }

        /**
         * <p>Describes a root volume replacement task. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/replace-root.html">Replace
         * a root volume</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeReplaceRootVolumeTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeReplaceRootVolumeTasksOutcome DescribeReplaceRootVolumeTasks(const Model::DescribeReplaceRootVolumeTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeReplaceRootVolumeTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeReplaceRootVolumeTasksRequestT = Model::DescribeReplaceRootVolumeTasksRequest>
        Model::DescribeReplaceRootVolumeTasksOutcomeCallable DescribeReplaceRootVolumeTasksCallable(const DescribeReplaceRootVolumeTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeReplaceRootVolumeTasks, request);
        }

        /**
         * An Async wrapper for DescribeReplaceRootVolumeTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeReplaceRootVolumeTasksRequestT = Model::DescribeReplaceRootVolumeTasksRequest>
        void DescribeReplaceRootVolumeTasksAsync(const DescribeReplaceRootVolumeTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeReplaceRootVolumeTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeReplaceRootVolumeTasks, request, handler, context);
        }

        /**
         * <p>Describes one or more of the Reserved Instances that you purchased.</p>
         * <p>For more information about Reserved Instances, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/concepts-on-demand-reserved-instances.html">Reserved
         * Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>  <p>The order of
         * the elements in the response, including those within nested structures, might
         * vary. Applications should not assume the elements appear in a particular
         * order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeReservedInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeReservedInstancesOutcome DescribeReservedInstances(const Model::DescribeReservedInstancesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeReservedInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeReservedInstancesRequestT = Model::DescribeReservedInstancesRequest>
        Model::DescribeReservedInstancesOutcomeCallable DescribeReservedInstancesCallable(const DescribeReservedInstancesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeReservedInstances, request);
        }

        /**
         * An Async wrapper for DescribeReservedInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeReservedInstancesRequestT = Model::DescribeReservedInstancesRequest>
        void DescribeReservedInstancesAsync(const DescribeReservedInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeReservedInstancesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeReservedInstances, request, handler, context);
        }

        /**
         * <p>Describes your account's Reserved Instance listings in the Reserved Instance
         * Marketplace.</p> <p>The Reserved Instance Marketplace matches sellers who want
         * to resell Reserved Instance capacity that they no longer need with buyers who
         * want to purchase additional capacity. Reserved Instances bought and sold through
         * the Reserved Instance Marketplace work like any other Reserved Instances.</p>
         * <p>As a seller, you choose to list some or all of your Reserved Instances, and
         * you specify the upfront price to receive for them. Your Reserved Instances are
         * then listed in the Reserved Instance Marketplace and are available for
         * purchase.</p> <p>As a buyer, you specify the configuration of the Reserved
         * Instance to purchase, and the Marketplace matches what you're searching for with
         * what's available. The Marketplace first sells the lowest priced Reserved
         * Instances to you, and continues to sell available Reserved Instance listings to
         * you until your demand is met. You are charged based on the total price of all of
         * the listings that you purchase.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ri-market-general.html">Sell
         * in the Reserved Instance Marketplace</a> in the <i>Amazon EC2 User
         * Guide</i>.</p>  <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeReservedInstancesListings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeReservedInstancesListingsOutcome DescribeReservedInstancesListings(const Model::DescribeReservedInstancesListingsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeReservedInstancesListings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeReservedInstancesListingsRequestT = Model::DescribeReservedInstancesListingsRequest>
        Model::DescribeReservedInstancesListingsOutcomeCallable DescribeReservedInstancesListingsCallable(const DescribeReservedInstancesListingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeReservedInstancesListings, request);
        }

        /**
         * An Async wrapper for DescribeReservedInstancesListings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeReservedInstancesListingsRequestT = Model::DescribeReservedInstancesListingsRequest>
        void DescribeReservedInstancesListingsAsync(const DescribeReservedInstancesListingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeReservedInstancesListingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeReservedInstancesListings, request, handler, context);
        }

        /**
         * <p>Describes the modifications made to your Reserved Instances. If no parameter
         * is specified, information about all your Reserved Instances modification
         * requests is returned. If a modification ID is specified, only information about
         * the specific modification is returned.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ri-modifying.html">Modify
         * Reserved Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>  <p>The
         * order of the elements in the response, including those within nested structures,
         * might vary. Applications should not assume the elements appear in a particular
         * order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeReservedInstancesModifications">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeReservedInstancesModificationsOutcome DescribeReservedInstancesModifications(const Model::DescribeReservedInstancesModificationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeReservedInstancesModifications that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeReservedInstancesModificationsRequestT = Model::DescribeReservedInstancesModificationsRequest>
        Model::DescribeReservedInstancesModificationsOutcomeCallable DescribeReservedInstancesModificationsCallable(const DescribeReservedInstancesModificationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeReservedInstancesModifications, request);
        }

        /**
         * An Async wrapper for DescribeReservedInstancesModifications that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeReservedInstancesModificationsRequestT = Model::DescribeReservedInstancesModificationsRequest>
        void DescribeReservedInstancesModificationsAsync(const DescribeReservedInstancesModificationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeReservedInstancesModificationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeReservedInstancesModifications, request, handler, context);
        }

        /**
         * <p>Describes Reserved Instance offerings that are available for purchase. With
         * Reserved Instances, you purchase the right to launch instances for a period of
         * time. During that time period, you do not receive insufficient capacity errors,
         * and you pay a lower usage rate than the rate charged for On-Demand instances for
         * the actual time used.</p> <p>If you have listed your own Reserved Instances for
         * sale in the Reserved Instance Marketplace, they will be excluded from these
         * results. This is to ensure that you do not purchase your own Reserved
         * Instances.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ri-market-general.html">Sell
         * in the Reserved Instance Marketplace</a> in the <i>Amazon EC2 User
         * Guide</i>.</p>  <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeReservedInstancesOfferings">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeReservedInstancesOfferingsOutcome DescribeReservedInstancesOfferings(const Model::DescribeReservedInstancesOfferingsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeReservedInstancesOfferings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeReservedInstancesOfferingsRequestT = Model::DescribeReservedInstancesOfferingsRequest>
        Model::DescribeReservedInstancesOfferingsOutcomeCallable DescribeReservedInstancesOfferingsCallable(const DescribeReservedInstancesOfferingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeReservedInstancesOfferings, request);
        }

        /**
         * An Async wrapper for DescribeReservedInstancesOfferings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeReservedInstancesOfferingsRequestT = Model::DescribeReservedInstancesOfferingsRequest>
        void DescribeReservedInstancesOfferingsAsync(const DescribeReservedInstancesOfferingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeReservedInstancesOfferingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeReservedInstancesOfferings, request, handler, context);
        }

        /**
         * <p>Describes one or more route server endpoints.</p> <p>A route server endpoint
         * is an Amazon Web Services-managed component inside a subnet that facilitates <a
         * href="https://en.wikipedia.org/wiki/Border_Gateway_Protocol">BGP (Border Gateway
         * Protocol)</a> connections between your route server and your BGP peers.</p>
         * <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeRouteServerEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeRouteServerEndpointsOutcome DescribeRouteServerEndpoints(const Model::DescribeRouteServerEndpointsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeRouteServerEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeRouteServerEndpointsRequestT = Model::DescribeRouteServerEndpointsRequest>
        Model::DescribeRouteServerEndpointsOutcomeCallable DescribeRouteServerEndpointsCallable(const DescribeRouteServerEndpointsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeRouteServerEndpoints, request);
        }

        /**
         * An Async wrapper for DescribeRouteServerEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeRouteServerEndpointsRequestT = Model::DescribeRouteServerEndpointsRequest>
        void DescribeRouteServerEndpointsAsync(const DescribeRouteServerEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeRouteServerEndpointsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeRouteServerEndpoints, request, handler, context);
        }

        /**
         * <p>Describes one or more route server peers.</p> <p>A route server peer is a
         * session between a route server endpoint and the device deployed in Amazon Web
         * Services (such as a firewall appliance or other network security function
         * running on an EC2 instance). The device must meet these requirements:</p> <ul>
         * <li> <p>Have an elastic network interface in the VPC</p> </li> <li> <p>Support
         * BGP (Border Gateway Protocol)</p> </li> <li> <p>Can initiate BGP sessions</p>
         * </li> </ul> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeRouteServerPeers">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeRouteServerPeersOutcome DescribeRouteServerPeers(const Model::DescribeRouteServerPeersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeRouteServerPeers that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeRouteServerPeersRequestT = Model::DescribeRouteServerPeersRequest>
        Model::DescribeRouteServerPeersOutcomeCallable DescribeRouteServerPeersCallable(const DescribeRouteServerPeersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeRouteServerPeers, request);
        }

        /**
         * An Async wrapper for DescribeRouteServerPeers that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeRouteServerPeersRequestT = Model::DescribeRouteServerPeersRequest>
        void DescribeRouteServerPeersAsync(const DescribeRouteServerPeersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeRouteServerPeersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeRouteServerPeers, request, handler, context);
        }

        /**
         * <p>Describes one or more route servers.</p> <p>Amazon VPC Route Server
         * simplifies routing for traffic between workloads that are deployed within a VPC
         * and its internet gateways. With this feature, VPC Route Server dynamically
         * updates VPC and internet gateway route tables with your preferred IPv4 or IPv6
         * routes to achieve routing fault tolerance for those workloads. This enables you
         * to automatically reroute traffic within a VPC, which increases the manageability
         * of VPC routing and interoperability with third-party workloads.</p> <p>Route
         * server supports the follow route table types:</p> <ul> <li> <p>VPC route tables
         * not associated with subnets</p> </li> <li> <p>Subnet route tables</p> </li> <li>
         * <p>Internet gateway route tables</p> </li> </ul> <p>Route server does not
         * support route tables associated with virtual private gateways. To propagate
         * routes into a transit gateway route table, use <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html">Transit
         * Gateway Connect</a>.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeRouteServers">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeRouteServersOutcome DescribeRouteServers(const Model::DescribeRouteServersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeRouteServers that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeRouteServersRequestT = Model::DescribeRouteServersRequest>
        Model::DescribeRouteServersOutcomeCallable DescribeRouteServersCallable(const DescribeRouteServersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeRouteServers, request);
        }

        /**
         * An Async wrapper for DescribeRouteServers that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeRouteServersRequestT = Model::DescribeRouteServersRequest>
        void DescribeRouteServersAsync(const DescribeRouteServersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeRouteServersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeRouteServers, request, handler, context);
        }

        /**
         * <p>Describes your route tables. The default is to describe all your route
         * tables. Alternatively, you can specify specific route table IDs or filter the
         * results to include only the route tables that match specific criteria.</p>
         * <p>Each subnet in your VPC must be associated with a route table. If a subnet is
         * not explicitly associated with any route table, it is implicitly associated with
         * the main route table. This command does not return the subnet ID for implicit
         * associations.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html">Route
         * tables</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeRouteTables">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeRouteTablesOutcome DescribeRouteTables(const Model::DescribeRouteTablesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeRouteTables that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeRouteTablesRequestT = Model::DescribeRouteTablesRequest>
        Model::DescribeRouteTablesOutcomeCallable DescribeRouteTablesCallable(const DescribeRouteTablesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeRouteTables, request);
        }

        /**
         * An Async wrapper for DescribeRouteTables that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeRouteTablesRequestT = Model::DescribeRouteTablesRequest>
        void DescribeRouteTablesAsync(const DescribeRouteTablesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeRouteTablesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeRouteTables, request, handler, context);
        }

        /**
         * <p>Finds available schedules that meet the specified criteria.</p> <p>You can
         * search for an available schedule no more than 3 months in advance. You must meet
         * the minimum required duration of 1,200 hours per year. For example, the minimum
         * daily schedule is 4 hours, the minimum weekly schedule is 24 hours, and the
         * minimum monthly schedule is 100 hours.</p> <p>After you find a schedule that
         * meets your needs, call <a>PurchaseScheduledInstances</a> to purchase Scheduled
         * Instances with that schedule.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeScheduledInstanceAvailability">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeScheduledInstanceAvailabilityOutcome DescribeScheduledInstanceAvailability(const Model::DescribeScheduledInstanceAvailabilityRequest& request) const;

        /**
         * A Callable wrapper for DescribeScheduledInstanceAvailability that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeScheduledInstanceAvailabilityRequestT = Model::DescribeScheduledInstanceAvailabilityRequest>
        Model::DescribeScheduledInstanceAvailabilityOutcomeCallable DescribeScheduledInstanceAvailabilityCallable(const DescribeScheduledInstanceAvailabilityRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeScheduledInstanceAvailability, request);
        }

        /**
         * An Async wrapper for DescribeScheduledInstanceAvailability that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeScheduledInstanceAvailabilityRequestT = Model::DescribeScheduledInstanceAvailabilityRequest>
        void DescribeScheduledInstanceAvailabilityAsync(const DescribeScheduledInstanceAvailabilityRequestT& request, const DescribeScheduledInstanceAvailabilityResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeScheduledInstanceAvailability, request, handler, context);
        }

        /**
         * <p>Describes the specified Scheduled Instances or all your Scheduled
         * Instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeScheduledInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeScheduledInstancesOutcome DescribeScheduledInstances(const Model::DescribeScheduledInstancesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeScheduledInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeScheduledInstancesRequestT = Model::DescribeScheduledInstancesRequest>
        Model::DescribeScheduledInstancesOutcomeCallable DescribeScheduledInstancesCallable(const DescribeScheduledInstancesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeScheduledInstances, request);
        }

        /**
         * An Async wrapper for DescribeScheduledInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeScheduledInstancesRequestT = Model::DescribeScheduledInstancesRequest>
        void DescribeScheduledInstancesAsync(const DescribeScheduledInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeScheduledInstancesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeScheduledInstances, request, handler, context);
        }

        /**
         * <p>Describes the VPCs on the other side of a VPC peering or Transit Gateway
         * connection that are referencing the security groups you've specified in this
         * request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSecurityGroupReferences">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSecurityGroupReferencesOutcome DescribeSecurityGroupReferences(const Model::DescribeSecurityGroupReferencesRequest& request) const;

        /**
         * A Callable wrapper for DescribeSecurityGroupReferences that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSecurityGroupReferencesRequestT = Model::DescribeSecurityGroupReferencesRequest>
        Model::DescribeSecurityGroupReferencesOutcomeCallable DescribeSecurityGroupReferencesCallable(const DescribeSecurityGroupReferencesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeSecurityGroupReferences, request);
        }

        /**
         * An Async wrapper for DescribeSecurityGroupReferences that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSecurityGroupReferencesRequestT = Model::DescribeSecurityGroupReferencesRequest>
        void DescribeSecurityGroupReferencesAsync(const DescribeSecurityGroupReferencesRequestT& request, const DescribeSecurityGroupReferencesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeSecurityGroupReferences, request, handler, context);
        }

        /**
         * <p>Describes one or more of your security group rules.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSecurityGroupRules">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSecurityGroupRulesOutcome DescribeSecurityGroupRules(const Model::DescribeSecurityGroupRulesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSecurityGroupRules that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSecurityGroupRulesRequestT = Model::DescribeSecurityGroupRulesRequest>
        Model::DescribeSecurityGroupRulesOutcomeCallable DescribeSecurityGroupRulesCallable(const DescribeSecurityGroupRulesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSecurityGroupRules, request);
        }

        /**
         * An Async wrapper for DescribeSecurityGroupRules that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSecurityGroupRulesRequestT = Model::DescribeSecurityGroupRulesRequest>
        void DescribeSecurityGroupRulesAsync(const DescribeSecurityGroupRulesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSecurityGroupRulesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSecurityGroupRules, request, handler, context);
        }

        /**
         * <p>Describes security group VPC associations made with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSecurityGroupVpc.html">AssociateSecurityGroupVpc</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSecurityGroupVpcAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSecurityGroupVpcAssociationsOutcome DescribeSecurityGroupVpcAssociations(const Model::DescribeSecurityGroupVpcAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSecurityGroupVpcAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSecurityGroupVpcAssociationsRequestT = Model::DescribeSecurityGroupVpcAssociationsRequest>
        Model::DescribeSecurityGroupVpcAssociationsOutcomeCallable DescribeSecurityGroupVpcAssociationsCallable(const DescribeSecurityGroupVpcAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSecurityGroupVpcAssociations, request);
        }

        /**
         * An Async wrapper for DescribeSecurityGroupVpcAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSecurityGroupVpcAssociationsRequestT = Model::DescribeSecurityGroupVpcAssociationsRequest>
        void DescribeSecurityGroupVpcAssociationsAsync(const DescribeSecurityGroupVpcAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSecurityGroupVpcAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSecurityGroupVpcAssociations, request, handler, context);
        }

        /**
         * <p>Describes the specified security groups or all of your security
         * groups.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSecurityGroups">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSecurityGroupsOutcome DescribeSecurityGroups(const Model::DescribeSecurityGroupsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSecurityGroups that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSecurityGroupsRequestT = Model::DescribeSecurityGroupsRequest>
        Model::DescribeSecurityGroupsOutcomeCallable DescribeSecurityGroupsCallable(const DescribeSecurityGroupsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSecurityGroups, request);
        }

        /**
         * An Async wrapper for DescribeSecurityGroups that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSecurityGroupsRequestT = Model::DescribeSecurityGroupsRequest>
        void DescribeSecurityGroupsAsync(const DescribeSecurityGroupsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSecurityGroupsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSecurityGroups, request, handler, context);
        }

        /**
         * <p>Describes the Outpost service link virtual interfaces.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeServiceLinkVirtualInterfaces">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeServiceLinkVirtualInterfacesOutcome DescribeServiceLinkVirtualInterfaces(const Model::DescribeServiceLinkVirtualInterfacesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeServiceLinkVirtualInterfaces that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeServiceLinkVirtualInterfacesRequestT = Model::DescribeServiceLinkVirtualInterfacesRequest>
        Model::DescribeServiceLinkVirtualInterfacesOutcomeCallable DescribeServiceLinkVirtualInterfacesCallable(const DescribeServiceLinkVirtualInterfacesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeServiceLinkVirtualInterfaces, request);
        }

        /**
         * An Async wrapper for DescribeServiceLinkVirtualInterfaces that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeServiceLinkVirtualInterfacesRequestT = Model::DescribeServiceLinkVirtualInterfacesRequest>
        void DescribeServiceLinkVirtualInterfacesAsync(const DescribeServiceLinkVirtualInterfacesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeServiceLinkVirtualInterfacesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeServiceLinkVirtualInterfaces, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified snapshot. You can specify
         * only one attribute at a time.</p> <p>For more information about EBS snapshots,
         * see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-snapshots.html">Amazon
         * EBS snapshots</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSnapshotAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSnapshotAttributeOutcome DescribeSnapshotAttribute(const Model::DescribeSnapshotAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeSnapshotAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSnapshotAttributeRequestT = Model::DescribeSnapshotAttributeRequest>
        Model::DescribeSnapshotAttributeOutcomeCallable DescribeSnapshotAttributeCallable(const DescribeSnapshotAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeSnapshotAttribute, request);
        }

        /**
         * An Async wrapper for DescribeSnapshotAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSnapshotAttributeRequestT = Model::DescribeSnapshotAttributeRequest>
        void DescribeSnapshotAttributeAsync(const DescribeSnapshotAttributeRequestT& request, const DescribeSnapshotAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeSnapshotAttribute, request, handler, context);
        }

        /**
         * <p>Describes the storage tier status of one or more Amazon EBS
         * snapshots.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSnapshotTierStatus">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSnapshotTierStatusOutcome DescribeSnapshotTierStatus(const Model::DescribeSnapshotTierStatusRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSnapshotTierStatus that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSnapshotTierStatusRequestT = Model::DescribeSnapshotTierStatusRequest>
        Model::DescribeSnapshotTierStatusOutcomeCallable DescribeSnapshotTierStatusCallable(const DescribeSnapshotTierStatusRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSnapshotTierStatus, request);
        }

        /**
         * An Async wrapper for DescribeSnapshotTierStatus that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSnapshotTierStatusRequestT = Model::DescribeSnapshotTierStatusRequest>
        void DescribeSnapshotTierStatusAsync(const DescribeSnapshotTierStatusResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSnapshotTierStatusRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSnapshotTierStatus, request, handler, context);
        }

        /**
         * <p>Describes the specified EBS snapshots available to you or all of the EBS
         * snapshots available to you.</p> <p>The snapshots available to you include public
         * snapshots, private snapshots that you own, and private snapshots owned by other
         * Amazon Web Services accounts for which you have explicit create volume
         * permissions.</p> <p>The create volume permissions fall into the following
         * categories:</p> <ul> <li> <p> <i>public</i>: The owner of the snapshot granted
         * create volume permissions for the snapshot to the <code>all</code> group. All
         * Amazon Web Services accounts have create volume permissions for these
         * snapshots.</p> </li> <li> <p> <i>explicit</i>: The owner of the snapshot granted
         * create volume permissions to a specific Amazon Web Services account.</p> </li>
         * <li> <p> <i>implicit</i>: An Amazon Web Services account has implicit create
         * volume permissions for all snapshots it owns.</p> </li> </ul> <p>The list of
         * snapshots returned can be filtered by specifying snapshot IDs, snapshot owners,
         * or Amazon Web Services accounts with create volume permissions. If no options
         * are specified, Amazon EC2 returns all snapshots for which you have create volume
         * permissions.</p> <p>If you specify one or more snapshot IDs, only snapshots that
         * have the specified IDs are returned. If you specify an invalid snapshot ID, an
         * error is returned. If you specify a snapshot ID for which you do not have
         * access, it is not included in the returned results.</p> <p>If you specify one or
         * more snapshot owners using the <code>OwnerIds</code> option, only snapshots from
         * the specified owners and for which you have access are returned. The results can
         * include the Amazon Web Services account IDs of the specified owners,
         * <code>amazon</code> for snapshots owned by Amazon, or <code>self</code> for
         * snapshots that you own.</p> <p>If you specify a list of restorable users, only
         * snapshots with create snapshot permissions for those users are returned. You can
         * specify Amazon Web Services account IDs (if you own the snapshots),
         * <code>self</code> for snapshots for which you own or have explicit permissions,
         * or <code>all</code> for public snapshots.</p> <p>If you are describing a long
         * list of snapshots, we recommend that you paginate the output to make the list
         * more manageable. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination">Pagination</a>.</p>
         * <p>To get the state of fast snapshot restores for a snapshot, use
         * <a>DescribeFastSnapshotRestores</a>.</p> <p>For more information about EBS
         * snapshots, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-snapshots.html">Amazon
         * EBS snapshots</a> in the <i>Amazon EBS User Guide</i>.</p>  <p>We
         * strongly recommend using only paginated requests. Unpaginated requests are
         * susceptible to throttling and timeouts.</p> <p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSnapshots">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSnapshotsOutcome DescribeSnapshots(const Model::DescribeSnapshotsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSnapshots that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSnapshotsRequestT = Model::DescribeSnapshotsRequest>
        Model::DescribeSnapshotsOutcomeCallable DescribeSnapshotsCallable(const DescribeSnapshotsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSnapshots, request);
        }

        /**
         * An Async wrapper for DescribeSnapshots that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSnapshotsRequestT = Model::DescribeSnapshotsRequest>
        void DescribeSnapshotsAsync(const DescribeSnapshotsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSnapshotsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSnapshots, request, handler, context);
        }

        /**
         * <p>Describes the data feed for Spot Instances. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-data-feeds.html">Spot
         * Instance data feed</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotDatafeedSubscription">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotDatafeedSubscriptionOutcome DescribeSpotDatafeedSubscription(const Model::DescribeSpotDatafeedSubscriptionRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSpotDatafeedSubscription that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotDatafeedSubscriptionRequestT = Model::DescribeSpotDatafeedSubscriptionRequest>
        Model::DescribeSpotDatafeedSubscriptionOutcomeCallable DescribeSpotDatafeedSubscriptionCallable(const DescribeSpotDatafeedSubscriptionRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotDatafeedSubscription, request);
        }

        /**
         * An Async wrapper for DescribeSpotDatafeedSubscription that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotDatafeedSubscriptionRequestT = Model::DescribeSpotDatafeedSubscriptionRequest>
        void DescribeSpotDatafeedSubscriptionAsync(const DescribeSpotDatafeedSubscriptionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSpotDatafeedSubscriptionRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotDatafeedSubscription, request, handler, context);
        }

        /**
         * <p>Describes the running instances for the specified Spot Fleet.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotFleetInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotFleetInstancesOutcome DescribeSpotFleetInstances(const Model::DescribeSpotFleetInstancesRequest& request) const;

        /**
         * A Callable wrapper for DescribeSpotFleetInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotFleetInstancesRequestT = Model::DescribeSpotFleetInstancesRequest>
        Model::DescribeSpotFleetInstancesOutcomeCallable DescribeSpotFleetInstancesCallable(const DescribeSpotFleetInstancesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotFleetInstances, request);
        }

        /**
         * An Async wrapper for DescribeSpotFleetInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotFleetInstancesRequestT = Model::DescribeSpotFleetInstancesRequest>
        void DescribeSpotFleetInstancesAsync(const DescribeSpotFleetInstancesRequestT& request, const DescribeSpotFleetInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotFleetInstances, request, handler, context);
        }

        /**
         * <p>Describes the events for the specified Spot Fleet request during the
         * specified time.</p> <p>Spot Fleet events are delayed by up to 30 seconds before
         * they can be described. This ensures that you can query by the last evaluated
         * time and not miss a recorded event. Spot Fleet events are available for 48
         * hours.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/fleet-monitor.html">Monitor
         * fleet events using Amazon EventBridge</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotFleetRequestHistory">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotFleetRequestHistoryOutcome DescribeSpotFleetRequestHistory(const Model::DescribeSpotFleetRequestHistoryRequest& request) const;

        /**
         * A Callable wrapper for DescribeSpotFleetRequestHistory that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotFleetRequestHistoryRequestT = Model::DescribeSpotFleetRequestHistoryRequest>
        Model::DescribeSpotFleetRequestHistoryOutcomeCallable DescribeSpotFleetRequestHistoryCallable(const DescribeSpotFleetRequestHistoryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotFleetRequestHistory, request);
        }

        /**
         * An Async wrapper for DescribeSpotFleetRequestHistory that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotFleetRequestHistoryRequestT = Model::DescribeSpotFleetRequestHistoryRequest>
        void DescribeSpotFleetRequestHistoryAsync(const DescribeSpotFleetRequestHistoryRequestT& request, const DescribeSpotFleetRequestHistoryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotFleetRequestHistory, request, handler, context);
        }

        /**
         * <p>Describes your Spot Fleet requests.</p> <p>Spot Fleet requests are deleted 48
         * hours after they are canceled and their instances are terminated.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotFleetRequests">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotFleetRequestsOutcome DescribeSpotFleetRequests(const Model::DescribeSpotFleetRequestsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSpotFleetRequests that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotFleetRequestsRequestT = Model::DescribeSpotFleetRequestsRequest>
        Model::DescribeSpotFleetRequestsOutcomeCallable DescribeSpotFleetRequestsCallable(const DescribeSpotFleetRequestsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotFleetRequests, request);
        }

        /**
         * An Async wrapper for DescribeSpotFleetRequests that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotFleetRequestsRequestT = Model::DescribeSpotFleetRequestsRequest>
        void DescribeSpotFleetRequestsAsync(const DescribeSpotFleetRequestsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSpotFleetRequestsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotFleetRequests, request, handler, context);
        }

        /**
         * <p>Describes the specified Spot Instance requests.</p> <p>You can use
         * <code>DescribeSpotInstanceRequests</code> to find a running Spot Instance by
         * examining the response. If the status of the Spot Instance is
         * <code>fulfilled</code>, the instance ID appears in the response and contains the
         * identifier of the instance. Alternatively, you can use <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances">DescribeInstances</a>
         * with a filter to look for instances where the instance lifecycle is
         * <code>spot</code>.</p> <p>We recommend that you set <code>MaxResults</code> to a
         * value between 5 and 1000 to limit the number of items returned. This paginates
         * the output, which makes the list more manageable and returns the items faster.
         * If the list of items exceeds your <code>MaxResults</code> value, then that
         * number of items is returned along with a <code>NextToken</code> value that can
         * be passed to a subsequent <code>DescribeSpotInstanceRequests</code> request to
         * retrieve the remaining items.</p> <p>Spot Instance requests are deleted four
         * hours after they are canceled and their instances are terminated.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotInstanceRequests">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotInstanceRequestsOutcome DescribeSpotInstanceRequests(const Model::DescribeSpotInstanceRequestsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSpotInstanceRequests that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotInstanceRequestsRequestT = Model::DescribeSpotInstanceRequestsRequest>
        Model::DescribeSpotInstanceRequestsOutcomeCallable DescribeSpotInstanceRequestsCallable(const DescribeSpotInstanceRequestsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotInstanceRequests, request);
        }

        /**
         * An Async wrapper for DescribeSpotInstanceRequests that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotInstanceRequestsRequestT = Model::DescribeSpotInstanceRequestsRequest>
        void DescribeSpotInstanceRequestsAsync(const DescribeSpotInstanceRequestsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSpotInstanceRequestsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotInstanceRequests, request, handler, context);
        }

        /**
         * <p>Describes the Spot price history. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances-history.html">Spot
         * Instance pricing history</a> in the <i>Amazon EC2 User Guide</i>.</p> <p>When
         * you specify a start and end time, the operation returns the prices of the
         * instance types within that time range. It also returns the last price change
         * before the start time, which is the effective price as of the start
         * time.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSpotPriceHistory">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSpotPriceHistoryOutcome DescribeSpotPriceHistory(const Model::DescribeSpotPriceHistoryRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSpotPriceHistory that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSpotPriceHistoryRequestT = Model::DescribeSpotPriceHistoryRequest>
        Model::DescribeSpotPriceHistoryOutcomeCallable DescribeSpotPriceHistoryCallable(const DescribeSpotPriceHistoryRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSpotPriceHistory, request);
        }

        /**
         * An Async wrapper for DescribeSpotPriceHistory that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSpotPriceHistoryRequestT = Model::DescribeSpotPriceHistoryRequest>
        void DescribeSpotPriceHistoryAsync(const DescribeSpotPriceHistoryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSpotPriceHistoryRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSpotPriceHistory, request, handler, context);
        }

        /**
         * <p>Describes the stale security group rules for security groups referenced
         * across a VPC peering connection, transit gateway connection, or with a security
         * group VPC association. Rules are stale when they reference a deleted security
         * group. Rules can also be stale if they reference a security group in a peer VPC
         * for which the VPC peering connection has been deleted, across a transit gateway
         * where the transit gateway has been deleted (or <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-vpc-attachments.html#vpc-attachment-security">the
         * transit gateway security group referencing feature</a> has been disabled), or if
         * a security group VPC association has been disassociated.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeStaleSecurityGroups">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeStaleSecurityGroupsOutcome DescribeStaleSecurityGroups(const Model::DescribeStaleSecurityGroupsRequest& request) const;

        /**
         * A Callable wrapper for DescribeStaleSecurityGroups that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeStaleSecurityGroupsRequestT = Model::DescribeStaleSecurityGroupsRequest>
        Model::DescribeStaleSecurityGroupsOutcomeCallable DescribeStaleSecurityGroupsCallable(const DescribeStaleSecurityGroupsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeStaleSecurityGroups, request);
        }

        /**
         * An Async wrapper for DescribeStaleSecurityGroups that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeStaleSecurityGroupsRequestT = Model::DescribeStaleSecurityGroupsRequest>
        void DescribeStaleSecurityGroupsAsync(const DescribeStaleSecurityGroupsRequestT& request, const DescribeStaleSecurityGroupsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeStaleSecurityGroups, request, handler, context);
        }

        /**
         * <p>Describes the progress of the AMI store tasks. You can describe the store
         * tasks for specified AMIs. If you don't specify the AMIs, you get a paginated
         * list of store tasks from the last 31 days.</p> <p>For each AMI task, the
         * response indicates if the task is <code>InProgress</code>,
         * <code>Completed</code>, or <code>Failed</code>. For tasks
         * <code>InProgress</code>, the response shows the estimated progress as a
         * percentage.</p> <p>Tasks are listed in reverse chronological order. Currently,
         * only tasks from the past 31 days can be viewed.</p> <p>To use this API, you must
         * have the required permissions. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/work-with-ami-store-restore.html#ami-s3-permissions">Permissions
         * for storing and restoring AMIs using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-store-restore.html">Store
         * and restore an AMI using S3</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeStoreImageTasks">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeStoreImageTasksOutcome DescribeStoreImageTasks(const Model::DescribeStoreImageTasksRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeStoreImageTasks that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeStoreImageTasksRequestT = Model::DescribeStoreImageTasksRequest>
        Model::DescribeStoreImageTasksOutcomeCallable DescribeStoreImageTasksCallable(const DescribeStoreImageTasksRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeStoreImageTasks, request);
        }

        /**
         * An Async wrapper for DescribeStoreImageTasks that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeStoreImageTasksRequestT = Model::DescribeStoreImageTasksRequest>
        void DescribeStoreImageTasksAsync(const DescribeStoreImageTasksResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeStoreImageTasksRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeStoreImageTasks, request, handler, context);
        }

        /**
         * <p>Describes your subnets. The default is to describe all your subnets.
         * Alternatively, you can specify specific subnet IDs or filter the results to
         * include only the subnets that match specific criteria.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/configure-subnets.html">Subnets</a>
         * in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeSubnets">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeSubnetsOutcome DescribeSubnets(const Model::DescribeSubnetsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeSubnets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeSubnetsRequestT = Model::DescribeSubnetsRequest>
        Model::DescribeSubnetsOutcomeCallable DescribeSubnetsCallable(const DescribeSubnetsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeSubnets, request);
        }

        /**
         * An Async wrapper for DescribeSubnets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeSubnetsRequestT = Model::DescribeSubnetsRequest>
        void DescribeSubnetsAsync(const DescribeSubnetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeSubnetsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeSubnets, request, handler, context);
        }

        /**
         * <p>Describes the specified tags for your EC2 resources.</p> <p>For more
         * information about tags, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tag
         * your Amazon EC2 resources</a> in the <i>Amazon Elastic Compute Cloud User
         * Guide</i>.</p>  <p>We strongly recommend using only paginated
         * requests. Unpaginated requests are susceptible to throttling and timeouts.</p>
         *   <p>The order of the elements in the response, including
         * those within nested structures, might vary. Applications should not assume the
         * elements appear in a particular order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTags">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTagsOutcome DescribeTags(const Model::DescribeTagsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTags that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTagsRequestT = Model::DescribeTagsRequest>
        Model::DescribeTagsOutcomeCallable DescribeTagsCallable(const DescribeTagsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTags, request);
        }

        /**
         * An Async wrapper for DescribeTags that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTagsRequestT = Model::DescribeTagsRequest>
        void DescribeTagsAsync(const DescribeTagsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTagsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTags, request, handler, context);
        }

        /**
         * <p>Describe traffic mirror filters that determine the traffic that is
         * mirrored.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTrafficMirrorFilterRules">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTrafficMirrorFilterRulesOutcome DescribeTrafficMirrorFilterRules(const Model::DescribeTrafficMirrorFilterRulesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTrafficMirrorFilterRules that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTrafficMirrorFilterRulesRequestT = Model::DescribeTrafficMirrorFilterRulesRequest>
        Model::DescribeTrafficMirrorFilterRulesOutcomeCallable DescribeTrafficMirrorFilterRulesCallable(const DescribeTrafficMirrorFilterRulesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTrafficMirrorFilterRules, request);
        }

        /**
         * An Async wrapper for DescribeTrafficMirrorFilterRules that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTrafficMirrorFilterRulesRequestT = Model::DescribeTrafficMirrorFilterRulesRequest>
        void DescribeTrafficMirrorFilterRulesAsync(const DescribeTrafficMirrorFilterRulesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTrafficMirrorFilterRulesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTrafficMirrorFilterRules, request, handler, context);
        }

        /**
         * <p>Describes one or more Traffic Mirror filters.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTrafficMirrorFilters">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTrafficMirrorFiltersOutcome DescribeTrafficMirrorFilters(const Model::DescribeTrafficMirrorFiltersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTrafficMirrorFilters that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTrafficMirrorFiltersRequestT = Model::DescribeTrafficMirrorFiltersRequest>
        Model::DescribeTrafficMirrorFiltersOutcomeCallable DescribeTrafficMirrorFiltersCallable(const DescribeTrafficMirrorFiltersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTrafficMirrorFilters, request);
        }

        /**
         * An Async wrapper for DescribeTrafficMirrorFilters that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTrafficMirrorFiltersRequestT = Model::DescribeTrafficMirrorFiltersRequest>
        void DescribeTrafficMirrorFiltersAsync(const DescribeTrafficMirrorFiltersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTrafficMirrorFiltersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTrafficMirrorFilters, request, handler, context);
        }

        /**
         * <p>Describes one or more Traffic Mirror sessions. By default, all Traffic Mirror
         * sessions are described. Alternatively, you can filter the results.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTrafficMirrorSessions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTrafficMirrorSessionsOutcome DescribeTrafficMirrorSessions(const Model::DescribeTrafficMirrorSessionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTrafficMirrorSessions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTrafficMirrorSessionsRequestT = Model::DescribeTrafficMirrorSessionsRequest>
        Model::DescribeTrafficMirrorSessionsOutcomeCallable DescribeTrafficMirrorSessionsCallable(const DescribeTrafficMirrorSessionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTrafficMirrorSessions, request);
        }

        /**
         * An Async wrapper for DescribeTrafficMirrorSessions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTrafficMirrorSessionsRequestT = Model::DescribeTrafficMirrorSessionsRequest>
        void DescribeTrafficMirrorSessionsAsync(const DescribeTrafficMirrorSessionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTrafficMirrorSessionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTrafficMirrorSessions, request, handler, context);
        }

        /**
         * <p>Information about one or more Traffic Mirror targets.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTrafficMirrorTargets">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTrafficMirrorTargetsOutcome DescribeTrafficMirrorTargets(const Model::DescribeTrafficMirrorTargetsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTrafficMirrorTargets that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTrafficMirrorTargetsRequestT = Model::DescribeTrafficMirrorTargetsRequest>
        Model::DescribeTrafficMirrorTargetsOutcomeCallable DescribeTrafficMirrorTargetsCallable(const DescribeTrafficMirrorTargetsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTrafficMirrorTargets, request);
        }

        /**
         * An Async wrapper for DescribeTrafficMirrorTargets that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTrafficMirrorTargetsRequestT = Model::DescribeTrafficMirrorTargetsRequest>
        void DescribeTrafficMirrorTargetsAsync(const DescribeTrafficMirrorTargetsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTrafficMirrorTargetsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTrafficMirrorTargets, request, handler, context);
        }

        /**
         * <p>Describes one or more attachments between resources and transit gateways. By
         * default, all attachments are described. Alternatively, you can filter the
         * results by attachment ID, attachment state, resource ID, or resource
         * owner.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayAttachments">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayAttachmentsOutcome DescribeTransitGatewayAttachments(const Model::DescribeTransitGatewayAttachmentsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayAttachments that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayAttachmentsRequestT = Model::DescribeTransitGatewayAttachmentsRequest>
        Model::DescribeTransitGatewayAttachmentsOutcomeCallable DescribeTransitGatewayAttachmentsCallable(const DescribeTransitGatewayAttachmentsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayAttachments, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayAttachments that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayAttachmentsRequestT = Model::DescribeTransitGatewayAttachmentsRequest>
        void DescribeTransitGatewayAttachmentsAsync(const DescribeTransitGatewayAttachmentsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayAttachmentsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayAttachments, request, handler, context);
        }

        /**
         * <p>Describes one or more Connect peers.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayConnectPeers">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayConnectPeersOutcome DescribeTransitGatewayConnectPeers(const Model::DescribeTransitGatewayConnectPeersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayConnectPeers that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayConnectPeersRequestT = Model::DescribeTransitGatewayConnectPeersRequest>
        Model::DescribeTransitGatewayConnectPeersOutcomeCallable DescribeTransitGatewayConnectPeersCallable(const DescribeTransitGatewayConnectPeersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayConnectPeers, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayConnectPeers that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayConnectPeersRequestT = Model::DescribeTransitGatewayConnectPeersRequest>
        void DescribeTransitGatewayConnectPeersAsync(const DescribeTransitGatewayConnectPeersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayConnectPeersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayConnectPeers, request, handler, context);
        }

        /**
         * <p>Describes one or more Connect attachments.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayConnects">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayConnectsOutcome DescribeTransitGatewayConnects(const Model::DescribeTransitGatewayConnectsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayConnects that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayConnectsRequestT = Model::DescribeTransitGatewayConnectsRequest>
        Model::DescribeTransitGatewayConnectsOutcomeCallable DescribeTransitGatewayConnectsCallable(const DescribeTransitGatewayConnectsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayConnects, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayConnects that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayConnectsRequestT = Model::DescribeTransitGatewayConnectsRequest>
        void DescribeTransitGatewayConnectsAsync(const DescribeTransitGatewayConnectsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayConnectsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayConnects, request, handler, context);
        }

        /**
         * <p>Describes one or more transit gateway multicast domains.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayMulticastDomains">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayMulticastDomainsOutcome DescribeTransitGatewayMulticastDomains(const Model::DescribeTransitGatewayMulticastDomainsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayMulticastDomains that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayMulticastDomainsRequestT = Model::DescribeTransitGatewayMulticastDomainsRequest>
        Model::DescribeTransitGatewayMulticastDomainsOutcomeCallable DescribeTransitGatewayMulticastDomainsCallable(const DescribeTransitGatewayMulticastDomainsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayMulticastDomains, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayMulticastDomains that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayMulticastDomainsRequestT = Model::DescribeTransitGatewayMulticastDomainsRequest>
        void DescribeTransitGatewayMulticastDomainsAsync(const DescribeTransitGatewayMulticastDomainsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayMulticastDomainsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayMulticastDomains, request, handler, context);
        }

        /**
         * <p>Describes your transit gateway peering attachments.</p><p><h3>See Also:</h3> 
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayPeeringAttachments">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayPeeringAttachmentsOutcome DescribeTransitGatewayPeeringAttachments(const Model::DescribeTransitGatewayPeeringAttachmentsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayPeeringAttachments that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayPeeringAttachmentsRequestT = Model::DescribeTransitGatewayPeeringAttachmentsRequest>
        Model::DescribeTransitGatewayPeeringAttachmentsOutcomeCallable DescribeTransitGatewayPeeringAttachmentsCallable(const DescribeTransitGatewayPeeringAttachmentsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayPeeringAttachments, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayPeeringAttachments that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayPeeringAttachmentsRequestT = Model::DescribeTransitGatewayPeeringAttachmentsRequest>
        void DescribeTransitGatewayPeeringAttachmentsAsync(const DescribeTransitGatewayPeeringAttachmentsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayPeeringAttachmentsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayPeeringAttachments, request, handler, context);
        }

        /**
         * <p>Describes one or more transit gateway route policy tables. </p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayPolicyTables">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayPolicyTablesOutcome DescribeTransitGatewayPolicyTables(const Model::DescribeTransitGatewayPolicyTablesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayPolicyTables that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayPolicyTablesRequestT = Model::DescribeTransitGatewayPolicyTablesRequest>
        Model::DescribeTransitGatewayPolicyTablesOutcomeCallable DescribeTransitGatewayPolicyTablesCallable(const DescribeTransitGatewayPolicyTablesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayPolicyTables, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayPolicyTables that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayPolicyTablesRequestT = Model::DescribeTransitGatewayPolicyTablesRequest>
        void DescribeTransitGatewayPolicyTablesAsync(const DescribeTransitGatewayPolicyTablesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayPolicyTablesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayPolicyTables, request, handler, context);
        }

        /**
         * <p>Describes one or more transit gateway route table
         * advertisements.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayRouteTableAnnouncements">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayRouteTableAnnouncementsOutcome DescribeTransitGatewayRouteTableAnnouncements(const Model::DescribeTransitGatewayRouteTableAnnouncementsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayRouteTableAnnouncements that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayRouteTableAnnouncementsRequestT = Model::DescribeTransitGatewayRouteTableAnnouncementsRequest>
        Model::DescribeTransitGatewayRouteTableAnnouncementsOutcomeCallable DescribeTransitGatewayRouteTableAnnouncementsCallable(const DescribeTransitGatewayRouteTableAnnouncementsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayRouteTableAnnouncements, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayRouteTableAnnouncements that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayRouteTableAnnouncementsRequestT = Model::DescribeTransitGatewayRouteTableAnnouncementsRequest>
        void DescribeTransitGatewayRouteTableAnnouncementsAsync(const DescribeTransitGatewayRouteTableAnnouncementsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayRouteTableAnnouncementsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayRouteTableAnnouncements, request, handler, context);
        }

        /**
         * <p>Describes one or more transit gateway route tables. By default, all transit
         * gateway route tables are described. Alternatively, you can filter the
         * results.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayRouteTables">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayRouteTablesOutcome DescribeTransitGatewayRouteTables(const Model::DescribeTransitGatewayRouteTablesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayRouteTables that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayRouteTablesRequestT = Model::DescribeTransitGatewayRouteTablesRequest>
        Model::DescribeTransitGatewayRouteTablesOutcomeCallable DescribeTransitGatewayRouteTablesCallable(const DescribeTransitGatewayRouteTablesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayRouteTables, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayRouteTables that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayRouteTablesRequestT = Model::DescribeTransitGatewayRouteTablesRequest>
        void DescribeTransitGatewayRouteTablesAsync(const DescribeTransitGatewayRouteTablesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayRouteTablesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayRouteTables, request, handler, context);
        }

        /**
         * <p>Describes one or more VPC attachments. By default, all VPC attachments are
         * described. Alternatively, you can filter the results.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGatewayVpcAttachments">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewayVpcAttachmentsOutcome DescribeTransitGatewayVpcAttachments(const Model::DescribeTransitGatewayVpcAttachmentsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGatewayVpcAttachments that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewayVpcAttachmentsRequestT = Model::DescribeTransitGatewayVpcAttachmentsRequest>
        Model::DescribeTransitGatewayVpcAttachmentsOutcomeCallable DescribeTransitGatewayVpcAttachmentsCallable(const DescribeTransitGatewayVpcAttachmentsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGatewayVpcAttachments, request);
        }

        /**
         * An Async wrapper for DescribeTransitGatewayVpcAttachments that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewayVpcAttachmentsRequestT = Model::DescribeTransitGatewayVpcAttachmentsRequest>
        void DescribeTransitGatewayVpcAttachmentsAsync(const DescribeTransitGatewayVpcAttachmentsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewayVpcAttachmentsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGatewayVpcAttachments, request, handler, context);
        }

        /**
         * <p>Describes one or more transit gateways. By default, all transit gateways are
         * described. Alternatively, you can filter the results.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTransitGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTransitGatewaysOutcome DescribeTransitGateways(const Model::DescribeTransitGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTransitGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTransitGatewaysRequestT = Model::DescribeTransitGatewaysRequest>
        Model::DescribeTransitGatewaysOutcomeCallable DescribeTransitGatewaysCallable(const DescribeTransitGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTransitGateways, request);
        }

        /**
         * An Async wrapper for DescribeTransitGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTransitGatewaysRequestT = Model::DescribeTransitGatewaysRequest>
        void DescribeTransitGatewaysAsync(const DescribeTransitGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTransitGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTransitGateways, request, handler, context);
        }

        /**
         * <p>Describes one or more network interface trunk associations.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeTrunkInterfaceAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeTrunkInterfaceAssociationsOutcome DescribeTrunkInterfaceAssociations(const Model::DescribeTrunkInterfaceAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeTrunkInterfaceAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeTrunkInterfaceAssociationsRequestT = Model::DescribeTrunkInterfaceAssociationsRequest>
        Model::DescribeTrunkInterfaceAssociationsOutcomeCallable DescribeTrunkInterfaceAssociationsCallable(const DescribeTrunkInterfaceAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeTrunkInterfaceAssociations, request);
        }

        /**
         * An Async wrapper for DescribeTrunkInterfaceAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeTrunkInterfaceAssociationsRequestT = Model::DescribeTrunkInterfaceAssociationsRequest>
        void DescribeTrunkInterfaceAssociationsAsync(const DescribeTrunkInterfaceAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeTrunkInterfaceAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeTrunkInterfaceAssociations, request, handler, context);
        }

        /**
         * <p>Describes the specified Amazon Web Services Verified Access
         * endpoints.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVerifiedAccessEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVerifiedAccessEndpointsOutcome DescribeVerifiedAccessEndpoints(const Model::DescribeVerifiedAccessEndpointsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVerifiedAccessEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVerifiedAccessEndpointsRequestT = Model::DescribeVerifiedAccessEndpointsRequest>
        Model::DescribeVerifiedAccessEndpointsOutcomeCallable DescribeVerifiedAccessEndpointsCallable(const DescribeVerifiedAccessEndpointsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVerifiedAccessEndpoints, request);
        }

        /**
         * An Async wrapper for DescribeVerifiedAccessEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVerifiedAccessEndpointsRequestT = Model::DescribeVerifiedAccessEndpointsRequest>
        void DescribeVerifiedAccessEndpointsAsync(const DescribeVerifiedAccessEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVerifiedAccessEndpointsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVerifiedAccessEndpoints, request, handler, context);
        }

        /**
         * <p>Describes the specified Verified Access groups.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVerifiedAccessGroups">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVerifiedAccessGroupsOutcome DescribeVerifiedAccessGroups(const Model::DescribeVerifiedAccessGroupsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVerifiedAccessGroups that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVerifiedAccessGroupsRequestT = Model::DescribeVerifiedAccessGroupsRequest>
        Model::DescribeVerifiedAccessGroupsOutcomeCallable DescribeVerifiedAccessGroupsCallable(const DescribeVerifiedAccessGroupsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVerifiedAccessGroups, request);
        }

        /**
         * An Async wrapper for DescribeVerifiedAccessGroups that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVerifiedAccessGroupsRequestT = Model::DescribeVerifiedAccessGroupsRequest>
        void DescribeVerifiedAccessGroupsAsync(const DescribeVerifiedAccessGroupsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVerifiedAccessGroupsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVerifiedAccessGroups, request, handler, context);
        }

        /**
         * <p>Describes the specified Amazon Web Services Verified Access
         * instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVerifiedAccessInstanceLoggingConfigurations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVerifiedAccessInstanceLoggingConfigurationsOutcome DescribeVerifiedAccessInstanceLoggingConfigurations(const Model::DescribeVerifiedAccessInstanceLoggingConfigurationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVerifiedAccessInstanceLoggingConfigurations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVerifiedAccessInstanceLoggingConfigurationsRequestT = Model::DescribeVerifiedAccessInstanceLoggingConfigurationsRequest>
        Model::DescribeVerifiedAccessInstanceLoggingConfigurationsOutcomeCallable DescribeVerifiedAccessInstanceLoggingConfigurationsCallable(const DescribeVerifiedAccessInstanceLoggingConfigurationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVerifiedAccessInstanceLoggingConfigurations, request);
        }

        /**
         * An Async wrapper for DescribeVerifiedAccessInstanceLoggingConfigurations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVerifiedAccessInstanceLoggingConfigurationsRequestT = Model::DescribeVerifiedAccessInstanceLoggingConfigurationsRequest>
        void DescribeVerifiedAccessInstanceLoggingConfigurationsAsync(const DescribeVerifiedAccessInstanceLoggingConfigurationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVerifiedAccessInstanceLoggingConfigurationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVerifiedAccessInstanceLoggingConfigurations, request, handler, context);
        }

        /**
         * <p>Describes the specified Amazon Web Services Verified Access
         * instances.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVerifiedAccessInstances">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVerifiedAccessInstancesOutcome DescribeVerifiedAccessInstances(const Model::DescribeVerifiedAccessInstancesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVerifiedAccessInstances that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVerifiedAccessInstancesRequestT = Model::DescribeVerifiedAccessInstancesRequest>
        Model::DescribeVerifiedAccessInstancesOutcomeCallable DescribeVerifiedAccessInstancesCallable(const DescribeVerifiedAccessInstancesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVerifiedAccessInstances, request);
        }

        /**
         * An Async wrapper for DescribeVerifiedAccessInstances that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVerifiedAccessInstancesRequestT = Model::DescribeVerifiedAccessInstancesRequest>
        void DescribeVerifiedAccessInstancesAsync(const DescribeVerifiedAccessInstancesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVerifiedAccessInstancesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVerifiedAccessInstances, request, handler, context);
        }

        /**
         * <p>Describes the specified Amazon Web Services Verified Access trust
         * providers.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVerifiedAccessTrustProviders">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVerifiedAccessTrustProvidersOutcome DescribeVerifiedAccessTrustProviders(const Model::DescribeVerifiedAccessTrustProvidersRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVerifiedAccessTrustProviders that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVerifiedAccessTrustProvidersRequestT = Model::DescribeVerifiedAccessTrustProvidersRequest>
        Model::DescribeVerifiedAccessTrustProvidersOutcomeCallable DescribeVerifiedAccessTrustProvidersCallable(const DescribeVerifiedAccessTrustProvidersRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVerifiedAccessTrustProviders, request);
        }

        /**
         * An Async wrapper for DescribeVerifiedAccessTrustProviders that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVerifiedAccessTrustProvidersRequestT = Model::DescribeVerifiedAccessTrustProvidersRequest>
        void DescribeVerifiedAccessTrustProvidersAsync(const DescribeVerifiedAccessTrustProvidersResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVerifiedAccessTrustProvidersRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVerifiedAccessTrustProviders, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified volume. You can specify
         * only one attribute at a time.</p> <p>For more information about EBS volumes, see
         * <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-volumes.html">Amazon
         * EBS volumes</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVolumeAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVolumeAttributeOutcome DescribeVolumeAttribute(const Model::DescribeVolumeAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeVolumeAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVolumeAttributeRequestT = Model::DescribeVolumeAttributeRequest>
        Model::DescribeVolumeAttributeOutcomeCallable DescribeVolumeAttributeCallable(const DescribeVolumeAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeVolumeAttribute, request);
        }

        /**
         * An Async wrapper for DescribeVolumeAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVolumeAttributeRequestT = Model::DescribeVolumeAttributeRequest>
        void DescribeVolumeAttributeAsync(const DescribeVolumeAttributeRequestT& request, const DescribeVolumeAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeVolumeAttribute, request, handler, context);
        }

        /**
         * <p>Describes the status of the specified volumes. Volume status provides the
         * result of the checks performed on your volumes to determine events that can
         * impair the performance of your volumes. The performance of a volume can be
         * affected if an issue occurs on the volume's underlying host. If the volume's
         * underlying host experiences a power outage or system issue, after the system is
         * restored, there could be data inconsistencies on the volume. Volume events
         * notify you if this occurs. Volume actions notify you if any action needs to be
         * taken in response to the event.</p> <p>The <code>DescribeVolumeStatus</code>
         * operation provides the following information about the specified volumes:</p>
         * <p> <i>Status</i>: Reflects the current status of the volume. The possible
         * values are <code>ok</code>, <code>impaired</code> , <code>warning</code>, or
         * <code>insufficient-data</code>. If all checks pass, the overall status of the
         * volume is <code>ok</code>. If the check fails, the overall status is
         * <code>impaired</code>. If the status is <code>insufficient-data</code>, then the
         * checks might still be taking place on your volume at the time. We recommend that
         * you retry the request. For more information about volume status, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/monitoring-volume-status.html">Monitor
         * the status of your volumes</a> in the <i>Amazon EBS User Guide</i>.</p> <p>
         * <i>Events</i>: Reflect the cause of a volume status and might require you to
         * take action. For example, if your volume returns an <code>impaired</code>
         * status, then the volume event might be
         * <code>potential-data-inconsistency</code>. This means that your volume has been
         * affected by an issue with the underlying host, has all I/O operations disabled,
         * and might have inconsistent data.</p> <p> <i>Actions</i>: Reflect the actions
         * you might have to take in response to an event. For example, if the status of
         * the volume is <code>impaired</code> and the volume event shows
         * <code>potential-data-inconsistency</code>, then the action shows
         * <code>enable-volume-io</code>. This means that you may want to enable the I/O
         * operations for the volume by calling the <a>EnableVolumeIO</a> action and then
         * check the volume for data consistency.</p> <p>Volume status is based on the
         * volume status checks, and does not reflect the volume state. Therefore, volume
         * status does not indicate volumes in the <code>error</code> state (for example,
         * when a volume is incapable of accepting I/O.)</p>  <p>The order of the
         * elements in the response, including those within nested structures, might vary.
         * Applications should not assume the elements appear in a particular order.</p>
         * <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVolumeStatus">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVolumeStatusOutcome DescribeVolumeStatus(const Model::DescribeVolumeStatusRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVolumeStatus that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVolumeStatusRequestT = Model::DescribeVolumeStatusRequest>
        Model::DescribeVolumeStatusOutcomeCallable DescribeVolumeStatusCallable(const DescribeVolumeStatusRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVolumeStatus, request);
        }

        /**
         * An Async wrapper for DescribeVolumeStatus that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVolumeStatusRequestT = Model::DescribeVolumeStatusRequest>
        void DescribeVolumeStatusAsync(const DescribeVolumeStatusResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVolumeStatusRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVolumeStatus, request, handler, context);
        }

        /**
         * <p>Describes the specified EBS volumes or all of your EBS volumes.</p> <p>If you
         * are describing a long list of volumes, we recommend that you paginate the output
         * to make the list more manageable. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination">Pagination</a>.</p>
         * <p>For more information about EBS volumes, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-volumes.html">Amazon
         * EBS volumes</a> in the <i>Amazon EBS User Guide</i>.</p>  <p>We
         * strongly recommend using only paginated requests. Unpaginated requests are
         * susceptible to throttling and timeouts.</p>   <p>The order of
         * the elements in the response, including those within nested structures, might
         * vary. Applications should not assume the elements appear in a particular
         * order.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVolumes">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVolumesOutcome DescribeVolumes(const Model::DescribeVolumesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVolumes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVolumesRequestT = Model::DescribeVolumesRequest>
        Model::DescribeVolumesOutcomeCallable DescribeVolumesCallable(const DescribeVolumesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVolumes, request);
        }

        /**
         * An Async wrapper for DescribeVolumes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVolumesRequestT = Model::DescribeVolumesRequest>
        void DescribeVolumesAsync(const DescribeVolumesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVolumesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVolumes, request, handler, context);
        }

        /**
         * <p>Describes the most recent volume modification request for the specified EBS
         * volumes.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/monitoring-volume-modifications.html">
         * Monitor the progress of volume modifications</a> in the <i>Amazon EBS User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVolumesModifications">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVolumesModificationsOutcome DescribeVolumesModifications(const Model::DescribeVolumesModificationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVolumesModifications that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVolumesModificationsRequestT = Model::DescribeVolumesModificationsRequest>
        Model::DescribeVolumesModificationsOutcomeCallable DescribeVolumesModificationsCallable(const DescribeVolumesModificationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVolumesModifications, request);
        }

        /**
         * An Async wrapper for DescribeVolumesModifications that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVolumesModificationsRequestT = Model::DescribeVolumesModificationsRequest>
        void DescribeVolumesModificationsAsync(const DescribeVolumesModificationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVolumesModificationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVolumesModifications, request, handler, context);
        }

        /**
         * <p>Describes the specified attribute of the specified VPC. You can specify only
         * one attribute at a time.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcAttribute">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcAttributeOutcome DescribeVpcAttribute(const Model::DescribeVpcAttributeRequest& request) const;

        /**
         * A Callable wrapper for DescribeVpcAttribute that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcAttributeRequestT = Model::DescribeVpcAttributeRequest>
        Model::DescribeVpcAttributeOutcomeCallable DescribeVpcAttributeCallable(const DescribeVpcAttributeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcAttribute, request);
        }

        /**
         * An Async wrapper for DescribeVpcAttribute that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcAttributeRequestT = Model::DescribeVpcAttributeRequest>
        void DescribeVpcAttributeAsync(const DescribeVpcAttributeRequestT& request, const DescribeVpcAttributeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcAttribute, request, handler, context);
        }

        /**
         * <p>Describe VPC Block Public Access (BPA) exclusions. A VPC BPA exclusion is a
         * mode that can be applied to a single VPC or subnet that exempts it from the
         * account’s BPA mode and will allow bidirectional or egress-only access. You can
         * create BPA exclusions for VPCs and subnets even when BPA is not enabled on the
         * account to ensure that there is no traffic disruption to the exclusions when VPC
         * BPA is turned on. To learn more about VPC BPA, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-vpc-bpa.html">Block
         * public access to VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcBlockPublicAccessExclusions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcBlockPublicAccessExclusionsOutcome DescribeVpcBlockPublicAccessExclusions(const Model::DescribeVpcBlockPublicAccessExclusionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcBlockPublicAccessExclusions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcBlockPublicAccessExclusionsRequestT = Model::DescribeVpcBlockPublicAccessExclusionsRequest>
        Model::DescribeVpcBlockPublicAccessExclusionsOutcomeCallable DescribeVpcBlockPublicAccessExclusionsCallable(const DescribeVpcBlockPublicAccessExclusionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcBlockPublicAccessExclusions, request);
        }

        /**
         * An Async wrapper for DescribeVpcBlockPublicAccessExclusions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcBlockPublicAccessExclusionsRequestT = Model::DescribeVpcBlockPublicAccessExclusionsRequest>
        void DescribeVpcBlockPublicAccessExclusionsAsync(const DescribeVpcBlockPublicAccessExclusionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcBlockPublicAccessExclusionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcBlockPublicAccessExclusions, request, handler, context);
        }

        /**
         * <p>Describe VPC Block Public Access (BPA) options. VPC Block Public Access (BPA)
         * enables you to block resources in VPCs and subnets that you own in a Region from
         * reaching or being reached from the internet through internet gateways and
         * egress-only internet gateways. To learn more about VPC BPA, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-vpc-bpa.html">Block
         * public access to VPCs and subnets</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcBlockPublicAccessOptions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcBlockPublicAccessOptionsOutcome DescribeVpcBlockPublicAccessOptions(const Model::DescribeVpcBlockPublicAccessOptionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcBlockPublicAccessOptions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcBlockPublicAccessOptionsRequestT = Model::DescribeVpcBlockPublicAccessOptionsRequest>
        Model::DescribeVpcBlockPublicAccessOptionsOutcomeCallable DescribeVpcBlockPublicAccessOptionsCallable(const DescribeVpcBlockPublicAccessOptionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcBlockPublicAccessOptions, request);
        }

        /**
         * An Async wrapper for DescribeVpcBlockPublicAccessOptions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcBlockPublicAccessOptionsRequestT = Model::DescribeVpcBlockPublicAccessOptionsRequest>
        void DescribeVpcBlockPublicAccessOptionsAsync(const DescribeVpcBlockPublicAccessOptionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcBlockPublicAccessOptionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcBlockPublicAccessOptions, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Describes the ClassicLink
         * status of the specified VPCs.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcClassicLink">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcClassicLinkOutcome DescribeVpcClassicLink(const Model::DescribeVpcClassicLinkRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcClassicLink that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcClassicLinkRequestT = Model::DescribeVpcClassicLinkRequest>
        Model::DescribeVpcClassicLinkOutcomeCallable DescribeVpcClassicLinkCallable(const DescribeVpcClassicLinkRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcClassicLink, request);
        }

        /**
         * An Async wrapper for DescribeVpcClassicLink that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcClassicLinkRequestT = Model::DescribeVpcClassicLinkRequest>
        void DescribeVpcClassicLinkAsync(const DescribeVpcClassicLinkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcClassicLinkRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcClassicLink, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Describes the ClassicLink
         * DNS support status of one or more VPCs. If enabled, the DNS hostname of a linked
         * EC2-Classic instance resolves to its private IP address when addressed from an
         * instance in the VPC to which it's linked. Similarly, the DNS hostname of an
         * instance in a VPC resolves to its private IP address when addressed from a
         * linked EC2-Classic instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcClassicLinkDnsSupport">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcClassicLinkDnsSupportOutcome DescribeVpcClassicLinkDnsSupport(const Model::DescribeVpcClassicLinkDnsSupportRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcClassicLinkDnsSupport that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcClassicLinkDnsSupportRequestT = Model::DescribeVpcClassicLinkDnsSupportRequest>
        Model::DescribeVpcClassicLinkDnsSupportOutcomeCallable DescribeVpcClassicLinkDnsSupportCallable(const DescribeVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcClassicLinkDnsSupport, request);
        }

        /**
         * An Async wrapper for DescribeVpcClassicLinkDnsSupport that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcClassicLinkDnsSupportRequestT = Model::DescribeVpcClassicLinkDnsSupportRequest>
        void DescribeVpcClassicLinkDnsSupportAsync(const DescribeVpcClassicLinkDnsSupportResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcClassicLinkDnsSupport, request, handler, context);
        }

        /**
         * <p>Describes the VPC resources, VPC endpoint services, Amazon Lattice services,
         * or service networks associated with the VPC endpoint.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointAssociations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointAssociationsOutcome DescribeVpcEndpointAssociations(const Model::DescribeVpcEndpointAssociationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointAssociations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointAssociationsRequestT = Model::DescribeVpcEndpointAssociationsRequest>
        Model::DescribeVpcEndpointAssociationsOutcomeCallable DescribeVpcEndpointAssociationsCallable(const DescribeVpcEndpointAssociationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointAssociations, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointAssociations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointAssociationsRequestT = Model::DescribeVpcEndpointAssociationsRequest>
        void DescribeVpcEndpointAssociationsAsync(const DescribeVpcEndpointAssociationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointAssociationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointAssociations, request, handler, context);
        }

        /**
         * <p>Describes the connection notifications for VPC endpoints and VPC endpoint
         * services.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointConnectionNotifications">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointConnectionNotificationsOutcome DescribeVpcEndpointConnectionNotifications(const Model::DescribeVpcEndpointConnectionNotificationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointConnectionNotifications that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointConnectionNotificationsRequestT = Model::DescribeVpcEndpointConnectionNotificationsRequest>
        Model::DescribeVpcEndpointConnectionNotificationsOutcomeCallable DescribeVpcEndpointConnectionNotificationsCallable(const DescribeVpcEndpointConnectionNotificationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointConnectionNotifications, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointConnectionNotifications that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointConnectionNotificationsRequestT = Model::DescribeVpcEndpointConnectionNotificationsRequest>
        void DescribeVpcEndpointConnectionNotificationsAsync(const DescribeVpcEndpointConnectionNotificationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointConnectionNotificationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointConnectionNotifications, request, handler, context);
        }

        /**
         * <p>Describes the VPC endpoint connections to your VPC endpoint services,
         * including any endpoints that are pending your acceptance.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointConnections">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointConnectionsOutcome DescribeVpcEndpointConnections(const Model::DescribeVpcEndpointConnectionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointConnections that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointConnectionsRequestT = Model::DescribeVpcEndpointConnectionsRequest>
        Model::DescribeVpcEndpointConnectionsOutcomeCallable DescribeVpcEndpointConnectionsCallable(const DescribeVpcEndpointConnectionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointConnections, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointConnections that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointConnectionsRequestT = Model::DescribeVpcEndpointConnectionsRequest>
        void DescribeVpcEndpointConnectionsAsync(const DescribeVpcEndpointConnectionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointConnectionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointConnections, request, handler, context);
        }

        /**
         * <p>Describes the VPC endpoint service configurations in your account (your
         * services).</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointServiceConfigurations">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointServiceConfigurationsOutcome DescribeVpcEndpointServiceConfigurations(const Model::DescribeVpcEndpointServiceConfigurationsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointServiceConfigurations that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointServiceConfigurationsRequestT = Model::DescribeVpcEndpointServiceConfigurationsRequest>
        Model::DescribeVpcEndpointServiceConfigurationsOutcomeCallable DescribeVpcEndpointServiceConfigurationsCallable(const DescribeVpcEndpointServiceConfigurationsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointServiceConfigurations, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointServiceConfigurations that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointServiceConfigurationsRequestT = Model::DescribeVpcEndpointServiceConfigurationsRequest>
        void DescribeVpcEndpointServiceConfigurationsAsync(const DescribeVpcEndpointServiceConfigurationsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointServiceConfigurationsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointServiceConfigurations, request, handler, context);
        }

        /**
         * <p>Describes the principals (service consumers) that are permitted to discover
         * your VPC endpoint service. Principal ARNs with path components aren't
         * supported.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointServicePermissions">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointServicePermissionsOutcome DescribeVpcEndpointServicePermissions(const Model::DescribeVpcEndpointServicePermissionsRequest& request) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointServicePermissions that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointServicePermissionsRequestT = Model::DescribeVpcEndpointServicePermissionsRequest>
        Model::DescribeVpcEndpointServicePermissionsOutcomeCallable DescribeVpcEndpointServicePermissionsCallable(const DescribeVpcEndpointServicePermissionsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointServicePermissions, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointServicePermissions that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointServicePermissionsRequestT = Model::DescribeVpcEndpointServicePermissionsRequest>
        void DescribeVpcEndpointServicePermissionsAsync(const DescribeVpcEndpointServicePermissionsRequestT& request, const DescribeVpcEndpointServicePermissionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointServicePermissions, request, handler, context);
        }

        /**
         * <p>Describes available services to which you can create a VPC endpoint.</p>
         * <p>When the service provider and the consumer have different accounts in
         * multiple Availability Zones, and the consumer views the VPC endpoint service
         * information, the response only includes the common Availability Zones. For
         * example, when the service provider account uses <code>us-east-1a</code> and
         * <code>us-east-1c</code> and the consumer uses <code>us-east-1a</code> and
         * <code>us-east-1b</code>, the response includes the VPC endpoint services in the
         * common Availability Zone, <code>us-east-1a</code>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpointServices">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointServicesOutcome DescribeVpcEndpointServices(const Model::DescribeVpcEndpointServicesRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpointServices that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointServicesRequestT = Model::DescribeVpcEndpointServicesRequest>
        Model::DescribeVpcEndpointServicesOutcomeCallable DescribeVpcEndpointServicesCallable(const DescribeVpcEndpointServicesRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpointServices, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpointServices that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointServicesRequestT = Model::DescribeVpcEndpointServicesRequest>
        void DescribeVpcEndpointServicesAsync(const DescribeVpcEndpointServicesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointServicesRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpointServices, request, handler, context);
        }

        /**
         * <p>Describes your VPC endpoints. The default is to describe all your VPC
         * endpoints. Alternatively, you can specify specific VPC endpoint IDs or filter
         * the results to include only the VPC endpoints that match specific
         * criteria.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcEndpoints">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcEndpointsOutcome DescribeVpcEndpoints(const Model::DescribeVpcEndpointsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcEndpoints that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcEndpointsRequestT = Model::DescribeVpcEndpointsRequest>
        Model::DescribeVpcEndpointsOutcomeCallable DescribeVpcEndpointsCallable(const DescribeVpcEndpointsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcEndpoints, request);
        }

        /**
         * An Async wrapper for DescribeVpcEndpoints that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcEndpointsRequestT = Model::DescribeVpcEndpointsRequest>
        void DescribeVpcEndpointsAsync(const DescribeVpcEndpointsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcEndpointsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcEndpoints, request, handler, context);
        }

        /**
         * <p>Describes your VPC peering connections. The default is to describe all your
         * VPC peering connections. Alternatively, you can specify specific VPC peering
         * connection IDs or filter the results to include only the VPC peering connections
         * that match specific criteria.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcPeeringConnections">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcPeeringConnectionsOutcome DescribeVpcPeeringConnections(const Model::DescribeVpcPeeringConnectionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcPeeringConnections that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcPeeringConnectionsRequestT = Model::DescribeVpcPeeringConnectionsRequest>
        Model::DescribeVpcPeeringConnectionsOutcomeCallable DescribeVpcPeeringConnectionsCallable(const DescribeVpcPeeringConnectionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcPeeringConnections, request);
        }

        /**
         * An Async wrapper for DescribeVpcPeeringConnections that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcPeeringConnectionsRequestT = Model::DescribeVpcPeeringConnectionsRequest>
        void DescribeVpcPeeringConnectionsAsync(const DescribeVpcPeeringConnectionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcPeeringConnectionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcPeeringConnections, request, handler, context);
        }

        /**
         * <p>Describes your VPCs. The default is to describe all your VPCs. Alternatively,
         * you can specify specific VPC IDs or filter the results to include only the VPCs
         * that match specific criteria.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpcs">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpcsOutcome DescribeVpcs(const Model::DescribeVpcsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpcs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpcsRequestT = Model::DescribeVpcsRequest>
        Model::DescribeVpcsOutcomeCallable DescribeVpcsCallable(const DescribeVpcsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpcs, request);
        }

        /**
         * An Async wrapper for DescribeVpcs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpcsRequestT = Model::DescribeVpcsRequest>
        void DescribeVpcsAsync(const DescribeVpcsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpcsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpcs, request, handler, context);
        }

        /**
         * <p>Describes one or more of your VPN connections.</p> <p>For more information,
         * see <a href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon
         * Web Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpnConnections">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpnConnectionsOutcome DescribeVpnConnections(const Model::DescribeVpnConnectionsRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpnConnections that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpnConnectionsRequestT = Model::DescribeVpnConnectionsRequest>
        Model::DescribeVpnConnectionsOutcomeCallable DescribeVpnConnectionsCallable(const DescribeVpnConnectionsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpnConnections, request);
        }

        /**
         * An Async wrapper for DescribeVpnConnections that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpnConnectionsRequestT = Model::DescribeVpnConnectionsRequest>
        void DescribeVpnConnectionsAsync(const DescribeVpnConnectionsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpnConnectionsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpnConnections, request, handler, context);
        }

        /**
         * <p>Describes one or more of your virtual private gateways.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html">Amazon Web
         * Services Site-to-Site VPN</a> in the <i>Amazon Web Services Site-to-Site VPN
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DescribeVpnGateways">AWS
         * API Reference</a></p>
         */
        virtual Model::DescribeVpnGatewaysOutcome DescribeVpnGateways(const Model::DescribeVpnGatewaysRequest& request = {}) const;

        /**
         * A Callable wrapper for DescribeVpnGateways that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DescribeVpnGatewaysRequestT = Model::DescribeVpnGatewaysRequest>
        Model::DescribeVpnGatewaysOutcomeCallable DescribeVpnGatewaysCallable(const DescribeVpnGatewaysRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DescribeVpnGateways, request);
        }

        /**
         * An Async wrapper for DescribeVpnGateways that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DescribeVpnGatewaysRequestT = Model::DescribeVpnGatewaysRequest>
        void DescribeVpnGatewaysAsync(const DescribeVpnGatewaysResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DescribeVpnGatewaysRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DescribeVpnGateways, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Unlinks (detaches) a linked
         * EC2-Classic instance from a VPC. After the instance has been unlinked, the VPC
         * security groups are no longer associated with it. An instance is automatically
         * unlinked from a VPC when it's stopped.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachClassicLinkVpc">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachClassicLinkVpcOutcome DetachClassicLinkVpc(const Model::DetachClassicLinkVpcRequest& request) const;

        /**
         * A Callable wrapper for DetachClassicLinkVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachClassicLinkVpcRequestT = Model::DetachClassicLinkVpcRequest>
        Model::DetachClassicLinkVpcOutcomeCallable DetachClassicLinkVpcCallable(const DetachClassicLinkVpcRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachClassicLinkVpc, request);
        }

        /**
         * An Async wrapper for DetachClassicLinkVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachClassicLinkVpcRequestT = Model::DetachClassicLinkVpcRequest>
        void DetachClassicLinkVpcAsync(const DetachClassicLinkVpcRequestT& request, const DetachClassicLinkVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachClassicLinkVpc, request, handler, context);
        }

        /**
         * <p>Detaches an internet gateway from a VPC, disabling connectivity between the
         * internet and the VPC. The VPC must not contain any running instances with
         * Elastic IP addresses or public IPv4 addresses.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachInternetGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachInternetGatewayOutcome DetachInternetGateway(const Model::DetachInternetGatewayRequest& request) const;

        /**
         * A Callable wrapper for DetachInternetGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachInternetGatewayRequestT = Model::DetachInternetGatewayRequest>
        Model::DetachInternetGatewayOutcomeCallable DetachInternetGatewayCallable(const DetachInternetGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachInternetGateway, request);
        }

        /**
         * An Async wrapper for DetachInternetGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachInternetGatewayRequestT = Model::DetachInternetGatewayRequest>
        void DetachInternetGatewayAsync(const DetachInternetGatewayRequestT& request, const DetachInternetGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachInternetGateway, request, handler, context);
        }

        /**
         * <p>Detaches a network interface from an instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachNetworkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachNetworkInterfaceOutcome DetachNetworkInterface(const Model::DetachNetworkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for DetachNetworkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachNetworkInterfaceRequestT = Model::DetachNetworkInterfaceRequest>
        Model::DetachNetworkInterfaceOutcomeCallable DetachNetworkInterfaceCallable(const DetachNetworkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachNetworkInterface, request);
        }

        /**
         * An Async wrapper for DetachNetworkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachNetworkInterfaceRequestT = Model::DetachNetworkInterfaceRequest>
        void DetachNetworkInterfaceAsync(const DetachNetworkInterfaceRequestT& request, const DetachNetworkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachNetworkInterface, request, handler, context);
        }

        /**
         * <p>Detaches the specified Amazon Web Services Verified Access trust provider
         * from the specified Amazon Web Services Verified Access instance.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachVerifiedAccessTrustProvider">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachVerifiedAccessTrustProviderOutcome DetachVerifiedAccessTrustProvider(const Model::DetachVerifiedAccessTrustProviderRequest& request) const;

        /**
         * A Callable wrapper for DetachVerifiedAccessTrustProvider that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachVerifiedAccessTrustProviderRequestT = Model::DetachVerifiedAccessTrustProviderRequest>
        Model::DetachVerifiedAccessTrustProviderOutcomeCallable DetachVerifiedAccessTrustProviderCallable(const DetachVerifiedAccessTrustProviderRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachVerifiedAccessTrustProvider, request);
        }

        /**
         * An Async wrapper for DetachVerifiedAccessTrustProvider that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachVerifiedAccessTrustProviderRequestT = Model::DetachVerifiedAccessTrustProviderRequest>
        void DetachVerifiedAccessTrustProviderAsync(const DetachVerifiedAccessTrustProviderRequestT& request, const DetachVerifiedAccessTrustProviderResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachVerifiedAccessTrustProvider, request, handler, context);
        }

        /**
         * <p>Detaches an EBS volume from an instance. Make sure to unmount any file
         * systems on the device within your operating system before detaching the volume.
         * Failure to do so can result in the volume becoming stuck in the
         * <code>busy</code> state while detaching. If this happens, detachment can be
         * delayed indefinitely until you unmount the volume, force detachment, reboot the
         * instance, or all three. If an EBS volume is the root device of an instance, it
         * can't be detached while the instance is running. To detach the root volume, stop
         * the instance first.</p> <p>When a volume with an Amazon Web Services Marketplace
         * product code is detached from an instance, the product code is no longer
         * associated with the instance.</p> <p>You can't detach or force detach volumes
         * that are attached to Amazon ECS or Fargate tasks. Attempting to do this results
         * in the <code>UnsupportedOperationException</code> exception with the
         * <code>Unable to detach volume attached to ECS tasks</code> error message.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-detaching-volume.html">Detach
         * an Amazon EBS volume</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachVolume">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachVolumeOutcome DetachVolume(const Model::DetachVolumeRequest& request) const;

        /**
         * A Callable wrapper for DetachVolume that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachVolumeRequestT = Model::DetachVolumeRequest>
        Model::DetachVolumeOutcomeCallable DetachVolumeCallable(const DetachVolumeRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachVolume, request);
        }

        /**
         * An Async wrapper for DetachVolume that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachVolumeRequestT = Model::DetachVolumeRequest>
        void DetachVolumeAsync(const DetachVolumeRequestT& request, const DetachVolumeResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachVolume, request, handler, context);
        }

        /**
         * <p>Detaches a virtual private gateway from a VPC. You do this if you're planning
         * to turn off the VPC and not use it anymore. You can confirm a virtual private
         * gateway has been completely detached from a VPC by describing the virtual
         * private gateway (any attachments to the virtual private gateway are also
         * described).</p> <p>You must wait for the attachment's state to switch to
         * <code>detached</code> before you can delete the VPC or attach a different VPC to
         * the virtual private gateway.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DetachVpnGateway">AWS
         * API Reference</a></p>
         */
        virtual Model::DetachVpnGatewayOutcome DetachVpnGateway(const Model::DetachVpnGatewayRequest& request) const;

        /**
         * A Callable wrapper for DetachVpnGateway that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DetachVpnGatewayRequestT = Model::DetachVpnGatewayRequest>
        Model::DetachVpnGatewayOutcomeCallable DetachVpnGatewayCallable(const DetachVpnGatewayRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DetachVpnGateway, request);
        }

        /**
         * An Async wrapper for DetachVpnGateway that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DetachVpnGatewayRequestT = Model::DetachVpnGatewayRequest>
        void DetachVpnGatewayAsync(const DetachVpnGatewayRequestT& request, const DetachVpnGatewayResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DetachVpnGateway, request, handler, context);
        }

        /**
         * <p>Disables Elastic IP address transfer. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro">Transfer
         * Elastic IP addresses</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableAddressTransfer">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableAddressTransferOutcome DisableAddressTransfer(const Model::DisableAddressTransferRequest& request) const;

        /**
         * A Callable wrapper for DisableAddressTransfer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableAddressTransferRequestT = Model::DisableAddressTransferRequest>
        Model::DisableAddressTransferOutcomeCallable DisableAddressTransferCallable(const DisableAddressTransferRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableAddressTransfer, request);
        }

        /**
         * An Async wrapper for DisableAddressTransfer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableAddressTransferRequestT = Model::DisableAddressTransferRequest>
        void DisableAddressTransferAsync(const DisableAddressTransferRequestT& request, const DisableAddressTransferResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableAddressTransfer, request, handler, context);
        }

        /**
         * <p>Disables Allowed AMIs for your account in the specified Amazon Web Services
         * Region. When set to <code>disabled</code>, the image criteria in your Allowed
         * AMIs settings do not apply, and no restrictions are placed on AMI
         * discoverability or usage. Users in your account can launch instances using any
         * public AMI or AMI shared with your account.</p>  <p>The Allowed AMIs
         * feature does not restrict the AMIs owned by your account. Regardless of the
         * criteria you set, the AMIs created by your account will always be discoverable
         * and usable by users in your account.</p>  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-allowed-amis.html">Control
         * the discovery and use of AMIs in Amazon EC2 with Allowed AMIs</a> in <i>Amazon
         * EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableAllowedImagesSettings">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableAllowedImagesSettingsOutcome DisableAllowedImagesSettings(const Model::DisableAllowedImagesSettingsRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableAllowedImagesSettings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableAllowedImagesSettingsRequestT = Model::DisableAllowedImagesSettingsRequest>
        Model::DisableAllowedImagesSettingsOutcomeCallable DisableAllowedImagesSettingsCallable(const DisableAllowedImagesSettingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableAllowedImagesSettings, request);
        }

        /**
         * An Async wrapper for DisableAllowedImagesSettings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableAllowedImagesSettingsRequestT = Model::DisableAllowedImagesSettingsRequest>
        void DisableAllowedImagesSettingsAsync(const DisableAllowedImagesSettingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableAllowedImagesSettingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableAllowedImagesSettings, request, handler, context);
        }

        /**
         * <p>Disables Infrastructure Performance metric subscriptions.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableAwsNetworkPerformanceMetricSubscription">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableAwsNetworkPerformanceMetricSubscriptionOutcome DisableAwsNetworkPerformanceMetricSubscription(const Model::DisableAwsNetworkPerformanceMetricSubscriptionRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableAwsNetworkPerformanceMetricSubscription that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableAwsNetworkPerformanceMetricSubscriptionRequestT = Model::DisableAwsNetworkPerformanceMetricSubscriptionRequest>
        Model::DisableAwsNetworkPerformanceMetricSubscriptionOutcomeCallable DisableAwsNetworkPerformanceMetricSubscriptionCallable(const DisableAwsNetworkPerformanceMetricSubscriptionRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableAwsNetworkPerformanceMetricSubscription, request);
        }

        /**
         * An Async wrapper for DisableAwsNetworkPerformanceMetricSubscription that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableAwsNetworkPerformanceMetricSubscriptionRequestT = Model::DisableAwsNetworkPerformanceMetricSubscriptionRequest>
        void DisableAwsNetworkPerformanceMetricSubscriptionAsync(const DisableAwsNetworkPerformanceMetricSubscriptionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableAwsNetworkPerformanceMetricSubscriptionRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableAwsNetworkPerformanceMetricSubscription, request, handler, context);
        }

        /**
         * <p>Disables EBS encryption by default for your account in the current
         * Region.</p> <p>After you disable encryption by default, you can still create
         * encrypted volumes by enabling encryption when you create each volume.</p>
         * <p>Disabling encryption by default does not change the encryption status of your
         * existing volumes.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableEbsEncryptionByDefault">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableEbsEncryptionByDefaultOutcome DisableEbsEncryptionByDefault(const Model::DisableEbsEncryptionByDefaultRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableEbsEncryptionByDefault that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableEbsEncryptionByDefaultRequestT = Model::DisableEbsEncryptionByDefaultRequest>
        Model::DisableEbsEncryptionByDefaultOutcomeCallable DisableEbsEncryptionByDefaultCallable(const DisableEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableEbsEncryptionByDefault, request);
        }

        /**
         * An Async wrapper for DisableEbsEncryptionByDefault that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableEbsEncryptionByDefaultRequestT = Model::DisableEbsEncryptionByDefaultRequest>
        void DisableEbsEncryptionByDefaultAsync(const DisableEbsEncryptionByDefaultResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableEbsEncryptionByDefault, request, handler, context);
        }

        /**
         * <p>Discontinue Windows fast launch for a Windows AMI, and clean up existing
         * pre-provisioned snapshots. After you disable Windows fast launch, the AMI uses
         * the standard launch process for each new instance. Amazon EC2 must remove all
         * pre-provisioned snapshots before you can enable Windows fast launch again.</p>
         *  <p>You can only change these settings for Windows AMIs that you own or
         * that have been shared with you.</p> <p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableFastLaunch">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableFastLaunchOutcome DisableFastLaunch(const Model::DisableFastLaunchRequest& request) const;

        /**
         * A Callable wrapper for DisableFastLaunch that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableFastLaunchRequestT = Model::DisableFastLaunchRequest>
        Model::DisableFastLaunchOutcomeCallable DisableFastLaunchCallable(const DisableFastLaunchRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableFastLaunch, request);
        }

        /**
         * An Async wrapper for DisableFastLaunch that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableFastLaunchRequestT = Model::DisableFastLaunchRequest>
        void DisableFastLaunchAsync(const DisableFastLaunchRequestT& request, const DisableFastLaunchResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableFastLaunch, request, handler, context);
        }

        /**
         * <p>Disables fast snapshot restores for the specified snapshots in the specified
         * Availability Zones.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableFastSnapshotRestores">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableFastSnapshotRestoresOutcome DisableFastSnapshotRestores(const Model::DisableFastSnapshotRestoresRequest& request) const;

        /**
         * A Callable wrapper for DisableFastSnapshotRestores that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableFastSnapshotRestoresRequestT = Model::DisableFastSnapshotRestoresRequest>
        Model::DisableFastSnapshotRestoresOutcomeCallable DisableFastSnapshotRestoresCallable(const DisableFastSnapshotRestoresRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableFastSnapshotRestores, request);
        }

        /**
         * An Async wrapper for DisableFastSnapshotRestores that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableFastSnapshotRestoresRequestT = Model::DisableFastSnapshotRestoresRequest>
        void DisableFastSnapshotRestoresAsync(const DisableFastSnapshotRestoresRequestT& request, const DisableFastSnapshotRestoresResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableFastSnapshotRestores, request, handler, context);
        }

        /**
         * <p>Sets the AMI state to <code>disabled</code> and removes all launch
         * permissions from the AMI. A disabled AMI can't be used for instance
         * launches.</p> <p>A disabled AMI can't be shared. If an AMI was public or
         * previously shared, it is made private. If an AMI was shared with an Amazon Web
         * Services account, organization, or Organizational Unit, they lose access to the
         * disabled AMI. </p> <p>A disabled AMI does not appear in <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html">DescribeImages</a>
         * API calls by default.</p> <p>Only the AMI owner can disable an AMI.</p> <p>You
         * can re-enable a disabled AMI using <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableImage.html">EnableImage</a>.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/disable-an-ami.html">Disable
         * an AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableImage">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableImageOutcome DisableImage(const Model::DisableImageRequest& request) const;

        /**
         * A Callable wrapper for DisableImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableImageRequestT = Model::DisableImageRequest>
        Model::DisableImageOutcomeCallable DisableImageCallable(const DisableImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableImage, request);
        }

        /**
         * An Async wrapper for DisableImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableImageRequestT = Model::DisableImageRequest>
        void DisableImageAsync(const DisableImageRequestT& request, const DisableImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableImage, request, handler, context);
        }

        /**
         * <p>Disables <i>block public access for AMIs</i> at the account level in the
         * specified Amazon Web Services Region. This removes the <i>block public
         * access</i> restriction from your account. With the restriction removed, you can
         * publicly share your AMIs in the specified Amazon Web Services Region.</p> <p>The
         * API can take up to 10 minutes to configure this setting. During this time, if
         * you run <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetImageBlockPublicAccessState.html">GetImageBlockPublicAccessState</a>,
         * the response will be <code>block-new-sharing</code>. When the API has completed
         * the configuration, the response will be <code>unblocked</code>.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-public-access-to-amis.html">Block
         * public access to your AMIs</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableImageBlockPublicAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableImageBlockPublicAccessOutcome DisableImageBlockPublicAccess(const Model::DisableImageBlockPublicAccessRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableImageBlockPublicAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableImageBlockPublicAccessRequestT = Model::DisableImageBlockPublicAccessRequest>
        Model::DisableImageBlockPublicAccessOutcomeCallable DisableImageBlockPublicAccessCallable(const DisableImageBlockPublicAccessRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableImageBlockPublicAccess, request);
        }

        /**
         * An Async wrapper for DisableImageBlockPublicAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableImageBlockPublicAccessRequestT = Model::DisableImageBlockPublicAccessRequest>
        void DisableImageBlockPublicAccessAsync(const DisableImageBlockPublicAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableImageBlockPublicAccessRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableImageBlockPublicAccess, request, handler, context);
        }

        /**
         * <p>Cancels the deprecation of the specified AMI.</p> <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-deprecate.html">Deprecate
         * an Amazon EC2 AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableImageDeprecation">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableImageDeprecationOutcome DisableImageDeprecation(const Model::DisableImageDeprecationRequest& request) const;

        /**
         * A Callable wrapper for DisableImageDeprecation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableImageDeprecationRequestT = Model::DisableImageDeprecationRequest>
        Model::DisableImageDeprecationOutcomeCallable DisableImageDeprecationCallable(const DisableImageDeprecationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableImageDeprecation, request);
        }

        /**
         * An Async wrapper for DisableImageDeprecation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableImageDeprecationRequestT = Model::DisableImageDeprecationRequest>
        void DisableImageDeprecationAsync(const DisableImageDeprecationRequestT& request, const DisableImageDeprecationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableImageDeprecation, request, handler, context);
        }

        /**
         * <p>Disables deregistration protection for an AMI. When deregistration protection
         * is disabled, the AMI can be deregistered.</p> <p>If you chose to include a
         * 24-hour cooldown period when you enabled deregistration protection for the AMI,
         * then, when you disable deregistration protection, you won’t immediately be able
         * to deregister the AMI.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-deregistration-protection.html">Protect
         * an Amazon EC2 AMI from deregistration</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableImageDeregistrationProtection">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableImageDeregistrationProtectionOutcome DisableImageDeregistrationProtection(const Model::DisableImageDeregistrationProtectionRequest& request) const;

        /**
         * A Callable wrapper for DisableImageDeregistrationProtection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableImageDeregistrationProtectionRequestT = Model::DisableImageDeregistrationProtectionRequest>
        Model::DisableImageDeregistrationProtectionOutcomeCallable DisableImageDeregistrationProtectionCallable(const DisableImageDeregistrationProtectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableImageDeregistrationProtection, request);
        }

        /**
         * An Async wrapper for DisableImageDeregistrationProtection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableImageDeregistrationProtectionRequestT = Model::DisableImageDeregistrationProtectionRequest>
        void DisableImageDeregistrationProtectionAsync(const DisableImageDeregistrationProtectionRequestT& request, const DisableImageDeregistrationProtectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableImageDeregistrationProtection, request, handler, context);
        }

        /**
         * <p>Disable the IPAM account. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/enable-integ-ipam.html">Enable
         * integration with Organizations</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableIpamOrganizationAdminAccount">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableIpamOrganizationAdminAccountOutcome DisableIpamOrganizationAdminAccount(const Model::DisableIpamOrganizationAdminAccountRequest& request) const;

        /**
         * A Callable wrapper for DisableIpamOrganizationAdminAccount that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableIpamOrganizationAdminAccountRequestT = Model::DisableIpamOrganizationAdminAccountRequest>
        Model::DisableIpamOrganizationAdminAccountOutcomeCallable DisableIpamOrganizationAdminAccountCallable(const DisableIpamOrganizationAdminAccountRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableIpamOrganizationAdminAccount, request);
        }

        /**
         * An Async wrapper for DisableIpamOrganizationAdminAccount that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableIpamOrganizationAdminAccountRequestT = Model::DisableIpamOrganizationAdminAccountRequest>
        void DisableIpamOrganizationAdminAccountAsync(const DisableIpamOrganizationAdminAccountRequestT& request, const DisableIpamOrganizationAdminAccountResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableIpamOrganizationAdminAccount, request, handler, context);
        }

        /**
         * <p>Disables route propagation from a route server to a specified route
         * table.</p> <p>When enabled, route server propagation installs the routes in the
         * FIB on the route table you've specified. Route server supports IPv4 and IPv6
         * route propagation.</p> <p>Amazon VPC Route Server simplifies routing for traffic
         * between workloads that are deployed within a VPC and its internet gateways. With
         * this feature, VPC Route Server dynamically updates VPC and internet gateway
         * route tables with your preferred IPv4 or IPv6 routes to achieve routing fault
         * tolerance for those workloads. This enables you to automatically reroute traffic
         * within a VPC, which increases the manageability of VPC routing and
         * interoperability with third-party workloads.</p> <p>Route server supports the
         * follow route table types:</p> <ul> <li> <p>VPC route tables not associated with
         * subnets</p> </li> <li> <p>Subnet route tables</p> </li> <li> <p>Internet gateway
         * route tables</p> </li> </ul> <p>Route server does not support route tables
         * associated with virtual private gateways. To propagate routes into a transit
         * gateway route table, use <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html">Transit
         * Gateway Connect</a>.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableRouteServerPropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableRouteServerPropagationOutcome DisableRouteServerPropagation(const Model::DisableRouteServerPropagationRequest& request) const;

        /**
         * A Callable wrapper for DisableRouteServerPropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableRouteServerPropagationRequestT = Model::DisableRouteServerPropagationRequest>
        Model::DisableRouteServerPropagationOutcomeCallable DisableRouteServerPropagationCallable(const DisableRouteServerPropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableRouteServerPropagation, request);
        }

        /**
         * An Async wrapper for DisableRouteServerPropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableRouteServerPropagationRequestT = Model::DisableRouteServerPropagationRequest>
        void DisableRouteServerPropagationAsync(const DisableRouteServerPropagationRequestT& request, const DisableRouteServerPropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableRouteServerPropagation, request, handler, context);
        }

        /**
         * <p>Disables access to the EC2 serial console of all instances for your account.
         * By default, access to the EC2 serial console is disabled for your account. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/configure-access-to-serial-console.html#serial-console-account-access">Manage
         * account access to the EC2 serial console</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableSerialConsoleAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableSerialConsoleAccessOutcome DisableSerialConsoleAccess(const Model::DisableSerialConsoleAccessRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableSerialConsoleAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableSerialConsoleAccessRequestT = Model::DisableSerialConsoleAccessRequest>
        Model::DisableSerialConsoleAccessOutcomeCallable DisableSerialConsoleAccessCallable(const DisableSerialConsoleAccessRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableSerialConsoleAccess, request);
        }

        /**
         * An Async wrapper for DisableSerialConsoleAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableSerialConsoleAccessRequestT = Model::DisableSerialConsoleAccessRequest>
        void DisableSerialConsoleAccessAsync(const DisableSerialConsoleAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableSerialConsoleAccessRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableSerialConsoleAccess, request, handler, context);
        }

        /**
         * <p>Disables the <i>block public access for snapshots</i> setting at the account
         * level for the specified Amazon Web Services Region. After you disable block
         * public access for snapshots in a Region, users can publicly share snapshots in
         * that Region.</p>  <p>Enabling block public access for snapshots in
         * <i>block-all-sharing</i> mode does not change the permissions for snapshots that
         * are already publicly shared. Instead, it prevents these snapshots from be
         * publicly visible and publicly accessible. Therefore, the attributes for these
         * snapshots still indicate that they are publicly shared, even though they are not
         * publicly available.</p> <p>If you disable block public access , these snapshots
         * will become publicly available again.</p>  <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/block-public-access-snapshots.html">
         * Block public access for snapshots</a> in the <i>Amazon EBS User Guide</i> .</p>
         * <p/><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableSnapshotBlockPublicAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableSnapshotBlockPublicAccessOutcome DisableSnapshotBlockPublicAccess(const Model::DisableSnapshotBlockPublicAccessRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableSnapshotBlockPublicAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableSnapshotBlockPublicAccessRequestT = Model::DisableSnapshotBlockPublicAccessRequest>
        Model::DisableSnapshotBlockPublicAccessOutcomeCallable DisableSnapshotBlockPublicAccessCallable(const DisableSnapshotBlockPublicAccessRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableSnapshotBlockPublicAccess, request);
        }

        /**
         * An Async wrapper for DisableSnapshotBlockPublicAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableSnapshotBlockPublicAccessRequestT = Model::DisableSnapshotBlockPublicAccessRequest>
        void DisableSnapshotBlockPublicAccessAsync(const DisableSnapshotBlockPublicAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableSnapshotBlockPublicAccessRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableSnapshotBlockPublicAccess, request, handler, context);
        }

        /**
         * <p>Disables the specified resource attachment from propagating routes to the
         * specified propagation route table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableTransitGatewayRouteTablePropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableTransitGatewayRouteTablePropagationOutcome DisableTransitGatewayRouteTablePropagation(const Model::DisableTransitGatewayRouteTablePropagationRequest& request) const;

        /**
         * A Callable wrapper for DisableTransitGatewayRouteTablePropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableTransitGatewayRouteTablePropagationRequestT = Model::DisableTransitGatewayRouteTablePropagationRequest>
        Model::DisableTransitGatewayRouteTablePropagationOutcomeCallable DisableTransitGatewayRouteTablePropagationCallable(const DisableTransitGatewayRouteTablePropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableTransitGatewayRouteTablePropagation, request);
        }

        /**
         * An Async wrapper for DisableTransitGatewayRouteTablePropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableTransitGatewayRouteTablePropagationRequestT = Model::DisableTransitGatewayRouteTablePropagationRequest>
        void DisableTransitGatewayRouteTablePropagationAsync(const DisableTransitGatewayRouteTablePropagationRequestT& request, const DisableTransitGatewayRouteTablePropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableTransitGatewayRouteTablePropagation, request, handler, context);
        }

        /**
         * <p>Disables a virtual private gateway (VGW) from propagating routes to a
         * specified route table of a VPC.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableVgwRoutePropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableVgwRoutePropagationOutcome DisableVgwRoutePropagation(const Model::DisableVgwRoutePropagationRequest& request) const;

        /**
         * A Callable wrapper for DisableVgwRoutePropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableVgwRoutePropagationRequestT = Model::DisableVgwRoutePropagationRequest>
        Model::DisableVgwRoutePropagationOutcomeCallable DisableVgwRoutePropagationCallable(const DisableVgwRoutePropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableVgwRoutePropagation, request);
        }

        /**
         * An Async wrapper for DisableVgwRoutePropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableVgwRoutePropagationRequestT = Model::DisableVgwRoutePropagationRequest>
        void DisableVgwRoutePropagationAsync(const DisableVgwRoutePropagationRequestT& request, const DisableVgwRoutePropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableVgwRoutePropagation, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Disables ClassicLink for a
         * VPC. You cannot disable ClassicLink for a VPC that has EC2-Classic instances
         * linked to it.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableVpcClassicLink">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableVpcClassicLinkOutcome DisableVpcClassicLink(const Model::DisableVpcClassicLinkRequest& request) const;

        /**
         * A Callable wrapper for DisableVpcClassicLink that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableVpcClassicLinkRequestT = Model::DisableVpcClassicLinkRequest>
        Model::DisableVpcClassicLinkOutcomeCallable DisableVpcClassicLinkCallable(const DisableVpcClassicLinkRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisableVpcClassicLink, request);
        }

        /**
         * An Async wrapper for DisableVpcClassicLink that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableVpcClassicLinkRequestT = Model::DisableVpcClassicLinkRequest>
        void DisableVpcClassicLinkAsync(const DisableVpcClassicLinkRequestT& request, const DisableVpcClassicLinkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisableVpcClassicLink, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Disables ClassicLink DNS
         * support for a VPC. If disabled, DNS hostnames resolve to public IP addresses
         * when addressed between a linked EC2-Classic instance and instances in the VPC to
         * which it's linked.</p> <p>You must specify a VPC ID in the
         * request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisableVpcClassicLinkDnsSupport">AWS
         * API Reference</a></p>
         */
        virtual Model::DisableVpcClassicLinkDnsSupportOutcome DisableVpcClassicLinkDnsSupport(const Model::DisableVpcClassicLinkDnsSupportRequest& request = {}) const;

        /**
         * A Callable wrapper for DisableVpcClassicLinkDnsSupport that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisableVpcClassicLinkDnsSupportRequestT = Model::DisableVpcClassicLinkDnsSupportRequest>
        Model::DisableVpcClassicLinkDnsSupportOutcomeCallable DisableVpcClassicLinkDnsSupportCallable(const DisableVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisableVpcClassicLinkDnsSupport, request);
        }

        /**
         * An Async wrapper for DisableVpcClassicLinkDnsSupport that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisableVpcClassicLinkDnsSupportRequestT = Model::DisableVpcClassicLinkDnsSupportRequest>
        void DisableVpcClassicLinkDnsSupportAsync(const DisableVpcClassicLinkDnsSupportResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisableVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisableVpcClassicLinkDnsSupport, request, handler, context);
        }

        /**
         * <p>Disassociates an Elastic IP address from the instance or network interface
         * it's associated with.</p> <p>This is an idempotent operation. If you perform the
         * operation more than once, Amazon EC2 doesn't return an error.</p> <p>An address
         * cannot be disassociated if the all of the following conditions are met:</p> <ul>
         * <li> <p>Network interface has a <code>publicDualStackDnsName</code>
         * publicDnsName</p> </li> <li> <p>Public IPv4 address is the primary public IPv4
         * address</p> </li> <li> <p>Network interface only has one remaining public IPv4
         * address</p> </li> </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateAddressOutcome DisassociateAddress(const Model::DisassociateAddressRequest& request = {}) const;

        /**
         * A Callable wrapper for DisassociateAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateAddressRequestT = Model::DisassociateAddressRequest>
        Model::DisassociateAddressOutcomeCallable DisassociateAddressCallable(const DisassociateAddressRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::DisassociateAddress, request);
        }

        /**
         * An Async wrapper for DisassociateAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateAddressRequestT = Model::DisassociateAddressRequest>
        void DisassociateAddressAsync(const DisassociateAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const DisassociateAddressRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::DisassociateAddress, request, handler, context);
        }

        /**
         * <p>Cancels a pending request to assign billing of the unused capacity of a
         * Capacity Reservation to a consumer account, or revokes a request that has
         * already been accepted. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html">Billing
         * assignment for shared Amazon EC2 Capacity Reservations</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateCapacityReservationBillingOwner">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateCapacityReservationBillingOwnerOutcome DisassociateCapacityReservationBillingOwner(const Model::DisassociateCapacityReservationBillingOwnerRequest& request) const;

        /**
         * A Callable wrapper for DisassociateCapacityReservationBillingOwner that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateCapacityReservationBillingOwnerRequestT = Model::DisassociateCapacityReservationBillingOwnerRequest>
        Model::DisassociateCapacityReservationBillingOwnerOutcomeCallable DisassociateCapacityReservationBillingOwnerCallable(const DisassociateCapacityReservationBillingOwnerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateCapacityReservationBillingOwner, request);
        }

        /**
         * An Async wrapper for DisassociateCapacityReservationBillingOwner that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateCapacityReservationBillingOwnerRequestT = Model::DisassociateCapacityReservationBillingOwnerRequest>
        void DisassociateCapacityReservationBillingOwnerAsync(const DisassociateCapacityReservationBillingOwnerRequestT& request, const DisassociateCapacityReservationBillingOwnerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateCapacityReservationBillingOwner, request, handler, context);
        }

        /**
         * <p>Disassociates a target network from the specified Client VPN endpoint. When
         * you disassociate the last target network from a Client VPN, the following
         * happens:</p> <ul> <li> <p>The route that was automatically added for the VPC is
         * deleted</p> </li> <li> <p>All active client connections are terminated</p> </li>
         * <li> <p>New client connections are disallowed</p> </li> <li> <p>The Client VPN
         * endpoint's status changes to <code>pending-associate</code> </p> </li>
         * </ul><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateClientVpnTargetNetwork">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateClientVpnTargetNetworkOutcome DisassociateClientVpnTargetNetwork(const Model::DisassociateClientVpnTargetNetworkRequest& request) const;

        /**
         * A Callable wrapper for DisassociateClientVpnTargetNetwork that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateClientVpnTargetNetworkRequestT = Model::DisassociateClientVpnTargetNetworkRequest>
        Model::DisassociateClientVpnTargetNetworkOutcomeCallable DisassociateClientVpnTargetNetworkCallable(const DisassociateClientVpnTargetNetworkRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateClientVpnTargetNetwork, request);
        }

        /**
         * An Async wrapper for DisassociateClientVpnTargetNetwork that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateClientVpnTargetNetworkRequestT = Model::DisassociateClientVpnTargetNetworkRequest>
        void DisassociateClientVpnTargetNetworkAsync(const DisassociateClientVpnTargetNetworkRequestT& request, const DisassociateClientVpnTargetNetworkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateClientVpnTargetNetwork, request, handler, context);
        }

        /**
         * <p>Disassociates an IAM role from an Certificate Manager (ACM) certificate.
         * Disassociating an IAM role from an ACM certificate removes the Amazon S3 object
         * that contains the certificate, certificate chain, and encrypted private key from
         * the Amazon S3 bucket. It also revokes the IAM role's permission to use the KMS
         * key used to encrypt the private key. This effectively revokes the role's
         * permission to use the certificate.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateEnclaveCertificateIamRole">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateEnclaveCertificateIamRoleOutcome DisassociateEnclaveCertificateIamRole(const Model::DisassociateEnclaveCertificateIamRoleRequest& request) const;

        /**
         * A Callable wrapper for DisassociateEnclaveCertificateIamRole that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateEnclaveCertificateIamRoleRequestT = Model::DisassociateEnclaveCertificateIamRoleRequest>
        Model::DisassociateEnclaveCertificateIamRoleOutcomeCallable DisassociateEnclaveCertificateIamRoleCallable(const DisassociateEnclaveCertificateIamRoleRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateEnclaveCertificateIamRole, request);
        }

        /**
         * An Async wrapper for DisassociateEnclaveCertificateIamRole that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateEnclaveCertificateIamRoleRequestT = Model::DisassociateEnclaveCertificateIamRoleRequest>
        void DisassociateEnclaveCertificateIamRoleAsync(const DisassociateEnclaveCertificateIamRoleRequestT& request, const DisassociateEnclaveCertificateIamRoleResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateEnclaveCertificateIamRole, request, handler, context);
        }

        /**
         * <p>Disassociates an IAM instance profile from a running or stopped instance.</p>
         * <p>Use <a>DescribeIamInstanceProfileAssociations</a> to get the association
         * ID.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateIamInstanceProfile">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateIamInstanceProfileOutcome DisassociateIamInstanceProfile(const Model::DisassociateIamInstanceProfileRequest& request) const;

        /**
         * A Callable wrapper for DisassociateIamInstanceProfile that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateIamInstanceProfileRequestT = Model::DisassociateIamInstanceProfileRequest>
        Model::DisassociateIamInstanceProfileOutcomeCallable DisassociateIamInstanceProfileCallable(const DisassociateIamInstanceProfileRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateIamInstanceProfile, request);
        }

        /**
         * An Async wrapper for DisassociateIamInstanceProfile that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateIamInstanceProfileRequestT = Model::DisassociateIamInstanceProfileRequest>
        void DisassociateIamInstanceProfileAsync(const DisassociateIamInstanceProfileRequestT& request, const DisassociateIamInstanceProfileResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateIamInstanceProfile, request, handler, context);
        }

        /**
         * <p>Disassociates one or more targets from an event window.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/event-windows.html">Define
         * event windows for scheduled events</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateInstanceEventWindow">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateInstanceEventWindowOutcome DisassociateInstanceEventWindow(const Model::DisassociateInstanceEventWindowRequest& request) const;

        /**
         * A Callable wrapper for DisassociateInstanceEventWindow that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateInstanceEventWindowRequestT = Model::DisassociateInstanceEventWindowRequest>
        Model::DisassociateInstanceEventWindowOutcomeCallable DisassociateInstanceEventWindowCallable(const DisassociateInstanceEventWindowRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateInstanceEventWindow, request);
        }

        /**
         * An Async wrapper for DisassociateInstanceEventWindow that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateInstanceEventWindowRequestT = Model::DisassociateInstanceEventWindowRequest>
        void DisassociateInstanceEventWindowAsync(const DisassociateInstanceEventWindowRequestT& request, const DisassociateInstanceEventWindowResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateInstanceEventWindow, request, handler, context);
        }

        /**
         * <p>Remove the association between your Autonomous System Number (ASN) and your
         * BYOIP CIDR. You may want to use this action to disassociate an ASN from a CIDR
         * or if you want to swap ASNs. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-byoasn.html">Tutorial:
         * Bring your ASN to IPAM</a> in the <i>Amazon VPC IPAM guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateIpamByoasn">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateIpamByoasnOutcome DisassociateIpamByoasn(const Model::DisassociateIpamByoasnRequest& request) const;

        /**
         * A Callable wrapper for DisassociateIpamByoasn that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateIpamByoasnRequestT = Model::DisassociateIpamByoasnRequest>
        Model::DisassociateIpamByoasnOutcomeCallable DisassociateIpamByoasnCallable(const DisassociateIpamByoasnRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateIpamByoasn, request);
        }

        /**
         * An Async wrapper for DisassociateIpamByoasn that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateIpamByoasnRequestT = Model::DisassociateIpamByoasnRequest>
        void DisassociateIpamByoasnAsync(const DisassociateIpamByoasnRequestT& request, const DisassociateIpamByoasnResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateIpamByoasn, request, handler, context);
        }

        /**
         * <p>Disassociates a resource discovery from an Amazon VPC IPAM. A resource
         * discovery is an IPAM component that enables IPAM to manage and monitor resources
         * that belong to the owning account.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateIpamResourceDiscovery">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateIpamResourceDiscoveryOutcome DisassociateIpamResourceDiscovery(const Model::DisassociateIpamResourceDiscoveryRequest& request) const;

        /**
         * A Callable wrapper for DisassociateIpamResourceDiscovery that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateIpamResourceDiscoveryRequestT = Model::DisassociateIpamResourceDiscoveryRequest>
        Model::DisassociateIpamResourceDiscoveryOutcomeCallable DisassociateIpamResourceDiscoveryCallable(const DisassociateIpamResourceDiscoveryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateIpamResourceDiscovery, request);
        }

        /**
         * An Async wrapper for DisassociateIpamResourceDiscovery that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateIpamResourceDiscoveryRequestT = Model::DisassociateIpamResourceDiscoveryRequest>
        void DisassociateIpamResourceDiscoveryAsync(const DisassociateIpamResourceDiscoveryRequestT& request, const DisassociateIpamResourceDiscoveryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateIpamResourceDiscovery, request, handler, context);
        }

        /**
         * <p>Disassociates secondary Elastic IP addresses (EIPs) from a public NAT
         * gateway. You cannot disassociate your primary EIP. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/nat-gateway-working-with.html#nat-gateway-edit-secondary">Edit
         * secondary IP address associations</a> in the <i>Amazon VPC User Guide</i>.</p>
         * <p>While disassociating is in progress, you cannot associate/disassociate
         * additional EIPs while the connections are being drained. You are, however,
         * allowed to delete the NAT gateway.</p> <p>An EIP is released only at the end of
         * MaxDrainDurationSeconds. It stays associated and supports the existing
         * connections but does not support any new connections (new connections are
         * distributed across the remaining associated EIPs). As the existing connections
         * drain out, the EIPs (and the corresponding private IP addresses mapped to them)
         * are released.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateNatGatewayAddress">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateNatGatewayAddressOutcome DisassociateNatGatewayAddress(const Model::DisassociateNatGatewayAddressRequest& request) const;

        /**
         * A Callable wrapper for DisassociateNatGatewayAddress that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateNatGatewayAddressRequestT = Model::DisassociateNatGatewayAddressRequest>
        Model::DisassociateNatGatewayAddressOutcomeCallable DisassociateNatGatewayAddressCallable(const DisassociateNatGatewayAddressRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateNatGatewayAddress, request);
        }

        /**
         * An Async wrapper for DisassociateNatGatewayAddress that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateNatGatewayAddressRequestT = Model::DisassociateNatGatewayAddressRequest>
        void DisassociateNatGatewayAddressAsync(const DisassociateNatGatewayAddressRequestT& request, const DisassociateNatGatewayAddressResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateNatGatewayAddress, request, handler, context);
        }

        /**
         * <p>Disassociates a route server from a VPC.</p> <p>A route server association is
         * the connection established between a route server and a VPC.</p> <p>For more
         * information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateRouteServer">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateRouteServerOutcome DisassociateRouteServer(const Model::DisassociateRouteServerRequest& request) const;

        /**
         * A Callable wrapper for DisassociateRouteServer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateRouteServerRequestT = Model::DisassociateRouteServerRequest>
        Model::DisassociateRouteServerOutcomeCallable DisassociateRouteServerCallable(const DisassociateRouteServerRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateRouteServer, request);
        }

        /**
         * An Async wrapper for DisassociateRouteServer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateRouteServerRequestT = Model::DisassociateRouteServerRequest>
        void DisassociateRouteServerAsync(const DisassociateRouteServerRequestT& request, const DisassociateRouteServerResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateRouteServer, request, handler, context);
        }

        /**
         * <p>Disassociates a subnet or gateway from a route table.</p> <p>After you
         * perform this action, the subnet no longer uses the routes in the route table.
         * Instead, it uses the routes in the VPC's main route table. For more information
         * about route tables, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html">Route
         * tables</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateRouteTableOutcome DisassociateRouteTable(const Model::DisassociateRouteTableRequest& request) const;

        /**
         * A Callable wrapper for DisassociateRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateRouteTableRequestT = Model::DisassociateRouteTableRequest>
        Model::DisassociateRouteTableOutcomeCallable DisassociateRouteTableCallable(const DisassociateRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateRouteTable, request);
        }

        /**
         * An Async wrapper for DisassociateRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateRouteTableRequestT = Model::DisassociateRouteTableRequest>
        void DisassociateRouteTableAsync(const DisassociateRouteTableRequestT& request, const DisassociateRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateRouteTable, request, handler, context);
        }

        /**
         * <p>Disassociates a security group from a VPC. You cannot disassociate the
         * security group if any Elastic network interfaces in the associated VPC are still
         * associated with the security group. Note that the disassociation is asynchronous
         * and you can check the status of the request with <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupVpcAssociations.html">DescribeSecurityGroupVpcAssociations</a>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateSecurityGroupVpc">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateSecurityGroupVpcOutcome DisassociateSecurityGroupVpc(const Model::DisassociateSecurityGroupVpcRequest& request) const;

        /**
         * A Callable wrapper for DisassociateSecurityGroupVpc that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateSecurityGroupVpcRequestT = Model::DisassociateSecurityGroupVpcRequest>
        Model::DisassociateSecurityGroupVpcOutcomeCallable DisassociateSecurityGroupVpcCallable(const DisassociateSecurityGroupVpcRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateSecurityGroupVpc, request);
        }

        /**
         * An Async wrapper for DisassociateSecurityGroupVpc that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateSecurityGroupVpcRequestT = Model::DisassociateSecurityGroupVpcRequest>
        void DisassociateSecurityGroupVpcAsync(const DisassociateSecurityGroupVpcRequestT& request, const DisassociateSecurityGroupVpcResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateSecurityGroupVpc, request, handler, context);
        }

        /**
         * <p>Disassociates a CIDR block from a subnet. Currently, you can disassociate an
         * IPv6 CIDR block only. You must detach or delete all gateways and resources that
         * are associated with the CIDR block before you can disassociate it.
         * </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateSubnetCidrBlock">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateSubnetCidrBlockOutcome DisassociateSubnetCidrBlock(const Model::DisassociateSubnetCidrBlockRequest& request) const;

        /**
         * A Callable wrapper for DisassociateSubnetCidrBlock that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateSubnetCidrBlockRequestT = Model::DisassociateSubnetCidrBlockRequest>
        Model::DisassociateSubnetCidrBlockOutcomeCallable DisassociateSubnetCidrBlockCallable(const DisassociateSubnetCidrBlockRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateSubnetCidrBlock, request);
        }

        /**
         * An Async wrapper for DisassociateSubnetCidrBlock that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateSubnetCidrBlockRequestT = Model::DisassociateSubnetCidrBlockRequest>
        void DisassociateSubnetCidrBlockAsync(const DisassociateSubnetCidrBlockRequestT& request, const DisassociateSubnetCidrBlockResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateSubnetCidrBlock, request, handler, context);
        }

        /**
         * <p>Disassociates the specified subnets from the transit gateway multicast
         * domain. </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateTransitGatewayMulticastDomain">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateTransitGatewayMulticastDomainOutcome DisassociateTransitGatewayMulticastDomain(const Model::DisassociateTransitGatewayMulticastDomainRequest& request) const;

        /**
         * A Callable wrapper for DisassociateTransitGatewayMulticastDomain that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateTransitGatewayMulticastDomainRequestT = Model::DisassociateTransitGatewayMulticastDomainRequest>
        Model::DisassociateTransitGatewayMulticastDomainOutcomeCallable DisassociateTransitGatewayMulticastDomainCallable(const DisassociateTransitGatewayMulticastDomainRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateTransitGatewayMulticastDomain, request);
        }

        /**
         * An Async wrapper for DisassociateTransitGatewayMulticastDomain that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateTransitGatewayMulticastDomainRequestT = Model::DisassociateTransitGatewayMulticastDomainRequest>
        void DisassociateTransitGatewayMulticastDomainAsync(const DisassociateTransitGatewayMulticastDomainRequestT& request, const DisassociateTransitGatewayMulticastDomainResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateTransitGatewayMulticastDomain, request, handler, context);
        }

        /**
         * <p>Removes the association between an an attachment and a policy
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateTransitGatewayPolicyTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateTransitGatewayPolicyTableOutcome DisassociateTransitGatewayPolicyTable(const Model::DisassociateTransitGatewayPolicyTableRequest& request) const;

        /**
         * A Callable wrapper for DisassociateTransitGatewayPolicyTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateTransitGatewayPolicyTableRequestT = Model::DisassociateTransitGatewayPolicyTableRequest>
        Model::DisassociateTransitGatewayPolicyTableOutcomeCallable DisassociateTransitGatewayPolicyTableCallable(const DisassociateTransitGatewayPolicyTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateTransitGatewayPolicyTable, request);
        }

        /**
         * An Async wrapper for DisassociateTransitGatewayPolicyTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateTransitGatewayPolicyTableRequestT = Model::DisassociateTransitGatewayPolicyTableRequest>
        void DisassociateTransitGatewayPolicyTableAsync(const DisassociateTransitGatewayPolicyTableRequestT& request, const DisassociateTransitGatewayPolicyTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateTransitGatewayPolicyTable, request, handler, context);
        }

        /**
         * <p>Disassociates a resource attachment from a transit gateway route
         * table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateTransitGatewayRouteTable">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateTransitGatewayRouteTableOutcome DisassociateTransitGatewayRouteTable(const Model::DisassociateTransitGatewayRouteTableRequest& request) const;

        /**
         * A Callable wrapper for DisassociateTransitGatewayRouteTable that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateTransitGatewayRouteTableRequestT = Model::DisassociateTransitGatewayRouteTableRequest>
        Model::DisassociateTransitGatewayRouteTableOutcomeCallable DisassociateTransitGatewayRouteTableCallable(const DisassociateTransitGatewayRouteTableRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateTransitGatewayRouteTable, request);
        }

        /**
         * An Async wrapper for DisassociateTransitGatewayRouteTable that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateTransitGatewayRouteTableRequestT = Model::DisassociateTransitGatewayRouteTableRequest>
        void DisassociateTransitGatewayRouteTableAsync(const DisassociateTransitGatewayRouteTableRequestT& request, const DisassociateTransitGatewayRouteTableResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateTransitGatewayRouteTable, request, handler, context);
        }

        /**
         * <p>Removes an association between a branch network interface with a trunk
         * network interface.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateTrunkInterface">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateTrunkInterfaceOutcome DisassociateTrunkInterface(const Model::DisassociateTrunkInterfaceRequest& request) const;

        /**
         * A Callable wrapper for DisassociateTrunkInterface that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateTrunkInterfaceRequestT = Model::DisassociateTrunkInterfaceRequest>
        Model::DisassociateTrunkInterfaceOutcomeCallable DisassociateTrunkInterfaceCallable(const DisassociateTrunkInterfaceRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateTrunkInterface, request);
        }

        /**
         * An Async wrapper for DisassociateTrunkInterface that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateTrunkInterfaceRequestT = Model::DisassociateTrunkInterfaceRequest>
        void DisassociateTrunkInterfaceAsync(const DisassociateTrunkInterfaceRequestT& request, const DisassociateTrunkInterfaceResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateTrunkInterface, request, handler, context);
        }

        /**
         * <p>Disassociates a CIDR block from a VPC. To disassociate the CIDR block, you
         * must specify its association ID. You can get the association ID by using
         * <a>DescribeVpcs</a>. You must detach or delete all gateways and resources that
         * are associated with the CIDR block before you can disassociate it. </p> <p>You
         * cannot disassociate the CIDR block with which you originally created the VPC
         * (the primary CIDR block).</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/DisassociateVpcCidrBlock">AWS
         * API Reference</a></p>
         */
        virtual Model::DisassociateVpcCidrBlockOutcome DisassociateVpcCidrBlock(const Model::DisassociateVpcCidrBlockRequest& request) const;

        /**
         * A Callable wrapper for DisassociateVpcCidrBlock that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename DisassociateVpcCidrBlockRequestT = Model::DisassociateVpcCidrBlockRequest>
        Model::DisassociateVpcCidrBlockOutcomeCallable DisassociateVpcCidrBlockCallable(const DisassociateVpcCidrBlockRequestT& request) const
        {
            return SubmitCallable(&EC2Client::DisassociateVpcCidrBlock, request);
        }

        /**
         * An Async wrapper for DisassociateVpcCidrBlock that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename DisassociateVpcCidrBlockRequestT = Model::DisassociateVpcCidrBlockRequest>
        void DisassociateVpcCidrBlockAsync(const DisassociateVpcCidrBlockRequestT& request, const DisassociateVpcCidrBlockResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::DisassociateVpcCidrBlock, request, handler, context);
        }

        /**
         * <p>Enables Elastic IP address transfer. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro">Transfer
         * Elastic IP addresses</a> in the <i>Amazon VPC User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableAddressTransfer">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableAddressTransferOutcome EnableAddressTransfer(const Model::EnableAddressTransferRequest& request) const;

        /**
         * A Callable wrapper for EnableAddressTransfer that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableAddressTransferRequestT = Model::EnableAddressTransferRequest>
        Model::EnableAddressTransferOutcomeCallable EnableAddressTransferCallable(const EnableAddressTransferRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableAddressTransfer, request);
        }

        /**
         * An Async wrapper for EnableAddressTransfer that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableAddressTransferRequestT = Model::EnableAddressTransferRequest>
        void EnableAddressTransferAsync(const EnableAddressTransferRequestT& request, const EnableAddressTransferResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableAddressTransfer, request, handler, context);
        }

        /**
         * <p>Enables Allowed AMIs for your account in the specified Amazon Web Services
         * Region. Two values are accepted:</p> <ul> <li> <p> <code>enabled</code>: The
         * image criteria in your Allowed AMIs settings are applied. As a result, only AMIs
         * matching these criteria are discoverable and can be used by your account to
         * launch instances.</p> </li> <li> <p> <code>audit-mode</code>: The image criteria
         * in your Allowed AMIs settings are not applied. No restrictions are placed on AMI
         * discoverability or usage. Users in your account can launch instances using any
         * public AMI or AMI shared with your account.</p> <p>The purpose of
         * <code>audit-mode</code> is to indicate which AMIs will be affected when Allowed
         * AMIs is <code>enabled</code>. In <code>audit-mode</code>, each AMI displays
         * either <code>"ImageAllowed": true</code> or <code>"ImageAllowed": false</code>
         * to indicate whether the AMI will be discoverable and available to users in the
         * account when Allowed AMIs is enabled.</p> </li> </ul>  <p>The Allowed AMIs
         * feature does not restrict the AMIs owned by your account. Regardless of the
         * criteria you set, the AMIs created by your account will always be discoverable
         * and usable by users in your account.</p>  <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-allowed-amis.html">Control
         * the discovery and use of AMIs in Amazon EC2 with Allowed AMIs</a> in <i>Amazon
         * EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableAllowedImagesSettings">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableAllowedImagesSettingsOutcome EnableAllowedImagesSettings(const Model::EnableAllowedImagesSettingsRequest& request) const;

        /**
         * A Callable wrapper for EnableAllowedImagesSettings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableAllowedImagesSettingsRequestT = Model::EnableAllowedImagesSettingsRequest>
        Model::EnableAllowedImagesSettingsOutcomeCallable EnableAllowedImagesSettingsCallable(const EnableAllowedImagesSettingsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableAllowedImagesSettings, request);
        }

        /**
         * An Async wrapper for EnableAllowedImagesSettings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableAllowedImagesSettingsRequestT = Model::EnableAllowedImagesSettingsRequest>
        void EnableAllowedImagesSettingsAsync(const EnableAllowedImagesSettingsRequestT& request, const EnableAllowedImagesSettingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableAllowedImagesSettings, request, handler, context);
        }

        /**
         * <p>Enables Infrastructure Performance subscriptions.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableAwsNetworkPerformanceMetricSubscription">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableAwsNetworkPerformanceMetricSubscriptionOutcome EnableAwsNetworkPerformanceMetricSubscription(const Model::EnableAwsNetworkPerformanceMetricSubscriptionRequest& request = {}) const;

        /**
         * A Callable wrapper for EnableAwsNetworkPerformanceMetricSubscription that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableAwsNetworkPerformanceMetricSubscriptionRequestT = Model::EnableAwsNetworkPerformanceMetricSubscriptionRequest>
        Model::EnableAwsNetworkPerformanceMetricSubscriptionOutcomeCallable EnableAwsNetworkPerformanceMetricSubscriptionCallable(const EnableAwsNetworkPerformanceMetricSubscriptionRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::EnableAwsNetworkPerformanceMetricSubscription, request);
        }

        /**
         * An Async wrapper for EnableAwsNetworkPerformanceMetricSubscription that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableAwsNetworkPerformanceMetricSubscriptionRequestT = Model::EnableAwsNetworkPerformanceMetricSubscriptionRequest>
        void EnableAwsNetworkPerformanceMetricSubscriptionAsync(const EnableAwsNetworkPerformanceMetricSubscriptionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const EnableAwsNetworkPerformanceMetricSubscriptionRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::EnableAwsNetworkPerformanceMetricSubscription, request, handler, context);
        }

        /**
         * <p>Enables EBS encryption by default for your account in the current Region.</p>
         * <p>After you enable encryption by default, the EBS volumes that you create are
         * always encrypted, either using the default KMS key or the KMS key that you
         * specified when you created each volume. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p> <p>You can specify
         * the default KMS key for encryption by default using
         * <a>ModifyEbsDefaultKmsKeyId</a> or <a>ResetEbsDefaultKmsKeyId</a>.</p>
         * <p>Enabling encryption by default has no effect on the encryption status of your
         * existing volumes.</p> <p>After you enable encryption by default, you can no
         * longer launch instances using instance types that do not support encryption. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption-requirements.html#ebs-encryption_supported_instances">Supported
         * instance types</a>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableEbsEncryptionByDefault">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableEbsEncryptionByDefaultOutcome EnableEbsEncryptionByDefault(const Model::EnableEbsEncryptionByDefaultRequest& request = {}) const;

        /**
         * A Callable wrapper for EnableEbsEncryptionByDefault that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableEbsEncryptionByDefaultRequestT = Model::EnableEbsEncryptionByDefaultRequest>
        Model::EnableEbsEncryptionByDefaultOutcomeCallable EnableEbsEncryptionByDefaultCallable(const EnableEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::EnableEbsEncryptionByDefault, request);
        }

        /**
         * An Async wrapper for EnableEbsEncryptionByDefault that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableEbsEncryptionByDefaultRequestT = Model::EnableEbsEncryptionByDefaultRequest>
        void EnableEbsEncryptionByDefaultAsync(const EnableEbsEncryptionByDefaultResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const EnableEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::EnableEbsEncryptionByDefault, request, handler, context);
        }

        /**
         * <p>When you enable Windows fast launch for a Windows AMI, images are
         * pre-provisioned, using snapshots to launch instances up to 65% faster. To create
         * the optimized Windows image, Amazon EC2 launches an instance and runs through
         * Sysprep steps, rebooting as required. Then it creates a set of reserved
         * snapshots that are used for subsequent launches. The reserved snapshots are
         * automatically replenished as they are used, depending on your settings for
         * launch frequency.</p>  <p>You can only change these settings for Windows
         * AMIs that you own or that have been shared with you.</p> <p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableFastLaunch">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableFastLaunchOutcome EnableFastLaunch(const Model::EnableFastLaunchRequest& request) const;

        /**
         * A Callable wrapper for EnableFastLaunch that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableFastLaunchRequestT = Model::EnableFastLaunchRequest>
        Model::EnableFastLaunchOutcomeCallable EnableFastLaunchCallable(const EnableFastLaunchRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableFastLaunch, request);
        }

        /**
         * An Async wrapper for EnableFastLaunch that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableFastLaunchRequestT = Model::EnableFastLaunchRequest>
        void EnableFastLaunchAsync(const EnableFastLaunchRequestT& request, const EnableFastLaunchResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableFastLaunch, request, handler, context);
        }

        /**
         * <p>Enables fast snapshot restores for the specified snapshots in the specified
         * Availability Zones.</p> <p>You get the full benefit of fast snapshot restores
         * after they enter the <code>enabled</code> state. To get the current state of
         * fast snapshot restores, use <a>DescribeFastSnapshotRestores</a>. To disable fast
         * snapshot restores, use <a>DisableFastSnapshotRestores</a>.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-fast-snapshot-restore.html">Amazon
         * EBS fast snapshot restore</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableFastSnapshotRestores">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableFastSnapshotRestoresOutcome EnableFastSnapshotRestores(const Model::EnableFastSnapshotRestoresRequest& request) const;

        /**
         * A Callable wrapper for EnableFastSnapshotRestores that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableFastSnapshotRestoresRequestT = Model::EnableFastSnapshotRestoresRequest>
        Model::EnableFastSnapshotRestoresOutcomeCallable EnableFastSnapshotRestoresCallable(const EnableFastSnapshotRestoresRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableFastSnapshotRestores, request);
        }

        /**
         * An Async wrapper for EnableFastSnapshotRestores that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableFastSnapshotRestoresRequestT = Model::EnableFastSnapshotRestoresRequest>
        void EnableFastSnapshotRestoresAsync(const EnableFastSnapshotRestoresRequestT& request, const EnableFastSnapshotRestoresResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableFastSnapshotRestores, request, handler, context);
        }

        /**
         * <p>Re-enables a disabled AMI. The re-enabled AMI is marked as
         * <code>available</code> and can be used for instance launches, appears in
         * describe operations, and can be shared. Amazon Web Services accounts,
         * organizations, and Organizational Units that lost access to the AMI when it was
         * disabled do not regain access automatically. Once the AMI is available, it can
         * be shared with them again.</p> <p>Only the AMI owner can re-enable a disabled
         * AMI.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/disable-an-ami.html">Disable
         * an Amazon EC2 AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableImage">AWS API
         * Reference</a></p>
         */
        virtual Model::EnableImageOutcome EnableImage(const Model::EnableImageRequest& request) const;

        /**
         * A Callable wrapper for EnableImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableImageRequestT = Model::EnableImageRequest>
        Model::EnableImageOutcomeCallable EnableImageCallable(const EnableImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableImage, request);
        }

        /**
         * An Async wrapper for EnableImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableImageRequestT = Model::EnableImageRequest>
        void EnableImageAsync(const EnableImageRequestT& request, const EnableImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableImage, request, handler, context);
        }

        /**
         * <p>Enables <i>block public access for AMIs</i> at the account level in the
         * specified Amazon Web Services Region. This prevents the public sharing of your
         * AMIs. However, if you already have public AMIs, they will remain publicly
         * available.</p> <p>The API can take up to 10 minutes to configure this setting.
         * During this time, if you run <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetImageBlockPublicAccessState.html">GetImageBlockPublicAccessState</a>,
         * the response will be <code>unblocked</code>. When the API has completed the
         * configuration, the response will be <code>block-new-sharing</code>.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-public-access-to-amis.html">Block
         * public access to your AMIs</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableImageBlockPublicAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableImageBlockPublicAccessOutcome EnableImageBlockPublicAccess(const Model::EnableImageBlockPublicAccessRequest& request) const;

        /**
         * A Callable wrapper for EnableImageBlockPublicAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableImageBlockPublicAccessRequestT = Model::EnableImageBlockPublicAccessRequest>
        Model::EnableImageBlockPublicAccessOutcomeCallable EnableImageBlockPublicAccessCallable(const EnableImageBlockPublicAccessRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableImageBlockPublicAccess, request);
        }

        /**
         * An Async wrapper for EnableImageBlockPublicAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableImageBlockPublicAccessRequestT = Model::EnableImageBlockPublicAccessRequest>
        void EnableImageBlockPublicAccessAsync(const EnableImageBlockPublicAccessRequestT& request, const EnableImageBlockPublicAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableImageBlockPublicAccess, request, handler, context);
        }

        /**
         * <p>Enables deprecation of the specified AMI at the specified date and time.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-deprecate.html">Deprecate
         * an AMI</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableImageDeprecation">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableImageDeprecationOutcome EnableImageDeprecation(const Model::EnableImageDeprecationRequest& request) const;

        /**
         * A Callable wrapper for EnableImageDeprecation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableImageDeprecationRequestT = Model::EnableImageDeprecationRequest>
        Model::EnableImageDeprecationOutcomeCallable EnableImageDeprecationCallable(const EnableImageDeprecationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableImageDeprecation, request);
        }

        /**
         * An Async wrapper for EnableImageDeprecation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableImageDeprecationRequestT = Model::EnableImageDeprecationRequest>
        void EnableImageDeprecationAsync(const EnableImageDeprecationRequestT& request, const EnableImageDeprecationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableImageDeprecation, request, handler, context);
        }

        /**
         * <p>Enables deregistration protection for an AMI. When deregistration protection
         * is enabled, the AMI can't be deregistered.</p> <p>To allow the AMI to be
         * deregistered, you must first disable deregistration protection using
         * <a>DisableImageDeregistrationProtection</a>.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ami-deregistration-protection.html">Protect
         * an Amazon EC2 AMI from deregistration</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableImageDeregistrationProtection">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableImageDeregistrationProtectionOutcome EnableImageDeregistrationProtection(const Model::EnableImageDeregistrationProtectionRequest& request) const;

        /**
         * A Callable wrapper for EnableImageDeregistrationProtection that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableImageDeregistrationProtectionRequestT = Model::EnableImageDeregistrationProtectionRequest>
        Model::EnableImageDeregistrationProtectionOutcomeCallable EnableImageDeregistrationProtectionCallable(const EnableImageDeregistrationProtectionRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableImageDeregistrationProtection, request);
        }

        /**
         * An Async wrapper for EnableImageDeregistrationProtection that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableImageDeregistrationProtectionRequestT = Model::EnableImageDeregistrationProtectionRequest>
        void EnableImageDeregistrationProtectionAsync(const EnableImageDeregistrationProtectionRequestT& request, const EnableImageDeregistrationProtectionResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableImageDeregistrationProtection, request, handler, context);
        }

        /**
         * <p>Enable an Organizations member account as the IPAM admin account. You cannot
         * select the Organizations management account as the IPAM admin account. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/enable-integ-ipam.html">Enable
         * integration with Organizations</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * </p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableIpamOrganizationAdminAccount">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableIpamOrganizationAdminAccountOutcome EnableIpamOrganizationAdminAccount(const Model::EnableIpamOrganizationAdminAccountRequest& request) const;

        /**
         * A Callable wrapper for EnableIpamOrganizationAdminAccount that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableIpamOrganizationAdminAccountRequestT = Model::EnableIpamOrganizationAdminAccountRequest>
        Model::EnableIpamOrganizationAdminAccountOutcomeCallable EnableIpamOrganizationAdminAccountCallable(const EnableIpamOrganizationAdminAccountRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableIpamOrganizationAdminAccount, request);
        }

        /**
         * An Async wrapper for EnableIpamOrganizationAdminAccount that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableIpamOrganizationAdminAccountRequestT = Model::EnableIpamOrganizationAdminAccountRequest>
        void EnableIpamOrganizationAdminAccountAsync(const EnableIpamOrganizationAdminAccountRequestT& request, const EnableIpamOrganizationAdminAccountResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableIpamOrganizationAdminAccount, request, handler, context);
        }

        /**
         * <p>Establishes a trust relationship between Reachability Analyzer and
         * Organizations. This operation must be performed by the management account for
         * the organization.</p> <p>After you establish a trust relationship, a user in the
         * management account or a delegated administrator account can run a cross-account
         * analysis using resources from the member accounts.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableReachabilityAnalyzerOrganizationSharing">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableReachabilityAnalyzerOrganizationSharingOutcome EnableReachabilityAnalyzerOrganizationSharing(const Model::EnableReachabilityAnalyzerOrganizationSharingRequest& request = {}) const;

        /**
         * A Callable wrapper for EnableReachabilityAnalyzerOrganizationSharing that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableReachabilityAnalyzerOrganizationSharingRequestT = Model::EnableReachabilityAnalyzerOrganizationSharingRequest>
        Model::EnableReachabilityAnalyzerOrganizationSharingOutcomeCallable EnableReachabilityAnalyzerOrganizationSharingCallable(const EnableReachabilityAnalyzerOrganizationSharingRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::EnableReachabilityAnalyzerOrganizationSharing, request);
        }

        /**
         * An Async wrapper for EnableReachabilityAnalyzerOrganizationSharing that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableReachabilityAnalyzerOrganizationSharingRequestT = Model::EnableReachabilityAnalyzerOrganizationSharingRequest>
        void EnableReachabilityAnalyzerOrganizationSharingAsync(const EnableReachabilityAnalyzerOrganizationSharingResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const EnableReachabilityAnalyzerOrganizationSharingRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::EnableReachabilityAnalyzerOrganizationSharing, request, handler, context);
        }

        /**
         * <p>Defines which route tables the route server can update with routes.</p>
         * <p>When enabled, route server propagation installs the routes in the FIB on the
         * route table you've specified. Route server supports IPv4 and IPv6 route
         * propagation.</p> <p>For more information see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/dynamic-routing-route-server.html">Dynamic
         * routing in your VPC with VPC Route Server</a> in the <i>Amazon VPC User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableRouteServerPropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableRouteServerPropagationOutcome EnableRouteServerPropagation(const Model::EnableRouteServerPropagationRequest& request) const;

        /**
         * A Callable wrapper for EnableRouteServerPropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableRouteServerPropagationRequestT = Model::EnableRouteServerPropagationRequest>
        Model::EnableRouteServerPropagationOutcomeCallable EnableRouteServerPropagationCallable(const EnableRouteServerPropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableRouteServerPropagation, request);
        }

        /**
         * An Async wrapper for EnableRouteServerPropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableRouteServerPropagationRequestT = Model::EnableRouteServerPropagationRequest>
        void EnableRouteServerPropagationAsync(const EnableRouteServerPropagationRequestT& request, const EnableRouteServerPropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableRouteServerPropagation, request, handler, context);
        }

        /**
         * <p>Enables access to the EC2 serial console of all instances for your account.
         * By default, access to the EC2 serial console is disabled for your account. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/configure-access-to-serial-console.html#serial-console-account-access">Manage
         * account access to the EC2 serial console</a> in the <i>Amazon EC2 User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableSerialConsoleAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableSerialConsoleAccessOutcome EnableSerialConsoleAccess(const Model::EnableSerialConsoleAccessRequest& request = {}) const;

        /**
         * A Callable wrapper for EnableSerialConsoleAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableSerialConsoleAccessRequestT = Model::EnableSerialConsoleAccessRequest>
        Model::EnableSerialConsoleAccessOutcomeCallable EnableSerialConsoleAccessCallable(const EnableSerialConsoleAccessRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::EnableSerialConsoleAccess, request);
        }

        /**
         * An Async wrapper for EnableSerialConsoleAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableSerialConsoleAccessRequestT = Model::EnableSerialConsoleAccessRequest>
        void EnableSerialConsoleAccessAsync(const EnableSerialConsoleAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const EnableSerialConsoleAccessRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::EnableSerialConsoleAccess, request, handler, context);
        }

        /**
         * <p>Enables or modifies the <i>block public access for snapshots</i> setting at
         * the account level for the specified Amazon Web Services Region. After you enable
         * block public access for snapshots in a Region, users can no longer request
         * public sharing for snapshots in that Region. Snapshots that are already publicly
         * shared are either treated as private or they remain publicly shared, depending
         * on the <b>State</b> that you specify.</p>  <p>Enabling block public
         * access for snapshots in <i>block all sharing</i> mode does not change the
         * permissions for snapshots that are already publicly shared. Instead, it prevents
         * these snapshots from be publicly visible and publicly accessible. Therefore, the
         * attributes for these snapshots still indicate that they are publicly shared,
         * even though they are not publicly available.</p> <p>If you later disable block
         * public access or change the mode to <i>block new sharing</i>, these snapshots
         * will become publicly available again.</p>  <p>For more information,
         * see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/block-public-access-snapshots.html">
         * Block public access for snapshots</a> in the <i>Amazon EBS User
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableSnapshotBlockPublicAccess">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableSnapshotBlockPublicAccessOutcome EnableSnapshotBlockPublicAccess(const Model::EnableSnapshotBlockPublicAccessRequest& request) const;

        /**
         * A Callable wrapper for EnableSnapshotBlockPublicAccess that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableSnapshotBlockPublicAccessRequestT = Model::EnableSnapshotBlockPublicAccessRequest>
        Model::EnableSnapshotBlockPublicAccessOutcomeCallable EnableSnapshotBlockPublicAccessCallable(const EnableSnapshotBlockPublicAccessRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableSnapshotBlockPublicAccess, request);
        }

        /**
         * An Async wrapper for EnableSnapshotBlockPublicAccess that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableSnapshotBlockPublicAccessRequestT = Model::EnableSnapshotBlockPublicAccessRequest>
        void EnableSnapshotBlockPublicAccessAsync(const EnableSnapshotBlockPublicAccessRequestT& request, const EnableSnapshotBlockPublicAccessResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableSnapshotBlockPublicAccess, request, handler, context);
        }

        /**
         * <p>Enables the specified attachment to propagate routes to the specified
         * propagation route table.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableTransitGatewayRouteTablePropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableTransitGatewayRouteTablePropagationOutcome EnableTransitGatewayRouteTablePropagation(const Model::EnableTransitGatewayRouteTablePropagationRequest& request) const;

        /**
         * A Callable wrapper for EnableTransitGatewayRouteTablePropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableTransitGatewayRouteTablePropagationRequestT = Model::EnableTransitGatewayRouteTablePropagationRequest>
        Model::EnableTransitGatewayRouteTablePropagationOutcomeCallable EnableTransitGatewayRouteTablePropagationCallable(const EnableTransitGatewayRouteTablePropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableTransitGatewayRouteTablePropagation, request);
        }

        /**
         * An Async wrapper for EnableTransitGatewayRouteTablePropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableTransitGatewayRouteTablePropagationRequestT = Model::EnableTransitGatewayRouteTablePropagationRequest>
        void EnableTransitGatewayRouteTablePropagationAsync(const EnableTransitGatewayRouteTablePropagationRequestT& request, const EnableTransitGatewayRouteTablePropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableTransitGatewayRouteTablePropagation, request, handler, context);
        }

        /**
         * <p>Enables a virtual private gateway (VGW) to propagate routes to the specified
         * route table of a VPC.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableVgwRoutePropagation">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableVgwRoutePropagationOutcome EnableVgwRoutePropagation(const Model::EnableVgwRoutePropagationRequest& request) const;

        /**
         * A Callable wrapper for EnableVgwRoutePropagation that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableVgwRoutePropagationRequestT = Model::EnableVgwRoutePropagationRequest>
        Model::EnableVgwRoutePropagationOutcomeCallable EnableVgwRoutePropagationCallable(const EnableVgwRoutePropagationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableVgwRoutePropagation, request);
        }

        /**
         * An Async wrapper for EnableVgwRoutePropagation that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableVgwRoutePropagationRequestT = Model::EnableVgwRoutePropagationRequest>
        void EnableVgwRoutePropagationAsync(const EnableVgwRoutePropagationRequestT& request, const EnableVgwRoutePropagationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableVgwRoutePropagation, request, handler, context);
        }

        /**
         * <p>Enables I/O operations for a volume that had I/O operations disabled because
         * the data on the volume was potentially inconsistent.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableVolumeIO">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableVolumeIOOutcome EnableVolumeIO(const Model::EnableVolumeIORequest& request) const;

        /**
         * A Callable wrapper for EnableVolumeIO that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableVolumeIORequestT = Model::EnableVolumeIORequest>
        Model::EnableVolumeIOOutcomeCallable EnableVolumeIOCallable(const EnableVolumeIORequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableVolumeIO, request);
        }

        /**
         * An Async wrapper for EnableVolumeIO that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableVolumeIORequestT = Model::EnableVolumeIORequest>
        void EnableVolumeIOAsync(const EnableVolumeIORequestT& request, const EnableVolumeIOResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableVolumeIO, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Enables a VPC for
         * ClassicLink. You can then link EC2-Classic instances to your ClassicLink-enabled
         * VPC to allow communication over private IP addresses. You cannot enable your VPC
         * for ClassicLink if any of your VPC route tables have existing routes for address
         * ranges within the <code>10.0.0.0/8</code> IP address range, excluding local
         * routes for VPCs in the <code>10.0.0.0/16</code> and <code>10.1.0.0/16</code> IP
         * address ranges.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableVpcClassicLink">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableVpcClassicLinkOutcome EnableVpcClassicLink(const Model::EnableVpcClassicLinkRequest& request) const;

        /**
         * A Callable wrapper for EnableVpcClassicLink that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableVpcClassicLinkRequestT = Model::EnableVpcClassicLinkRequest>
        Model::EnableVpcClassicLinkOutcomeCallable EnableVpcClassicLinkCallable(const EnableVpcClassicLinkRequestT& request) const
        {
            return SubmitCallable(&EC2Client::EnableVpcClassicLink, request);
        }

        /**
         * An Async wrapper for EnableVpcClassicLink that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableVpcClassicLinkRequestT = Model::EnableVpcClassicLinkRequest>
        void EnableVpcClassicLinkAsync(const EnableVpcClassicLinkRequestT& request, const EnableVpcClassicLinkResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::EnableVpcClassicLink, request, handler, context);
        }

        /**
         *  <p>This action is deprecated.</p>  <p>Enables a VPC to support DNS
         * hostname resolution for ClassicLink. If enabled, the DNS hostname of a linked
         * EC2-Classic instance resolves to its private IP address when addressed from an
         * instance in the VPC to which it's linked. Similarly, the DNS hostname of an
         * instance in a VPC resolves to its private IP address when addressed from a
         * linked EC2-Classic instance.</p> <p>You must specify a VPC ID in the
         * request.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/EnableVpcClassicLinkDnsSupport">AWS
         * API Reference</a></p>
         */
        virtual Model::EnableVpcClassicLinkDnsSupportOutcome EnableVpcClassicLinkDnsSupport(const Model::EnableVpcClassicLinkDnsSupportRequest& request = {}) const;

        /**
         * A Callable wrapper for EnableVpcClassicLinkDnsSupport that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename EnableVpcClassicLinkDnsSupportRequestT = Model::EnableVpcClassicLinkDnsSupportRequest>
        Model::EnableVpcClassicLinkDnsSupportOutcomeCallable EnableVpcClassicLinkDnsSupportCallable(const EnableVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::EnableVpcClassicLinkDnsSupport, request);
        }

        /**
         * An Async wrapper for EnableVpcClassicLinkDnsSupport that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename EnableVpcClassicLinkDnsSupportRequestT = Model::EnableVpcClassicLinkDnsSupportRequest>
        void EnableVpcClassicLinkDnsSupportAsync(const EnableVpcClassicLinkDnsSupportResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const EnableVpcClassicLinkDnsSupportRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::EnableVpcClassicLinkDnsSupport, request, handler, context);
        }

        /**
         * <p>Downloads the client certificate revocation list for the specified Client VPN
         * endpoint.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportClientVpnClientCertificateRevocationList">AWS
         * API Reference</a></p>
         */
        virtual Model::ExportClientVpnClientCertificateRevocationListOutcome ExportClientVpnClientCertificateRevocationList(const Model::ExportClientVpnClientCertificateRevocationListRequest& request) const;

        /**
         * A Callable wrapper for ExportClientVpnClientCertificateRevocationList that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ExportClientVpnClientCertificateRevocationListRequestT = Model::ExportClientVpnClientCertificateRevocationListRequest>
        Model::ExportClientVpnClientCertificateRevocationListOutcomeCallable ExportClientVpnClientCertificateRevocationListCallable(const ExportClientVpnClientCertificateRevocationListRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ExportClientVpnClientCertificateRevocationList, request);
        }

        /**
         * An Async wrapper for ExportClientVpnClientCertificateRevocationList that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ExportClientVpnClientCertificateRevocationListRequestT = Model::ExportClientVpnClientCertificateRevocationListRequest>
        void ExportClientVpnClientCertificateRevocationListAsync(const ExportClientVpnClientCertificateRevocationListRequestT& request, const ExportClientVpnClientCertificateRevocationListResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ExportClientVpnClientCertificateRevocationList, request, handler, context);
        }

        /**
         * <p>Downloads the contents of the Client VPN endpoint configuration file for the
         * specified Client VPN endpoint. The Client VPN endpoint configuration file
         * includes the Client VPN endpoint and certificate information clients need to
         * establish a connection with the Client VPN endpoint.</p><p><h3>See Also:</h3>  
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportClientVpnClientConfiguration">AWS
         * API Reference</a></p>
         */
        virtual Model::ExportClientVpnClientConfigurationOutcome ExportClientVpnClientConfiguration(const Model::ExportClientVpnClientConfigurationRequest& request) const;

        /**
         * A Callable wrapper for ExportClientVpnClientConfiguration that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ExportClientVpnClientConfigurationRequestT = Model::ExportClientVpnClientConfigurationRequest>
        Model::ExportClientVpnClientConfigurationOutcomeCallable ExportClientVpnClientConfigurationCallable(const ExportClientVpnClientConfigurationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ExportClientVpnClientConfiguration, request);
        }

        /**
         * An Async wrapper for ExportClientVpnClientConfiguration that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ExportClientVpnClientConfigurationRequestT = Model::ExportClientVpnClientConfigurationRequest>
        void ExportClientVpnClientConfigurationAsync(const ExportClientVpnClientConfigurationRequestT& request, const ExportClientVpnClientConfigurationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ExportClientVpnClientConfiguration, request, handler, context);
        }

        /**
         * <p>Exports an Amazon Machine Image (AMI) to a VM file. For more information, see
         * <a
         * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport_image.html">Exporting
         * a VM directly from an Amazon Machine Image (AMI)</a> in the <i>VM Import/Export
         * User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportImage">AWS API
         * Reference</a></p>
         */
        virtual Model::ExportImageOutcome ExportImage(const Model::ExportImageRequest& request) const;

        /**
         * A Callable wrapper for ExportImage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ExportImageRequestT = Model::ExportImageRequest>
        Model::ExportImageOutcomeCallable ExportImageCallable(const ExportImageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ExportImage, request);
        }

        /**
         * An Async wrapper for ExportImage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ExportImageRequestT = Model::ExportImageRequest>
        void ExportImageAsync(const ExportImageRequestT& request, const ExportImageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ExportImage, request, handler, context);
        }

        /**
         * <p>Exports routes from the specified transit gateway route table to the
         * specified S3 bucket. By default, all routes are exported. Alternatively, you can
         * filter by CIDR range.</p> <p>The routes are saved to the specified bucket in a
         * JSON file. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-route-tables.html#tgw-export-route-tables">Export
         * route tables to Amazon S3</a> in the <i>Amazon Web Services Transit Gateways
         * Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportTransitGatewayRoutes">AWS
         * API Reference</a></p>
         */
        virtual Model::ExportTransitGatewayRoutesOutcome ExportTransitGatewayRoutes(const Model::ExportTransitGatewayRoutesRequest& request) const;

        /**
         * A Callable wrapper for ExportTransitGatewayRoutes that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ExportTransitGatewayRoutesRequestT = Model::ExportTransitGatewayRoutesRequest>
        Model::ExportTransitGatewayRoutesOutcomeCallable ExportTransitGatewayRoutesCallable(const ExportTransitGatewayRoutesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ExportTransitGatewayRoutes, request);
        }

        /**
         * An Async wrapper for ExportTransitGatewayRoutes that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ExportTransitGatewayRoutesRequestT = Model::ExportTransitGatewayRoutesRequest>
        void ExportTransitGatewayRoutesAsync(const ExportTransitGatewayRoutesRequestT& request, const ExportTransitGatewayRoutesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ExportTransitGatewayRoutes, request, handler, context);
        }

        /**
         * <p>Exports the client configuration for a Verified Access
         * instance.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ExportVerifiedAccessInstanceClientConfiguration">AWS
         * API Reference</a></p>
         */
        virtual Model::ExportVerifiedAccessInstanceClientConfigurationOutcome ExportVerifiedAccessInstanceClientConfiguration(const Model::ExportVerifiedAccessInstanceClientConfigurationRequest& request) const;

        /**
         * A Callable wrapper for ExportVerifiedAccessInstanceClientConfiguration that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename ExportVerifiedAccessInstanceClientConfigurationRequestT = Model::ExportVerifiedAccessInstanceClientConfigurationRequest>
        Model::ExportVerifiedAccessInstanceClientConfigurationOutcomeCallable ExportVerifiedAccessInstanceClientConfigurationCallable(const ExportVerifiedAccessInstanceClientConfigurationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::ExportVerifiedAccessInstanceClientConfiguration, request);
        }

        /**
         * An Async wrapper for ExportVerifiedAccessInstanceClientConfiguration that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename ExportVerifiedAccessInstanceClientConfigurationRequestT = Model::ExportVerifiedAccessInstanceClientConfigurationRequest>
        void ExportVerifiedAccessInstanceClientConfigurationAsync(const ExportVerifiedAccessInstanceClientConfigurationRequestT& request, const ExportVerifiedAccessInstanceClientConfigurationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::ExportVerifiedAccessInstanceClientConfiguration, request, handler, context);
        }

        /**
         * <p>Returns the currently negotiated security parameters for an active VPN
         * tunnel, including IKE version, DH groups, encryption algorithms, and integrity
         * algorithms.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetActiveVpnTunnelStatus">AWS
         * API Reference</a></p>
         */
        virtual Model::GetActiveVpnTunnelStatusOutcome GetActiveVpnTunnelStatus(const Model::GetActiveVpnTunnelStatusRequest& request) const;

        /**
         * A Callable wrapper for GetActiveVpnTunnelStatus that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetActiveVpnTunnelStatusRequestT = Model::GetActiveVpnTunnelStatusRequest>
        Model::GetActiveVpnTunnelStatusOutcomeCallable GetActiveVpnTunnelStatusCallable(const GetActiveVpnTunnelStatusRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetActiveVpnTunnelStatus, request);
        }

        /**
         * An Async wrapper for GetActiveVpnTunnelStatus that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetActiveVpnTunnelStatusRequestT = Model::GetActiveVpnTunnelStatusRequest>
        void GetActiveVpnTunnelStatusAsync(const GetActiveVpnTunnelStatusRequestT& request, const GetActiveVpnTunnelStatusResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetActiveVpnTunnelStatus, request, handler, context);
        }

        /**
         * <p>Gets the current state of the Allowed AMIs setting and the list of Allowed
         * AMIs criteria at the account level in the specified Region.</p>  <p>The
         * Allowed AMIs feature does not restrict the AMIs owned by your account.
         * Regardless of the criteria you set, the AMIs created by your account will always
         * be discoverable and usable by users in your account.</p>  <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-allowed-amis.html">Control
         * the discovery and use of AMIs in Amazon EC2 with Allowed AMIs</a> in <i>Amazon
         * EC2 User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetAllowedImagesSettings">AWS
         * API Reference</a></p>
         */
        virtual Model::GetAllowedImagesSettingsOutcome GetAllowedImagesSettings(const Model::GetAllowedImagesSettingsRequest& request = {}) const;

        /**
         * A Callable wrapper for GetAllowedImagesSettings that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetAllowedImagesSettingsRequestT = Model::GetAllowedImagesSettingsRequest>
        Model::GetAllowedImagesSettingsOutcomeCallable GetAllowedImagesSettingsCallable(const GetAllowedImagesSettingsRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::GetAllowedImagesSettings, request);
        }

        /**
         * An Async wrapper for GetAllowedImagesSettings that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetAllowedImagesSettingsRequestT = Model::GetAllowedImagesSettingsRequest>
        void GetAllowedImagesSettingsAsync(const GetAllowedImagesSettingsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const GetAllowedImagesSettingsRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::GetAllowedImagesSettings, request, handler, context);
        }

        /**
         * <p>Returns the IAM roles that are associated with the specified ACM (ACM)
         * certificate. It also returns the name of the Amazon S3 bucket and the Amazon S3
         * object key where the certificate, certificate chain, and encrypted private key
         * bundle are stored, and the ARN of the KMS key that's used to encrypt the private
         * key.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetAssociatedEnclaveCertificateIamRoles">AWS
         * API Reference</a></p>
         */
        virtual Model::GetAssociatedEnclaveCertificateIamRolesOutcome GetAssociatedEnclaveCertificateIamRoles(const Model::GetAssociatedEnclaveCertificateIamRolesRequest& request) const;

        /**
         * A Callable wrapper for GetAssociatedEnclaveCertificateIamRoles that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetAssociatedEnclaveCertificateIamRolesRequestT = Model::GetAssociatedEnclaveCertificateIamRolesRequest>
        Model::GetAssociatedEnclaveCertificateIamRolesOutcomeCallable GetAssociatedEnclaveCertificateIamRolesCallable(const GetAssociatedEnclaveCertificateIamRolesRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetAssociatedEnclaveCertificateIamRoles, request);
        }

        /**
         * An Async wrapper for GetAssociatedEnclaveCertificateIamRoles that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetAssociatedEnclaveCertificateIamRolesRequestT = Model::GetAssociatedEnclaveCertificateIamRolesRequest>
        void GetAssociatedEnclaveCertificateIamRolesAsync(const GetAssociatedEnclaveCertificateIamRolesRequestT& request, const GetAssociatedEnclaveCertificateIamRolesResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetAssociatedEnclaveCertificateIamRoles, request, handler, context);
        }

        /**
         * <p>Gets information about the IPv6 CIDR block associations for a specified IPv6
         * address pool.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetAssociatedIpv6PoolCidrs">AWS
         * API Reference</a></p>
         */
        virtual Model::GetAssociatedIpv6PoolCidrsOutcome GetAssociatedIpv6PoolCidrs(const Model::GetAssociatedIpv6PoolCidrsRequest& request) const;

        /**
         * A Callable wrapper for GetAssociatedIpv6PoolCidrs that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetAssociatedIpv6PoolCidrsRequestT = Model::GetAssociatedIpv6PoolCidrsRequest>
        Model::GetAssociatedIpv6PoolCidrsOutcomeCallable GetAssociatedIpv6PoolCidrsCallable(const GetAssociatedIpv6PoolCidrsRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetAssociatedIpv6PoolCidrs, request);
        }

        /**
         * An Async wrapper for GetAssociatedIpv6PoolCidrs that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetAssociatedIpv6PoolCidrsRequestT = Model::GetAssociatedIpv6PoolCidrsRequest>
        void GetAssociatedIpv6PoolCidrsAsync(const GetAssociatedIpv6PoolCidrsRequestT& request, const GetAssociatedIpv6PoolCidrsResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetAssociatedIpv6PoolCidrs, request, handler, context);
        }

        /**
         * <p>Gets network performance data.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetAwsNetworkPerformanceData">AWS
         * API Reference</a></p>
         */
        virtual Model::GetAwsNetworkPerformanceDataOutcome GetAwsNetworkPerformanceData(const Model::GetAwsNetworkPerformanceDataRequest& request = {}) const;

        /**
         * A Callable wrapper for GetAwsNetworkPerformanceData that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetAwsNetworkPerformanceDataRequestT = Model::GetAwsNetworkPerformanceDataRequest>
        Model::GetAwsNetworkPerformanceDataOutcomeCallable GetAwsNetworkPerformanceDataCallable(const GetAwsNetworkPerformanceDataRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::GetAwsNetworkPerformanceData, request);
        }

        /**
         * An Async wrapper for GetAwsNetworkPerformanceData that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetAwsNetworkPerformanceDataRequestT = Model::GetAwsNetworkPerformanceDataRequest>
        void GetAwsNetworkPerformanceDataAsync(const GetAwsNetworkPerformanceDataResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const GetAwsNetworkPerformanceDataRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::GetAwsNetworkPerformanceData, request, handler, context);
        }

        /**
         * <p>Gets usage information about a Capacity Reservation. If the Capacity
         * Reservation is shared, it shows usage information for the Capacity Reservation
         * owner and each Amazon Web Services account that is currently using the shared
         * capacity. If the Capacity Reservation is not shared, it shows only the Capacity
         * Reservation owner's usage.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetCapacityReservationUsage">AWS
         * API Reference</a></p>
         */
        virtual Model::GetCapacityReservationUsageOutcome GetCapacityReservationUsage(const Model::GetCapacityReservationUsageRequest& request) const;

        /**
         * A Callable wrapper for GetCapacityReservationUsage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetCapacityReservationUsageRequestT = Model::GetCapacityReservationUsageRequest>
        Model::GetCapacityReservationUsageOutcomeCallable GetCapacityReservationUsageCallable(const GetCapacityReservationUsageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetCapacityReservationUsage, request);
        }

        /**
         * An Async wrapper for GetCapacityReservationUsage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetCapacityReservationUsageRequestT = Model::GetCapacityReservationUsageRequest>
        void GetCapacityReservationUsageAsync(const GetCapacityReservationUsageRequestT& request, const GetCapacityReservationUsageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetCapacityReservationUsage, request, handler, context);
        }

        /**
         * <p>Describes the allocations from the specified customer-owned address
         * pool.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetCoipPoolUsage">AWS
         * API Reference</a></p>
         */
        virtual Model::GetCoipPoolUsageOutcome GetCoipPoolUsage(const Model::GetCoipPoolUsageRequest& request) const;

        /**
         * A Callable wrapper for GetCoipPoolUsage that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetCoipPoolUsageRequestT = Model::GetCoipPoolUsageRequest>
        Model::GetCoipPoolUsageOutcomeCallable GetCoipPoolUsageCallable(const GetCoipPoolUsageRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetCoipPoolUsage, request);
        }

        /**
         * An Async wrapper for GetCoipPoolUsage that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetCoipPoolUsageRequestT = Model::GetCoipPoolUsageRequest>
        void GetCoipPoolUsageAsync(const GetCoipPoolUsageRequestT& request, const GetCoipPoolUsageResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetCoipPoolUsage, request, handler, context);
        }

        /**
         * <p>Gets the console output for the specified instance. For Linux instances, the
         * instance console output displays the exact console output that would normally be
         * displayed on a physical monitor attached to a computer. For Windows instances,
         * the instance console output includes the last three system event log errors.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-console.html#instance-console-console-output">Instance
         * console output</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetConsoleOutput">AWS
         * API Reference</a></p>
         */
        virtual Model::GetConsoleOutputOutcome GetConsoleOutput(const Model::GetConsoleOutputRequest& request) const;

        /**
         * A Callable wrapper for GetConsoleOutput that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetConsoleOutputRequestT = Model::GetConsoleOutputRequest>
        Model::GetConsoleOutputOutcomeCallable GetConsoleOutputCallable(const GetConsoleOutputRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetConsoleOutput, request);
        }

        /**
         * An Async wrapper for GetConsoleOutput that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetConsoleOutputRequestT = Model::GetConsoleOutputRequest>
        void GetConsoleOutputAsync(const GetConsoleOutputRequestT& request, const GetConsoleOutputResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetConsoleOutput, request, handler, context);
        }

        /**
         * <p>Retrieve a JPG-format screenshot of a running instance to help with
         * troubleshooting.</p> <p>The returned content is Base64-encoded.</p> <p>For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/troubleshoot-unreachable-instance.html#instance-console-console-output">Instance
         * console output</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetConsoleScreenshot">AWS
         * API Reference</a></p>
         */
        virtual Model::GetConsoleScreenshotOutcome GetConsoleScreenshot(const Model::GetConsoleScreenshotRequest& request) const;

        /**
         * A Callable wrapper for GetConsoleScreenshot that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetConsoleScreenshotRequestT = Model::GetConsoleScreenshotRequest>
        Model::GetConsoleScreenshotOutcomeCallable GetConsoleScreenshotCallable(const GetConsoleScreenshotRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetConsoleScreenshot, request);
        }

        /**
         * An Async wrapper for GetConsoleScreenshot that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetConsoleScreenshotRequestT = Model::GetConsoleScreenshotRequest>
        void GetConsoleScreenshotAsync(const GetConsoleScreenshotRequestT& request, const GetConsoleScreenshotResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetConsoleScreenshot, request, handler, context);
        }

        /**
         * <p>Retrieves a summary of the account status report.</p> <p>To view the full
         * report, download it from the Amazon S3 bucket where it was saved. Reports are
         * accessible only when they have the <code>complete</code> status. Reports with
         * other statuses (<code>running</code>, <code>cancelled</code>, or
         * <code>error</code>) are not available in the S3 bucket. For more information
         * about downloading objects from an S3 bucket, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/download-objects.html">Downloading
         * objects</a> in the <i>Amazon Simple Storage Service User Guide</i>.</p> <p>For
         * more information, see <a
         * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_policies_declarative_status-report.html">Generating
         * the account status report for declarative policies</a> in the <i>Amazon Web
         * Services Organizations User Guide</i>.</p><p><h3>See Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetDeclarativePoliciesReportSummary">AWS
         * API Reference</a></p>
         */
        virtual Model::GetDeclarativePoliciesReportSummaryOutcome GetDeclarativePoliciesReportSummary(const Model::GetDeclarativePoliciesReportSummaryRequest& request) const;

        /**
         * A Callable wrapper for GetDeclarativePoliciesReportSummary that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetDeclarativePoliciesReportSummaryRequestT = Model::GetDeclarativePoliciesReportSummaryRequest>
        Model::GetDeclarativePoliciesReportSummaryOutcomeCallable GetDeclarativePoliciesReportSummaryCallable(const GetDeclarativePoliciesReportSummaryRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetDeclarativePoliciesReportSummary, request);
        }

        /**
         * An Async wrapper for GetDeclarativePoliciesReportSummary that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetDeclarativePoliciesReportSummaryRequestT = Model::GetDeclarativePoliciesReportSummaryRequest>
        void GetDeclarativePoliciesReportSummaryAsync(const GetDeclarativePoliciesReportSummaryRequestT& request, const GetDeclarativePoliciesReportSummaryResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetDeclarativePoliciesReportSummary, request, handler, context);
        }

        /**
         * <p>Describes the default credit option for CPU usage of a burstable performance
         * instance family.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
         * performance instances</a> in the <i>Amazon EC2 User Guide</i>.</p><p><h3>See
         * Also:</h3>   <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetDefaultCreditSpecification">AWS
         * API Reference</a></p>
         */
        virtual Model::GetDefaultCreditSpecificationOutcome GetDefaultCreditSpecification(const Model::GetDefaultCreditSpecificationRequest& request) const;

        /**
         * A Callable wrapper for GetDefaultCreditSpecification that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetDefaultCreditSpecificationRequestT = Model::GetDefaultCreditSpecificationRequest>
        Model::GetDefaultCreditSpecificationOutcomeCallable GetDefaultCreditSpecificationCallable(const GetDefaultCreditSpecificationRequestT& request) const
        {
            return SubmitCallable(&EC2Client::GetDefaultCreditSpecification, request);
        }

        /**
         * An Async wrapper for GetDefaultCreditSpecification that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetDefaultCreditSpecificationRequestT = Model::GetDefaultCreditSpecificationRequest>
        void GetDefaultCreditSpecificationAsync(const GetDefaultCreditSpecificationRequestT& request, const GetDefaultCreditSpecificationResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const
        {
            return SubmitAsync(&EC2Client::GetDefaultCreditSpecification, request, handler, context);
        }

        /**
         * <p>Describes the default KMS key for EBS encryption by default for your account
         * in this Region. You can change the default KMS key for encryption by default
         * using <a>ModifyEbsDefaultKmsKeyId</a> or <a>ResetEbsDefaultKmsKeyId</a>.</p>
         * <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetEbsDefaultKmsKeyId">AWS
         * API Reference</a></p>
         */
        virtual Model::GetEbsDefaultKmsKeyIdOutcome GetEbsDefaultKmsKeyId(const Model::GetEbsDefaultKmsKeyIdRequest& request = {}) const;

        /**
         * A Callable wrapper for GetEbsDefaultKmsKeyId that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetEbsDefaultKmsKeyIdRequestT = Model::GetEbsDefaultKmsKeyIdRequest>
        Model::GetEbsDefaultKmsKeyIdOutcomeCallable GetEbsDefaultKmsKeyIdCallable(const GetEbsDefaultKmsKeyIdRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::GetEbsDefaultKmsKeyId, request);
        }

        /**
         * An Async wrapper for GetEbsDefaultKmsKeyId that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetEbsDefaultKmsKeyIdRequestT = Model::GetEbsDefaultKmsKeyIdRequest>
        void GetEbsDefaultKmsKeyIdAsync(const GetEbsDefaultKmsKeyIdResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const GetEbsDefaultKmsKeyIdRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::GetEbsDefaultKmsKeyId, request, handler, context);
        }

        /**
         * <p>Describes whether EBS encryption by default is enabled for your account in
         * the current Region.</p> <p>For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
         * EBS encryption</a> in the <i>Amazon EBS User Guide</i>.</p><p><h3>See Also:</h3>
         * <a
         * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/GetEbsEncryptionByDefault">AWS
         * API Reference</a></p>
         */
        virtual Model::GetEbsEncryptionByDefaultOutcome GetEbsEncryptionByDefault(const Model::GetEbsEncryptionByDefaultRequest& request = {}) const;

        /**
         * A Callable wrapper for GetEbsEncryptionByDefault that returns a future to the operation so that it can be executed in parallel to other requests.
         */
        template<typename GetEbsEncryptionByDefaultRequestT = Model::GetEbsEncryptionByDefaultRequest>
        Model::GetEbsEncryptionByDefaultOutcomeCallable GetEbsEncryptionByDefaultCallable(const GetEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitCallable(&EC2Client::GetEbsEncryptionByDefault, request);
        }

        /**
         * An Async wrapper for GetEbsEncryptionByDefault that queues the request into a thread executor and triggers associated callback when operation has finished.
         */
        template<typename GetEbsEncryptionByDefaultRequestT = Model::GetEbsEncryptionByDefaultRequest>
        void GetEbsEncryptionByDefaultAsync(const GetEbsEncryptionByDefaultResponseReceivedHandler& handler, const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr, const GetEbsEncryptionByDefaultRequestT& request = {}) const
        {
            return SubmitAsync(&EC2Client::GetEbsEncryptionByDefault, request, handler, context);
        }

        /**
         * <p>Generates a CloudFormation template that streamlines and automates the
         * integration of VPC flow logs with Amazon Athena. This make it easier for you to
         * query and gain insights from VPC flow logs data. Based on the information that
         * you provide, we configure resources in the template to do the following:</p>
         * <ul> <li> <p>Create a table in Athena that maps fields to a custom log
         * format</p> </li> <li> <p>Create a Lambda function that updates the table with
    