﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Xml
{
  class XmlNode;
} // namespace Xml
} // namespace Utils
namespace S3Crt
{
namespace Model
{

  /**
   * <p>Specifies encryption-related information for an Amazon S3 bucket that is a
   * destination for replicated objects.</p>  <p>If you're specifying a
   * customer managed KMS key, we recommend using a fully qualified KMS key ARN. If
   * you use a KMS key alias instead, then KMS resolves the key within the
   * requester’s account. This behavior can result in data that's encrypted with a
   * KMS key that belongs to the requester, and not the bucket owner.</p>
   * <p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/EncryptionConfiguration">AWS
   * API Reference</a></p>
   */
  class EncryptionConfiguration
  {
  public:
    AWS_S3CRT_API EncryptionConfiguration() = default;
    AWS_S3CRT_API EncryptionConfiguration(const Aws::Utils::Xml::XmlNode& xmlNode);
    AWS_S3CRT_API EncryptionConfiguration& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

    AWS_S3CRT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;


    ///@{
    /**
     * <p>Specifies the ID (Key ARN or Alias ARN) of the customer managed Amazon Web
     * Services KMS key stored in Amazon Web Services Key Management Service (KMS) for
     * the destination bucket. Amazon S3 uses this key to encrypt replica objects.
     * Amazon S3 only supports symmetric encryption KMS keys. For more information, see
     * <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/symmetric-asymmetric.html">Asymmetric
     * keys in Amazon Web Services KMS</a> in the <i>Amazon Web Services Key Management
     * Service Developer Guide</i>.</p>
     */
    inline const Aws::String& GetReplicaKmsKeyID() const { return m_replicaKmsKeyID; }
    inline bool ReplicaKmsKeyIDHasBeenSet() const { return m_replicaKmsKeyIDHasBeenSet; }
    template<typename ReplicaKmsKeyIDT = Aws::String>
    void SetReplicaKmsKeyID(ReplicaKmsKeyIDT&& value) { m_replicaKmsKeyIDHasBeenSet = true; m_replicaKmsKeyID = std::forward<ReplicaKmsKeyIDT>(value); }
    template<typename ReplicaKmsKeyIDT = Aws::String>
    EncryptionConfiguration& WithReplicaKmsKeyID(ReplicaKmsKeyIDT&& value) { SetReplicaKmsKeyID(std::forward<ReplicaKmsKeyIDT>(value)); return *this;}
    ///@}
  private:

    Aws::String m_replicaKmsKeyID;
    bool m_replicaKmsKeyIDHasBeenSet = false;
  };

} // namespace Model
} // namespace S3Crt
} // namespace Aws
