/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2024-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#pragma once

#include <sys/types.h>
#include <functional>

#include "agg_types.h"

using update_func_ptr = std::function<int()>;

enum ExitOnError
{
    CONTINUE_ON_ERROR,
    EXIT_ON_ERROR
};

// Abstract class
class FileParserFabricAbs
{
   protected:
    string m_file_name;
    time_t m_st_mtime_;   // file creation time
    time_t m_last_mtime_;
    TopologyType m_topology_type;
    ExitOnError m_exit_on_error;
    bool m_should_reparse_file;

    update_func_ptr m_update_start;
    update_func_ptr m_update_end;
    update_func_ptr m_update_failed;

    CsvParser* m_csv_parser;
    FabricGraph* m_fabric_graph_;

   public:
    FileParserFabricAbs(const string& file_name,
                        TopologyType topology_type,
                        ExitOnError exit_on_error,
                        update_func_ptr update_start,
                        update_func_ptr update_end,
                        update_func_ptr update_failed,
                        CsvParser* csv_parser,
                        FabricGraph* fabric_graph)
        : m_file_name(file_name),
          m_st_mtime_(0),
          m_last_mtime_(0),
          m_topology_type(topology_type),
          m_exit_on_error(exit_on_error),
          m_should_reparse_file(false),
          m_update_start(update_start),
          m_update_end(update_end),
          m_update_failed(update_failed),
          m_csv_parser(csv_parser),
          m_fabric_graph_(fabric_graph)
    {}

    virtual ~FileParserFabricAbs() = default;

    virtual int Load() = 0;

    bool CheckFileHasChanged();
    int OpenFile(ifstream& ifs);

    static int CheckFileExist(const string& file_name, uint8_t num_retries, uint8_t timeout);

    const char* GetFileName() { return m_file_name.c_str(); }
    ExitOnError GetExitOnError() { return m_exit_on_error; }

    int LoadFabric();
    virtual int Init() { return 0; }
};
