"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;
var _uuid = require("uuid");
var _utils = require("./repository/utils");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    this.client = client;
  }
  async build(integration, options) {
    const instance = await (0, _utils.deepCheck)(integration);
    if (!instance.ok) {
      return Promise.reject(instance.error);
    }
    const assets = await integration.getAssets();
    if (!assets.ok) {
      return Promise.reject(assets.error);
    }
    const remapped = this.remapIDs(this.getSavedObjectBundles(assets.value, options.workflows));
    const assets1 = this.addMDSReference(remapped, options.dataSource, options.dataSourceMDSId, options.dataSourceMDSLabel);
    const withDataSource = this.remapDataSource(assets1, options.indexPattern);
    const withSubstitutedQueries = this.substituteQueries(withDataSource, options.dataSource, options.tableName);
    const refs = await this.postAssets(withSubstitutedQueries);
    const builtInstance = await this.buildInstance(integration, refs, options);
    return builtInstance;
  }
  substituteSavedSearch(assets, dataSource, tableName) {
    if (!dataSource || !tableName) {
      return assets;
    }
    assets = assets.map(asset => {
      if (asset.type === 'search') {
        var _searchSource$query;
        const searchSourceMeta = asset.attributes.kibanaSavedObjectMeta.searchSourceJSON;
        let searchSource;
        try {
          searchSource = JSON.parse(searchSourceMeta);
        } catch (error) {
          console.error('Invalid JSON in searchSourceJSON:', error);
          return asset;
        }
        if ((_searchSource$query = searchSource.query) !== null && _searchSource$query !== void 0 && _searchSource$query.query && searchSource.query.language === 'SQL') {
          searchSource.query.query = searchSource.query.query.replaceAll('{table_name}', tableName);
        }
        if (searchSourceMeta.dataset.type === 's3glue') {
          asset.attributes.title = `${dataSource}.default.${tableName}`;
        }
        asset.attributes.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSource);
      }
      return asset;
    });
    return assets;
  }

  // If we have a data source or table specified, hunt for saved queries and update them with the
  // new DS/table.
  substituteQueries(assets, dataSource, tableName) {
    if (!dataSource) {
      return assets;
    }
    assets = assets.map(asset => {
      if (asset.type === 'observability-search') {
        const savedQuery = asset.attributes.savedQuery;
        if (!savedQuery.data_sources) {
          return asset;
        }
        const dataSources = JSON.parse(savedQuery.data_sources);
        for (const ds of dataSources) {
          if (ds.type !== 's3glue') {
            continue; // Nothing to do
          }
          // TODO is there a distinction between these where we should only set one? They're all
          // equivalent in every export I've seen.
          ds.name = dataSource;
          ds.label = dataSource;
          ds.value = dataSource;
        }
        savedQuery.data_sources = JSON.stringify(dataSources);
        if (savedQuery.query_lang === 'SQL' && tableName) {
          savedQuery.query = savedQuery.query.replaceAll('{table_name}', tableName);
        }
      }
      return asset;
    });
    return assets;
  }
  getSavedObjectBundles(assets, includeWorkflows) {
    return assets.filter(asset => {
      // At this stage we only care about installing bundles
      if (asset.type !== 'savedObjectBundle') {
        return false;
      }
      // If no workflows present: default to all workflows
      // Otherwise only install if workflow is present
      if (!asset.workflows || !includeWorkflows) {
        return true;
      }
      return includeWorkflows.some(w => {
        var _asset$workflows;
        return (_asset$workflows = asset.workflows) === null || _asset$workflows === void 0 ? void 0 : _asset$workflows.includes(w);
      });
    }).map(asset => asset.data).flat();
  }
  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    return assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }
      return asset;
    });
  }
  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }
      item.id = idMap.get(item.id);
      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;
        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }
        item.references[ref].id = idMap.get(refId);
      }
      return item;
    });
  }
  addMDSReference(assets, dataSource, dataSourceMDSId, dataSourceMDSLabel) {
    if (!dataSourceMDSId) {
      return assets;
    }
    return assets.map(asset => {
      // Check if the asset type is 'index-pattern' or if the title contains 'Timeline' visualization
      if (asset !== null && asset !== void 0 && asset.type && (asset.type === 'index-pattern' || asset.type === 'visualization' && asset.attributes.visState.type === 'timelion')) {
        const dataSourceIndex = asset.references.findIndex(ref => ref.type === 'data-source');
        if (dataSourceIndex !== -1) {
          // If a data-source reference exists, update it
          asset.references[dataSourceIndex] = {
            id: dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '',
            name: dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster',
            type: 'data-source'
          };
        } else {
          // If no data-source reference exists, add a new one
          asset.references.push({
            id: dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '',
            name: dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster',
            type: 'data-source'
          });
        }
      }
      if (asset.type === 'search') {
        var _asset$attributes;
        if (asset !== null && asset !== void 0 && (_asset$attributes = asset.attributes) !== null && _asset$attributes !== void 0 && (_asset$attributes = _asset$attributes.kibanaSavedObjectMeta) !== null && _asset$attributes !== void 0 && _asset$attributes.searchSourceJSON) {
          var _searchSourceJSON$que;
          const searchSourceJSON = JSON.parse(asset.attributes.kibanaSavedObjectMeta.searchSourceJSON);
          if (searchSourceJSON !== null && searchSourceJSON !== void 0 && (_searchSourceJSON$que = searchSourceJSON.query) !== null && _searchSourceJSON$que !== void 0 && (_searchSourceJSON$que = _searchSourceJSON$que.dataset) !== null && _searchSourceJSON$que !== void 0 && _searchSourceJSON$que.dataSource) {
            searchSourceJSON.query.dataset.dataSource.id = dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '';
            searchSourceJSON.query.dataset.dataSource.name = dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster';
            searchSourceJSON.query.dataset.dataSource.type = 'data-source';
          }
          asset.attributes.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
        }
      }
      return asset;
    });
  }
  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;
        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }
  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }
    const instance = {
      name: options.name,
      templateName: config.value.name,
      dataSource: options.indexPattern,
      creationDate: new Date().toISOString(),
      assets: refs
    };
    if (options.dataSourceMDSId) {
      instance.references = [{
        id: options.dataSourceMDSId || '',
        name: options.dataSourceMDSLabel || 'Local cluster',
        type: 'data-source'
      }];
    }
    return Promise.resolve(instance);
  }
}
exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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