"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileSystemDataAdaptor = void 0;
var fs = _interopRequireWildcard(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var _json = _interopRequireDefault(require("json5"));
var _utils = require("./utils");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
// Check if a location is a directory without an exception if location not found
const safeIsDirectory = async maybeDirectory => {
  try {
    return (await fs.lstat(maybeDirectory)).isDirectory();
  } catch (_err) {
    return false;
  }
};

/**
 * A CatalogDataAdaptor that reads from the local filesystem.
 * Used to read default Integrations shipped in the in-product catalog at `__data__`.
 */
class FileSystemDataAdaptor {
  /**
   * Creates a new FileSystemCatalogDataAdaptor instance.
   *
   * @param directory The base directory from which to read files. This is not sanitized.
   */
  constructor(directory) {
    _defineProperty(this, "isConfigLocalized", false);
    _defineProperty(this, "directory", void 0);
    this.directory = directory;
  }
  async readFile(filename, type) {
    let content;
    try {
      content = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename), {
        encoding: 'utf-8'
      });
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    // First try to parse as JSON, then NDJSON, then fail.
    try {
      const parsed = _json.default.parse(content);
      return {
        ok: true,
        value: parsed
      };
    } catch (err) {
      const parsed = await (0, _utils.tryParseNDJson)(content);
      if (parsed) {
        return {
          ok: true,
          value: parsed
        };
      }
      return {
        ok: false,
        error: new Error('Unable to parse file as JSON or NDJson', {
          cause: err
        })
      };
    }
  }
  async readFileRaw(filename, type) {
    try {
      const buffer = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename));
      return {
        ok: true,
        value: buffer
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async findIntegrations(dirname = '.') {
    try {
      const integrations = [];
      await this.collectIntegrationsRecursive(dirname, integrations);
      return {
        ok: true,
        value: integrations
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async collectIntegrationsRecursive(dirname, integrations) {
    const entries = await fs.readdir(_path.default.join(this.directory, dirname));
    for (const entry of entries) {
      const fullPath = _path.default.join(dirname, entry);
      const isDirectory = (await this.getDirectoryType(fullPath)) === 'integration';
      if (isDirectory) {
        integrations.push(fullPath);
      } else if ((await this.getDirectoryType(fullPath)) === 'repository') {
        await this.collectIntegrationsRecursive(fullPath, integrations);
      }
    }
  }
  async findIntegrationVersions(dirname = '.') {
    let files;
    const integPath = _path.default.join(this.directory, dirname);
    try {
      files = await fs.readdir(integPath);
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    const versions = [];
    for (const file of files) {
      // TODO handle nested repositories -- assumes integrations are 1 level deep
      if (_path.default.extname(file) === '.json' && file.startsWith(`${_path.default.basename(integPath)}-`)) {
        const version = file.substring(_path.default.basename(integPath).length + 1, file.length - 5);
        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }
        versions.push(version);
      }
    }
    return {
      ok: true,
      value: versions
    };
  }
  async getDirectoryType(dirname) {
    const isDir = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.'));
    if (!isDir) {
      return 'unknown';
    }
    // Sloppily just check for one mandatory integration directory to distinguish.
    // Improve if we need to distinguish a repository with an integration named "schemas".
    const hasSchemas = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.', 'schemas'));
    return hasSchemas ? 'integration' : 'repository';
  }
  join(filename) {
    return new FileSystemDataAdaptor(_path.default.join(this.directory, filename));
  }
}
exports.FileSystemDataAdaptor = FileSystemDataAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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