"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JwtAuthentication = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = void 0;
var _authentication_type = require("../authentication_type");
var _routes = require("./routes");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _jwt_helper = require("./jwt_helper");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = {
  cookiePrefix: 'security_authentication_jwt',
  additionalCookies: 5
};
class JwtAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    var _this$config$jwt;
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", 'jwt');
    _defineProperty(this, "authHeaderName", void 0);
    this.authHeaderName = ((_this$config$jwt = this.config.jwt) === null || _this$config$jwt === void 0 ? void 0 : _this$config$jwt.header.toLowerCase()) || 'authorization';
  }
  async init() {
    this.createExtraStorage();
    const routes = new _routes.JwtAuthRoutes(this.router, this.sessionStorageFactory, this.config);
    routes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const {
      cookiePrefix,
      additionalCookies
    } = this.getExtraAuthStorageOptions();
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= additionalCookies; i++) {
      hapiServer.states.add(cookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    var _this$config$jwt2, _this$config$jwt3;
    const extraAuthStorageOptions = {
      cookiePrefix: ((_this$config$jwt2 = this.config.jwt) === null || _this$config$jwt2 === void 0 ? void 0 : _this$config$jwt2.extra_storage.cookie_prefix) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.cookiePrefix,
      additionalCookies: ((_this$config$jwt3 = this.config.jwt) === null || _this$config$jwt3 === void 0 ? void 0 : _this$config$jwt3.extra_storage.additional_cookies) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.additionalCookies,
      logger: this.logger
    };
    return extraAuthStorageOptions;
  }
  getTokenFromUrlParam(request) {
    var _this$config$jwt4;
    const urlParamName = (_this$config$jwt4 = this.config.jwt) === null || _this$config$jwt4 === void 0 ? void 0 : _this$config$jwt4.url_param;
    if (urlParamName) {
      const token = request.url.searchParams.get(urlParamName);
      return token || undefined;
    }
    return undefined;
  }
  getBearerToken(request) {
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      return `Bearer ${token}`;
    }

    // no token in url parameter, try to get token from header
    return request.headers[this.authHeaderName] || undefined;
  }
  requestIncludesAuthInfo(request) {
    var _this$config$jwt5;
    if (request.headers[this.authHeaderName]) {
      return true;
    }
    const urlParamName = (_this$config$jwt5 = this.config.jwt) === null || _this$config$jwt5 === void 0 ? void 0 : _this$config$jwt5.url_param;
    if (urlParamName && request.url.searchParams.get(urlParamName)) {
      return true;
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    const header = {};
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      header[this.authHeaderName] = `Bearer ${token}`;
    }
    return header;
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, this.getBearerToken(request) || '', this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: (0, _jwt_helper.getExpirationDate)(this.getBearerToken(request), Date.now() + this.config.session.ttl)
    };
  }
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    const hasAuthHeaderValue = ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie);
    return cookie.authType === this.type && cookie.username && cookie.expiryTime && hasAuthHeaderValue;
  }
  getKeepAliveExpiry(cookie, request) {
    return (0, _jwt_helper.getExpirationDate)(this.buildAuthHeaderFromCookie(cookie, request)[this.authHeaderName], Date.now() + this.config.session.ttl);
  }
  handleUnauthedRequest(request, response, toolkit) {
    return response.unauthorized();
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
      }
    }
    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header[this.authHeaderName] = authHeaderValue;
    }
    return header;
  }
}
exports.JwtAuthentication = JwtAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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