import { Adapter, type TsType } from "../..";
import type { Backup } from "../../../models";
import * as ZSpec from "../../../zspec";
import type { Eui64, ExtendedPanId, PanId } from "../../../zspec/tstypes";
import * as Zcl from "../../../zspec/zcl";
import * as Zdo from "../../../zspec/zdo";
import type * as ZdoTypes from "../../../zspec/zdo/definition/tstypes";
import type { ZclPayload } from "../../events";
import { IEEE802154CcaMode } from "../enums";
import type { EmberKeyData, EmberNetworkParameters } from "../types";
export type NetworkCache = {
    eui64: Eui64;
    parameters: EmberNetworkParameters;
};
/**
 * Use for a link key backup.
 *
 * Each entry notes the EUI64 of the device it is paired to and the key data.
 *   This key may be hashed and not the actual link key currently in use.
 */
export type LinkKeyBackupData = {
    deviceEui64: Eui64;
    key: EmberKeyData;
    outgoingFrameCounter: number;
    incomingFrameCounter: number;
};
type StackConfig = {
    CONCENTRATOR_RAM_TYPE: "high" | "low";
    /**
     * Minimum Time between broadcasts (in seconds) <1-60>
     * Default: 10
     * The minimum amount of time that must pass between MTORR broadcasts.
     */
    CONCENTRATOR_MIN_TIME: number;
    /**
     * Maximum Time between broadcasts (in seconds) <30-300>
     * Default: 60
     * The maximum amount of time that can pass between MTORR broadcasts.
     */
    CONCENTRATOR_MAX_TIME: number;
    /**
     * Route Error Threshold <1-100>
     * Default: 3
     * The number of route errors that will trigger a re-broadcast of the MTORR.
     */
    CONCENTRATOR_ROUTE_ERROR_THRESHOLD: number;
    /**
     * Delivery Failure Threshold <1-100>
     * Default: 1
     * The number of APS delivery failures that will trigger a re-broadcast of the MTORR.
     */
    CONCENTRATOR_DELIVERY_FAILURE_THRESHOLD: number;
    /**
     * Maximum number of hops for Broadcast <0-30>
     * Default: 0
     * The maximum number of hops that the MTORR broadcast will be allowed to have.
     * A value of 0 will be converted to the EMBER_MAX_HOPS value set by the stack.
     */
    CONCENTRATOR_MAX_HOPS: number;
    /** <6-64> (Default: 6) @see EzspConfigId.MAX_END_DEVICE_CHILDREN */
    MAX_END_DEVICE_CHILDREN: number;
    /** <-> (Default: 10000) @see EzspValueId.TRANSIENT_DEVICE_TIMEOUT */
    TRANSIENT_DEVICE_TIMEOUT: number;
    /** <0-14> (Default: 8) @see EzspConfigId.END_DEVICE_POLL_TIMEOUT */
    END_DEVICE_POLL_TIMEOUT: number;
    /** <0-65535> (Default: 300) @see EzspConfigId.TRANSIENT_KEY_TIMEOUT_S */
    TRANSIENT_KEY_TIMEOUT_S: number;
    /**@see Ezsp.ezspSetRadioIeee802154CcaMode */
    CCA_MODE?: keyof typeof IEEE802154CcaMode;
};
/**
 * Default stack configuration values.
 * @see https://www.silabs.com/documents/public/user-guides/ug100-ezsp-reference-guide.pdf 2.3.1 for descriptions/RAM costs
 *
 * https://github.com/darkxst/silabs-firmware-builder/tree/main/manifests
 * https://github.com/NabuCasa/silabs-firmware/wiki/Zigbee-EmberZNet-NCP-firmware-configuration#skyconnect
 * https://github.com/SiliconLabs/UnifySDK/blob/main/applications/zigbeed/project_files/zigbeed.slcp
 */
export declare const DEFAULT_STACK_CONFIG: Readonly<StackConfig>;
/**
 * NOTE: This from SDK is currently ignored here because of issues in below links:
 * - BUGZID 12261: Concentrators use MTORRs for route discovery and should not enable route discovery in the APS options.
 * - https://community.silabs.com/s/question/0D58Y00008DRfDCSA1/coordinator-cant-send-unicast-to-sleepy-node-after-reboot
 * - https://community.silabs.com/s/question/0D58Y0000B4nTb7SQE/largedense-network-communication-problem-source-route-table-not-big-enough
 *
 * Removing `ENABLE_ROUTE_DISCOVERY` leads to devices that won't reconnect/go offline, and various other issues. Keeping it for now.
 */
export declare const DEFAULT_APS_OPTIONS: number;
/**
 * Relay calls between Z2M and EZSP-layer and handle any error that might occur via queue & waitress.
 *
 * Anything post `start` that requests anything from the EZSP layer must run through the request queue for proper execution flow.
 */
export declare class EmberAdapter extends Adapter {
    /** Current manufacturer code assigned to the coordinator. Used for join workarounds... */
    private manufacturerCode;
    readonly stackConfig: StackConfig;
    private ezsp;
    private version;
    private readonly queue;
    private readonly oneWaitress;
    /** Periodically retrieve counters then clear them. */
    private watchdogCountersHandle?;
    /** Sequence number used for ZDO requests. static uint8_t  */
    private zdoRequestSequence;
    private interpanLock;
    /**
     * Cached network params to avoid NCP calls. Prevents frequent EZSP transactions.
     * NOTE: Do not use directly, use getter functions for it that check if valid or need retrieval from NCP.
     */
    private networkCache;
    private multicastTable;
    constructor(networkOptions: TsType.NetworkOptions, serialPortOptions: TsType.SerialPortOptions, backupPath: string, adapterOptions: TsType.AdapterOptions);
    private loadStackConfig;
    /**
     * Emitted from @see Ezsp.ezspStackStatusHandler
     * @param status
     */
    private onStackStatus;
    /**
     * Emitted from @see Ezsp.ezspMessageSentHandler
     * WARNING: Cannot rely on `ezspMessageSentHandler` > `ezspIncomingMessageHandler` order, some devices mix it up!
     *
     * @param type
     * @param indexOrDestination
     * @param apsFrame
     * @param messageTag
     * @param status
     */
    private onMessageSent;
    /**
     * Emitted from @see Ezsp.ezspIncomingMessageHandler
     *
     * @param apsFrame The APS frame associated with the response.
     * @param sender The sender of the response. Should match `payload.nodeId` in many responses.
     * @param messageContents The content of the response.
     */
    private onZDOResponse;
    /**
     * Emitted from @see Ezsp.ezspIncomingMessageHandler @see Ezsp.ezspGpepIncomingMessageHandler
     *
     * @param type
     * @param apsFrame
     * @param lastHopLqi
     * @param sender
     * @param messageContents
     */
    private onIncomingMessage;
    /**
     * Emitted from @see Ezsp.ezspMacFilterMatchMessageHandler when the message is a valid InterPAN touchlink message.
     *
     * @param _sourcePanId
     * @param sourceAddress
     * @param groupId
     * @param lastHopLqi
     * @param messageContents
     */
    private onTouchlinkMessage;
    /**
     * Emitted from @see Ezsp.ezspTrustCenterJoinHandler
     * Also from @see Ezsp.ezspIdConflictHandler as a DEVICE_LEFT
     *
     * @param newNodeId
     * @param newNodeEui64
     * @param status
     * @param policyDecision
     * @param parentOfNewNodeId
     */
    private onTrustCenterJoin;
    private watchdogCounters;
    /**
     * Proceed to execute the long list of commands required to setup comms between Host<>NCP.
     * This is called by start and on internal reset.
     */
    private initEzsp;
    /**
     * NCP concentrator init. Also enables source route discovery mode with RESCHEDULE.
     *
     * From AN1233:
     * To function correctly in a Zigbee PRO network, a trust center also requires that:
     *
     * 1. The trust center application must act as a concentrator (either high or low RAM).
     * 2. The trust center application must have support for source routing.
     *    It must record the source routes and properly handle requests by the stack for a particular source route.
     * 3. The trust center application must use an address cache for security, in order to maintain a mapping of IEEE address to short ID.
     *
     * Failure to satisfy all of the above requirements may result in failures when joining/rejoining devices to the network across multiple hops
     * (through a target node that is neither the trust center nor one of its neighboring routers.)
     */
    private initNCPConcentrator;
    /**
     * Register fixed endpoints and set any related multicast entries that need to be.
     */
    private registerFixedEndpoints;
    /**
     *
     * @returns True if the network needed to be formed.
     */
    private initTrustCenter;
    /**
     * Form a network using given parameters.
     */
    private formNetwork;
    /**
     * Loads currently stored backup and returns it in internal backup model.
     */
    private getStoredBackup;
    /**
     * Export link keys for backup.
     *
     * @return List of keys data with AES hashed keys
     */
    exportLinkKeys(): Promise<LinkKeyBackupData[]>;
    /**
     * Import link keys from backup.
     *
     * @param backupData
     */
    importLinkKeys(backupData: LinkKeyBackupData[]): Promise<void>;
    /**
     * Routine to update the network key and broadcast the update to the network after a set time.
     * NOTE: This should run at a large interval, but before the uint32_t of the frame counter is able to reach all Fs (can't wrap to 0).
     *       This may disrupt sleepy end devices that miss the update, but they should be able to TC rejoin (in most cases...).
     *       On the other hand, the more often this runs, the more secure the network is...
     */
    broadcastNetworkKeyUpdate(): Promise<void>;
    /**
     * Received when EZSP layer alerts of a problem that needs the NCP to be reset.
     * @param status
     */
    private onNcpNeedsResetAndInit;
    /**
     * Clear the cached network values (set to invalid values).
     */
    clearNetworkCache(): void;
    /**
     * Return the EUI 64 of the local node
     * This call caches the results on the host to prevent frequent EZSP transactions.
     * Check against BLANK_EUI64 for validity.
     */
    emberGetEui64(): Promise<Eui64>;
    /**
     * Return the PAN ID of the local node.
     * This call caches the results on the host to prevent frequent EZSP transactions.
     * Check against INVALID_PAN_ID for validity.
     */
    emberGetPanId(): Promise<PanId>;
    /**
     * Return the Extended PAN ID of the local node.
     * This call caches the results on the host to prevent frequent EZSP transactions.
     * Check against BLANK_EXTENDED_PAN_ID for validity.
     */
    emberGetExtendedPanId(): Promise<ExtendedPanId>;
    /**
     * Return the radio channel (uint8_t) of the current network.
     * This call caches the results on the host to prevent frequent EZSP transactions.
     * Check against INVALID_RADIO_CHANNEL for validity.
     */
    emberGetRadioChannel(): Promise<number>;
    /**
     * Ensure the Host & NCP are aligned on protocols using version.
     * Cache the retrieved information.
     *
     * NOTE: currently throws on mismatch until support for lower versions is implemented (not planned atm)
     *
     * Does nothing if ncpNeedsResetAndInit == true.
     */
    private emberVersion;
    /**
     * This function sets an EZSP config value.
     * WARNING: Do not call for values that cannot be set after init without first resetting NCP (like table sizes).
     *          To avoid an extra NCP call, this does not check for it.
     * @param configId
     * @param value uint16_t
     * @returns
     */
    private emberSetEzspConfigValue;
    /**
     * This function sets an EZSP value.
     * @param valueId
     * @param valueLength uint8_t
     * @param value uint8_t *
     * @returns
     */
    private emberSetEzspValue;
    /**
     * This function sets an EZSP policy.
     * @param policyId
     * @param decisionId Can be bitop
     * @returns
     */
    private emberSetEzspPolicy;
    /**
     * Set the trust center policy bitmask using decision.
     * @param decision
     * @returns
     */
    private emberSetJoinPolicy;
    /**
     * ZDO
     * Get the next device request sequence number.
     *
     * Requests have sequence numbers so that they can be matched up with the
     * responses. To avoid complexities, the library uses numbers with the high
     * bit clear and the stack uses numbers with the high bit set.
     *
     * @return uint8_t The next device request sequence number
     */
    private nextZDORequestSequence;
    start(): Promise<TsType.StartResult>;
    stop(): Promise<void>;
    getCoordinatorIEEE(): Promise<string>;
    getCoordinatorVersion(): Promise<TsType.CoordinatorVersion>;
    reset(type: "soft" | "hard"): Promise<void>;
    supportsBackup(): Promise<boolean>;
    backup(_ieeeAddressesInDatabase: string[]): Promise<Backup>;
    getNetworkParameters(): Promise<TsType.NetworkParameters>;
    addInstallCode(ieeeAddress: string, key: Buffer, hashed: boolean): Promise<void>;
    /** WARNING: Adapter impl. Starts timer immediately upon returning */
    waitFor(networkAddress: number | undefined, endpoint: number, _frameType: Zcl.FrameType, _direction: Zcl.Direction, transactionSequenceNumber: number | undefined, clusterID: number, commandIdentifier: number, timeout: number): {
        promise: Promise<ZclPayload>;
        cancel: () => void;
    };
    sendZdo(ieeeAddress: string, networkAddress: number, clusterId: Zdo.ClusterId, payload: Buffer, disableResponse: true): Promise<void>;
    sendZdo<K extends keyof ZdoTypes.RequestToResponseMap>(ieeeAddress: string, networkAddress: number, clusterId: K, payload: Buffer, disableResponse: false): Promise<ZdoTypes.RequestToResponseMap[K]>;
    permitJoin(seconds: number, networkAddress?: number): Promise<void>;
    sendZclFrameToEndpoint(ieeeAddr: string, networkAddress: number, endpoint: number, zclFrame: Zcl.Frame, timeout: number, disableResponse: boolean, disableRecovery: boolean, sourceEndpoint?: number): Promise<ZclPayload | undefined>;
    sendZclFrameToGroup(groupID: number, zclFrame: Zcl.Frame, sourceEndpoint?: number): Promise<void>;
    sendZclFrameToAll(endpoint: number, zclFrame: Zcl.Frame, sourceEndpoint: number, destination: ZSpec.BroadcastAddress): Promise<void>;
    setChannelInterPAN(channel: number): Promise<void>;
    sendZclFrameInterPANToIeeeAddr(zclFrame: Zcl.Frame, ieeeAddress: string): Promise<void>;
    sendZclFrameInterPANBroadcast(zclFrame: Zcl.Frame, timeout: number): Promise<ZclPayload>;
    restoreChannelInterPAN(): Promise<void>;
    private checkInterpanLock;
}
export {};
//# sourceMappingURL=emberAdapter.d.ts.map