/* This file is part of the 'atomes' software

'atomes' is free software: you can redistribute it and/or modify it under the terms
of the GNU Affero General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.

'atomes' is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU Affero General Public License along with 'atomes'.
If not, see <https://www.gnu.org/licenses/>

Copyright (C) 2022-2024 by CNRS and University of Strasbourg */

/*!
* @file cff91.c
* @short CFF91 force field, from 'CFF91.frc'
* @author Sébastien Le Roux <sebastien.leroux@ipcms.unistra.fr>
*/

/*
* This file: 'cff91.c'
*
* Contains:
*

  CFF91 force field, data from taken from:

       - the file: 'CFF91.frc'

  This file contains several tables:

    Atoms            : CFF91_atoms[93][5]
    Equivalence      : CFF91_equivalence_auto[97][11]
    Equivalence      : CFF91_equivalence[95][7]
    Quadractic bonds : CFF91_bonds_auto[667][5]
    Quartic bonds    : CFF91_bonds[58][7]
    Quadratic angles : CFF91_angles_auto[330][6]
    Quartic angles   : CFF91_angles[155][8]
    Torsions         : CFF91_torsions_auto[216][8]
    Torsions         : CFF91_torsions[294][11]
    Inversions       : CFF91_inversions[70][7]
    Non bonded       : CFF91_vdw[40][4]
    Bond increments  : CFF91_bond_increments[560][5]

*/

#include "global.h"

int CFF91_objects[14] = {93, 97, 95, 667, 58, 0, 330, 155, 216, 294, 70, 0, 40, 560};
int CFF91_dim[14] = {5, 11, 7, 5, 7, 0, 6, 8, 8, 11, 7, 0, 4, 5};

/*
 Atoms
  0= Element, 1= Mass, 2= Key, 3= Description, 4= CFF91 info
*/
char * CFF91_atoms[93][5]= {{"C", "12.01115", "c", "generic SP3 carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c3", "sp3 carbon with 3 hHs 1 heavy", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c2", "sp3 carbon with 2 H's, 2 Heavy's", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "co", "sp3 carbon in acetals", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c3m", "sp3 carbon in 3-membered ring", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c4m", "sp3 carbon in 4-membered ring", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "coh", "sp3 carbon in acetals with hydrogen", "Ver. 2.0, Ref. 1"},
                            {"C", "12.01115", "c3h", "sp3 carbon in 3-membered ring with hydrogens", "Ver. 2.0, Ref. 2"},
                            {"C", "12.01115", "c4h", "sp3 carbon in 4-membered ring with hydrogens", "Ver. 2.0, Ref. 2"},
                            {"C", "12.01115", "c1", "sp3 carbon with 1 H 3 heavies", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "ca", "general amino acid alpha carbon (sp3)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "cg", "sp3 alpha carbon in glycine", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c=", "non aromatic end doubly bonded carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c=1", "non aromatic, next to end doubly bonded carbon", "Ver. 2.0, Ref. 3"},
                            {"C", "12.01115", "c=2", "non aromatic doubly bonded carbon", "Ver. 2.0, Ref. 3"},
                            {"C", "12.01115", "c*", "carbon in carbonyl group, non_amides", "Ver. 2.0, Ref. 4"},
                            {"C", "12.01115", "c\"", "carbon in carbonyl group, non_amides", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c'", "carbon in carbonyl group of amides", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "cp", "sp2 aromatic carbon", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c5", "sp2 aromatic carbon in 5-membered ring", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c-", "C in charged carboxylate", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "cr", "C in neutral arginine", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "c+", "C in guanidinium group", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "cs", "sp2 aromatic carbon in 5 membered ring next to S", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "ci", "sp2 aromatic carbon in charged imidazole ring (His+)", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "ct", "sp carbon involved in a triple bond", "Ver. 1.0, Ref. 1"},
                            {"C", "12.01115", "ct2", "sp carbon in CO2", "Ver. 2.0, Ref. 6"},
                            {"N", "14.00670", "na", "sp3 nitrogen in amines", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n3m", "sp3 nitrogen in 3- membered ring", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n4m", "sp3 nitrogen in 4- membered ring", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n3n", "sp2 nitrogen in 3- membered ring", "Ver. 2.0, Ref. 2"},
                            {"N", "14.00670", "n4n", "sp2 nitrogen in 4- membered ring", "Ver. 2.0, Ref. 2"},
                            {"N", "14.00670", "nb", "sp2 nitrogen in aromatic amines", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "nn", "sp2 nitrogen in aromatic amines", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n", "generic sp2 nitrogen (in amids))", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "np", "sp2 nitrogen in 5- or 6-membered ring", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "npc", "sp2 nitrogen in 5- or 6-membered ring bonded to a heavy atom", "Ver. 2.0, Ref. 2"},
                            {"N", "14.00670", "nh", "sp2 nitrogen in 5- or 6- membered ring with hydrogen attached", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "nho", "sp2 nitrogen in 6- membered ring next to a carbonyl group and with a hydrogen", "Ver. 2.0, Ref. 2"},
                            {"N", "14.00670", "nh+", "protonated nitrogen in 6- membered ring with hydrogen attached", "Ver. 2.0, Ref. 2"},
                            {"N", "14.00670", "n+", "sp3 nitrogen in protonated amines", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n4", "sp3 nitrogen in protonated amines", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "nr", "sp2 nitrogen (NH2) in guanidinium group (HN=C(NH2)2)", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n=", "non aromatic end doubly bonded nitrogen", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n=1", "non aromatic, next to end doubly bonded carbon", "Ver. 2.0, Ref. 3"},
                            {"N", "14.00670", "n=2", "non aromatic doubly bonded nitrogen", "Ver. 2.0, Ref. 3"},
                            {"N", "14.00670", "ni", "nitrogen in charged imidazole ring", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n1", "sp2 nitrogen in charged arginine", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "n2", "sp2 nitrogen (NH2) in guanidinium group (HN=C(NH2)2)", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "nt", "sp nitrogen involved in a triple bond", "Ver. 1.0, Ref. 1"},
                            {"N", "14.00670", "nz", "sp nitrogen in N2", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o", "generic SP3 oxygen", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "oh", "oxygen bonded to hydrogen", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "oc", "sp3 oxygen in ether or acetals", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "oe", "sp3 oxygen in ester", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o3e", "sp3 oxygen in three membered ring", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o4e", "sp3 oxygen in four membered ring", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o'", "oxygen in carbonyl group", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "op", "sp2 aromatic in 5 membered ring", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o*", "oxygen in water", "Ver. 1.0, Ref. 1"},
                            {"O", "15.99940", "o-", "partial double oxygen bonded to something then bonded to another partial double oxygen", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "h", "generic hydrogen bound to C, Si,or H", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "h*", "hydrogen bonded to nitrogen, Oxygen", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "h+", "charged hydrogen in cations", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hs", "hydrogen bonded to sulfur", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hc", "hydrogen bonded to carbon", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hp", "hydrogen bonded to phosphorus", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "ho", "hydrogen bonded to oxygen", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hn", "hydrogen bonded to nitrogen", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hi", "Hydrogen in charged imidazole ring", "Ver. 1.0, Ref. 1"},
                            {"H", "1.007970", "hw", "hydrogen in water", "Ver. 1.0, Ref. 1"},
                            {"D", "2.014000", "dw", "deuterium in heivy water", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "s", "sp3 sulfur", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "sc", "sp3 sulfur in methionines (C-S-C) group", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "s3e", "sulfur in three membered ring", "Ver. 2.0, Ref. 2"},
                            {"S", "32.06400", "s4e", "sulfur in four membered ring", "Ver. 2.0, Ref. 2"},
                            {"S", "32.06400", "s1", "sp3 sulfur involved in (S-S) group of disulfides", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "sh", "sp3 sulfur in sulfhydryl (-SH) group (e.g. cysteine)", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "sp", "sulfur in an aromatic ring (e.g. thiophene)", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "s'", "S in thioketone group", "Ver. 1.0, Ref. 1"},
                            {"S", "32.06400", "s-", "partial double sulfur bonded to something then bonded to another partial double oxygen or sulfur", "Ver. 1.0, Ref. 1"},
                            {"P", "30.97380", "p", "general phosphorous atom", "Ver. 1.0, Ref. 1"},
                            {"Si", "28.08600", "si", "silicon atom", "Ver. 1.0, Ref. 1"},
                            {"Ca", "40.08000", "ca+", "calcium ion", "Ver. 1.0, Ref. 1"},
                            {"F", "18.99840", "f", "fluorine atom", "Ver. 1.0, Ref. 1"},
                            {"Cl", "35.45300", "cl", "chlorine atom", "Ver. 1.0, Ref. 1"},
                            {"Cl", "35.45300", "Cl", "chlorine ion", "Ver. 1.0, Ref. 1"},
                            {"Br", "79.90900", "br", "bromine atom", "Ver. 1.0, Ref. 1"},
                            {"Br", "79.90900", "Br", "bromine ion", "Ver. 1.0, Ref. 1"},
                            {"I", "126.9044", "i", "iodine atom", "Ver. 1.0, Ref. 1"},
                            {"Na", "22.98980", "Na", "sodium ion", "Ver. 1.0, Ref. 1"},
                            {"L", "1.000000", "lp", "lone pair", "Ver. 1.0, Ref. 1"},
                            {"Ar", "39.94800", "ar", "Argon atom", "Ver. 1.0, Ref. 1"}};

/*
 Equivalence table for key symbols CFF91 auto with:
   0 = Key in atom list
   1 = Non-bonded
   2 = Bond increments
   3 = Bonds
   4 = Angles end atom
   5 = Angles appex atom
   6 = Torsion end atom
   7 = Torsion center atom
   8 = Inversion end atom
   9 = Inversion center atom
  10 = CFF91 info
*/
char * CFF91_equivalence_auto[97][11]= {{"h", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"d", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hc", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hn", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"ho", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hp", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hs", "h", "h", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"h*", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hw", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"hi", "h*", "hi", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"h+", "h+", "h+", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"dw", "h*", "h*", "h_", "h_", "h_", "h_", "h_", "h_", "h_", "Ver. 2.0, Ref. 2"},
                                         {"c", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"cg", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"ca", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c3", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"cn", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c2", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c1", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"co", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c3m", "c", "c", "c3m_", "c3m_", "c3m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c4m", "c", "c", "c4m_", "c4m_", "c4m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"coh", "c", "c", "c_", "c_", "c_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c3h", "c", "c", "c3m_", "c3m_", "c3m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"c4h", "c", "c", "c4m_", "c4m_", "c4m_", "c_", "c_", "c_", "c_", "Ver. 2.0, Ref. 2"},
                                         {"cp", "cp", "cp", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 2"},
                                         {"c5", "cp", "cp", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 2"},
                                         {"cs", "cp", "cp", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 2"},
                                         {"ci", "cp", "ci", "cp_", "c_", "cp_", "c_", "cp_", "c_", "cp_", "Ver. 2.0, Ref. 2"},
                                         {"c*", "c*", "c*", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 4"},
                                         {"c\"", "c*", "c*", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 4"},
                                         {"c'", "c'", "c'", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 2"},
                                         {"cr", "cr", "cr", "cr_", "c_", "c'_", "c_", "c=_3", "c_", "c'_", "Ver. 2.0, Ref. 2"},
                                         {"c-", "c-", "c-", "c'_", "c_", "c'_", "c_", "c'_", "c_", "c'_", "Ver. 2.0, Ref. 2"},
                                         {"c+", "c+", "c+", "cr_", "c_", "c'_", "c_", "c+_", "c_", "c'_", "Ver. 2.0, Ref. 2"},
                                         {"c=", "c=", "c=", "c=_3", "c_", "c=_", "c_", "c=_3", "c_", "c=_", "Ver. 2.0, Ref. 2"},
                                         {"c=1", "c=", "c=", "c=_1", "c_", "c=_", "c_", "c=_1", "c_", "c=_", "Ver. 2.0, Ref. 2"},
                                         {"c=2", "c=", "c=", "c=_2", "c_", "c=_", "c_", "c=_2", "c_", "c=_", "Ver. 2.0, Ref. 2"},
                                         {"ct", "c=", "ct", "ct_", "c_", "ct_", "c_", "ct_", "c_", "ct_", "Ver. 2.0, Ref. 7"},
                                         {"ct2", "c=", "ct", "ct2_", "c_", "ct2_", "c_", "ct_", "c_", "ct_", "Ver. 2.0, Ref. 6"},
                                         {"na", "na", "na", "na_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 2"},
                                         {"n3m", "na", "na", "n3m_", "n3m_", "n3m_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 2"},
                                         {"n4m", "na", "na", "n4m_", "n4m_", "n4m_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 2"},
                                         {"np", "np", "np", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"npc", "nh", "nh", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"nh", "nh", "nh", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"nho", "nh", "nh", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"nh+", "nh", "nh+", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"ni", "nh", "nh", "np_", "n_", "np_", "n_", "np_", "n_", "np_", "Ver. 2.0, Ref. 2"},
                                         {"nn", "nn", "nn", "na_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 2"},
                                         {"nb", "nn", "nn", "np_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 2"},
                                         {"n+", "n+", "n+", "n+_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 2"},
                                         {"n4", "n+", "n+", "n+_", "n_", "na_", "n_", "na_", "n_", "na_", "Ver. 2.0, Ref. 2"},
                                         {"n", "n", "n", "n_", "n_", "n_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 2"},
                                         {"n3n", "n", "n", "n3m_", "n3m_", "n3m_", "n_", "n3n_", "n_", "n_", "Ver. 2.0, Ref. 2"},
                                         {"n4n", "n", "n", "n4m_", "n4m_", "n4m_", "n_", "n_", "n_", "n_", "Ver. 2.0, Ref. 2"},
                                         {"nr", "nr", "nr", "n_", "n_", "n_", "n_", "n_", "n_", "nr_", "Ver. 2.0, Ref. 2"},
                                         {"n2", "nr", "nr", "n_", "n_", "n_", "n_", "n_", "n_", "nr_", "Ver. 2.0, Ref. 2"},
                                         {"n1", "nr", "nr", "n_", "n_", "n_", "n_", "n_", "n_", "nr_", "Ver. 2.0, Ref. 2"},
                                         {"n=", "n=", "n=", "n=_3", "n_", "n_", "n_", "n=_3", "n_", "n=_", "Ver. 2.0, Ref. 2"},
                                         {"n=1", "n=", "n=", "n=_1", "n_", "n_", "n_", "n=_1", "n_", "n=_", "Ver. 2.0, Ref. 2"},
                                         {"n=2", "n=", "n=", "n=_2", "n_", "n_", "n_", "n=_2", "n_", "n=_", "Ver. 2.0, Ref. 2"},
                                         {"nt", "nt", "nt", "nt_", "n_", "nt_", "n_", "nt_", "n_", "nt_", "Ver. 2.0, Ref. 2"},
                                         {"nz", "nz", "nz", "nz_", "n_", "nz_", "n_", "nz_", "n_", "nz_", "Ver. 2.0, Ref. 2"},
                                         {"o", "o", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"o*", "o*", "o*", "o_", "o_", "o*_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"oh", "o", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"oc", "o", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"oe", "o", "o", "o_", "o_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"o3e", "o", "o", "o3e_", "o3e_", "o3e_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"o4e", "o", "o", "o4e_", "o4e_", "o4e_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"op", "op", "op", "op_", "o_", "op_", "o_", "o_", "o_", "op_", "Ver. 2.0, Ref. 2"},
                                         {"o'", "o'", "o'", "o'_", "o'_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"o-", "o-", "o-", "o-_", "o'_", "o_", "o_", "o_", "o_", "o_", "Ver. 2.0, Ref. 2"},
                                         {"s", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"s'", "s", "s'", "s'_", "s'_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"s-", "s", "s-", "s-_", "s'_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"sc", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"s3e", "s", "s", "s3e_", "s3e_", "s3e_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"s4e", "s", "s", "s4e_", "s4e_", "s4e_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"s1", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"sh", "s", "s", "s_", "s_", "s_", "s_", "s_", "s_", "s_", "Ver. 2.0, Ref. 2"},
                                         {"sp", "sp", "sp", "sp_", "s_", "sp_", "s_", "sp_", "s_", "sp_", "Ver. 2.0, Ref. 2"},
                                         {"p", "p", "p", "p_", "p_", "p_", "p_", "p_", "p_", "p_", "Ver. 2.0, Ref. 2"},
                                         {"pz", "p", "p", "p_", "p_", "p_", "p_", "p_", "p_", "p_", "Ver. 2.0, Ref. 2"},
                                         {"f", "f", "f", "f_", "f_", "f_", "f_", "f_", "f_", "f_", "Ver. 2.0, Ref. 2"},
                                         {"i", "i", "i", "i_", "f_", "i_", "i_", "i_", "i_", "i_", "Ver. 2.0, Ref. 2"},
                                         {"cl", "cl", "cl", "cl_", "f_", "cl_", "cl_", "cl_", "cl_", "cl_", "Ver. 2.0, Ref. 2"},
                                         {"br", "br", "br", "br_", "f_", "br_", "br_", "br_", "br_", "br_", "Ver. 2.0, Ref. 2"},
                                         {"si", "si", "si", "si_", "si_", "si_", "si_", "si_", "si_", "si_", "Ver. 2.0, Ref. 2"},
                                         {"sz", "sz", "sz", "sz_", "sz_", "sz_", "sz_", "sz_", "sz_", "sz_", "Ver. 2.0, Ref. 2"},
                                         {"nu", "nu", "nu", "nu_", "nu_", "nu_", "nu_", "nu_", "nu_", "nu_", "Ver. 2.0, Ref. 2"},
                                         {"Cl", "Cl", "Cl", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Cl_", "Ver. 2.0, Ref. 2"},
                                         {"Br", "Br", "Br", "Br_", "Br_", "Br_", "Br_", "Br_", "Br_", "Br_", "Ver. 2.0, Ref. 2"},
                                         {"Na", "Na", "Na", "Na_", "Na_", "Na_", "Na_", "Na_", "Na_", "Na_", "Ver. 2.0, Ref. 2"},
                                         {"ar", "ar", "ar", "ar_", "ar_", "ar_", "ar_", "ar_", "ar_", "ar_", "Ver. 2.0, Ref. 2"},
                                         {"ca+", "ca+", "ca+", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "ca+_", "Ver. 2.0, Ref. 9"}};

/*
 Equivalence table for key symbols CFF91 with:
   0 = Key in atom list
   1 = Non-bonded
   2 = Bonds
   3 = Angles
   4 = Torsion
   5 = Inversion
   6 = CFF91 info
*/
char * CFF91_equivalence[95][7]= {{"h", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                   {"hs", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                   {"hc", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                   {"hp", "h", "h", "h", "h", "h", "Ver. 1.0, Ref. 1"},
                                   {"h*", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"hn", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"hi", "h*", "hi", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"ho", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"hw", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"h+", "h+", "h+", "h+", "h+", "h+", "Ver. 1.0, Ref. 1"},
                                   {"dw", "h*", "h*", "h*", "h*", "h*", "Ver. 1.0, Ref. 1"},
                                   {"c", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"c3", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"c2", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"co", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"c3m", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"c4m", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"coh", "c", "c", "c", "c", "c", "Ver. 2.0, Ref. 2"},
                                   {"c3h", "c", "c", "c", "c", "c", "Ver. 2.0, Ref. 2"},
                                   {"c4h", "c", "c", "c", "c", "c", "Ver. 2.0, Ref. 2"},
                                   {"c1", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"ca", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"cg", "c", "c", "c", "c", "c", "Ver. 1.0, Ref. 1"},
                                   {"c=", "c=", "c=", "c=", "c=", "c=", "Ver. 1.0, Ref. 1"},
                                   {"c=1", "c=", "c=1", "c=", "c=1", "c=", "Ver. 2.0, Ref. 3"},
                                   {"c=2", "c=", "c=2", "c=", "c=2", "c=", "Ver. 2.0, Ref. 3"},
                                   {"c*", "c*", "c*", "c*", "c*", "c*", "Ver. 2.0, Ref. 4"},
                                   {"c\"", "c*", "c*", "c*", "c*", "c*", "Ver. 2.0, Ref. 4"},
                                   {"c'", "c'", "c'", "c'", "c'", "c'", "Ver. 1.0, Ref. 1"},
                                   {"cp", "cp", "cp", "cp", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                   {"c5", "cp", "cp", "cp", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                   {"cs", "cp", "cp", "cp", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                   {"ci", "cp", "cp", "cp", "cp", "cp", "Ver. 1.0, Ref. 1"},
                                   {"cr", "cr", "cr", "cr", "cr", "cr", "Ver. 1.0, Ref. 1"},
                                   {"c+", "c+", "c+", "c+", "c+", "c+", "Ver. 1.0, Ref. 1"},
                                   {"c-", "c-", "c-", "c-", "c-", "c-", "Ver. 1.0, Ref. 1"},
                                   {"ct", "c=", "ct", "ct", "ct", "ct", "Ver. 2.0, Ref. 7"},
                                   {"ct2", "c=", "ct2", "ct2", "ct2", "ct2", "Ver. 2.0, Ref. 6"},
                                   {"n", "n", "n", "n", "n", "n", "Ver. 1.0, Ref. 1"},
                                   {"n3n", "n", "n", "n", "n", "n", "Ver. 2.0, Ref. 2"},
                                   {"n4n", "n", "n", "n", "n", "n", "Ver. 2.0, Ref. 2"},
                                   {"na", "na", "na", "na", "na", "na", "Ver. 1.0, Ref. 1"},
                                   {"n3m", "na", "na", "na", "na", "na", "Ver. 1.0, Ref. 1"},
                                   {"n4m", "na", "na", "na", "na", "na", "Ver. 1.0, Ref. 1"},
                                   {"nn", "nn", "nn", "nn", "nn", "nn", "Ver. 1.0, Ref. 1"},
                                   {"nb", "nn", "nn", "nn", "nn", "nn", "Ver. 1.0, Ref. 1"},
                                   {"n+", "n+", "n+", "n+", "n+", "n+", "Ver. 1.0, Ref. 1"},
                                   {"n4", "n+", "n+", "n+", "n+", "n+", "Ver. 1.0, Ref. 1"},
                                   {"np", "np", "np", "np", "np", "np", "Ver. 1.0, Ref. 1"},
                                   {"npc", "nh", "nh", "nh", "nh", "nh", "Ver. 2.0, Ref. 2"},
                                   {"nh", "nh", "nh", "nh", "nh", "nh", "Ver. 1.0, Ref. 1"},
                                   {"nho", "nh", "nh", "nh", "nh", "nh", "Ver. 2.0, Ref. 2"},
                                   {"nh+", "nh", "nh+", "nh", "nh", "nh", "Ver. 2.0, Ref. 2"},
                                   {"nr", "nr", "nr", "nr", "nr", "nr", "Ver. 1.0, Ref. 1"},
                                   {"n2", "nr", "nr", "nr", "nr", "nr", "Ver. 1.0, Ref. 1"},
                                   {"n=", "n=", "n=", "n=", "n=", "n=", "Ver. 1.0, Ref. 1"},
                                   {"n=1", "n=", "n=1", "n=", "n=1", "n=", "Ver. 1.0, Ref. 1"},
                                   {"n=2", "n=", "n=2", "n=", "n=2", "n=", "Ver. 1.0, Ref. 1"},
                                   {"n1", "nr", "nr", "nr", "nr", "nr", "Ver. 1.0, Ref. 1"},
                                   {"ni", "nh", "nh", "nh", "nh", "nh", "Ver. 1.0, Ref. 1"},
                                   {"nt", "n=", "nt", "nt", "nt", "nt", "Ver. 2.0, Ref. 5"},
                                   {"nz", "n=", "nz", "nz", "nz", "nz", "Ver. 1.0, Ref. 10"},
                                   {"o'", "o'", "o'", "o'", "o'", "o'", "Ver. 1.0, Ref. 1"},
                                   {"op", "op", "op", "op", "op", "op", "Ver. 1.0, Ref. 1"},
                                   {"o", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"oc", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"oe", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"o3e", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"o4e", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"o-", "o-", "o-", "o-", "o-", "o-", "Ver. 1.0, Ref. 1"},
                                   {"o*", "o*", "o*", "o*", "o*", "o*", "Ver. 1.0, Ref. 1"},
                                   {"oh", "o", "o", "o", "o", "o", "Ver. 1.0, Ref. 1"},
                                   {"s", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                   {"sc", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                   {"s3e", "s", "s", "s", "s", "s", "Ver. 2.0, Ref. 2"},
                                   {"s4e", "s", "s", "s", "s", "s", "Ver. 2.0, Ref. 2"},
                                   {"s1", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                   {"sh", "s", "s", "s", "s", "s", "Ver. 1.0, Ref. 1"},
                                   {"sp", "sp", "sp", "sp", "sp", "sp", "Ver. 1.0, Ref. 1"},
                                   {"s'", "sp", "s'", "s'", "s'", "s'", "Ver. 1.0, Ref. 11"},
                                   {"s-", "sp", "s-", "s-", "s-", "s-", "Ver. 1.0, Ref. 11"},
                                   {"p", "p", "p", "p", "p", "p", "Ver. 1.0, Ref. 1"},
                                   {"pz", "p", "p", "p", "p", "p", "Ver. 1.0, Ref. 1"},
                                   {"f", "f", "f", "f", "f", "f", "Ver. 1.0, Ref. 1"},
                                   {"i", "i", "i", "i", "i", "i", "Ver. 1.0, Ref. 1"},
                                   {"cl", "cl", "cl", "cl", "cl", "cl", "Ver. 1.0, Ref. 1"},
                                   {"br", "br", "br", "br", "br", "br", "Ver. 1.0, Ref. 1"},
                                   {"si", "si", "si", "si", "si", "si", "Ver. 1.0, Ref. 1"},
                                   {"sz", "sz", "sz", "sz", "sz", "sz", "Ver. 1.0, Ref. 1"},
                                   {"nu", "nu", "nu", "nu", "nu", "nu", "Ver. 1.0, Ref. 1"},
                                   {"Cl", "Cl", "Cl", "Cl", "Cl", "Cl", "Ver. 1.0, Ref. 1"},
                                   {"Br", "Br", "Br", "Br", "Br", "Br", "Ver. 1.0, Ref. 1"},
                                   {"Na", "Na", "Na", "Na", "Na", "Na", "Ver. 1.0, Ref. 1"},
                                   {"ar", "ar", "ar", "ar", "ar", "ar", "Ver. 1.0, Ref. 1"},
                                   {"ca+", "ca+", "ca+", "ca+", "ca+", "ca+", "Ver. 2.0, Ref. 9"}};

/*
 Quadratic bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= K2 (kcal mol-1), 4= CFF91 info

  E = K2 * (R - R0)^2
*/
char * CFF91_bonds_auto[667][5]= {{"c3m_", "c3m_", "1.5100", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "c4m_", "1.5260", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "ct_", "1.4000", "340.000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "ct2_", "1.4000", "340.0000", "Ver. 2.0, Ref. 6"},
                                  {"c3m_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "n3m_", "1.4850", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "n4m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c3m_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "o3e_", "1.4340", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "o4e_", "1.4250", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "s3e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "s4e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 2"},
                                  {"c3m_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "c4m_", "1.5520", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "ct_", "1.4000", "340.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "ct2_", "1.4000", "340.0000", "Ver. 2.0, Ref. 6"},
                                  {"c4m_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "n3m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "n4m_", "1.4670", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c4m_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "o3e_", "1.4250", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "o4e_", "1.4462", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "s3e_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "s4e_", "1.8470", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 2"},
                                  {"c4m_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "n3m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c_", "n4m_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "n3m_", "1.4460", "272.0000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "n4m_", "1.4000", "332.0000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "s3e_", "1.7700", "175.0035", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "s4e_", "1.7700", "175.0035", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n3m_", "1.4200", "280.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n4m_", "1.4200", "280.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "s3e", "1.7300", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "s4e", "1.7300", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c=", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 2"},
                                  {"c=", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 2"},
                                  {"c=_1", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "n3m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "n4m_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 2"},
                                  {"c=", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 2"},
                                  {"c=_1", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "s3e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "s4e_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"ct_", "n3m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 2"},
                                  {"ct2_", "n3m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 6"},
                                  {"ct_", "n4m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 2"},
                                  {"ct2_", "n4m_", "1.3820", "286.8096", "Ver. 2.0, Ref. 6"},
                                  {"ct_", "s3e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 2"},
                                  {"ct2_", "s3e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 6"},
                                  {"ct_", "s4e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 2"},
                                  {"ct2_", "s4e_", "1.7200", "271.4328", "Ver. 2.0, Ref. 6"},
                                  {"na_", "n3m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"na_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"na_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"na_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "n3m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n3m_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n3m_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n3m_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 2"},
                                  {"n3m_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "n4m_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n4m_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n4m_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"n4m_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "s3e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "s4e_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 2"},
                                  {"n4m_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 2"},
                                  {"n_", "s3e_", "1.7050", "210.6208", "Ver. 2.0, Ref. 2"},
                                  {"n_", "s4e_", "1.7050", "210.6208", "Ver. 2.0, Ref. 2"},
                                  {"np_", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"np_", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"n=", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"n=", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "s3e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "s4e_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"o_", "s3e_", "1.6930", "288.0848", "Ver. 2.0, Ref. 2"},
                                  {"o_", "s4e_", "1.6930", "288.0848", "Ver. 2.0, Ref. 2"},
                                  {"op_", "s3e_", "1.6800", "266.7748", "Ver. 2.0, Ref. 2"},
                                  {"op_", "s4e_", "1.6800", "266.7748", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "s3e_", "1.6500", "239.5744", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "s4e_", "1.6500", "239.5744", "Ver. 2.0, Ref. 2"},
                                  {"s_", "s3e_", "2.1000", "160.0000", "Ver. 2.0, Ref. 2"},
                                  {"s_", "s4e_", "2.1000", "160.0000", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "s3e_", "2.0400", "175.1260", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "s4e_", "2.0400", "175.1260", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "s3e_", "2.0400", "189.9260", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "s4e_", "2.0400", "189.9260", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "s3e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "s4e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 2"},
                                  {"s3e_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "s4e_", "2.0000", "180.0000", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 2"},
                                  {"s4e_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 2"},
                                  {"c_", "c_", "1.5260", "322.7158", "Ver. 2.0, Ref. 2"},
                                  {"c_", "c'_", "1.5200", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c_", "cp_", "1.5100", "283.0924", "Ver. 2.0, Ref. 2"},
                                  {"c_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "ct_", "1.4000", "340.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "ct2_", "1.4000", "340.0000", "Ver. 2.0, Ref. 6"},
                                  {"c_", "na_", "1.4700", "356.5988", "Ver. 2.0, Ref. 2"},
                                  {"c_", "n_", "1.4600", "377.5752", "Ver. 2.0, Ref. 2"},
                                  {"c_", "np_", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "n=", "1.4750", "336.8000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "n=_1", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "n=_2", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "n=_3", "1.4750", "336.8000", "Ver. 2.0, Ref. 3"},
                                  {"c_", "n+_", "1.4620", "270.8836", "Ver. 2.0, Ref. 2"},
                                  {"c_", "o_", "1.4250", "273.2000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "op_", "1.3800", "346.5484", "Ver. 2.0, Ref. 2"},
                                  {"c_", "o'_", "1.3800", "318.9484", "Ver. 2.0, Ref. 2"},
                                  {"c_", "s_", "1.8000", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "sp_", "1.7700", "242.5324", "Ver. 2.0, Ref. 2"},
                                  {"c_", "s'_", "1.7700", "257.3324", "Ver. 2.0, Ref. 2"},
                                  {"c_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 2"},
                                  {"c_", "p_", "1.7500", "249.1344", "Ver. 2.0, Ref. 2"},
                                  {"c_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "si_", "1.8090", "238.0000", "Ver. 2.0, Ref. 2"},
                                  {"c_", "i_", "2.1200", "200.0000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "c'_", "1.5000", "266.4000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "cp_", "1.5000", "284.3316", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "ct_", "1.4200", "311.1492", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "ct2_", "1.4200", "311.1492", "Ver. 2.0, Ref. 6"},
                                  {"c'_", "n_", "1.3600", "388.0000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "n=", "1.4050", "296.2996", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "n=_1", "1.4050", "296.2996", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "n=_2", "1.4050", "296.2996", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "n=_3", "1.4050", "296.2996", "Ver. 2.0, Ref. 3"},
                                  {"c'_", "np_", "1.4050", "296.2996", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "o_", "1.3400", "400.0000", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "o'_", "1.2200", "615.3220", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "o-_", "1.2500", "572.8860", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "op_", "1.3500", "294.1008", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "s_", "1.7700", "175.0035", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "s'_", "1.6110", "510.2775", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "s-_", "1.6800", "280.3060", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "sp_", "1.7400", "215.3532", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "h_", "1.1050", "340.6175", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "p_", "1.7200", "241.3820", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "f_", "1.3900", "217.7092", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "cl_", "1.7270", "249.8588", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "br_", "1.8750", "228.2808", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "si_", "1.8300", "239.3552", "Ver. 2.0, Ref. 2"},
                                  {"c'_", "i_", "2.0610", "208.6024", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "cp_", "1.3900", "480.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "c=", "1.5000", "322.8000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "c=_1", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "c=_2", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "c=_3", "1.5000", "322.8000", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "ct_", "1.4000", "321.6716", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "ct2_", "1.4000", "321.6716", "Ver. 2.0, Ref. 6"},
                                  {"cp_", "na_", "1.4120", "257.7752", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n_", "1.4200", "280.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n=", "1.3850", "316.0380", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n=_1", "1.3850", "316.0380", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "n=_2", "1.3850", "316.0380", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "n=_3", "1.3850", "316.0380", "Ver. 2.0, Ref. 3"},
                                  {"cp_", "np_", "1.3500", "440.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "n+_", "1.4120", "251.3344", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "o_", "1.3700", "384.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "o'_", "1.3300", "297.4852", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "op_", "1.3700", "420.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "s_", "1.7300", "228.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "s'_", "1.7200", "239.8024", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "sp_", "1.7106", "320.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "h_", "1.0800", "363.4164", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "p_", "1.7000", "235.0428", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "f_", "1.3630", "496.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "cl_", "1.7610", "314.0000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "br_", "1.9200", "223.6000", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "i_", "2.0410", "217.4512", "Ver. 2.0, Ref. 2"},
                                  {"cp_", "si_", "1.8100", "225.1676", "Ver. 2.0, Ref. 2"},
                                  {"ci", "ci", "1.3900", "280.0000", "Ver. 2.0, Ref. 2"},
                                  {"ci", "ni", "1.3800", "320.0000", "Ver. 2.0, Ref. 2"},
                                  {"ci", "h_", "1.0800", "363.4164", "Ver. 2.0, Ref. 2"},
                                  {"c=", "c=", "1.3300", "655.2000", "Ver. 2.0, Ref. 2"},
                                  {"c=_3", "c=_3", "1.3300", "655.2000", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "c=_3", "1.3300", "655.2000", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "c=_2", "1.4100", "480.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "c=_2", "1.4800", "320.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "c=_1", "1.4800", "320.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "c=_3", "1.4800", "320.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 2"},
                                  {"c=", "ct2_", "1.3300", "655.2000", "Ver. 2.0, Ref. 6"},
                                  {"c=", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 2"},
                                  {"c=", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 2"},
                                  {"c=_1", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "ct2_", "1.4250", "337.6424", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "ct2_", "1.3300", "655.2000", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "ct_", "1.4250", "337.6424", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "ct2_", "1.3300", "655.2000", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "na_", "1.4370", "273.7168", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "n_", "1.4100", "279.0740", "Ver. 2.0, Ref. 3"},
                                  {"c=", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 2"},
                                  {"c=_3", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "n=_1", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n=_2", "1.3430", "493.5268", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "n=_2", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n=_1", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "n=_1", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "n=_2", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "n=_3", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 2"},
                                  {"c=", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 2"},
                                  {"c=", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 2"},
                                  {"c=", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 2"},
                                  {"c=", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 2"},
                                  {"c=", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 2"},
                                  {"c=", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 2"},
                                  {"c=", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 2"},
                                  {"c=", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 2"},
                                  {"c=", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 2"},
                                  {"c=", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 2"},
                                  {"c=", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 2"},
                                  {"c=", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 2"},
                                  {"c=", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 2"},
                                  {"c=_1", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 3"},
                                  {"c=_1", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 3"},
                                  {"c=_2", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "np_", "1.4100", "331.8740", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "o_", "1.3680", "355.1988", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "op_", "1.3550", "340.5116", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "o'_", "1.3550", "312.9116", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "h_", "1.0900", "361.6000", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "p_", "1.7250", "250.9988", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "s_", "1.7750", "254.9440", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "sp_", "1.7450", "240.9880", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "s'_", "1.7450", "255.7880", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "f_", "1.3950", "285.1320", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "cl_", "1.7320", "278.5132", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "br_", "1.8800", "253.7008", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "i_", "2.0660", "233.4432", "Ver. 2.0, Ref. 3"},
                                  {"c=_3", "si_", "1.8350", "241.0424", "Ver. 2.0, Ref. 3"},
                                  {"cr_", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 2"},
                                  {"cr_", "n=_1", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"cr_", "n=_2", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"cr_", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 3"},
                                  {"cr_", "n_", "1.3200", "388.0000", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "ct_", "1.2040", "800.0000", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "nt_", "1.1580", "880.0000", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "na_", "1.3820", "286.8096", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "n_", "1.3550", "289.4448", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "n=", "1.3550", "342.2448", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "n=_1", "1.3550", "342.2448", "Ver. 2.0, Ref. 3"},
                                  {"ct_", "n=_2", "1.3550", "342.2448", "Ver. 2.0, Ref. 3"},
                                  {"ct_", "n=_3", "1.3550", "342.2448", "Ver. 2.0, Ref. 3"},
                                  {"ct_", "n+_", "1.3820", "278.3768", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "o_", "1.3130", "367.8164", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "s_", "1.7200", "271.4328", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "op_", "1.3000", "346.4680", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "o'_", "1.3000", "318.8680", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "sp_", "1.6900", "256.0344", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "s'_", "1.6900", "270.8344", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "p_", "1.6700", "273.4668", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "h_", "1.0530", "316.9016", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "f_", "1.3400", "278.3932", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "cl_", "1.6770", "292.1952", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "br_", "1.8250", "268.8528", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "i_", "2.0110", "248.9268", "Ver. 2.0, Ref. 2"},
                                  {"ct_", "si_", "1.7800", "267.2964", "Ver. 2.0, Ref. 2"},
                                  {"ct2_", "ct2_", "1.2040", "800.0000", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "nt_", "1.1580", "880.0000", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "na_", "1.3820", "286.8096", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n_", "1.3550", "289.4448", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n=", "1.2600", "560.0000", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n=_1", "1.3550", "342.2448", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n=_2", "1.2600", "560.0000", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n=_3", "1.2600", "560.0000", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "n+_", "1.3820", "278.3768", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "o_", "1.3130", "367.8164", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "s_", "1.7200", "271.4328", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "op_", "1.3000", "346.4680", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "o'_", "1.1600", "615.3220", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "sp_", "1.6900", "256.0344", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "s'_", "1.6000", "510.2775", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "p_", "1.6700", "273.4668", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "h_", "1.0530", "316.9016", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "f_", "1.3400", "278.3932", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "cl_", "1.6770", "292.1952", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "br_", "1.8250", "268.8528", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "i_", "2.0110", "248.9268", "Ver. 2.0, Ref. 6"},
                                  {"ct2_", "si_", "1.7800", "267.2964", "Ver. 2.0, Ref. 6"},
                                  {"na_", "na_", "1.3940", "220.8000", "Ver. 2.0, Ref. 2"},
                                  {"na_", "n_", "1.3670", "221.6968", "Ver. 2.0, Ref. 2"},
                                  {"na_", "np_", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"na_", "n=", "1.3670", "274.4968", "Ver. 2.0, Ref. 2"},
                                  {"na_", "n=_1", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"na_", "n=_2", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"na_", "n=_3", "1.3670", "274.4968", "Ver. 2.0, Ref. 3"},
                                  {"na_", "n+_", "1.3940", "211.1592", "Ver. 2.0, Ref. 2"},
                                  {"na_", "o_", "1.3250", "301.3500", "Ver. 2.0, Ref. 2"},
                                  {"na_", "s_", "1.7320", "206.9404", "Ver. 2.0, Ref. 2"},
                                  {"na_", "op_", "1.3120", "274.1760", "Ver. 2.0, Ref. 2"},
                                  {"na_", "o'_", "1.3120", "246.5760", "Ver. 2.0, Ref. 2"},
                                  {"na_", "sp_", "1.7020", "190.9752", "Ver. 2.0, Ref. 2"},
                                  {"na_", "s'_", "1.7020", "205.7752", "Ver. 2.0, Ref. 2"},
                                  {"na_", "p_", "1.6820", "210.5400", "Ver. 2.0, Ref. 2"},
                                  {"na_", "h_", "1.0260", "457.4592", "Ver. 2.0, Ref. 2"},
                                  {"na_", "f_", "1.3520", "200.9852", "Ver. 2.0, Ref. 2"},
                                  {"na_", "cl_", "1.6890", "226.4260", "Ver. 2.0, Ref. 2"},
                                  {"na_", "br_", "1.8370", "203.8340", "Ver. 2.0, Ref. 2"},
                                  {"na_", "i_", "2.0230", "184.0104", "Ver. 2.0, Ref. 2"},
                                  {"na_", "si_", "1.7920", "204.4236", "Ver. 2.0, Ref. 2"},
                                  {"n_", "n_", "1.3400", "220.0000", "Ver. 2.0, Ref. 2"},
                                  {"n_", "np_", "1.3400", "272.8000", "Ver. 2.0, Ref. 2"},
                                  {"n_", "n=", "1.3400", "272.8000", "Ver. 2.0, Ref. 2"},
                                  {"n_", "n=_1", "1.3400", "272.8000", "Ver. 2.0, Ref. 3"},
                                  {"n_", "n=_2", "1.3400", "272.8000", "Ver. 2.0, Ref. 3"},
                                  {"n_", "n=_3", "1.3400", "272.8000", "Ver. 2.0, Ref. 3"},
                                  {"n_", "n+_", "1.3670", "210.6360", "Ver. 2.0, Ref. 2"},
                                  {"n_", "o_", "1.2980", "301.2632", "Ver. 2.0, Ref. 2"},
                                  {"n_", "s_", "1.7050", "210.6208", "Ver. 2.0, Ref. 2"},
                                  {"n_", "op_", "1.2850", "267.1080", "Ver. 2.0, Ref. 2"},
                                  {"n_", "o'_", "1.2850", "239.5080", "Ver. 2.0, Ref. 2"},
                                  {"n_", "sp_", "1.6750", "193.4844", "Ver. 2.0, Ref. 2"},
                                  {"n_", "s'_", "1.6750", "208.2844", "Ver. 2.0, Ref. 2"},
                                  {"n_", "p_", "1.6550", "219.9524", "Ver. 2.0, Ref. 2"},
                                  {"n_", "h_", "1.0260", "483.4512", "Ver. 2.0, Ref. 2"},
                                  {"n_", "f_", "1.3250", "189.3856", "Ver. 2.0, Ref. 2"},
                                  {"n_", "cl_", "1.6620", "227.5604", "Ver. 2.0, Ref. 2"},
                                  {"n_", "br_", "1.8100", "206.3980", "Ver. 2.0, Ref. 2"},
                                  {"n_", "i_", "1.9960", "186.6972", "Ver. 2.0, Ref. 2"},
                                  {"n_", "si_", "1.7650", "216.8064", "Ver. 2.0, Ref. 2"},
                                  {"np_", "np_", "1.3400", "408.0000", "Ver. 2.0, Ref. 2"},
                                  {"np_", "n=", "1.3400", "325.6000", "Ver. 2.0, Ref. 2"},
                                  {"np_", "n=_1", "1.3400", "325.6000", "Ver. 2.0, Ref. 3"},
                                  {"np_", "n=_2", "1.3400", "325.6000", "Ver. 2.0, Ref. 3"},
                                  {"np_", "n=_3", "1.3400", "325.6000", "Ver. 2.0, Ref. 3"},
                                  {"np_", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 2"},
                                  {"np_", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 2"},
                                  {"np_", "o'_", "1.2850", "292.3080", "Ver. 2.0, Ref. 2"},
                                  {"np_", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 2"},
                                  {"np_", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 2"},
                                  {"np_", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"np_", "s'_", "1.6750", "261.0844", "Ver. 2.0, Ref. 2"},
                                  {"np_", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 2"},
                                  {"np_", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 2"},
                                  {"np_", "h_", "1.0260", "483.4512", "Ver. 2.0, Ref. 2"},
                                  {"np_", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 2"},
                                  {"np_", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 2"},
                                  {"np_", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 2"},
                                  {"np_", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 2"},
                                  {"np_", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 2"},
                                  {"n=", "n=", "1.2100", "651.2000", "Ver. 2.0, Ref. 2"},
                                  {"n=_3", "n=_3", "1.2100", "651.2000", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "n=_3", "1.2100", "651.2000", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "n=_2", "1.2760", "488.0000", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "n=_2", "1.3400", "325.6000", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "n=_1", "1.3400", "325.6000", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "n=_3", "1.3400", "325.6000", "Ver. 2.0, Ref. 2"},
                                  {"n=", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 2"},
                                  {"n=", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "n+_", "1.3670", "263.4360", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "o_", "1.2980", "354.0632", "Ver. 2.0, Ref. 3"},
                                  {"n=", "o'_", "1.1600", "575.8720", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "o'_", "1.2850", "292.3080", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "o'_", "1.2225", "434.0900", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "o'_", "1.1600", "575.8720", "Ver. 2.0, Ref. 3"},
                                  {"n=", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "o-_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "op_", "1.2850", "319.9080", "Ver. 2.0, Ref. 3"},
                                  {"n=", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "s_", "1.7050", "263.4208", "Ver. 2.0, Ref. 2"},
                                  {"n=", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "sp_", "1.6750", "246.2844", "Ver. 2.0, Ref. 2"},
                                  {"n=", "s'_", "1.5900", "489.2400", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "s'_", "1.6750", "261.0844", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "s'_", "1.6325", "375.1624", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "s'_", "1.5900", "489.2400", "Ver. 2.0, Ref. 3"},
                                  {"n=", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 2"},
                                  {"n=", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 2"},
                                  {"n=", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 2"},
                                  {"n=", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 2"},
                                  {"n=", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 2"},
                                  {"n=", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 2"},
                                  {"n=", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 2"},
                                  {"n=_1", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 3"},
                                  {"n=_1", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 3"},
                                  {"n=_2", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "p_", "1.6550", "272.7524", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "h_", "1.0380", "551.2061", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "f_", "1.3250", "242.1856", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "cl_", "1.6620", "280.3604", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "br_", "1.8100", "259.1980", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "i_", "1.9960", "239.4972", "Ver. 2.0, Ref. 3"},
                                  {"n=_3", "si_", "1.7650", "269.6064", "Ver. 2.0, Ref. 3"},
                                  {"n+_", "n+_", "1.3940", "193.5604", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "o_", "1.3250", "291.0020", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "s_", "1.7320", "200.1168", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "op_", "1.3120", "261.7060", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "o'_", "1.3120", "234.1060", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "sp_", "1.7020", "183.2952", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "s'_", "1.7020", "198.0952", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "p_", "1.6820", "209.1536", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "h_", "1.0650", "461.1848", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "f_", "1.3520", "178.0372", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "cl_", "1.6890", "217.7248", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "br_", "1.8370", "196.5940", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "i_", "2.0230", "177.2764", "Ver. 2.0, Ref. 2"},
                                  {"n+_", "si_", "1.7920", "206.9412", "Ver. 2.0, Ref. 2"},
                                  {"nz_", "nz_", "1.0976", "1632.4955", "Ver. 2.0, Ref. 2"},
                                  {"nt_", "nt_", "1.0976", "1632.4955", "Ver. 2.0, Ref. 2"},
                                  {"o_", "o_", "1.2080", "833.6868", "Ver. 2.0, Ref. 2"},
                                  {"o_", "s_", "1.6930", "288.0848", "Ver. 2.0, Ref. 2"},
                                  {"o_", "op_", "1.2430", "350.7720", "Ver. 2.0, Ref. 2"},
                                  {"o_", "o'_", "1.2430", "323.1720", "Ver. 2.0, Ref. 2"},
                                  {"o_", "sp_", "1.6330", "271.9584", "Ver. 2.0, Ref. 2"},
                                  {"o_", "s'_", "1.6330", "286.7584", "Ver. 2.0, Ref. 2"},
                                  {"o_", "p_", "1.6100", "245.2000", "Ver. 2.0, Ref. 2"},
                                  {"o_", "si_", "1.6650", "392.8000", "Ver. 2.0, Ref. 2"},
                                  {"o_", "h_", "0.9600", "493.8480", "Ver. 2.0, Ref. 2"},
                                  {"o_", "f_", "1.4180", "224.0000", "Ver. 2.0, Ref. 2"},
                                  {"o_", "cl_", "1.6500", "307.0632", "Ver. 2.0, Ref. 2"},
                                  {"o_", "br_", "1.7980", "284.7988", "Ver. 2.0, Ref. 2"},
                                  {"o_", "i_", "1.9840", "264.9868", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "o'_", "1.1100", "484.8000", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "op_", "1.2300", "272.4000", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "s_", "1.5900", "360.4188", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "sp_", "1.5600", "341.2736", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "s'_", "1.5400", "421.5188", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "p_", "1.4800", "524.0000", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "h_", "0.9830", "458.4610", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "f_", "1.2700", "204.0505", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "cl_", "1.6070", "251.7939", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "br_", "1.7550", "233.2954", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "i_", "1.9410", "213.2317", "Ver. 2.0, Ref. 2"},
                                  {"o'_", "si_", "1.6500", "454.7464", "Ver. 2.0, Ref. 2"},
                                  {"op_", "op_", "1.2300", "300.0000", "Ver. 2.0, Ref. 2"},
                                  {"op_", "s_", "1.6800", "266.7748", "Ver. 2.0, Ref. 2"},
                                  {"op_", "sp_", "1.6500", "247.8440", "Ver. 2.0, Ref. 2"},
                                  {"op_", "s'_", "1.6200", "262.9744", "Ver. 2.0, Ref. 2"},
                                  {"op_", "p_", "1.6300", "286.0904", "Ver. 2.0, Ref. 2"},
                                  {"op_", "h_", "0.9830", "486.0610", "Ver. 2.0, Ref. 2"},
                                  {"op_", "f_", "1.2700", "231.6505", "Ver. 2.0, Ref. 2"},
                                  {"op_", "cl_", "1.6070", "279.3939", "Ver. 2.0, Ref. 2"},
                                  {"op_", "br_", "1.7550", "260.8954", "Ver. 2.0, Ref. 2"},
                                  {"op_", "i_", "1.9410", "240.8317", "Ver. 2.0, Ref. 2"},
                                  {"op_", "si", "1.5870", "292.2400", "Ver. 2.0, Ref. 2"},
                                  {"o-_", "p_", "1.4800", "428.0000", "Ver. 2.0, Ref. 2"},
                                  {"s_", "s_", "2.0547", "180.0000", "Ver. 2.0, Ref. 2"},
                                  {"s_", "sp_", "2.0400", "175.1260", "Ver. 2.0, Ref. 2"},
                                  {"s_", "s'_", "2.0400", "189.9260", "Ver. 2.0, Ref. 2"},
                                  {"s_", "p_", "2.0200", "186.8792", "Ver. 2.0, Ref. 2"},
                                  {"s_", "h_", "1.3300", "274.1288", "Ver. 2.0, Ref. 2"},
                                  {"s_", "f_", "1.6900", "204.8184", "Ver. 2.0, Ref. 2"},
                                  {"s_", "cl_", "2.0270", "212.0812", "Ver. 2.0, Ref. 2"},
                                  {"s_", "br_", "2.1750", "187.8836", "Ver. 2.0, Ref. 2"},
                                  {"s_", "i_", "2.3610", "167.7624", "Ver. 2.0, Ref. 2"},
                                  {"s_", "si_", "2.1300", "177.2928", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "s'_", "1.9700", "320.0000", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "p_", "1.9700", "255.2524", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "h_", "1.3730", "316.8138", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "f_", "1.6600", "195.3021", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "cl_", "1.9970", "211.1368", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "br_", "2.1450", "187.6017", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "i_", "2.3310", "167.6272", "Ver. 2.0, Ref. 2"},
                                  {"s'_", "si_", "2.0800", "247.5744", "Ver. 2.0, Ref. 2"},
                                  {"s-_", "p_", "1.9800", "210.9800", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "sp_", "2.0100", "160.0000", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "s'_", "1.9900", "240.0000", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "p_", "1.9900", "175.1796", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "h_", "1.3730", "236.5449", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "f_", "1.6600", "180.5021", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "cl_", "1.9970", "196.3368", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "br_", "2.1450", "172.8017", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "i_", "2.3310", "152.8272", "Ver. 2.0, Ref. 2"},
                                  {"sp_", "si_", "2.1000", "167.4260", "Ver. 2.0, Ref. 2"},
                                  {"p_", "p_", "1.9700", "176.0000", "Ver. 2.0, Ref. 2"},
                                  {"p_", "h_", "1.4300", "224.0000", "Ver. 2.0, Ref. 2"},
                                  {"p_", "f_", "1.5400", "230.3664", "Ver. 2.0, Ref. 2"},
                                  {"p_", "cl_", "2.0430", "208.8228", "Ver. 2.0, Ref. 2"},
                                  {"p_", "br_", "2.1800", "183.1472", "Ver. 2.0, Ref. 2"},
                                  {"p_", "i_", "2.3110", "162.7080", "Ver. 2.0, Ref. 2"},
                                  {"p_", "si_", "1.9170", "168.2072", "Ver. 2.0, Ref. 2"},
                                  {"h_", "h_", "0.7461", "398.7301", "Ver. 2.0, Ref. 2"},
                                  {"h_", "f_", "1.0230", "520.7304", "Ver. 2.0, Ref. 2"},
                                  {"h_", "cl_", "1.3600", "345.9024", "Ver. 2.0, Ref. 2"},
                                  {"h_", "br_", "1.5080", "314.1728", "Ver. 2.0, Ref. 2"},
                                  {"h_", "i_", "1.6940", "292.0432", "Ver. 2.0, Ref. 2"},
                                  {"h_", "si_", "1.4630", "288.3168", "Ver. 2.0, Ref. 2"},
                                  {"dw", "dw", "0.7416", "398.2392", "Ver. 2.0, Ref. 2"},
                                  {"f_", "f_", "1.4170", "259.0683", "Ver. 2.0, Ref. 2"},
                                  {"f_", "cl_", "1.6470", "207.1180", "Ver. 2.0, Ref. 2"},
                                  {"f_", "br_", "1.7950", "193.5000", "Ver. 2.0, Ref. 2"},
                                  {"f_", "i_", "1.9810", "174.1664", "Ver. 2.0, Ref. 2"},
                                  {"f_", "si_", "1.5870", "297.3400", "Ver. 2.0, Ref. 2"},
                                  {"cl_", "cl_", "1.9880", "236.5339", "Ver. 2.0, Ref. 2"},
                                  {"cl_", "br_", "2.1320", "209.1876", "Ver. 2.0, Ref. 2"},
                                  {"cl_", "i_", "2.3180", "189.3396", "Ver. 2.0, Ref. 2"},
                                  {"cl_", "si_", "2.0870", "207.9748", "Ver. 2.0, Ref. 2"},
                                  {"br_", "br_", "2.2900", "175.6329", "Ver. 2.0, Ref. 2"},
                                  {"br_", "i_", "2.4660", "165.6156", "Ver. 2.0, Ref. 2"},
                                  {"br_", "si_", "2.2350", "180.2320", "Ver. 2.0, Ref. 2"},
                                  {"i_", "i_", "2.6620", "123.2110", "Ver. 2.0, Ref. 2"},
                                  {"i_", "si_", "2.4210", "158.9664", "Ver. 2.0, Ref. 2"},
                                  {"si_", "si_", "2.1900", "144.0000", "Ver. 2.0, Ref. 2"}};

/*
 Quartic bonds
  0= Key_a, 1= Key_b, 2= R0 (A), 3= K2 (kcal mol-1), 4= K3 (kcal mol-1), 5= K4 (kcal mol-1), 6= CFF91 info

  E = K2 * (R - R0)^2  +  K3 * (R - R0)^3  +  K4 * (R - R0)^4
*/
char * CFF91_bonds[58][7]= {{"c", "h", "1.1010", "341.0000", "-691.8900", "844.6000", "Ver. 1.0, Ref. 1"},
                             {"c", "c", "1.5330", "299.6700", "-501.7700", "679.8100", "Ver. 1.0, Ref. 1"},
                             {"c=", "c=", "1.3521", "545.2663", "-1005.6330", "1225.7415", "Ver. 1.0, Ref. 1"},
                             {"c=", "c=1", "1.3521", "545.2663", "-1005.6330", "1225.7415", "Ver. 2.0, Ref. 3"},
                             {"c=", "c=2", "1.3521", "545.2663", "-1005.6330", "1225.7415", "Ver. 2.0, Ref. 3"},
                             {"c", "c=", "1.5060", "312.3517", "-582.1861", "339.8971", "Ver. 1.0, Ref. 1"},
                             {"c", "c=1", "1.5060", "312.3517", "-582.1861", "339.8971", "Ver. 2.0, Ref. 3"},
                             {"c", "c=2", "1.5060", "312.3517", "-582.1861", "339.8971", "Ver. 2.0, Ref. 3"},
                             {"c=", "h", "1.0883", "365.7679", "-725.5404", "781.6621", "Ver. 1.0, Ref. 1"},
                             {"c=1", "h", "1.0883", "365.7679", "-725.5404", "781.6621", "Ver. 2.0, Ref. 3"},
                             {"c=2", "h", "1.0883", "365.7679", "-725.5404", "781.6621", "Ver. 2.0, Ref. 3"},
                             {"c", "o", "1.4200", "400.3954", "-835.1951", "1313.0142", "Ver. 1.0, Ref. 1"},
                             {"h*", "o", "0.9650", "532.5062", "-1282.9050", "2004.7658", "Ver. 1.0, Ref. 1"},
                             {"c", "na", "1.4570", "365.8052", "-699.6368", "998.4842", "Ver. 1.0, Ref. 1"},
                             {"h*", "na", "1.0060", "466.7400", "-1073.6018", "1251.1056", "Ver. 1.0, Ref. 1"},
                             {"c", "s", "1.8230", "225.2768", "-327.7057", "488.9722", "Ver. 1.0, Ref. 1"},
                             {"h", "s", "1.3261", "275.1123", "-531.3181", "562.9630", "Ver. 1.0, Ref. 1"},
                             {"s", "s", "2.0559", "197.6560", "-196.1366", "644.4103", "Ver. 1.0, Ref. 1"},
                             {"c", "c*", "1.5140", "312.3719", "-465.8290", "473.8300", "Ver. 1.0, Ref. 4"},
                             {"c*", "h", "1.1220", "304.8631", "-623.3705", "700.2828", "Ver. 1.0, Ref. 4"},
                             {"c*", "o'", "1.2160", "823.7948", "-1878.7940", "2303.5311", "Ver. 1.0, Ref. 4"},
                             {"c*", "o", "1.3649", "368.7309", "-832.4784", "1274.0231", "Ver. 1.0, Ref. 4"},
                             {"c'", "o'", "1.2195", "820.7018", "-1875.1000", "2303.7600", "Ver. 1.0, Ref. 1"},
                             {"c'", "n", "1.4160", "359.1591", "-558.4730", "1146.3810", "Ver. 1.0, Ref. 1"},
                             {"c", "c'", "1.5202", "253.7067", "-423.0370", "396.9000", "Ver. 1.0, Ref. 1"},
                             {"c", "n", "1.4520", "327.1657", "-547.8990", "526.5000", "Ver. 1.0, Ref. 1"},
                             {"h*", "n", "1.0100", "462.7500", "-1053.6300", "1545.7570", "Ver. 1.0, Ref. 1"},
                             {"c'", "h", "1.1110", "325.5717", "-632.1990", "726.0003", "Ver. 1.0, Ref. 1"},
                             {"cp", "cp", "1.4314", "356.0904", "-627.6179", "1327.6345", "Ver. 1.0, Ref. 1"},
                             {"cp", "h", "1.0862", "377.7644", "-803.4526", "894.3173", "Ver. 1.0, Ref. 1"},
                             {"cp", "np", "1.3485", "508.8587", "-977.6914", "1772.5134", "Ver. 1.0, Ref. 1"},
                             {"np", "np", "1.3121", "513.0111", "-873.6366", "1634.3437", "Ver. 1.0, Ref. 1"},
                             {"h*", "nh", "1.0053", "463.9230", "-1050.8070", "1284.7262", "Ver. 1.0, Ref. 1"},
                             {"hi", "nh", "1.0053", "463.9230", "-1050.8070", "1284.7262", "Ver. 1.1, Ref. 1"},
                             {"cp", "nh", "1.3749", "477.5202", "-848.5592", "1022.3909", "Ver. 1.0, Ref. 1"},
                             {"nh", "np", "1.3204", "646.7598", "-1639.8800", "6799.7099", "Ver. 1.0, Ref. 1"},
                             {"cp", "op", "1.3597", "547.5169", "-834.0665", "544.3090", "Ver. 1.0, Ref. 1"},
                             {"cp", "sp", "1.7053", "364.2568", "-457.7758", "291.1498", "Ver. 1.0, Ref. 1"},
                             {"cp", "o", "1.3768", "428.8798", "-738.2351", "1114.9655", "Ver. 1.0, Ref. 1"},
                             {"c", "cp", "1.5010", "321.9021", "-521.8208", "572.1628", "Ver. 1.0, Ref. 1"},
                             {"cp", "nn", "1.3912", "447.0438", "-784.5346", "886.1671", "Ver. 1.0, Ref. 1"},
                             {"h*", "nn", "1.0012", "465.8608", "-1066.2360", "1496.5647", "Ver. 1.0, Ref. 1"},
                             {"h+", "n+", "1.0119", "448.6300", "-963.1917", "1248.4000", "Ver. 1.0, Ref. 1"},
                             {"c", "n+", "1.5185", "293.1700", "-603.7882", "629.6900", "Ver. 1.0, Ref. 1"},
                             {"c", "c-", "1.5483", "253.0800", "-449.0300", "457.3200", "Ver. 1.0, Ref. 1"},
                             {"c-", "o-", "1.2339", "711.3500", "-1543.9000", "1858.6000", "Ver. 1.0, Ref. 1"},
                             {"c-", "h", "1.1331", "241.0600", "-574.7800", "853.7500", "Ver. 1.0, Ref. 1"},
                             {"c+", "nr", "1.3834", "380.4600", "-814.4300", "1153.3000", "Ver. 1.0, Ref. 1"},
                             {"h*", "nr", "1.0023", "462.3900", "-1044.6000", "1468.7000", "Ver. 1.0, Ref. 1"},
                             {"c", "nr", "1.4695", "340.2400", "-589.4800", "854.5300", "Ver. 1.0, Ref. 1"},
                             {"nr", "cr", "1.3200", "388.0000", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                             {"n=", "cr", "1.2600", "560.0000", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                             {"n=1", "cr", "1.2600", "560.0000", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                             {"n=2", "cr", "1.2600", "560.0000", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                             {"n=", "c", "1.4750", "336.0000", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                             {"n=1", "c", "1.4750", "336.0000", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                             {"n=2", "c", "1.4750", "336.0000", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                             {"h*", "o*", "0.9700", "563.2800", "-1428.2200", "1902.1200", "Ver. 1.1, Ref. 3"}};

/*
 Quadratic angles
  0= Key_a, 1= Key_b, 2= Key_c, 3= Theta0, 4= K2, 5= CFF91 info

  E = K2 * (Theta - Theta0)^2
  With:
   - Theta = Angle
   - Theta0 = Equilibrium angle
   - K2 = Spring constant
*/
char * CFF91_angles_auto[330][6]= {{"c3m_", "c3m_", "c3m_", "60.0000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c3m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c3m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c3m_", "c3m_", "60.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c3m_", "n3m_", "60.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c3m_", "c_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c3m_", "c3m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c3m_", "c4m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o3e_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o3e_", "c3m_", "c3m_", "60.0000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o4e_", "c3m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c3m_", "c3m_", "60.0000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c3m_", "s3e_", "60.0000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c3m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c3m_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c3m_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c3m_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "c3m_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "c3m_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "c4m_", "95.0000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "n4m_", "88.3400", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "o4e_", "91.8400", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "s4e_", "94.5900", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c4m_", "o_", "121.0000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c4m_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c4m_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c4m_", "n4m_", "88.4000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c4m_", "o4e_", "90.0000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c4m_", "s4e_", "89.0000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c4m_", "c_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c4m_", "c3m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c4m_", "c4m_", "112.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o3e_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o4e_", "c4m_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o4e_", "c4m_", "o4e_", "90.0000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"o4e_", "c4m_", "s4e_", "89.0000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c4m_", "s4e_", "91.0000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c4m_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c4m_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c4m_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c4m_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "c4m_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "c4m_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c_", "c3m_", "114.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c_", "c4m_", "114.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "n3m_", "125.0000", "145.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "n4m_", "123.0000", "145.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "n3m_", "c3m_", "113.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "n3m_", "c3m_", "60.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "n3m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "n3m_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "n3m_", "c3m_", "110.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "n3m_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "n3m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "n4m_", "*", "110.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "n4m_", "c4m_", "91.3800", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "n4m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "n4m_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "n4m_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "n4m_", "c4m_", "110.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "n4m_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "n4m_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "o3e_", "c3m_", "58.9580", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "o4e_", "c4m_", "91.7370", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "o_", "*", "120.0000", "72.0000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "o_", "*", "120.0010", "72.0000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "op_", "*", "108.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "op_", "*", "106.0000", "27.5000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "s_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "s_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "s3e_", "c3m_", "44.5000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "s3e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "s3e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "s3e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "s3e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "s3e_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "s3e_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "s3e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "s4e_", "c4m_", "85.9200", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "s4e_", "*", "99.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"n3m_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"n4m_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "s4e_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"s3e_", "s4e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s4e_", "s4e_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "s4e_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "s4e_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "s4e_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"c3m_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 2"},
                                   {"c4m_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c_", "*", "109.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c_", "c_", "114.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "c_", "*", "109.5000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "c_", "*", "109.5000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c_", "o_", "110.5000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c_", "s_", "115.0000", "46.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c_", "*", "109.5000", "44.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c_", "*", "107.8000", "95.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c_", "h_", "107.1000", "62.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "c_", "*", "112.3000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "c_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c'_", "c_", "115.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c'_", "*", "120.0000", "53.5000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "c'_", "*", "110.0000", "122.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "c'_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "c'_", "*", "123.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c'_", "*", "110.0000", "55.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c'_", "n_", "120.0000", "102.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c'_", "c_", "114.0000", "82.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "o_", "123.0000", "145.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "h_", "120.0000", "55.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c'_", "n_", "123.0000", "145.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c'_", "h_", "117.0200", "26.3900", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c'_", "*", "120.0000", "99.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "c'_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "c'_", "*", "120.0000", "65.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "cp_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "cp_", "*", "120.0000", "102.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "cp_", "*", "120.0000", "89.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "cp_", "c_", "114.0000", "89.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "cp_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "cp_", "*", "120.0000", "37.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "cp_", "*", "120.0000", "99.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "cp_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "cp_", "*", "120.0000", "65.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "c=_", "*", "120.0000", "36.2000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "c=_", "*", "120.0000", "90.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "c=_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "c=_", "*", "120.0000", "68.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "c=_", "*", "120.0000", "40.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "c=_", "*", "120.0000", "37.5000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "c=_", "*", "120.0000", "96.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "c=_", "*", "120.0000", "34.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "c=_", "*", "120.0000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "ct_", "*", "180.0000", "200.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "ct2_", "*", "180.0000", "200.0000", "Ver. 2.0, Ref. 6"},
                                   {"c_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "na_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "na_", "*", "114.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "na_", "*", "110.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "na_", "*", "109.0000", "41.6000", "Ver. 2.0, Ref. 2"},
                                   {"*", "na_", "*", "109.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "n_", "*", "120.0000", "80.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "n_", "*", "120.0000", "70.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "n_", "*", "122.0000", "35.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "n_", "*", "120.0000", "35.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "n_", "*", "120.0000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "np_", "*", "120.0000", "27.5000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "np_", "*", "120.0000", "27.5000", "Ver. 2.0, Ref. 2"},
                                   {"*", "np_", "*", "120.0000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "nt_", "*", "180.0", "50.0", "Ver. 2.0, Ref. 2"},
                                   {"c_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "o_", "*", "120.0000", "72.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "o_", "*", "109.0000", "58.5000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "o*_", "h_", "104.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "o_", "*", "124.1000", "56.4000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "o_", "si", "149.8000", "31.1000", "Ver. 2.0, Ref. 2"},
                                   {"*", "o_", "*", "109.5000", "60.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "s_", "*", "102.0000", "58.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "s_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "s_", "*", "103.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "s_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "s_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "s_", "*", "109.5000", "75.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "s_", "*", "109.5000", "48.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "s_", "*", "109.5000", "50.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"n_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"o_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"s_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"h_", "sp_", "*", "96.0000", "48.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "sp_", "*", "92.5670", "126.5060", "Ver. 2.0, Ref. 2"},
                                   {"si_", "sp_", "*", "96.0000", "48.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "sp_", "*", "92.5670", "120.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"o'_", "p_", "*", "120.0000", "110.0000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"s'_", "p_", "*", "120.0000", "100.0000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "p_", "*", "109.5000", "30.0000", "Ver. 2.0, Ref. 2"},
                                   {"*", "p_", "*", "109.5000", "45.0000", "Ver. 2.0, Ref. 2"},
                                   {"c_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 2"},
                                   {"n_", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 2"},
                                   {"o_", "si_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"s_", "si_", "*", "113.1000", "42.3000", "Ver. 2.0, Ref. 2"},
                                   {"h_", "si_", "*", "112.0000", "31.8000", "Ver. 2.0, Ref. 2"},
                                   {"f_", "si_", "*", "117.3000", "44.1000", "Ver. 2.0, Ref. 2"},
                                   {"si_", "si_", "*", "113.4000", "33.3000", "Ver. 2.0, Ref. 2"},
                                   {"*", "si_", "*", "113.5000", "44.4000", "Ver. 2.0, Ref. 2"}};

/*
 Quartic angles
  0= Key_a, 1= Key_b, 2= Key_c, 3= Theta0, 4= K2, 5= K3, 6= K4, 7= CFF91 info

  Delta = Theta - Theta0
  E = K2 * Delta^2  +  K3 * Delta^3  +  K4 * Delta^4
  With:
   - Theta = Angle
   - Theta0 = Equilibrium angle
*/
char * CFF91_angles[155][8]= {{"h", "c", "h", "107.6600", "39.6410", "-12.9210", "-2.4318", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "h", "110.7700", "41.4530", "-10.6040", "5.1290", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "c", "112.6700", "39.5160", "-7.4430", "-9.5583", "Ver. 1.0, Ref. 1"},
                              {"c", "c=", "c=", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 1.0, Ref. 1"},
                              {"c", "c=", "c=1", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=", "c=2", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=1", "c=", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=1", "c=1", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=1", "c=2", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=2", "c=", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=2", "c=1", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c", "c=2", "c=2", "126.2600", "43.8250", "-27.7266", "1.0056", "Ver. 2.0, Ref. 3"},
                              {"c=", "c=", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 1.0, Ref. 1"},
                              {"c=1", "c=", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c=", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=", "c=1", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=1", "c=1", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c=1", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=", "c=2", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=1", "c=2", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c=2", "h", "124.8800", "35.2766", "-17.7740", "-1.6215", "Ver. 2.0, Ref. 3"},
                              {"c", "c=", "h", "117.2700", "30.0944", "-8.0826", "-8.6781", "Ver. 1.0, Ref. 1"},
                              {"c", "c=1", "h", "117.2700", "30.0944", "-8.0826", "-8.6781", "Ver. 2.0, Ref. 3"},
                              {"c", "c=2", "h", "117.2700", "30.0944", "-8.0826", "-8.6781", "Ver. 2.0, Ref. 3"},
                              {"c", "c", "c=", "111.7600", "45.7026", "-10.6396", "-9.9121", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "c=1", "111.7600", "45.7026", "-10.6396", "-9.9121", "Ver. 2.0, Ref. 3"},
                              {"c", "c", "c=2", "111.7600", "45.7026", "-10.6396", "-9.9121", "Ver. 2.0, Ref. 3"},
                              {"c=", "c", "h", "110.0600", "41.2784", "-14.2963", "5.2229", "Ver. 1.0, Ref. 1"},
                              {"c=1", "c", "h", "110.0600", "41.2784", "-14.2963", "5.2229", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c", "h", "110.0600", "41.2784", "-14.2963", "5.2229", "Ver. 2.0, Ref. 3"},
                              {"h", "c=", "h", "115.4900", "29.6363", "-12.4853", "-6.2218", "Ver. 1.0, Ref. 1"},
                              {"h", "c=1", "h", "115.4900", "29.6363", "-12.4853", "-6.2218", "Ver. 2.0, Ref. 3"},
                              {"h", "c=2", "h", "115.4900", "29.6363", "-12.4853", "-6.2218", "Ver. 2.0, Ref. 3"},
                              {"c", "c=", "c", "121.0100", "29.2704", "-10.1175", "-6.7906", "Ver. 1.0, Ref. 1"},
                              {"c", "c=1", "c", "121.0100", "29.2704", "-10.1175", "-6.7906", "Ver. 2.0, Ref. 3"},
                              {"c", "c=2", "c", "121.0100", "29.2704", "-10.1175", "-6.7906", "Ver. 2.0, Ref. 3"},
                              {"c=", "c", "c=", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 1.0, Ref. 1"},
                              {"c=1", "c", "c=", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c", "c=", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 2.0, Ref. 3"},
                              {"c=1", "c", "c=1", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 2.0, Ref. 3"},
                              {"c=2", "c", "c=2", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 2.0, Ref. 3"},
                              {"c=1", "c", "c=2", "113.0100", "44.2251", "-10.2683", "-9.5886", "Ver. 2.0, Ref. 3"},
                              {"c", "o", "c", "104.5000", "35.7454", "-10.0067", "-6.2729", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "o", "108.7280", "58.5446", "-10.8088", "-12.4006", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "o", "111.2700", "54.5381", "-8.3642", "-13.0838", "Ver. 1.0, Ref. 1"},
                              {"c", "o", "h*", "105.8000", "52.7061", "-12.1090", "-9.8681", "Ver. 1.0, Ref. 1"},
                              {"c", "na", "c", "112.4436", "47.2337", "-10.6612", "-10.2062", "Ver. 1.0, Ref. 1"},
                              {"c", "na", "h*", "110.9538", "50.8652", "-4.4522", "-10.0298", "Ver. 1.0, Ref. 1"},
                              {"h*", "na", "h*", "107.0671", "45.2520", "-7.5558", "-9.5120", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "na", "111.9100", "60.7147", "-13.3366", "-13.0785", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "na", "110.6204", "51.3137", "-6.7198", "-2.6003", "Ver. 1.0, Ref. 1"},
                              {"c", "s", "h", "96.8479", "56.7336", "14.2713", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "s", "112.5642", "47.0276", "-10.6790", "-10.1687", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "s", "107.8522", "51.4949", "-13.5270", "7.0260", "Ver. 1.0, Ref. 1"},
                              {"c", "s", "c", "97.5000", "57.6938", "-5.0559", "-11.8206", "Ver. 1.0, Ref. 1"},
                              {"c", "s", "s", "100.3000", "57.2900", "-6.5301", "-11.8204", "Ver. 1.0, Ref. 1"},
                              {"s", "c", "s", "111.5000", "27.9677", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "s", "s", "97.2876", "54.4281", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "s", "h", "94.3711", "54.9676", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "c*", "108.4000", "43.9594", "-8.3924", "-9.3379", "Ver. 1.0, Ref. 4"},
                              {"c*", "c", "h", "107.8594", "38.0833", "-17.5074", "0.0000", "Ver. 1.0, Ref. 4"},
                              {"c", "c*", "h", "106.2000", "27.4878", "-1.9350", "14.5545", "Ver. 1.0, Ref. 4"},
                              {"c", "c*", "o'", "119.3000", "65.1016", "-17.9766", "0.0000", "Ver. 1.0, Ref. 4"},
                              {"h", "c*", "o'", "116.5000", "59.6420", "-21.9179", "0.0000", "Ver. 1.0, Ref. 4"},
                              {"c", "c*", "c", "110.5884", "34.6214", "-7.0939", "-7.4032", "Ver. 1.0, Ref. 4"},
                              {"h", "c*", "h", "108.6000", "40.0764", "-6.8139", "-8.4349", "Ver. 1.0, Ref. 4"},
                              {"o", "c*", "o'", "118.9855", "98.6813", "-22.2485", "10.3673", "Ver. 1.0, Ref. 4"},
                              {"c", "c*", "o", "100.3182", "38.8631", "-3.8323", "-7.9802", "Ver. 1.0, Ref. 4"},
                              {"c*", "o", "h*", "111.2537", "53.5303", "-11.8454", "-11.5405", "Ver. 1.0, Ref. 4"},
                              {"c*", "c", "o", "106.1764", "74.4143", "-12.6018", "-48.7850", "Ver. 1.0, Ref. 4"},
                              {"c", "o", "c*", "102.9695", "38.9739", "-6.2595", "-8.1710", "Ver. 1.0, Ref. 4"},
                              {"h", "c*", "o", "94.5209", "32.3751", "-0.6174", "-6.5639", "Ver. 1.0, Ref. 4"},
                              {"n", "c'", "o'", "125.5420", "92.5720", "-34.4800", "-11.1871", "Ver. 1.0, Ref. 1"},
                              {"c", "c'", "o'", "123.1451", "55.5431", "-17.2123", "0.1348", "Ver. 1.0, Ref. 1"},
                              {"c", "c'", "n", "116.9257", "39.4193", "-10.9945", "-8.7733", "Ver. 1.0, Ref. 1"},
                              {"c", "n", "c'", "111.0372", "31.8958", "-6.6942", "-6.8370", "Ver. 1.0, Ref. 1"},
                              {"c'", "n", "h*", "117.9607", "37.4964", "-8.1837", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "n", "h*", "113.8683", "45.9271", "-20.0824", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c'", "c", "n", "100.5663", "52.0966", "-5.2642", "-10.7045", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "n", "114.3018", "42.6589", "-10.5464", "-9.3243", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "n", "108.9372", "57.4010", "2.9374", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "c'", "108.5295", "51.9747", "-9.4851", "-10.9985", "Ver. 1.0, Ref. 1"},
                              {"c'", "c", "h", "107.7336", "40.6099", "-28.8121", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "c'", "o'", "117.8326", "45.9187", "-22.5264", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "c'", "n", "110.3935", "52.1647", "-18.4845", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h*", "n", "h*", "116.9402", "37.5749", "-8.6676", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "n", "c", "111.5635", "39.6084", "-8.5459", "-8.5152", "Ver. 1.0, Ref. 1"},
                              {"n", "c'", "n", "122.5292", "104.0857", "-36.7315", "-24.2616", "Ver. 1.0, Ref. 1"},
                              {"c'", "n", "c'", "121.9556", "76.3105", "-26.3166", "-17.6944", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "cp", "118.9000", "61.0226", "-34.9931", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "h", "117.9400", "35.1558", "-12.4682", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "np", "cp", "111.6800", "84.5159", "-48.5528", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "np", "116.5100", "76.6970", "-35.3868", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "cp", "np", "110.5500", "40.8275", "-13.0318", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"np", "cp", "np", "115.3800", "85.4923", "-6.5225", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "np", "np", "106.0400", "109.1158", "-42.9319", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "nh", "h*", "109.3800", "47.0120", "-17.3556", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "nh", "cp", "106.0100", "109.7746", "-9.0636", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "nh", "112.5900", "78.6418", "-16.3888", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "cp", "nh", "109.8000", "43.8408", "-9.5153", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"nh", "cp", "np", "108.9100", "98.8519", "-5.7502", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "nh", "np", "109.3900", "119.1811", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h*", "nh", "np", "99.4530", "41.6499", "-5.7422", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "np", "nh", "108.2200", "119.0383", "-24.2061", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "op", "cp", "103.4300", "112.4497", "-25.7326", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "op", "104.1700", "101.3206", "-17.3072", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "cp", "op", "106.1700", "48.0995", "-9.0144", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"np", "cp", "op", "104.0100", "104.4800", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "sp", "cp", "88.2540", "130.6992", "-18.4789", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "sp", "105.5300", "96.2006", "-44.9267", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "cp", "sp", "115.9800", "36.7902", "-13.1342", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"np", "cp", "sp", "114.2300", "92.3110", "-35.5956", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"nh", "cp", "nh", "105.0080", "107.0693", "-27.7154", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "o", "123.4200", "73.6781", "-21.6787", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "o", "h*", "108.1900", "53.1250", "-8.5016", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "cp", "cp", "120.0500", "44.7148", "-22.7352", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "c", "h", "111.0000", "44.3234", "-9.4454", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "cp", "nn", "121.4584", "61.0647", "-21.6172", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cp", "nn", "h*", "111.8725", "40.8369", "-15.6673", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h*", "nn", "h*", "107.5130", "42.5182", "-21.7566", "-4.3372", "Ver. 1.0, Ref. 1"},
                              {"nn", "cp", "np", "118.5414", "56.9522", "-7.2655", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h+", "n+", "h+", "106.1100", "45.3280", "-14.0474", "1.9350", "Ver. 1.0, Ref. 1"},
                              {"c", "n+", "h+", "110.5100", "49.2170", "-12.2153", "-18.9667", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "n+", "112.1300", "66.4520", "4.8694", "37.7860", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "n+", "105.8500", "72.2630", "-28.1923", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "n+", "c", "109.7700", "44.8220", "-3.9132", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c-", "c", "h", "109.6700", "37.9190", "-7.3877", "-8.0694", "Ver. 1.0, Ref. 1"},
                              {"c", "c-", "o-", "115.0600", "59.0960", "-15.1430", "-12.9820", "Ver. 1.0, Ref. 1"},
                              {"o-", "c-", "o-", "130.0100", "111.2900", "-52.3390", "-28.1070", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "c-", "104.4900", "31.3750", "-4.4023", "-6.5271", "Ver. 1.0, Ref. 1"},
                              {"h", "c-", "o-", "112.7500", "61.1530", "-14.0190", "-13.2380", "Ver. 1.0, Ref. 1"},
                              {"c+", "nr", "h*", "119.0700", "45.8110", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h*", "nr", "h*", "110.9100", "31.0910", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"nr", "c+", "nr", "117.4500", "83.9840", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "c", "nr", "117.3500", "55.0400", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"h", "c", "nr", "107.5000", "62.6790", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "nr", "c+", "117.0900", "31.4400", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"c", "nr", "h*", "117.2000", "37.2620", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                              {"cr", "nr", "h*", "122.9480", "40.482", "-16.2028", "0.0", "Ver. 1.1, Ref. 1"},
                              {"n=", "cr", "nr", "125.532", "101.8765", "-41.8094", "0.0", "Ver. 1.1, Ref. 1"},
                              {"n=1", "cr", "nr", "125.532", "101.8765", "-41.8094", "0.0", "Ver. 2.0, Ref. 3"},
                              {"n=2", "cr", "nr", "125.532", "101.8765", "-41.8094", "0.0", "Ver. 2.0, Ref. 3"},
                              {"nr", "cr", "nr", "122.5292", "104.0857", "-36.7315", "-24.2616", "Ver. 1.1, Ref. 1"},
                              {"c", "n=", "cr", "117.0900", "31.6888", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                              {"n=", "c", "h", "107.4989", "62.7484", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                              {"n=1", "c", "h", "107.4989", "62.7484", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                              {"n=2", "c", "h", "107.4989", "62.7484", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                              {"c", "c", "n=", "117.2847", "55.4431", "0.0000", "0.0000", "Ver. 1.1, Ref. 1"},
                              {"h*", "o*", "h*", "103.7000", "49.8400", "-11.6000", "-8.0000", "Ver. 1.2, Ref. 1"},
                              {"c'", "c", "n+", "100.5663", "52.0966", "-5.2642", "-10.7045", "Ver. 1.3, Ref. 1"},
                              {"c", "c", "cp", "108.4000", "43.9594", "-8.3924", "-9.3379", "Ver. 1.3, Ref. 1"},
                              {"c", "cp", "np", "120.0500", "44.7148", "-22.7352", "0.0000", "Ver. 1.3, Ref. 1"},
                              {"c-", "c", "n", "100.5663", "52.0966", "-5.2642", "-10.7045", "Ver. 1.3, Ref. 1"},
                              {"c", "o", "cp", "102.9695", "38.9739", "-6.2595", "-8.1710", "Ver. 1.4, Ref. 1"},
                              {"cp", "c", "cp", "111.0000", "44.3234", "-9.4454", "0.0000", "Ver. 1.5, Ref. 1"}};

/*
 Dihedrals / torsions
 0, 1, 2, 3 = Keys, 4= Kphi, 5= n, 6= Phi0, 7= CFF91 info
 E = Kphi * [ 1 + cos(n*Phi - Phi0) ]
 With:
  - Kphi = Rotation barrier
  - n = Periodicity / multiplicity
  - Phi0 = Equilibrium angle
*/
char * CFF91_torsions_auto[216][8]= {{"*", "c", "n3n", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "n3n", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "n3n", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "n3n", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=1", "n3n", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "n3n", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "n3n", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "ct", "n3n", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "n3n", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n3n", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=1", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=_3", "n3n", "*", "0.0500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "o", "n3n", "*", "0.3000", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "s", "n3n", "*", "0.3000", "2", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "si", "n3n", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "c", "*", "0.1580", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "c'", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "cp", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "c=", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "c=1", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "c=2", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "c=_3", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "na", "*", "0.0500", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "n", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "np", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "n=", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "n=1", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "n=2", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "n=_3", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c", "o", "*", "0.1300", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "s", "*", "0.1367", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "p", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c", "si", "*", "0.1111", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "c'", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "cp", "*", "2.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "c=", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "c=1", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "c=2", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "c=_3", "*", "0.4500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "n", "*", "3.2000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "n", "h", "1.2000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "n=", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "n=1", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "n=2", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "n=_3", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c'", "np", "*", "5.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "np", "h", "1.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "o", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "op", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "sp", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c'", "si", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "cp", "*", "3.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "c=", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "c=1", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "c=2", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "c=_3", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "na", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "n", "*", "2.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "np", "*", "2.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "np", "h", "1.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "n=", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "n=1", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "n=2", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "n=_3", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "cp", "o", "*", "1.8000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "o", "h", "0.7500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "op", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "sp", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "si", "*", "0.1667", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "cp", "p", "*", "0.2500", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "c=", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=_3", "c=_3", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "c=_3", "*", "4.0750", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "c=2", "*", "3.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "c=1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "c=2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "c=_3", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "na", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "n", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "np", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "np", "h", "0.7500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=1", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "na", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "n", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "np", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "np", "h", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "na", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "n", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "np", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "np", "h", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "na", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "n", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "np", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "np", "h", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=", "n=", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=_3", "n=_3", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "n=_3", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "n=1", "*", "8.1500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "n=2", "*", "2.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "n=1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "n=2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "n=1", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "n=_3", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "n=2", "*", "0.6250", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=", "o", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "sp", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "si", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=", "p", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "c=1", "o", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "sp", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "si", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=1", "p", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "o", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "sp", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "si", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=2", "p", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "o", "*", "0.9000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "op", "*", "4.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "s", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "sp", "*", "6.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "si", "*", "0.2110", "3", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c=_3", "p", "*", "1.2500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "c+", "n", "*", "3.4000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "ct", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "na", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "n", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "np", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "o", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "s", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "ct", "si", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "na", "*", "0.2500", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "n", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "np", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "n=", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "n=1", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "na", "n=2", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "na", "n=_3", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 3"},
                                     {"*", "na", "o", "*", "0.0975", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "s", "*", "0.0975", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "na", "si", "*", "0.0667", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "n", "*", "0.3750", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "np", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "np", "h", "0.3750", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "n=", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "n=1", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n", "n=2", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n", "n=_3", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n", "o", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "s", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n", "si", "*", "0.0000", "0", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "n=", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "n=1", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "np", "n=2", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "np", "n=_3", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "np", "np", "*", "11.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "o", "*", "1.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "op", "*", "11.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "s", "*", "1.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "sp", "*", "10.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "np", "si", "*", "0.2500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "n=", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "n=1", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"h", "np", "n=2", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"h", "np", "n=_3", "*", "0.7500", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"h", "np", "np", "*", "5.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "o", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "op", "*", "5.50000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "s", "*", "0.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "sp", "*", "5.5000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"h", "np", "si", "*", "0.1250", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=", "n=", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=_3", "n=_3", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=1", "n=_3", "*", "15.0000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "n=2", "*", "7.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=1", "n=1", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=1", "n=2", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "n=_3", "*", "1.5000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=", "o", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=", "s", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=", "si", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 2"},
                                     {"*", "n=1", "o", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=1", "s", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=1", "si", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "o", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "s", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=2", "si", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=_3", "o", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=_3", "s", "*", "0.7000", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "n=_3", "si", "*", "0.2333", "2", "180.", "Ver. 2.0, Ref. 3"},
                                     {"*", "o", "o", "*", "5.0000", "2", "0.", "Ver. 2.0, Ref. 8"},
                                     {"*", "o", "s", "*", "5.0000", "2", "0.", "Ver. 2.0, Ref. 8"},
                                     {"*", "o", "si", "*", "0.3333", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "o", "p", "*", "0.3750", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "s", "s", "*", "5.5000", "2", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "s", "si", "*", "0.2333", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "s", "p", "*", "0.3750", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "si", "si", "*", "0.1667", "3", "0.", "Ver. 2.0, Ref. 2"},
                                     {"*", "si", "p", "*", "0.0000", "3", "0.", "Ver. 2.0, Ref. 2"}};

/*
 Dihedrals / torsions
 0, 1, 2, 3 = Keys, 4= V1, 5= Phi0, 6= V2, 7= Phi0, 8= V3, 9= Phi0, 10= CFF91 info
 E = SUM(n=1,3) { V(n) * [ 1 - cos(n*Phi - Phi0(n)) ] }
 With:
  - Phi0 = Equilibrium angle
*/
char * CFF91_torsions[294][11]= {{"h", "c", "c", "h", "-0.2432", "0.0", "0.0617", "0.0", "-0.1383", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "h", "0.0000", "0.0", "0.0316", "0.0", "-0.1781", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "c", "0.1223", "0.0", "0.0514", "0.0", "-0.2230", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c=", "c=", "c", "0.0860", "0.0", "5.1995", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c=", "c=1", "c", "0.0860", "0.0", "5.1995", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c=", "c=2", "c", "0.0860", "0.0", "5.1995", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c=", "c=", "h", "0.0000", "0.0", "5.2097", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c=", "c=1", "h", "0.0000", "0.0", "5.2097", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c=", "c=2", "h", "0.0000", "0.0", "5.2097", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c=1", "c=", "h", "0.0000", "0.0", "5.2097", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c=2", "c=", "h", "0.0000", "0.0", "5.2097", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c=", "c=", "h", "0.0000", "0.0", "4.8974", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c=", "c=1", "h", "0.0000", "0.0", "4.8974", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c=", "c=2", "h", "0.0000", "0.0", "4.8974", "0.0", "0.0000", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=", "c=", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c=", "c=1", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=", "c=2", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=1", "c=", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=1", "c=1", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=1", "c=2", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=2", "c=", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=2", "c=1", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=2", "c=2", "0.2433", "0.0", "0.0000", "0.0", "0.1040", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=", "c=", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c=", "c=1", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=", "c=2", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=1", "c=", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=1", "c=1", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=1", "c=2", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=2", "c=", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=2", "c=1", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=2", "c=2", "0.1143", "0.0", "0.0000", "0.0", "0.1854", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=", "h", "-0.2433", "0.0", "0.0000", "0.0", "-0.3281", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c=1", "h", "-0.2433", "0.0", "0.0000", "0.0", "-0.3281", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=2", "h", "-0.2433", "0.0", "0.0000", "0.0", "-0.3281", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=", "h", "-0.1143", "0.0", "0.0000", "0.0", "-0.1349", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c=1", "h", "-0.1143", "0.0", "0.0000", "0.0", "-0.1349", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=2", "h", "-0.1143", "0.0", "0.0000", "0.0", "-0.1349", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c", "c=", "0.0883", "0.0", "0.0000", "0.0", "-0.0198", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "c=1", "0.0883", "0.0", "0.0000", "0.0", "-0.0198", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c", "c=2", "0.0883", "0.0", "0.0000", "0.0", "-0.0198", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c", "h", "0.0000", "0.0", "0.0000", "0.0", "-0.1166", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c", "c", "h", "0.0000", "0.0", "0.0000", "0.0", "-0.1166", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c", "h", "0.0000", "0.0", "0.0000", "0.0", "-0.1166", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=", "c", "-0.1143", "0.0", "0.0000", "0.0", "-0.1682", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c=1", "c", "-0.1143", "0.0", "0.0000", "0.0", "-0.1682", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c=2", "c", "-0.1143", "0.0", "0.0000", "0.0", "-0.1682", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=", "c", "-0.0871", "0.0", "0.0619", "0.0", "-0.7371", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c=1", "c", "-0.0871", "0.0", "0.0619", "0.0", "-0.7371", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "c=2", "c", "-0.0871", "0.0", "0.0619", "0.0", "-0.7371", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c=", "c", "c=1", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=2", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=1", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=2", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=1", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=2", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=1", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=2", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=1", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=2", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=1", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=2", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=1", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=2", "c=", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=1", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=2", "c=1", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=1", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=2", "c=2", "0.0025", "0.0", "0.0000", "0.0", "0.0173", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c", "c=", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=1", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=1", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=1", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c=2", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c=2", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c=2", "h", "0.0025", "0.0", "0.0000", "0.0", "-0.2911", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=", "c", "c", "c=", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c=1", "c", "c", "c=", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c", "c=", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=1", "c", "c", "c=1", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c", "c=1", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c=2", "c", "c", "c=2", "0.0000", "0.0", "0.0000", "0.0", "-0.3160", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "o", "c", "0.5302", "0.0", "0.0000", "0.0", "-0.2836", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "o", "c", "-0.5203", "0.0", "-0.3028", "0.0", "-0.3450", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "o", "-0.1435", "0.0", "0.2530", "0.0", "-0.0905", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "o", "0.7137", "0.0", "0.2660", "0.0", "-0.2545", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"o", "c", "c", "o", "-0.1820", "0.0", "-0.1084", "0.0", "-0.7047", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "o", "h*", "0.1863", "0.0", "-0.4338", "0.0", "-0.2121", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "o", "h*", "-0.6732", "0.0", "-0.4778", "0.0", "-0.1670", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "na", "c", "0.1904", "0.0", "-0.1342", "0.0", "-0.2460", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "na", "h*", "-0.5187", "0.0", "-0.4837", "0.0", "-0.1692", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "na", "h*", "-1.1506", "0.0", "-0.6344", "0.0", "-0.1845", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "na", "c", "-0.1406", "0.0", "0.4168", "0.0", "0.0150", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "na", "-0.2428", "0.0", "0.4065", "0.0", "-0.3079", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "na", "0.1764", "0.0", "0.1766", "0.0", "-0.5206", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"na", "c", "c", "na", "0.3805", "0.0", "0.3547", "0.0", "-0.1102", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "s", "h", "-0.4871", "0.0", "-0.4514", "0.0", "-0.1428", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "s", "h", "-0.5374", "0.0", "-0.5091", "0.0", "-0.1361", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"s", "c", "c", "s", "-1.2002", "0.0", "-1.2999", "0.0", "-0.1626", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "s", "-0.2078", "0.0", "-0.1060", "0.0", "-0.3595", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "s", "c", "-0.5073", "0.0", "0.0155", "0.0", "-0.0671", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "s", "c", "-0.3338", "0.0", "-0.0684", "0.0", "-0.1706", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "s", "s", "c", "-1.8578", "0.0", "-3.8321", "0.0", "-0.4469", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "s", "s", "-0.6269", "0.0", "-0.9598", "0.0", "-0.4957", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "s", "s", "-0.0610", "0.0", "-0.6387", "0.0", "-0.3072", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"s", "c", "s", "c", "-1.9835", "0.0", "-1.9213", "0.0", "-0.3816", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "s", "-0.7017", "0.0", "0.0201", "0.0", "0.1040", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "s", "s", "h", "-0.4956", "0.0", "-3.4601", "0.0", "-0.2482", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "s", "s", "h", "-0.0528", "0.0", "-3.5171", "0.0", "-0.2394", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"s", "c", "s", "h", "-0.0591", "0.0", "-0.6235", "0.0", "-0.0788", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "c*", "0.0929", "0.0", "0.0689", "0.0", "-0.2466", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c*", "c", "c", "h", "-0.1569", "0.0", "-0.2030", "0.0", "-0.1599", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c", "c*", "h", "0.3055", "0.0", "0.6703", "0.0", "-0.2816", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c", "c*", "o'", "0.6989", "0.0", "1.2089", "0.0", "-0.0445", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c", "c*", "h", "-0.5265", "0.0", "0.3627", "0.0", "-0.1792", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c", "c*", "o'", "-0.9975", "0.0", "0.9060", "0.0", "-0.0195", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c", "c*", "c", "0.2904", "0.0", "1.0177", "0.0", "-0.1085", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c", "c*", "c", "-0.4768", "0.0", "0.6051", "0.0", "-0.1205", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c*", "c", "c", "c*", "-1.2352", "0.0", "-0.2551", "0.0", "-0.1335", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c", "c*", "o", "-0.6359", "0.0", "1.4807", "0.0", "-0.0438", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"o'", "c*", "o", "h*", "-2.7332", "0.0", "2.9646", "0.0", "-0.0155", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c*", "o", "h*", "-2.9126", "0.0", "1.0199", "0.0", "-0.2077", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"o", "c", "c*", "o'", "0.6444", "0.0", "0.7897", "0.0", "1.0604", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"o'", "c*", "o", "c", "0.8905", "0.0", "3.2644", "0.0", "0.2646", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c", "o", "c*", "0.9513", "0.0", "0.1155", "0.0", "0.0720", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c*", "o", "c", "-2.5594", "0.0", "2.2013", "0.0", "0.0325", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c", "c*", "o", "1.8341", "0.0", "2.0603", "0.0", "-0.0195", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c", "o", "c*", "0.1302", "0.0", "-0.3250", "0.0", "0.1134", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c*", "o", "c", "-0.9993", "0.0", "0.2131", "0.0", "-0.4274", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c*", "c", "o", "c*", "-1.8534", "0.0", "0.1981", "0.0", "0.2423", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"o", "c", "c*", "h", "-0.0390", "0.0", "1.4052", "0.0", "0.0757", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c*", "o", "h*", "-1.4540", "0.0", "0.8387", "0.0", "-0.4912", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c*", "c", "c", "o", "-0.0858", "0.0", "-0.1320", "0.0", "-0.5909", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"o'", "c'", "n", "c", "0.8297", "0.0", "3.7234", "0.0", "-0.0495", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"o'", "c'", "n", "h*", "-1.6938", "0.0", "2.7386", "0.0", "-0.3360", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c'", "n", "c", "-0.7532", "0.0", "2.7392", "0.0", "0.0907", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c'", "n", "h*", "-0.8236", "0.0", "2.1467", "0.0", "-0.2142", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c'", "o'", "-0.1804", "0.0", "0.0012", "0.0", "0.0371", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c'", "n", "0.1693", "0.0", "-0.0090", "0.0", "-0.0687", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "n", "c'", "-0.0688", "0.0", "0.0762", "0.0", "-0.0618", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "n", "c'", "0.0143", "0.0", "-0.0132", "0.0", "0.0091", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n", "c'", "0.0219", "0.0", "-0.0260", "0.0", "0.0714", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "n", "h*", "0.0546", "0.0", "0.0756", "0.0", "0.0728", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "n", "h*", "-0.0483", "0.0", "-0.0077", "0.0", "-0.0014", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n", "h*", "-0.0148", "0.0", "-0.0791", "0.0", "-0.0148", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"n", "c", "c'", "o'", "0.0899", "0.0", "0.1220", "0.0", "0.0905", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"n", "c", "c'", "n", "-0.0892", "0.0", "0.1259", "0.0", "-0.0884", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c'", "o'", "0.0442", "0.0", "0.0292", "0.0", "0.0562", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c'", "n", "-0.0368", "0.0", "0.0389", "0.0", "-0.0529", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "n", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "c", "h", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c'", "n", "c", "0.3345", "0.0", "2.5838", "0.0", "-0.4006", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c'", "n", "h*", "-0.0078", "0.0", "2.6186", "0.0", "-0.0900", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "c'", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "n", "c", "-0.0017", "0.0", "-0.0072", "0.0", "0.0008", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n", "c", "0.0406", "0.0", "0.0354", "0.0", "-0.1649", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "c", "c'", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "c", "n", "c", "-0.0036", "0.0", "0.0049", "0.0", "0.0039", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"n", "c'", "n", "h*", "-0.7358", "0.0", "0.4643", "0.0", "-1.1098", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"o'", "c'", "n", "c'", "-0.4066", "0.0", "1.2513", "0.0", "-0.7507", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c'", "n", "c'", "0.1907", "0.0", "1.1212", "0.0", "0.0426", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "cp", "8.3667", "0.0", "1.1932", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "h", "0.0000", "0.0", "3.9661", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "h", "0.0000", "0.0", "1.8769", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "np", "cp", "0.0000", "0.0", "6.8517", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "cp", "0.0000", "0.0", "5.6183", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "np", "0.0000", "0.0", "4.7675", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "np", "0.0000", "0.0", "3.6689", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "np", "cp", "0.0000", "0.0", "7.4664", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "cp", "np", "0.0000", "0.0", "8.8338", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "np", "np", "cp", "0.0000", "0.0", "13.7232", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "np", "np", "0.0000", "0.0", "12.0680", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "np", "0.0000", "0.0", "6.3346", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "nh", "h*", "0.0000", "0.0", "1.8202", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "nh", "h*", "0.0000", "0.0", "1.3342", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "nh", "cp", "10.8765", "0.0", "6.3475", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "nh", "cp", "0.0000", "0.0", "3.7848", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "nh", "11.5270", "0.0", "2.7183", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "nh", "0.0000", "0.0", "3.0202", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "nh", "cp", "0.0000", "0.0", "20.0173", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "nh", "h*", "0.0000", "0.0", "3.5096", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nh", "cp", "np", "cp", "0.0000", "0.0", "9.7830", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nh", "cp", "cp", "np", "0.0000", "0.0", "4.8266", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "nh", "np", "cp", "0.0000", "0.0", "14.4766", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h*", "nh", "np", "cp", "0.0000", "0.0", "7.4549", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "nh", "np", "0.0000", "0.0", "10.7803", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "nh", "np", "0.0000", "0.0", "4.1751", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "np", "nh", "0.0000", "0.0", "5.1942", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "nh", "0.0000", "0.0", "5.2455", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "op", "cp", "0.0000", "0.0", "27.5402", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "op", "cp", "0.0000", "0.0", "2.6043", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "op", "0.0000", "0.0", "10.6923", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "op", "0.0000", "0.0", "3.3516", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "op", "cp", "0.0000", "0.0", "30.4292", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"op", "cp", "np", "cp", "0.0000", "0.0", "15.3660", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "cp", "op", "0.0000", "0.0", "9.7415", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"op", "cp", "np", "np", "0.0000", "0.0", "11.8577", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "sp", "cp", "0.0000", "0.0", "31.5576", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "sp", "cp", "0.0000", "0.0", "4.2145", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "sp", "0.0000", "0.0", "9.2097", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "sp", "0.0000", "0.0", "3.7957", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "sp", "cp", "0.0000", "0.0", "21.1715", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"sp", "cp", "np", "cp", "0.0000", "0.0", "13.6743", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "cp", "sp", "0.0000", "0.0", "8.5974", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"sp", "cp", "np", "np", "0.0000", "0.0", "11.5762", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nh", "cp", "nh", "cp", "0.0000", "0.0", "23.0443", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nh", "cp", "nh", "h*", "0.0000", "0.0", "7.0570", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nh", "cp", "cp", "nh", "0.0000", "0.0", "12.7748", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "o", "0.0000", "0.0", "4.8498", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "o", "0.0000", "0.0", "1.7234", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "o", "h*", "-0.6900", "0.0", "0.5097", "0.0", "0.0095", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "cp", "cp", "cp", "0.0000", "0.0", "4.4072", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "cp", "cp", "h", "0.0000", "0.0", "1.5590", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "cp", "cp", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "nn", "0.0000", "0.0", "5.3826", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "cp", "nn", "0.0000", "0.0", "1.3331", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "nn", "h*", "0.0000", "0.0", "1.2190", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nn", "cp", "np", "cp", "0.0000", "0.0", "6.7119", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"np", "cp", "nn", "h*", "0.0000", "0.0", "2.0184", "0.0", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "n+", "h+", "-0.8792", "0.0", "-0.5978", "0.0", "-0.3242", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n+", "h+", "-0.2458", "0.0", "-0.2789", "0.0", "-0.0294", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "n+", "-1.9394", "0.0", "0.0086", "0.0", "0.2775", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "n+", "-0.2179", "0.0", "-0.4127", "0.0", "-0.1252", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n+", "c", "0.3624", "0.0", "0.1012", "0.0", "-0.2486", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "n+", "c", "0.7077", "0.0", "-0.3744", "0.0", "-0.1914", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c-", "o-", "-2.5999", "0.0", "1.0488", "0.0", "-0.2089", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c", "c-", "-0.4054", "0.0", "0.4825", "0.0", "0.4345", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c-", "c", "c", "h", "-1.2767", "0.0", "0.5949", "0.0", "0.2379", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "c-", "o-", "1.7311", "0.0", "1.8510", "0.0", "-0.1933", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nr", "c+", "nr", "h*", "-7.2378", "0.0", "1.9150", "0.0", "0.1436", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "c", "nr", "-0.5336", "0.0", "-0.0444", "0.0", "-0.1432", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "nr", "c+", "-5.4418", "0.0", "-0.0437", "0.0", "0.8035", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c", "nr", "h*", "-5.0724", "0.0", "-0.4980", "0.0", "-0.4381", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "nr", "c+", "2.8272", "0.0", "-0.3930", "0.0", "-0.3847", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "nr", "h*", "1.2659", "0.0", "-0.7739", "0.0", "0.0378", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nr", "c+", "nr", "c", "-0.6197", "0.0", "3.0539", "0.0", "0.1861", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c", "n=", "cr", "0.5865", "0.0", "-0.5482", "0.0", "-0.2767", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"h", "c", "n=1", "cr", "0.5865", "0.0", "-0.5482", "0.0", "-0.2767", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "n=2", "cr", "0.5865", "0.0", "-0.5482", "0.0", "-0.2767", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "n=", "cr", "nr", "-7.3589", "0.0", "0.4643", "0.0", "-0.1098", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"c", "n=1", "cr", "nr", "-7.3589", "0.0", "0.4643", "0.0", "-0.1098", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "n=2", "cr", "nr", "-7.3589", "0.0", "0.4643", "0.0", "-0.1098", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "n=", "cr", "0.0143", "0.0", "-0.0132", "0.0", "0.0091", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"c", "c", "n=1", "cr", "0.0143", "0.0", "-0.0132", "0.0", "0.0091", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"c", "c", "n=2", "cr", "0.0143", "0.0", "-0.0132", "0.0", "0.0091", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c", "n=", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"h", "c", "c", "n=1", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h", "c", "c", "n=2", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h*", "nr", "cr", "n=", "-0.7358", "0.0", "0.4643", "0.0", "-1.1098", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"h*", "nr", "cr", "n=1", "-0.7358", "0.0", "0.4643", "0.0", "-1.1098", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"h*", "nr", "cr", "n=2", "-0.7358", "0.0", "0.4643", "0.0", "-1.1098", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"n=", "c", "c", "c", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"n=1", "c", "c", "c", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"n=2", "c", "c", "c", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 2.0, Ref. 3"},
                                 {"nr", "cr", "nr", "h*", "-7.2378", "0.0", "1.9150", "0.0", "0.1436", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"c'", "c", "n+", "h+", "-0.8792", "0.0", "-0.5978", "0.0", "-0.3242", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c*", "c", "n+", "h+", "-0.8792", "0.0", "-0.5978", "0.0", "-0.3242", "0.0", "Ver. 1.3, Ref. 4"},
                                 {"n+", "c", "c'", "o'", "0.0899", "0.0", "0.1220", "0.0", "0.0905", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"n", "c", "c", "o", "-0.1820", "0.0", "-0.1084", "0.0", "-0.7047", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"cp", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"cp", "c", "c", "h", "-0.0228", "0.0", "0.0280", "0.0", "-0.1863", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c'", "c", "c", "cp", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c", "c", "cp", "cp", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c", "c", "cp", "np", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"h", "c", "cp", "np", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c", "cp", "cp", "nh", "0.0000", "0.0", "3.0202", "0.0", "0.0000", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c", "cp", "np", "cp", "0.0000", "0.0", "5.6183", "0.0", "0.0000", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c-", "c", "n", "c'", "-0.0688", "0.0", "0.0762", "0.0", "-0.0618", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c-", "c", "n", "h*", "0.0546", "0.0", "0.0756", "0.0", "0.0728", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"n", "c", "c-", "o-", "0.0899", "0.0", "0.1220", "0.0", "0.0905", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c-", "c", "c", "cp", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"n+", "c", "c'", "n", "-0.0892", "0.0", "0.1259", "0.0", "-0.0884", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c'", "c", "c", "c-", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"s", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c'", "c", "c", "s", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c-", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c'", "c", "c", "o", "-0.0858", "0.0", "-0.1320", "0.0", "-0.5909", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"c*", "c", "c", "n", "0.0972", "0.0", "0.0722", "0.0", "-0.2581", "0.0", "Ver. 1.3, Ref. 4"},
                                 {"c*", "c", "c", "c'", "-1.2352", "0.0", "-0.2551", "0.0", "-0.1335", "0.0", "Ver. 1.3, Ref. 4"},
                                 {"h", "c", "o", "cp", "0.9513", "0.0", "0.1155", "0.0", "0.0720", "0.0", "Ver. 1.4, Ref. 1"},
                                 {"cp", "cp", "o", "c", "0.0000", "0.0", "1.8000", "0.0", "0.0000", "0.0", "Ver. 1.4, Ref. 1"},
                                 {"cp", "c", "cp", "cp", "-0.2802", "0.0", "-0.0678", "0.0", "-0.0122", "0.0", "Ver. 1.5, Ref. 1"}};

/*
 Inversions
  0, 1, 2, 3 = Keys, 4= K, 5= Chi0, 6= CFF91 info
  E = K * (Chi - Chi0)^2
*/
char * CFF91_inversions[70][7]= {{"c", "c=", "c=", "h", "2.0765", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c=", "c=", "h", "h", "2.8561", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c=", "c", "c=", "2.0568", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "na", "c", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "na", "h*", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "na", "c", "c", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c*", "h", "o'", "38.4069", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c*", "c", "o'", "38.7949", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c*", "h", "o'", "37.8733", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c*", "o", "o'", "46.9264", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"h", "c*", "o", "o'", "45.3800", "0.0", "Ver. 1.0, Ref. 4"},
                                 {"c", "c'", "n", "o'", "24.3329", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "n", "c'", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c'", "n", "o'", "23.1691", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "n", "h*", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "n", "c", "c'", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"n", "c'", "n", "o'", "27.0615", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c'", "n", "c'", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "h", "7.6012", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "h", "np", "10.4920", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "np", "14.3802", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "nh", "cp", "h*", "5.9154", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "h", "nh", "7.9682", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "nh", "np", "8.8464", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "nh", "h*", "np", "4.9959", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "h", "op", "8.1654", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "op", "10.8102", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "h", "sp", "5.5684", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "np", "sp", "7.3414", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "cp", "nh", "nh", "10.2877", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "nh", "8.3206", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "cp", "7.1794", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "o", "13.0421", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "cp", "cp", "cp", "6.9644", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "cp", "cp", "nn", "10.7855", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cp", "nn", "h*", "h*", "0.0000", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nn", "cp", "np", "np", "15.0921", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "c-", "o-", "o-", "44.1500", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"h", "c-", "o-", "o-", "45.7520", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c+", "nr", "h*", "h*", "1.0035", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"nr", "c+", "nr", "nr", "54.4060", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"c", "nr", "c+", "h*", "0.3176", "0.0", "Ver. 1.0, Ref. 1"},
                                 {"cr", "nr", "h*", "h*", "1.0035", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"n=", "cr", "nr", "nr", "54.4060", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"c", "n=", "cr", "h*", "0.3176", "0.0", "Ver. 1.1, Ref. 1"},
                                 {"cp", "cp", "c", "np", "10.4920", "0.0", "Ver. 1.3, Ref. 1"},
                                 {"*", "c'", "*", "*", "36.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "cp", "*", "*", "10.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "c=", "*", "*", "2.0568", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "c+", "*", "*", "54.4060", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "na", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "n", "*", "*", "0.1000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "np", "*", "*", "5.5000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "n=", "*", "*", "0.5000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "nr", "*", "*", "1.0035", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "o", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "s", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "p", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "c'", "*", "*", "36.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "cp", "*", "*", "10.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "c=", "*", "*", "2.0568", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "c+", "*", "*", "54.4060", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "na", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "n", "*", "*", "0.1000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "np", "*", "*", "5.5000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "n=", "*", "*", "0.5000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "nr", "*", "*", "1.0035", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "o", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "s", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"},
                                 {"*", "p", "*", "*", "0.0000", "0.0", "Ver. 2.0, Ref. 1"}};

/*
  Non-bonded (9-6)
 0= Key, 1= r, 2= eps, 3= CFF91 info
 E = eps(ij) [2(R(ij)/r(ij))^9 - 3(R(ij)/r(ij))^6]
 where    R(ij) = [(r(i)^6 + r(j)^6))/2]^(1/6)

        eps(ij) = 2 sqrt(eps(i) * eps(j)) *
                   r(i)^3 * r(j)^3/[r(i)^6 + r(j)^6]
*/
char * CFF91_vdw[40][4]= {{"c", "4.0100", "0.0540", "Ver. 2.0, Ref. 1"},
                          {"c*", "3.3080", "0.1200", "Ver. 2.0, Ref. 4"},
                          {"c'", "3.3080", "0.1200", "Ver. 2.0, Ref. 1"},
                          {"cp", "4.0100", "0.0640", "Ver. 2.0, Ref. 1"},
                          {"c=", "4.0100", "0.0640", "Ver. 2.0, Ref. 1"},
                          {"c=1", "4.0100", "0.0640", "Ver. 2.0, Ref. 3"},
                          {"c=2", "4.0100", "0.0640", "Ver. 2.0, Ref. 3"},
                          {"c-", "3.9080", "0.1200", "Ver. 2.0, Ref. 1"},
                          {"c+", "3.3080", "0.1200", "Ver. 2.0, Ref. 1"},
                          {"cr", "3.3080", "0.1200", "Ver. 2.0, Ref. 1"},
                          {"na", "4.0700", "0.0650", "Ver. 2.0, Ref. 1"},
                          {"n", "4.0700", "0.1060", "Ver. 2.0, Ref. 1"},
                          {"nr", "4.0700", "0.1060", "Ver. 2.0, Ref. 1"},
                          {"np", "3.5700", "0.0410", "Ver. 2.0, Ref. 1"},
                          {"nh", "4.0700", "0.1340", "Ver. 2.0, Ref. 1"},
                          {"nn", "4.0700", "0.0650", "Ver. 2.0, Ref. 1"},
                          {"n=", "4.1210", "0.0620", "Ver. 2.0, Ref. 1"},
                          {"n=1", "4.1210", "0.0620", "Ver. 2.0, Ref. 3"},
                          {"n=2", "4.1210", "0.0620", "Ver. 2.0, Ref. 3"},
                          {"n+", "3.2620", "0.0650", "Ver. 2.0, Ref. 1"},
                          {"o", "3.5350", "0.2400", "Ver. 2.0, Ref. 1"},
                          {"o*", "3.6080", "0.2740", "Ver. 2.0, Ref. 1"},
                          {"o'", "3.5350", "0.2670", "Ver. 2.0, Ref. 1"},
                          {"op", "3.5350", "0.1090", "Ver. 2.0, Ref. 1"},
                          {"o-", "3.5960", "0.1670", "Ver. 2.0, Ref. 1"},
                          {"s", "4.0270", "0.0710", "Ver. 2.0, Ref. 1"},
                          {"sp", "4.0270", "0.0710", "Ver. 2.0, Ref. 1"},
                          {"h", "2.9950", "0.0200", "Ver. 2.0, Ref. 1"},
                          {"h*", "1.0980", "0.0130", "Ver. 2.0, Ref. 1"},
                          {"h+", "1.0980", "0.0130", "Ver. 2.0, Ref. 1"},
                          {"p", "4.2950", "0.2150", "Ver. 2.0, Ref. 2"},
                          {"f", "3.2850", "0.3050", "Ver. 2.0, Ref. 2"},
                          {"cl", "3.9150", "0.3050", "Ver. 2.0, Ref. 2"},
                          {"Cl", "4.0000", "0.4000", "Ver. 2.0, Ref. 2"},
                          {"br", "4.2150", "0.3050", "Ver. 2.0, Ref. 2"},
                          {"Br", "4.3000", "0.3600", "Ver. 2.0, Ref. 2"},
                          {"i", "4.8000", "0.4000", "Ver. 2.0, Ref. 2"},
                          {"si", "4.4350", "0.0950", "Ver. 2.0, Ref. 2"},
                          {"Na", "3.1440", "0.5000", "Ver. 2.0, Ref. 2"},
                          {"ca+", "3.4720", "0.0500", "Ver. 2.0, Ref. 2"}};

/*
 Bond increments
 To implement atomic charges
*/
char * CFF91_bond_increments[560][5]= {{"c", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"c*", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                                       {"c'", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"cp", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"c=", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"c=1", "c", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=2", "c", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"cr", "c", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"c-", "c", "0.2300", "-0.2300", "Ver. 1.0, Ref. 1"},
                                       {"ct", "c", "-0.0400", "0.0400", "Ver. 1.0, Ref. 1"},
                                       {"ct2", "c", "-0.0400", "0.0400", "Ver. 2.0, Ref. 6"},
                                       {"na", "c", "-0.0827", "0.0827", "Ver. 1.0, Ref. 1"},
                                       {"n", "c", "-0.2108", "0.2108", "Ver. 1.0, Ref. 1"},
                                       {"nn", "c", "-0.2108", "0.2108", "Ver. 1.0, Ref. 2"},
                                       {"nr", "c", "-0.4802", "0.4802", "Ver. 1.0, Ref. 1"},
                                       {"np", "c", "-0.1100", "0.1100", "Ver. 1.0, Ref. 2"},
                                       {"nh", "c", "-0.0610", "0.0610", "Ver. 1.0, Ref. 2"},
                                       {"n=", "c", "-0.3000", "0.3000", "Ver. 1.0, Ref. 1"},
                                       {"n=1", "c", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "c", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "c", "-0.1571", "0.4071", "Ver. 1.0, Ref. 1"},
                                       {"nt", "c", "-0.3640", "0.3640", "Ver. 1.0, Ref. 2"},
                                       {"o", "c", "-0.1133", "0.1133", "Ver. 1.0, Ref. 1"},
                                       {"op", "c", "-0.3957", "0.3957", "Ver. 2.0, Ref. 2"},
                                       {"s", "c", "-0.0650", "0.0650", "Ver. 1.0, Ref. 1"},
                                       {"s'", "c", "-0.1180", "0.1180", "Ver. 2.0, Ref. 2"},
                                       {"sp", "c", "-0.1180", "0.1180", "Ver. 2.0, Ref. 2"},
                                       {"h", "c", "0.0530", "-0.0530", "Ver. 1.0, Ref. 1"},
                                       {"p", "c", "-0.0110", "0.0110", "Ver. 2.0, Ref. 2"},
                                       {"f", "c", "-0.2750", "0.2750", "Ver. 2.0, Ref. 2"},
                                       {"cl", "c", "-0.2260", "0.2260", "Ver. 2.0, Ref. 2"},
                                       {"br", "c", "-0.1920", "0.1920", "Ver. 2.0, Ref. 2"},
                                       {"i", "c", "-0.1120", "0.1120", "Ver. 2.0, Ref. 2"},
                                       {"si", "c", "0.1767", "-0.1767", "Ver. 2.0, Ref. 2"},
                                       {"c'", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"c*", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"c*", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"cp", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"c=", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"c=1", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=2", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c-", "c'", "0.1368", "-0.1368", "Ver. 2.0, Ref. 2"},
                                       {"ct", "c'", "0.0927", "-0.0927", "Ver. 2.0, Ref. 2"},
                                       {"ct2", "c'", "0.0927", "-0.0927", "Ver. 2.0, Ref. 6"},
                                       {"n", "c'", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"nn", "c'", "-0.0362", "0.0362", "Ver. 2.0, Ref. 2"},
                                       {"np", "c'", "-0.0362", "0.0362", "Ver. 2.0, Ref. 2"},
                                       {"nh", "c'", "0.1422", "-0.1422", "Ver. 2.0, Ref. 2"},
                                       {"n=", "c'", "-0.0362", "0.0362", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "c'", "-0.0362", "0.0362", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "c'", "-0.0362", "0.0362", "Ver. 2.0, Ref. 3"},
                                       {"n+", "c'", "0.1169", "0.1331", "Ver. 2.0, Ref. 2"},
                                       {"nt", "c'", "-0.1641", "0.1641", "Ver. 2.0, Ref. 2"},
                                       {"cp", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"c=", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"c=1", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"c=2", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"c-", "c*", "0.1368", "-0.1368", "Ver. 2.0, Ref. 4"},
                                       {"ct", "c*", "0.0927", "-0.0927", "Ver. 2.0, Ref. 4"},
                                       {"ct2", "c*", "0.0927", "-0.0927", "Ver. 2.0, Ref. 6"},
                                       {"n", "c*", "0.0000", "0.0000", "Ver. 1.0, Ref. 4"},
                                       {"nn", "c*", "-0.0362", "0.0362", "Ver. 2.0, Ref. 4"},
                                       {"np", "c*", "-0.0362", "0.0362", "Ver. 2.0, Ref. 4"},
                                       {"nh", "c*", "0.1422", "-0.1422", "Ver. 2.0, Ref. 4"},
                                       {"n=", "c*", "-0.0362", "0.0362", "Ver. 2.0, Ref. 4"},
                                       {"n=1", "c*", "-0.0362", "0.0362", "Ver. 2.0, Ref. 4"},
                                       {"n=2", "c*", "-0.0362", "0.0362", "Ver. 2.0, Ref. 4"},
                                       {"n+", "c*", "0.1169", "0.1331", "Ver. 2.0, Ref. 4"},
                                       {"nt", "c*", "-0.1641", "0.1641", "Ver. 2.0, Ref. 4"},
                                       {"o", "c'", "-0.0030", "0.0030", "Ver. 1.0, Ref. 1"},
                                       {"o", "c*", "-0.0030", "0.0030", "Ver. 1.0, Ref. 4"},
                                       {"o'", "c'", "-0.3964", "0.3964", "Ver. 1.0, Ref. 1"},
                                       {"o'", "c*", "-0.3964", "0.3964", "Ver. 1.0, Ref. 4"},
                                       {"op", "c'", "-0.0283", "0.0283", "Ver. 1.0, Ref. 1"},
                                       {"sp", "c'", "0.0130", "-0.0130", "Ver. 1.0, Ref. 1"},
                                       {"s", "c'", "0.0140", "-0.0140", "Ver. 2.0, Ref. 2"},
                                       {"s'", "c'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"op", "c*", "-0.0283", "0.0283", "Ver. 1.0, Ref. 4"},
                                       {"sp", "c*", "0.0130", "-0.0130", "Ver. 1.0, Ref. 4"},
                                       {"s", "c*", "0.0140", "-0.0140", "Ver. 2.0, Ref. 4"},
                                       {"s'", "c*", "0.0000", "0.0000", "Ver. 2.0, Ref. 4"},
                                       {"h", "c'", "0.0456", "-0.0456", "Ver. 1.0, Ref. 1"},
                                       {"h", "c*", "0.0456", "-0.0456", "Ver. 1.0, Ref. 4"},
                                       {"p", "c'", "0.2396", "-0.2396", "Ver. 2.0, Ref. 2"},
                                       {"f", "c'", "-0.1300", "0.1300", "Ver. 2.0, Ref. 2"},
                                       {"cl", "c'", "-0.1020", "0.1020", "Ver. 2.0, Ref. 2"},
                                       {"br", "c'", "-0.0800", "0.0800", "Ver. 2.0, Ref. 2"},
                                       {"i", "c'", "0.1291", "-0.1291", "Ver. 2.0, Ref. 2"},
                                       {"si", "c'", "0.4405", "-0.4405", "Ver. 2.0, Ref. 2"},
                                       {"p", "c*", "0.2396", "-0.2396", "Ver. 2.0, Ref. 4"},
                                       {"f", "c*", "-0.1300", "0.1300", "Ver. 2.0, Ref. 4"},
                                       {"cl", "c*", "-0.1020", "0.1020", "Ver. 2.0, Ref. 4"},
                                       {"br", "c*", "-0.0800", "0.0800", "Ver. 2.0, Ref. 4"},
                                       {"i", "c*", "0.1291", "-0.1291", "Ver. 2.0, Ref. 4"},
                                       {"si", "c*", "0.4405", "-0.4405", "Ver. 2.0, Ref. 4"},
                                       {"cp", "cp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"c=", "cp", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"c=1", "cp", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=2", "cp", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c-", "cp", "-0.0424", "0.0424", "Ver. 2.0, Ref. 2"},
                                       {"ct", "cp", "-0.0852", "0.0852", "Ver. 2.0, Ref. 2"},
                                       {"ct2", "cp", "-0.0852", "0.0852", "Ver. 2.0, Ref. 6"},
                                       {"na", "cp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"n", "cp", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"n+", "cp", "-0.0489", "0.2989", "Ver. 2.0, Ref. 2"},
                                       {"nn", "cp", "-0.0827", "0.0827", "Ver. 1.0, Ref. 1"},
                                       {"np", "cp", "-0.2405", "0.2405", "Ver. 1.0, Ref. 1"},
                                       {"nh", "cp", "0.0500", "-0.0500", "Ver. 1.0, Ref. 1"},
                                       {"n=", "cp", "-0.1993", "0.1993", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "cp", "-0.1993", "0.1993", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "cp", "-0.1993", "0.1993", "Ver. 2.0, Ref. 3"},
                                       {"o", "cp", "-0.0265", "0.0265", "Ver. 1.0, Ref. 1"},
                                       {"o'", "cp", "-0.3964", "0.3964", "Ver. 2.0, Ref. 2"},
                                       {"op", "cp", "-0.0283", "0.0283", "Ver. 1.0, Ref. 1"},
                                       {"sp", "cp", "0.0130", "-0.0130", "Ver. 1.0, Ref. 1"},
                                       {"s", "cp", "0.0120", "-0.0120", "Ver. 2.0, Ref. 2"},
                                       {"s'", "cp", "-0.0732", "0.0732", "Ver. 2.0, Ref. 2"},
                                       {"h", "cp", "0.1268", "-0.1268", "Ver. 1.0, Ref. 1"},
                                       {"p", "cp", "0.0380", "-0.0380", "Ver. 1.0, Ref. 2"},
                                       {"f", "cp", "-0.1300", "0.1300", "Ver. 2.0, Ref. 2"},
                                       {"cl", "cp", "-0.1020", "0.1020", "Ver. 2.0, Ref. 2"},
                                       {"br", "cp", "-0.0800", "0.0800", "Ver. 2.0, Ref. 2"},
                                       {"i", "cp", "-0.0642", "0.0642", "Ver. 2.0, Ref. 2"},
                                       {"si", "cp", "0.2270", "-0.2270", "Ver. 2.0, Ref. 2"},
                                       {"c=", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 1"},
                                       {"c=1", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=2", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=1", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=1", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"c=2", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"ct", "c=", "-0.0852", "0.0852", "Ver. 2.0, Ref. 2"},
                                       {"ct2", "c=", "-0.0852", "0.0852", "Ver. 2.0, Ref. 6"},
                                       {"na", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"n", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"nn", "c=", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n+", "c=", "-0.0489", "0.2989", "Ver. 2.0, Ref. 2"},
                                       {"np", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"nh", "c=", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"n=", "c=", "-0.3000", "0.3000", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "c=", "-0.3000", "0.3000", "Ver. 2.0, Ref. 2"},
                                       {"n=2", "c=", "-0.3000", "0.3000", "Ver. 2.0, Ref. 2"},
                                       {"n=", "c=1", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=1", "c=1", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "c=1", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=", "c=2", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=1", "c=2", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "c=2", "-0.3000", "0.3000", "Ver. 2.0, Ref. 3"},
                                       {"o", "c=", "-0.0265", "0.0265", "Ver. 1.0, Ref. 1"},
                                       {"op", "c=", "-0.3583", "0.3583", "Ver. 2.0, Ref. 2"},
                                       {"o'", "c=", "-0.3583", "0.3583", "Ver. 2.0, Ref. 2"},
                                       {"s", "c=", "0.0120", "-0.0120", "Ver. 2.0, Ref. 2"},
                                       {"sp", "c=", "-0.0732", "0.0732", "Ver. 2.0, Ref. 2"},
                                       {"s'", "c=", "-0.0732", "0.0732", "Ver. 2.0, Ref. 2"},
                                       {"h", "c=", "0.1268", "-0.1268", "Ver. 1.0, Ref. 1"},
                                       {"p", "c=", "0.0380", "-0.0380", "Ver. 2.0, Ref. 2"},
                                       {"f", "c=", "-0.1300", "0.1300", "Ver. 2.0, Ref. 2"},
                                       {"cl", "c=", "-0.1020", "0.1020", "Ver. 2.0, Ref. 2"},
                                       {"br", "c=", "-0.0800", "0.0800", "Ver. 2.0, Ref. 2"},
                                       {"i", "c=", "-0.0642", "0.0642", "Ver. 2.0, Ref. 2"},
                                       {"si", "c=", "0.2270", "-0.2270", "Ver. 2.0, Ref. 2"},
                                       {"ct", "c=1", "-0.0852", "0.0852", "Ver. 2.0, Ref. 3"},
                                       {"ct2", "c=1", "-0.0852", "0.0852", "Ver. 2.0, Ref. 6"},
                                       {"na", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"nn", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "c=1", "-0.0489", "0.2989", "Ver. 2.0, Ref. 3"},
                                       {"np", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"nh", "c=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"o", "c=1", "-0.0265", "0.0265", "Ver. 2.0, Ref. 3"},
                                       {"op", "c=1", "-0.3583", "0.3583", "Ver. 2.0, Ref. 3"},
                                       {"o'", "c=1", "-0.3583", "0.3583", "Ver. 2.0, Ref. 3"},
                                       {"s", "c=1", "0.0120", "-0.0120", "Ver. 2.0, Ref. 3"},
                                       {"sp", "c=1", "-0.0732", "0.0732", "Ver. 2.0, Ref. 3"},
                                       {"s'", "c=1", "-0.0732", "0.0732", "Ver. 2.0, Ref. 3"},
                                       {"h", "c=1", "0.1268", "-0.1268", "Ver. 2.0, Ref. 3"},
                                       {"p", "c=1", "0.0380", "-0.0380", "Ver. 2.0, Ref. 3"},
                                       {"f", "c=1", "-0.1300", "0.1300", "Ver. 2.0, Ref. 3"},
                                       {"cl", "c=1", "-0.1020", "0.1020", "Ver. 2.0, Ref. 3"},
                                       {"br", "c=1", "-0.0800", "0.0800", "Ver. 2.0, Ref. 3"},
                                       {"i", "c=1", "-0.0642", "0.0642", "Ver. 2.0, Ref. 3"},
                                       {"si", "c=1", "0.2270", "-0.2270", "Ver. 2.0, Ref. 3"},
                                       {"ct", "c=2", "-0.0852", "0.0852", "Ver. 2.0, Ref. 3"},
                                       {"ct2", "c=2", "-0.0852", "0.0852", "Ver. 2.0, Ref. 6"},
                                       {"na", "c=2", "0.0000", "0.0000", "Ver. 1.0, Ref. 3"},
                                       {"n", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"nn", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "c=2", "-0.0489", "0.2989", "Ver. 2.0, Ref. 3"},
                                       {"np", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"nh", "c=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"o", "c=2", "-0.0265", "0.0265", "Ver. 2.0, Ref. 3"},
                                       {"op", "c=2", "-0.3583", "0.3583", "Ver. 2.0, Ref. 3"},
                                       {"o'", "c=2", "-0.3583", "0.3583", "Ver. 2.0, Ref. 3"},
                                       {"s", "c=2", "0.0120", "-0.0120", "Ver. 2.0, Ref. 3"},
                                       {"sp", "c=2", "-0.0732", "0.0732", "Ver. 2.0, Ref. 3"},
                                       {"s'", "c=2", "-0.0732", "0.0732", "Ver. 2.0, Ref. 3"},
                                       {"h", "c=2", "0.1268", "-0.1268", "Ver. 2.0, Ref. 3"},
                                       {"p", "c=2", "0.0380", "-0.0380", "Ver. 2.0, Ref. 3"},
                                       {"f", "c=2", "-0.1300", "0.1300", "Ver. 2.0, Ref. 3"},
                                       {"cl", "c=2", "-0.1020", "0.1020", "Ver. 2.0, Ref. 3"},
                                       {"br", "c=2", "-0.0800", "0.0800", "Ver. 2.0, Ref. 3"},
                                       {"i", "c=2", "-0.0642", "0.0642", "Ver. 2.0, Ref. 3"},
                                       {"si", "c=2", "0.2270", "-0.2270", "Ver. 2.0, Ref. 3"},
                                       {"n=", "cr", "-0.4000", "0.4000", "Ver. 1.0, Ref. 1"},
                                       {"n=1", "cr", "-0.4000", "0.4000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "cr", "-0.4000", "0.4000", "Ver. 2.0, Ref. 3"},
                                       {"nr", "cr", "-0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"o-", "c-", "-0.5337", "0.0337", "Ver. 1.0, Ref. 1"},
                                       {"h", "c-", "0.0530", "-0.0530", "Ver. 1.0, Ref. 1"},
                                       {"c-", "c-", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"ct", "c-", "-0.0432", "0.0432", "Ver. 2.0, Ref. 2"},
                                       {"ct2", "c-", "-0.0432", "0.0432", "Ver. 2.0, Ref. 6"},
                                       {"n", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 2"},
                                       {"n+", "c-", "-0.0097", "0.2597", "Ver. 2.0, Ref. 2"},
                                       {"nn", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 2"},
                                       {"np", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 2"},
                                       {"n=", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "c-", "-0.1607", "0.1607", "Ver. 2.0, Ref. 3"},
                                       {"nh", "c-", "0.0176", "-0.0176", "Ver. 2.0, Ref. 2"},
                                       {"o", "c-", "-0.1012", "0.1012", "Ver. 2.0, Ref. 2"},
                                       {"op", "c-", "-0.3241", "0.3241", "Ver. 2.0, Ref. 2"},
                                       {"o'", "c-", "-0.3241", "0.3241", "Ver. 2.0, Ref. 2"},
                                       {"s", "c-", "0.0087", "-0.0087", "Ver. 2.0, Ref. 2"},
                                       {"s-", "c-", "-0.3777", "-0.1223", "Ver. 2.0, Ref. 2"},
                                       {"p", "c-", "0.0857", "-0.0857", "Ver. 2.0, Ref. 2"},
                                       {"f", "c-", "-0.2241", "0.2241", "Ver. 2.0, Ref. 2"},
                                       {"cl", "c-", "-0.0747", "0.0747", "Ver. 2.0, Ref. 2"},
                                       {"br", "c-", "-0.0281", "0.0281", "Ver. 2.0, Ref. 2"},
                                       {"i", "c-", "-0.0185", "0.0185", "Ver. 2.0, Ref. 2"},
                                       {"si", "c-", "0.2775", "-0.2775", "Ver. 2.0, Ref. 2"},
                                       {"nr", "c+", "0.0680", "0.2653", "Ver. 1.0, Ref. 1"},
                                       {"ct", "ct", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"ct2", "ct2", "0.0000", "0.0000", "Ver. 1.0, Ref. 6"},
                                       {"na", "ct", "0.0636", "-0.0636", "Ver. 2.0, Ref. 2"},
                                       {"n", "ct", "-0.1204", "0.1204", "Ver. 2.0, Ref. 2"},
                                       {"nn", "ct", "-0.0920", "0.0920", "Ver. 2.0, Ref. 2"},
                                       {"n+", "ct", "0.1508", "0.0992", "Ver. 2.0, Ref. 2"},
                                       {"nt", "ct", "-0.2454", "0.2454", "Ver. 2.0, Ref. 2"},
                                       {"np", "ct", "-0.1204", "0.1204", "Ver. 2.0, Ref. 2"},
                                       {"nh", "ct", "0.0568", "-0.0568", "Ver. 2.0, Ref. 2"},
                                       {"n=", "ct", "-0.1204", "0.1204", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "ct", "-0.1204", "0.1204", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "ct", "-0.1204", "0.1204", "Ver. 2.0, Ref. 3"},
                                       {"o", "ct", "-0.0675", "0.0675", "Ver. 2.0, Ref. 2"},
                                       {"op", "ct", "-0.2874", "0.2874", "Ver. 2.0, Ref. 2"},
                                       {"o'", "ct", "-0.2874", "0.2874", "Ver. 2.0, Ref. 2"},
                                       {"s", "ct", "0.0581", "-0.0581", "Ver. 2.0, Ref. 2"},
                                       {"sp", "ct", "0.0135", "-0.0135", "Ver. 2.0, Ref. 2"},
                                       {"s'", "ct", "0.0135", "-0.0135", "Ver. 2.0, Ref. 2"},
                                       {"p", "ct", "0.1335", "-0.1335", "Ver. 2.0, Ref. 2"},
                                       {"h", "ct", "0.2052", "-0.2052", "Ver. 2.0, Ref. 2"},
                                       {"f", "ct", "-0.1873", "0.1873", "Ver. 2.0, Ref. 2"},
                                       {"cl", "ct", "-0.0319", "0.0319", "Ver. 2.0, Ref. 2"},
                                       {"br", "ct", "0.0173", "-0.0173", "Ver. 2.0, Ref. 2"},
                                       {"i", "ct", "0.0281", "-0.0281", "Ver. 2.0, Ref. 2"},
                                       {"na", "ct2", "0.0636", "-0.0636", "Ver. 2.0, Ref. 6"},
                                       {"n", "ct2", "-0.1204", "0.1204", "Ver. 2.0, Ref. 6"},
                                       {"nn", "ct2", "-0.0920", "0.0920", "Ver. 2.0, Ref. 6"},
                                       {"n+", "ct2", "0.1508", "0.0992", "Ver. 2.0, Ref. 6"},
                                       {"nt", "ct2", "-0.2454", "0.2454", "Ver. 2.0, Ref. 2"},
                                       {"np", "ct2", "-0.1204", "0.1204", "Ver. 2.0, Ref. 6"},
                                       {"nh", "ct2", "0.0568", "-0.0568", "Ver. 2.0, Ref. 6"},
                                       {"n=", "ct2", "-0.1204", "0.1204", "Ver. 2.0, Ref. 6"},
                                       {"n=1", "ct2", "-0.1204", "0.1204", "Ver. 2.0, Ref. 6"},
                                       {"n=2", "ct2", "-0.1204", "0.1204", "Ver. 2.0, Ref. 6"},
                                       {"o", "ct2", "-0.0675", "0.0675", "Ver. 2.0, Ref. 6"},
                                       {"op", "ct2", "-0.2874", "0.2874", "Ver. 2.0, Ref. 6"},
                                       {"o'", "ct2", "-0.2874", "0.2874", "Ver. 2.0, Ref. 6"},
                                       {"s", "ct2", "0.0581", "-0.0581", "Ver. 2.0, Ref. 6"},
                                       {"sp", "ct2", "0.0135", "-0.0135", "Ver. 2.0, Ref. 6"},
                                       {"s'", "ct2", "0.0135", "-0.0135", "Ver. 2.0, Ref. 6"},
                                       {"p", "ct2", "0.1335", "-0.1335", "Ver. 2.0, Ref. 6"},
                                       {"h", "ct2", "0.2052", "-0.2052", "Ver. 2.0, Ref. 6"},
                                       {"f", "ct2", "-0.1873", "0.1873", "Ver. 2.0, Ref. 6"},
                                       {"cl", "ct2", "-0.0319", "0.0319", "Ver. 2.0, Ref. 6"},
                                       {"br", "ct2", "0.0173", "-0.0173", "Ver. 2.0, Ref. 6"},
                                       {"i", "ct2", "0.0281", "-0.0281", "Ver. 2.0, Ref. 6"},
                                       {"si", "ct2", "0.3266", "-0.3266", "Ver. 2.0, Ref. 6"},
                                       {"na", "na", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n", "na", "-0.0742", "0.0742", "Ver. 2.0, Ref. 2"},
                                       {"nn", "na", "-0.0371", "0.0371", "Ver. 2.0, Ref. 2"},
                                       {"np", "na", "-0.0742", "0.0742", "Ver. 2.0, Ref. 2"},
                                       {"nh", "na", "0.0919", "-0.0919", "Ver. 2.0, Ref. 2"},
                                       {"n+", "na", "0.0850", "0.1650", "Ver. 2.0, Ref. 2"},
                                       {"nt", "na", "-0.1927", "0.1927", "Ver. 2.0, Ref. 2"},
                                       {"n=", "na", "-0.0742", "0.0742", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "na", "-0.0742", "0.0742", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "na", "-0.0742", "0.0742", "Ver. 2.0, Ref. 3"},
                                       {"o", "na", "-0.0254", "0.0254", "Ver. 2.0, Ref. 2"},
                                       {"op", "na", "-0.2369", "0.2369", "Ver. 2.0, Ref. 2"},
                                       {"o'", "na", "-0.2369", "0.2369", "Ver. 2.0, Ref. 2"},
                                       {"s", "na", "0.0966", "-0.0966", "Ver. 2.0, Ref. 2"},
                                       {"sp", "na", "0.0551", "-0.0551", "Ver. 2.0, Ref. 2"},
                                       {"s'", "na", "0.0551", "-0.0551", "Ver. 2.0, Ref. 2"},
                                       {"p", "na", "0.2518", "-0.2518", "Ver. 2.0, Ref. 2"},
                                       {"f", "na", "-0.1415", "0.1415", "Ver. 2.0, Ref. 2"},
                                       {"cl", "na", "0.0117", "-0.0117", "Ver. 2.0, Ref. 2"},
                                       {"br", "na", "0.0601", "-0.0601", "Ver. 2.0, Ref. 2"},
                                       {"i", "na", "0.0714", "-0.0714", "Ver. 2.0, Ref. 2"},
                                       {"si", "na", "0.3501", "-0.3501", "Ver. 2.0, Ref. 2"},
                                       {"h*", "na", "0.2487", "-0.2487", "Ver. 1.0, Ref. 1"},
                                       {"n", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"nn", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"np", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"nh", "n", "0.1637", "-0.1637", "Ver. 2.0, Ref. 2"},
                                       {"n=", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "n", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "n", "0.1617", "0.0883", "Ver. 2.0, Ref. 2"},
                                       {"nt", "n", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"o", "n", "0.0427", "-0.0427", "Ver. 2.0, Ref. 2"},
                                       {"op", "n", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o'", "n", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"s", "n", "0.1753", "-0.1753", "Ver. 2.0, Ref. 2"},
                                       {"sp", "n", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"s'", "n", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"h*", "n", "0.4395", "-0.4395", "Ver. 1.0, Ref. 1"},
                                       {"p", "n", "0.3359", "-0.3359", "Ver. 2.0, Ref. 2"},
                                       {"f", "n", "-0.0731", "0.0731", "Ver. 2.0, Ref. 2"},
                                       {"cl", "n", "0.0897", "-0.0897", "Ver. 2.0, Ref. 2"},
                                       {"br", "n", "0.1422", "-0.1422", "Ver. 2.0, Ref. 2"},
                                       {"i", "n", "0.1554", "-0.1554", "Ver. 2.0, Ref. 2"},
                                       {"si", "n", "0.4368", "-0.4368", "Ver. 2.0, Ref. 2"},
                                       {"nn", "nn", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"np", "nn", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"nh", "nn", "0.1637", "-0.1637", "Ver. 2.0, Ref. 2"},
                                       {"n=", "nn", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "nn", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "nn", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "nn", "0.1617", "0.0883", "Ver. 2.0, Ref. 2"},
                                       {"nt", "nn", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"o", "nn", "0.0427", "-0.0427", "Ver. 2.0, Ref. 2"},
                                       {"op", "nn", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o'", "nn", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"s", "nn", "0.1753", "-0.1753", "Ver. 2.0, Ref. 2"},
                                       {"sp", "nn", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"s'", "nn", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"h*", "nn", "0.2487", "-0.2487", "Ver. 1.0, Ref. 1"},
                                       {"p", "nn", "0.3359", "-0.3359", "Ver. 2.0, Ref. 2"},
                                       {"f", "nn", "-0.0731", "0.0731", "Ver. 2.0, Ref. 2"},
                                       {"cl", "nn", "0.0897", "-0.0897", "Ver. 2.0, Ref. 2"},
                                       {"br", "nn", "0.1422", "-0.1422", "Ver. 2.0, Ref. 2"},
                                       {"i", "nn", "0.1554", "-0.1554", "Ver. 2.0, Ref. 2"},
                                       {"si", "nn", "0.4368", "-0.4368", "Ver. 2.0, Ref. 2"},
                                       {"np", "np", "0.0000", "0.0000", "Ver. 1.0, Ref. 1"},
                                       {"nh", "np", "-0.2823", "0.2823", "Ver. 1.0, Ref. 1"},
                                       {"n=", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "np", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "np", "0.1617", "0.0883", "Ver. 2.0, Ref. 2"},
                                       {"nt", "np", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"o", "np", "0.0427", "-0.0427", "Ver. 2.0, Ref. 2"},
                                       {"op", "np", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o'", "np", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o-", "np", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"s", "np", "0.1753", "-0.1753", "Ver. 2.0, Ref. 2"},
                                       {"sp", "np", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"s'", "np", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"h*", "np", "0.3278", "-0.3278", "Ver. 2.0, Ref. 2"},
                                       {"p", "np", "0.3359", "-0.3359", "Ver. 2.0, Ref. 2"},
                                       {"f", "np", "-0.0731", "0.0731", "Ver. 2.0, Ref. 2"},
                                       {"cl", "np", "0.0897", "-0.0897", "Ver. 2.0, Ref. 2"},
                                       {"br", "np", "0.1422", "-0.1422", "Ver. 2.0, Ref. 2"},
                                       {"i", "np", "0.1554", "-0.1554", "Ver. 2.0, Ref. 2"},
                                       {"si", "np", "0.4368", "-0.4368", "Ver. 2.0, Ref. 2"},
                                       {"nh", "nh", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n=", "nh", "-0.1637", "0.1637", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "nh", "-0.1637", "0.1637", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "nh", "-0.1637", "0.1637", "Ver. 2.0, Ref. 3"},
                                       {"n+", "nh", "-0.0044", "0.2544", "Ver. 2.0, Ref. 2"},
                                       {"nt", "nh", "-0.2783", "0.2783", "Ver. 2.0, Ref. 2"},
                                       {"o", "nh", "-0.1090", "0.1090", "Ver. 2.0, Ref. 2"},
                                       {"op", "nh", "-0.3148", "0.3148", "Ver. 2.0, Ref. 2"},
                                       {"o'", "nh", "-0.3148", "0.3148", "Ver. 2.0, Ref. 2"},
                                       {"s", "nh", "-0.0046", "0.0046", "Ver. 2.0, Ref. 2"},
                                       {"sp", "nh", "-0.0454", "0.0454", "Ver. 2.0, Ref. 2"},
                                       {"s'", "nh", "-0.0454", "0.0454", "Ver. 2.0, Ref. 2"},
                                       {"h*", "nh", "0.3925", "-0.3925", "Ver. 1.0, Ref. 1"},
                                       {"hi", "nh", "0.3925", "0.1075", "Ver. 1.0, Ref. 1"},
                                       {"p", "nh", "0.1375", "-0.1375", "Ver. 2.0, Ref. 2"},
                                       {"f", "nh", "-0.2220", "0.2220", "Ver. 2.0, Ref. 2"},
                                       {"cl", "nh", "-0.0854", "0.0854", "Ver. 2.0, Ref. 2"},
                                       {"br", "nh", "-0.0438", "0.0438", "Ver. 2.0, Ref. 2"},
                                       {"i", "nh", "-0.0356", "0.0358", "Ver. 2.0, Ref. 2"},
                                       {"si", "nh", "0.2278", "-0.2278", "Ver. 2.0, Ref. 2"},
                                       {"n=", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"n=1", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "n=", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=1", "n=1", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=2", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n=1", "n=2", "0.0000", "0.0000", "Ver. 2.0, Ref. 3"},
                                       {"n+", "n=", "0.1617", "0.0883", "Ver. 2.0, Ref. 2"},
                                       {"nt", "n=", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"nt", "n=1", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"nt", "n=2", "-0.1186", "0.1186", "Ver. 2.0, Ref. 2"},
                                       {"o", "n=", "0.0427", "-0.0427", "Ver. 2.0, Ref. 2"},
                                       {"op", "n=", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o'", "n=", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"o-", "n=", "-0.1684", "0.1684", "Ver. 2.0, Ref. 2"},
                                       {"s", "n=", "0.1753", "-0.1753", "Ver. 2.0, Ref. 2"},
                                       {"sp", "n=", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"s'", "n=", "0.1346", "-0.1346", "Ver. 2.0, Ref. 2"},
                                       {"h*", "n=", "0.3278", "-0.3278", "Ver. 2.0, Ref. 2"},
                                       {"p", "n=", "0.3359", "-0.3359", "Ver. 2.0, Ref. 2"},
                                       {"f", "n=", "-0.0731", "0.0731", "Ver. 2.0, Ref. 2"},
                                       {"cl", "n=", "0.0897", "-0.0897", "Ver. 2.0, Ref. 2"},
                                       {"br", "n=", "0.1422", "-0.1422", "Ver. 2.0, Ref. 2"},
                                       {"i", "n=", "0.1554", "-0.1554", "Ver. 2.0, Ref. 2"},
                                       {"si", "n=", "0.4368", "-0.4368", "Ver. 2.0, Ref. 2"},
                                       {"n+", "n=1", "0.1617", "0.0883", "Ver. 2.0, Ref. 3"},
                                       {"o", "n=1", "0.0427", "-0.0427", "Ver. 2.0, Ref. 3"},
                                       {"op", "n=1", "-0.1684", "0.1684", "Ver. 2.0, Ref. 3"},
                                       {"o'", "n=1", "-0.1684", "0.1684", "Ver. 2.0, Ref. 3"},
                                       {"s", "n=1", "0.1753", "-0.1753", "Ver. 2.0, Ref. 3"},
                                       {"sp", "n=1", "0.1346", "-0.1346", "Ver. 2.0, Ref. 3"},
                                       {"s'", "n=1", "0.1346", "-0.1346", "Ver. 2.0, Ref. 3"},
                                       {"h*", "n=1", "0.3278", "-0.3278", "Ver. 2.0, Ref. 3"},
                                       {"p", "n=1", "0.3359", "-0.3359", "Ver. 2.0, Ref. 3"},
                                       {"f", "n=1", "-0.0731", "0.0731", "Ver. 2.0, Ref. 3"},
                                       {"cl", "n=1", "0.0897", "-0.0897", "Ver. 2.0, Ref. 3"},
                                       {"br", "n=1", "0.1422", "-0.1422", "Ver. 2.0, Ref. 3"},
                                       {"i", "n=1", "0.1554", "-0.1554", "Ver. 2.0, Ref. 3"},
                                       {"si", "n=1", "0.4368", "-0.4368", "Ver. 2.0, Ref. 3"},
                                       {"n+", "n=2", "0.1617", "0.0883", "Ver. 2.0, Ref. 3"},
                                       {"o", "n=2", "0.0427", "-0.0427", "Ver. 2.0, Ref. 3"},
                                       {"op", "n=2", "-0.1684", "0.1684", "Ver. 2.0, Ref. 3"},
                                       {"s", "n=2", "0.1753", "-0.1753", "Ver. 2.0, Ref. 3"},
                                       {"sp", "n=2", "0.1346", "-0.1346", "Ver. 2.0, Ref. 3"},
                                       {"o'", "n=2", "-0.1684", "0.1684", "Ver. 2.0, Ref. 3"},
                                       {"s'", "n=2", "0.1346", "-0.1346", "Ver. 2.0, Ref. 3"},
                                       {"h*", "n=2", "0.3278", "-0.3278", "Ver. 2.0, Ref. 3"},
                                       {"p", "n=2", "0.3359", "-0.3359", "Ver. 2.0, Ref. 3"},
                                       {"f", "n=2", "-0.0731", "0.0731", "Ver. 2.0, Ref. 3"},
                                       {"cl", "n=2", "0.0897", "-0.0897", "Ver. 2.0, Ref. 3"},
                                       {"br", "n=2", "0.1422", "-0.1422", "Ver. 2.0, Ref. 3"},
                                       {"i", "n=2", "0.1554", "-0.1554", "Ver. 2.0, Ref. 3"},
                                       {"si", "n=2", "0.4368", "-0.4368", "Ver. 2.0, Ref. 3"},
                                       {"n+", "n+", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"o", "n+", "0.1250", "0.1250", "Ver. 2.0, Ref. 2"},
                                       {"op", "n+", "-0.0918", "0.3418", "Ver. 2.0, Ref. 2"},
                                       {"o'", "n+", "-0.0918", "0.3418", "Ver. 2.0, Ref. 2"},
                                       {"s", "n+", "0.2755", "-0.0255", "Ver. 2.0, Ref. 2"},
                                       {"sp", "n+", "0.2341", "0.0159", "Ver. 2.0, Ref. 2"},
                                       {"s'", "n+", "0.2341", "0.0159", "Ver. 2.0, Ref. 2"},
                                       {"h+", "n+", "0.2800", "-0.0300", "Ver. 1.0, Ref. 1"},
                                       {"p", "n+", "0.4494", "-0.1994", "Ver. 2.0, Ref. 2"},
                                       {"f", "n+", "0.0062", "0.2438", "Ver. 2.0, Ref. 2"},
                                       {"cl", "n+", "0.1858", "0.0642", "Ver. 2.0, Ref. 2"},
                                       {"br", "n+", "0.2452", "0.0048", "Ver. 2.0, Ref. 2"},
                                       {"i", "n+", "0.2615", "-0.0115", "Ver. 2.0, Ref. 2"},
                                       {"si", "n+", "0.5583", "-0.3083", "Ver. 2.0, Ref. 2"},
                                       {"h*", "nr", "0.4068", "-0.4068", "Ver. 1.0, Ref. 1"},
                                       {"nt", "nt", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"o", "nt", "0.1523", "-0.1523", "Ver. 2.0, Ref. 2"},
                                       {"op", "nt", "-0.0585", "0.0585", "Ver. 2.0, Ref. 2"},
                                       {"o'", "nt", "-0.0585", "0.0585", "Ver. 2.0, Ref. 2"},
                                       {"s", "nt", "0.3010", "-0.3010", "Ver. 2.0, Ref. 2"},
                                       {"sp", "nt", "0.2612", "-0.2612", "Ver. 2.0, Ref. 2"},
                                       {"s'", "nt", "0.2612", "-0.2612", "Ver. 2.0, Ref. 2"},
                                       {"p", "nt", "0.4691", "-0.4691", "Ver. 2.0, Ref. 2"},
                                       {"h", "nt", "0.4688", "-0.4688", "Ver. 2.0, Ref. 2"},
                                       {"f", "nt", "0.0367", "-0.0367", "Ver. 2.0, Ref. 2"},
                                       {"cl", "nt", "0.2141", "-0.2141", "Ver. 2.0, Ref. 2"},
                                       {"br", "nt", "0.2727", "-0.2727", "Ver. 2.0, Ref. 2"},
                                       {"i", "nt", "0.2889", "-0.2889", "Ver. 2.0, Ref. 2"},
                                       {"si", "nt", "0.5738", "-0.5738", "Ver. 2.0, Ref. 2"},
                                       {"h*", "o*", "0.3991", "-0.3991", "Ver. 1.0, Ref. 1"},
                                       {"h*", "o", "0.4241", "-0.4241", "Ver. 1.0, Ref. 1"},
                                       {"o", "o", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"op", "o", "-0.1962", "0.1962", "Ver. 2.0, Ref. 2"},
                                       {"o'", "o", "-0.1962", "0.1962", "Ver. 2.0, Ref. 2"},
                                       {"s", "o", "0.1143", "-0.1143", "Ver. 2.0, Ref. 2"},
                                       {"sp", "o", "0.0766", "-0.0766", "Ver. 2.0, Ref. 2"},
                                       {"s'", "o", "0.0766", "-0.0766", "Ver. 2.0, Ref. 2"},
                                       {"p", "o", "0.2548", "-0.2548", "Ver. 2.0, Ref. 2"},
                                       {"f", "o", "-0.1077", "0.1077", "Ver. 2.0, Ref. 2"},
                                       {"cl", "o", "0.0367", "-0.0367", "Ver. 2.0, Ref. 2"},
                                       {"br", "o", "0.0818", "-0.0818", "Ver. 2.0, Ref. 2"},
                                       {"i", "o", "0.0924", "-0.0924", "Ver. 2.0, Ref. 2"},
                                       {"si", "o", "0.3425", "-0.3425", "Ver. 2.0, Ref. 2"},
                                       {"op", "op", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"o'", "op", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"s", "op", "0.3386", "-0.3386", "Ver. 2.0, Ref. 2"},
                                       {"sp", "op", "0.3024", "-0.3024", "Ver. 2.0, Ref. 2"},
                                       {"s'", "op", "0.3024", "-0.3024", "Ver. 2.0, Ref. 2"},
                                       {"p", "op", "0.4933", "-0.4933", "Ver. 2.0, Ref. 2"},
                                       {"h", "op", "0.4943", "-0.4943", "Ver. 2.0, Ref. 2"},
                                       {"f", "op", "0.0888", "-0.0888", "Ver. 2.0, Ref. 2"},
                                       {"cl", "op", "0.2585", "-0.2585", "Ver. 2.0, Ref. 2"},
                                       {"br", "op", "0.3140", "-0.3140", "Ver. 2.0, Ref. 2"},
                                       {"i", "op", "0.3297", "-0.3297", "Ver. 2.0, Ref. 2"},
                                       {"si", "op", "0.5883", "-0.5883", "Ver. 2.0, Ref. 2"},
                                       {"o'", "o'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"s", "o'", "0.3386", "-0.3386", "Ver. 2.0, Ref. 2"},
                                       {"sp", "o'", "0.3024", "-0.3024", "Ver. 2.0, Ref. 2"},
                                       {"s'", "o'", "0.3024", "-0.3024", "Ver. 2.0, Ref. 2"},
                                       {"p", "o'", "0.4933", "-0.4933", "Ver. 2.0, Ref. 2"},
                                       {"h", "o'", "0.4943", "-0.4943", "Ver. 2.0, Ref. 2"},
                                       {"f", "o'", "0.0888", "-0.0888", "Ver. 2.0, Ref. 2"},
                                       {"cl", "o'", "0.2585", "-0.2585", "Ver. 2.0, Ref. 2"},
                                       {"br", "o'", "0.3140", "-0.3140", "Ver. 2.0, Ref. 2"},
                                       {"i", "o'", "0.3297", "-0.3297", "Ver. 2.0, Ref. 2"},
                                       {"si", "o'", "0.5883", "-0.5883", "Ver. 2.0, Ref. 2"},
                                       {"p", "o-", "0.3500", "-0.8500", "Ver. 2.0, Ref. 2"},
                                       {"s", "s", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"sp", "s", "-0.0455", "0.0455", "Ver. 2.0, Ref. 2"},
                                       {"s'", "s", "-0.0455", "0.0455", "Ver. 2.0, Ref. 2"},
                                       {"p", "s", "0.1600", "-0.1600", "Ver. 2.0, Ref. 2"},
                                       {"h", "s", "0.1392", "-0.1392", "Ver. 2.0, Ref. 2"},
                                       {"f", "s", "-0.2380", "0.2380", "Ver. 2.0, Ref. 2"},
                                       {"cl", "s", "-0.0898", "0.0898", "Ver. 2.0, Ref. 2"},
                                       {"br", "s", "-0.0437", "0.0437", "Ver. 2.0, Ref. 2"},
                                       {"i", "s", "-0.0345", "0.0345", "Ver. 2.0, Ref. 2"},
                                       {"si", "s", "0.2634", "-0.2634", "Ver. 2.0, Ref. 2"},
                                       {"sp", "sp", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"s'", "sp", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"p", "sp", "0.2106", "-0.2106", "Ver. 2.0, Ref. 2"},
                                       {"h", "sp", "0.1932", "-0.1932", "Ver. 2.0, Ref. 2"},
                                       {"f", "sp", "-0.2011", "0.2011", "Ver. 2.0, Ref. 2"},
                                       {"cl", "sp", "-0.0457", "0.0457", "Ver. 2.0, Ref. 2"},
                                       {"br", "sp", "0.0034", "-0.0034", "Ver. 2.0, Ref. 2"},
                                       {"i", "sp", "0.0140", "-0.0140", "Ver. 2.0, Ref. 2"},
                                       {"si", "sp", "0.3172", "-0.3172", "Ver. 2.0, Ref. 2"},
                                       {"s'", "s'", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"p", "s'", "0.2106", "-0.2106", "Ver. 2.0, Ref. 2"},
                                       {"h", "s'", "0.1932", "-0.1932", "Ver. 2.0, Ref. 2"},
                                       {"f", "s'", "-0.2011", "0.2011", "Ver. 2.0, Ref. 2"},
                                       {"cl", "s'", "-0.0457", "0.0457", "Ver. 2.0, Ref. 2"},
                                       {"br", "s'", "0.0034", "-0.0034", "Ver. 2.0, Ref. 2"},
                                       {"i", "s'", "0.0140", "-0.0140", "Ver. 2.0, Ref. 2"},
                                       {"si", "s'", "0.3172", "-0.3172", "Ver. 2.0, Ref. 2"},
                                       {"p", "s-", "0.1824", "-0.6824", "Ver. 2.0, Ref. 2"},
                                       {"p", "p", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"h", "p", "-0.0356", "0.0356", "Ver. 2.0, Ref. 2"},
                                       {"f", "p", "-0.3869", "0.3869", "Ver. 2.0, Ref. 2"},
                                       {"cl", "p", "-0.2544", "0.2544", "Ver. 2.0, Ref. 2"},
                                       {"br", "p", "-0.2156", "0.2156", "Ver. 2.0, Ref. 2"},
                                       {"i", "p", "-0.2110", "0.2110", "Ver. 2.0, Ref. 2"},
                                       {"si", "p", "0.1069", "-0.1069", "Ver. 2.0, Ref. 2"},
                                       {"h", "h", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"f", "h", "-0.3823", "0.3823", "Ver. 2.0, Ref. 2"},
                                       {"cl", "h", "-0.2404", "0.2404", "Ver. 2.0, Ref. 2"},
                                       {"br", "h", "-0.1978", "0.1978", "Ver. 2.0, Ref. 2"},
                                       {"i", "h", "-0.1923", "0.1923", "Ver. 2.0, Ref. 2"},
                                       {"si", "h", "0.1537", "-0.1537", "Ver. 2.0, Ref. 2"},
                                       {"f", "f", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"cl", "f", "0.1589", "-0.1589", "Ver. 2.0, Ref. 2"},
                                       {"br", "f", "0.2099", "-0.2099", "Ver. 2.0, Ref. 2"},
                                       {"i", "f", "0.2234", "-0.2234", "Ver. 2.0, Ref. 2"},
                                       {"si", "f", "0.4789", "-0.4789", "Ver. 2.0, Ref. 2"},
                                       {"cl", "cl", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"br", "cl", "0.0507", "-0.0507", "Ver. 2.0, Ref. 2"},
                                       {"i", "cl", "0.0623", "-0.0623", "Ver. 2.0, Ref. 2"},
                                       {"si", "cl", "0.3598", "-0.3598", "Ver. 2.0, Ref. 2"},
                                       {"br", "br", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"i", "br", "0.0110", "-0.0110", "Ver. 2.0, Ref. 2"},
                                       {"si", "br", "0.3273", "-0.3273", "Ver. 2.0, Ref. 2"},
                                       {"i", "i", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"},
                                       {"si", "i", "0.3263", "-0.3263", "Ver. 2.0, Ref. 2"},
                                       {"si", "si", "0.0000", "0.0000", "Ver. 2.0, Ref. 2"}};
