"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@oclif/core");
const chalk_1 = __importDefault(require("chalk"));
const change_case_1 = require("change-case");
const node_path_1 = require("node:path");
const generator_1 = require("../../generator");
class GenerateCommand extends generator_1.GeneratorCommand {
    static args = {
        name: core_1.Args.string({ description: 'name of command', required: true }),
    };
    static description = 'Add a command to an existing CLI or plugin.';
    static flags = {
        'commands-dir': core_1.Flags.string({ default: 'src/commands', description: 'The directory to create the command in.' }),
        force: core_1.Flags.boolean({ description: 'Overwrite existing files.' }),
    };
    async run() {
        const packageJSON = await (0, generator_1.readPJSON)(process.cwd());
        if (!packageJSON)
            throw new core_1.Errors.CLIError('not in a project directory');
        const topicSeparator = packageJSON.oclif?.topicSeparator ?? ':';
        this.log(`Adding ${chalk_1.default.dim(this.args.name.replaceAll(':', topicSeparator))} to ${packageJSON.name}!`);
        const cmdPath = this.args.name.split(':').join('/');
        const destination = (0, node_path_1.join)(process.cwd(), this.flags['commands-dir'], `${cmdPath}.ts`);
        let bin = packageJSON.oclif?.bin ?? packageJSON.oclif?.dirname ?? packageJSON.name;
        if (bin.includes('/'))
            bin = bin.split('/').at(-1);
        const opts = {
            bin,
            className: (0, change_case_1.pascalCase)(this.args.name),
            cmd: `${bin} ${this.args.name}`,
            name: this.args.name,
            path: destination,
            type: 'command',
        };
        await this.template((0, node_path_1.join)(this.templatesDir, 'src', 'command.ts.ejs'), destination, opts);
        if (packageJSON.devDependencies?.mocha) {
            const testTemplatePath = (0, node_path_1.join)(this.templatesDir, 'test', 'command.test.ts.ejs');
            const testDestination = (0, node_path_1.join)(process.cwd(), 'test', 'commands', `${cmdPath}.test.ts`);
            await this.template(testTemplatePath, testDestination, opts);
        }
    }
}
exports.default = GenerateCommand;
