/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.gui;

import docking.ComponentProvider;
import docking.Tool;
import generic.theme.GColor;
import generic.util.WindowUtilities;
import generic.util.image.ImageUtils;
import ghidra.framework.plugintool.Plugin;
import ghidra.framework.plugintool.PluginInfo;
import ghidra.framework.plugintool.PluginTool;
import ghidra.framework.plugintool.util.PluginStatus;
import ghidra.util.Msg;
import ghidra.util.Swing;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.AWTEventListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;

@PluginInfo(status=PluginStatus.STABLE, packageName="Developer", category="Support", shortDescription="Window Locations", description="Shows all known window and screen geometry")
public class WindowLocationPlugin
extends Plugin {
    private static final Color BG_COLOR = new GColor("color.bg.plugin.windowlocation");
    private static final Color BG_COLOR_BOUNDS_VIRTUAL = new GColor("color.bg.plugin.windowlocation.bounds.virtual");
    private static final Color BG_COLOR_BOUNDS_VISIBLE = new GColor("color.bg.plugin.windowlocation.bounds.visible");
    private static final Color BG_COLOR_SCREENS = new GColor("color.bg.plugin.windowlocation.screens");
    private static final Color BG_COLOR_WINDOW_SELECTED = new GColor("color.bg.plugin.windowlocation.window.selected");
    private static final Color FG_COLOR_WINDOW_TEXT = new GColor("color.fg.plugin.windowlocation.window.text");
    static final String NAME = "Window Locations";
    private WindowLocationProvider provider;
    private Map<Window, WindowInfo> visibleWindows = new HashMap<Window, WindowInfo>();

    public WindowLocationPlugin(PluginTool tool) {
        super(tool);
        this.provider = new WindowLocationProvider((Tool)tool);
        tool.addComponentProvider((ComponentProvider)this.provider, false);
    }

    private class WindowLocationProvider
    extends ComponentProvider {
        private WindowLocationPanel windowPanel;

        public WindowLocationProvider(Tool tool) {
            super(tool, WindowLocationPlugin.NAME, WindowLocationPlugin.this.getName());
            this.build();
        }

        private void build() {
            this.windowPanel = new WindowLocationPanel();
            this.windowPanel.setPreferredSize(new Dimension(1000, 600));
            this.windowPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
            Toolkit toolkit = Toolkit.getDefaultToolkit();
            AWTEventListener listener = event -> this.windowPanel.repaint();
            toolkit.addAWTEventListener(listener, 32L);
            toolkit.addAWTEventListener(listener, 16L);
        }

        void repaint() {
            this.windowPanel.repaint();
        }

        public void componentHidden() {
            this.windowPanel.clear();
        }

        public JComponent getComponent() {
            return this.windowPanel;
        }
    }

    private class MouseListener
    extends MouseAdapter {
        private WindowInfo draggedInfo;
        private Point lastPoint;

        private MouseListener() {
        }

        @Override
        public void mouseDragged(MouseEvent e) {
            WindowInfo info;
            Point newLocation = e.getPoint();
            this.draggedInfo = info = this.getDraggedInfo(e);
            if (info == null) {
                return;
            }
            if (this.lastPoint == null) {
                this.lastPoint = newLocation;
            }
            double dx = newLocation.getX() - this.lastPoint.getX();
            double dy = newLocation.getY() - this.lastPoint.getY();
            this.lastPoint = newLocation;
            info.move(dx, dy);
        }

        void clear() {
            this.draggedInfo = null;
            this.lastPoint = null;
        }

        private WindowInfo getDraggedInfo(MouseEvent e) {
            if (this.draggedInfo != null) {
                return this.draggedInfo;
            }
            List intersection = WindowLocationPlugin.this.visibleWindows.values().stream().filter(w -> w.contains(e.getPoint())).sorted((w1, w2) -> w1.zOrder - w2.zOrder).collect(Collectors.toList());
            if (intersection.isEmpty()) {
                return null;
            }
            WindowInfo info = (WindowInfo)intersection.get(0);
            if (!info.isSelected()) {
                return null;
            }
            return info;
        }

        @Override
        public void mouseClicked(MouseEvent e) {
            this.draggedInfo = null;
            this.lastPoint = e.getPoint();
            boolean isRightClick = e.getButton() == 3;
            List<WindowInfo> intersection = WindowLocationPlugin.this.visibleWindows.values().stream().filter(w -> w.contains(e.getPoint())).sorted((w1, w2) -> w1.zOrder - w2.zOrder).collect(Collectors.toList());
            this.clearSelection();
            if (intersection.isEmpty()) {
                if (isRightClick) {
                    this.lastPoint = null;
                    this.resetAllLocations();
                }
                WindowLocationPlugin.this.provider.repaint();
                return;
            }
            if (intersection.size() == 1) {
                if (isRightClick) {
                    this.lastPoint = null;
                    intersection.get(0).resetLocation();
                    return;
                }
                ((WindowInfo)intersection.get(0)).setSelected(true);
                WindowLocationPlugin.this.provider.repaint();
                return;
            }
            if (e.isShiftDown()) {
                this.cycleZOrder(intersection);
            } else {
                intersection.get(0).setSelected(true);
            }
            WindowLocationPlugin.this.provider.repaint();
        }

        private void resetAllLocations() {
            WindowLocationPlugin.this.visibleWindows.values().forEach(info -> info.resetLocation());
        }

        private void clearSelection() {
            WindowLocationPlugin.this.visibleWindows.values().forEach(info -> info.setSelected(false));
        }

        private void cycleZOrder(List<WindowInfo> topWindows) {
            ArrayList<WindowInfo> reSorted = new ArrayList<WindowInfo>(WindowLocationPlugin.this.visibleWindows.values());
            for (int i = topWindows.size() - 2; i >= 0; --i) {
                WindowInfo info = topWindows.get(i);
                reSorted.remove(info);
                reSorted.add(0, info);
            }
            WindowInfo lastInfo = topWindows.get(topWindows.size() - 1);
            reSorted.remove(lastInfo);
            reSorted.add(0, lastInfo);
            int zOrder = 0;
            for (WindowInfo info : reSorted) {
                info.setZOrder(zOrder++);
            }
            ((WindowInfo)reSorted.get(0)).setSelected(true);
        }

        @Override
        public void mouseReleased(MouseEvent e) {
            this.draggedInfo = null;
            this.lastPoint = null;
        }

        @Override
        public void mouseExited(MouseEvent e) {
            this.draggedInfo = null;
            this.lastPoint = null;
        }

        @Override
        public void mouseEntered(MouseEvent e) {
            this.draggedInfo = null;
            this.lastPoint = null;
        }
    }

    private class WindowInfo {
        private Window window;
        private Rectangle startBounds;
        private int zOrder;
        private boolean isSelected;
        private AffineTransform xform;
        private String infoName;
        private Image windowImage;

        WindowInfo(Window w, AffineTransform xform, int zOrder) {
            this.window = w;
            this.xform = xform;
            this.zOrder = zOrder;
            this.startBounds = this.window.getBounds();
            this.infoName = WindowUtilities.getTitle((Window)w);
            if (this.infoName == null) {
                this.infoName = "no title";
            }
        }

        int getZ() {
            return this.zOrder;
        }

        void resetLocation() {
            this.window.setBounds(this.startBounds);
        }

        void setZOrder(int zOrder) {
            this.zOrder = zOrder;
        }

        void setSelected(boolean isSelected) {
            this.isSelected = isSelected;
            if (!isSelected) {
                this.windowImage = null;
            }
        }

        boolean isSelected() {
            return this.isSelected;
        }

        void move(double dx, double dy) {
            double sx = 1.0 / this.xform.getScaleX() * dx;
            double sy = 1.0 / this.xform.getScaleY() * dy;
            Point oldLocation = this.window.getLocation();
            double newx = oldLocation.getX() + sx;
            double newy = oldLocation.getY() + sy;
            oldLocation.setLocation(newx, newy);
            this.window.setLocation(oldLocation);
        }

        boolean contains(Point location) {
            Rectangle bounds = this.window.getBounds();
            Point xlocation = new Point();
            try {
                this.xform.inverseTransform(location, xlocation);
            }
            catch (NoninvertibleTransformException e) {
                Msg.debug((Object)this, (Object)"Unexpected exception transforming point", (Throwable)e);
            }
            return bounds.contains(xlocation);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void paint(Graphics2D g2d) {
            Rectangle b = this.window.getBounds();
            g2d.drawString(this.infoName, (float)b.getX(), (float)(b.getY() - 10.0));
            g2d.draw(b);
            FontMetrics fm = g2d.getFontMetrics();
            String coords = b.getX() + ", " + b.getY();
            Rectangle2D sb = fm.getStringBounds(this.infoName, g2d);
            g2d.drawString(coords, (float)b.getX(), (float)(b.getY() + b.getHeight() + 10.0 + sb.getHeight()));
            if (this.isSelected) {
                Color bg = g2d.getColor();
                try {
                    g2d.setColor(BG_COLOR_WINDOW_SELECTED);
                    g2d.fill(b);
                }
                finally {
                    g2d.setColor(bg);
                }
                Image image = this.getImage();
                if (image != null) {
                    g2d.drawImage(image, (int)b.getX(), (int)b.getY(), null);
                }
            }
        }

        private Image getImage() {
            if (this.windowImage != null) {
                return this.windowImage;
            }
            Swing.runLater(() -> {
                this.windowImage = this.createImage();
                WindowLocationPlugin.this.provider.repaint();
            });
            return null;
        }

        private Image createImage() {
            return ImageUtils.createImage((Component)this.window);
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.window == null ? 0 : this.window.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            WindowInfo other = (WindowInfo)obj;
            return Objects.equals(this.window, other.window);
        }

        public String toString() {
            return this.infoName + " - selected? " + this.isSelected;
        }
    }

    private class WindowLocationPanel
    extends JPanel {
        private MouseListener mousy;

        WindowLocationPanel() {
            this.mousy = new MouseListener();
            this.setFocusable(true);
            this.addMouseMotionListener(this.mousy);
            this.addMouseListener(this.mousy);
        }

        void clear() {
            WindowLocationPlugin.this.visibleWindows.clear();
            this.mousy.clear();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        protected void paintComponent(Graphics g) {
            Dimension size = this.getSize();
            double panelWidth = size.getWidth();
            double panelHeight = size.getHeight();
            this.setBackground(BG_COLOR);
            g.fillRect(0, 0, (int)panelWidth, (int)panelHeight);
            Graphics2D g2d = (Graphics2D)g;
            AffineTransform orig = g2d.getTransform();
            AffineTransform clone = (AffineTransform)orig.clone();
            try {
                AffineTransform newxform = this.createScreenTransform();
                clone.concatenate(newxform);
                g2d.setTransform(clone);
                this.paintVirtualBounds(g2d, BG_COLOR_BOUNDS_VIRTUAL);
                this.paintVisibleBounds(g2d, BG_COLOR_BOUNDS_VISIBLE);
                this.paintScreens(g2d, BG_COLOR_SCREENS);
                this.paintWindows(g2d, newxform);
            }
            finally {
                g2d.setTransform(orig);
            }
        }

        private AffineTransform createScreenTransform() {
            Area area = this.getFullScreenArea();
            Rectangle ab = area.getBounds();
            double fullWidth = ab.getWidth();
            double fullHeight = ab.getHeight();
            Dimension size = this.getSize();
            double panelWidth = size.getWidth();
            double panelHeight = size.getHeight();
            double dw = panelWidth / fullWidth;
            double dh = panelHeight / fullHeight;
            double scale = Math.min(dw, dh);
            double tx = ab.x;
            double ty = ab.y;
            double stx = (tx -= 100.0) * scale;
            double sty = (ty -= 100.0) * scale;
            AffineTransform xtranslate = AffineTransform.getTranslateInstance(-stx, -sty);
            AffineTransform xscale = AffineTransform.getScaleInstance(scale, scale);
            AffineTransform newxform = new AffineTransform();
            newxform.concatenate(xtranslate);
            newxform.concatenate(xscale);
            return newxform;
        }

        private void paintWindows(Graphics2D g2d, AffineTransform xform) {
            Font f = g2d.getFont();
            Font biggerFont = f.deriveFont(40.0f);
            g2d.setFont(biggerFont);
            g2d.setColor(FG_COLOR_WINDOW_TEXT);
            Window[] windows = Window.getWindows();
            int z = 0;
            Collection<WindowInfo> infos = WindowLocationPlugin.this.visibleWindows.values();
            for (WindowInfo info : infos) {
                int infoz = info.getZ();
                z = Math.max(infoz, z);
            }
            for (Window w : windows) {
                if (!w.isShowing()) {
                    WindowLocationPlugin.this.visibleWindows.remove(w);
                    continue;
                }
                WindowInfo info = WindowLocationPlugin.this.visibleWindows.get(w);
                if (info == null) {
                    info = new WindowInfo(w, g2d.getTransform(), ++z);
                    WindowLocationPlugin.this.visibleWindows.put(w, info);
                } else {
                    info.xform = xform;
                }
                info.paint(g2d);
            }
        }

        private void paintScreens(Graphics2D g2d, Color color) {
            g2d.setColor(color);
            Collection<Rectangle> screens = this.getScreens();
            for (Rectangle screen : screens) {
                g2d.draw(screen);
            }
        }

        private void paintVirtualBounds(Graphics2D g2d, Color color) {
            g2d.setColor(color);
            Rectangle virtualBounds = WindowUtilities.getVirtualScreenBounds();
            g2d.draw(virtualBounds);
        }

        private void paintVisibleBounds(Graphics2D g2d, Color color) {
            g2d.setColor(color);
            Shape visibleShape = WindowUtilities.getVisibleScreenBounds();
            g2d.draw(visibleShape.getBounds());
        }

        private Area getFullScreenArea() {
            Window[] windows;
            Rectangle virtualBounds = WindowUtilities.getVirtualScreenBounds();
            Shape visibleShape = WindowUtilities.getVisibleScreenBounds();
            Rectangle visibleBounds = visibleShape.getBounds();
            double tx = virtualBounds.x;
            double ty = virtualBounds.y;
            tx = Math.min(tx, (double)visibleBounds.x);
            ty = Math.min(ty, (double)visibleBounds.y);
            Area area = new Area();
            area.add(new Area(virtualBounds));
            area.add(new Area(visibleShape));
            for (Window w : windows = Window.getWindows()) {
                if (!w.isVisible()) continue;
                Rectangle bounds = w.getBounds();
                area.add(new Area(bounds));
                tx = Math.min(tx, (double)bounds.x);
                ty = Math.min(ty, (double)bounds.y);
            }
            Rectangle fullBounds = area.getBounds();
            int width = fullBounds.width + (int)(-(tx -= 100.0) * 2.0);
            int height = fullBounds.height + (int)(-(ty -= 100.0) * 2.0);
            area.add(new Area(new Rectangle(0, 0, width, height)));
            return area;
        }

        private Collection<Rectangle> getScreens() {
            GraphicsDevice[] gs;
            ArrayList<Rectangle> screens = new ArrayList<Rectangle>();
            GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
            for (GraphicsDevice gd : gs = ge.getScreenDevices()) {
                GraphicsConfiguration gc = gd.getDefaultConfiguration();
                Rectangle gcBounds = gc.getBounds();
                screens.add(gcBounds);
            }
            return screens;
        }
    }
}

