/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.ext4;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteArrayProvider;
import ghidra.app.util.bin.ByteProvider;
import ghidra.file.formats.ext4.Ext4BlockMapHelper;
import ghidra.file.formats.ext4.Ext4DirEntry;
import ghidra.file.formats.ext4.Ext4DirEntry2;
import ghidra.file.formats.ext4.Ext4ExtentsHelper;
import ghidra.file.formats.ext4.Ext4File;
import ghidra.file.formats.ext4.Ext4FileSystemFactory;
import ghidra.file.formats.ext4.Ext4GroupDescriptor;
import ghidra.file.formats.ext4.Ext4Inode;
import ghidra.file.formats.ext4.Ext4SuperBlock;
import ghidra.formats.gfilesystem.AbstractFileSystem;
import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.FSUtilities;
import ghidra.formats.gfilesystem.FileSystemService;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.formats.gfilesystem.fileinfo.FileAttributeType;
import ghidra.formats.gfilesystem.fileinfo.FileAttributes;
import ghidra.formats.gfilesystem.fileinfo.FileType;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.BitSet;
import java.util.Date;

@FileSystemInfo(type="ext4", description="EXT4", factory=Ext4FileSystemFactory.class)
public class Ext4FileSystem
extends AbstractFileSystem<Ext4File> {
    public static final Charset EXT4_DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private int blockSize;
    private ByteProvider provider;
    private String volumeName;
    private String uuid;
    private Ext4SuperBlock superBlock;
    private static final int MAX_SYMLINK_LOOKUP_COUNT = 100;

    public Ext4FileSystem(FSRLRoot fsrl, ByteProvider provider) {
        super(fsrl, FileSystemService.getInstance());
        this.provider = provider;
    }

    public void mountFS(TaskMonitor monitor) throws IOException, CancelledException {
        BinaryReader reader = new BinaryReader(this.provider, true);
        reader.setPointerIndex(1024);
        this.superBlock = new Ext4SuperBlock(reader);
        this.volumeName = this.superBlock.getVolumeName();
        this.uuid = NumericUtilities.convertBytesToString((byte[])this.superBlock.getS_uuid());
        long blockCount = this.superBlock.getS_blocks_count();
        int s_log_block_size = this.superBlock.getS_log_block_size();
        this.blockSize = (int)Math.pow(2.0, 10 + s_log_block_size);
        int groupSize = this.blockSize * this.superBlock.getS_blocks_per_group();
        if (groupSize <= 0) {
            throw new IOException("Invalid groupSize: " + groupSize);
        }
        int numGroups = (int)(blockCount / (long)this.superBlock.getS_blocks_per_group());
        if (blockCount % (long)this.superBlock.getS_blocks_per_group() != 0L) {
            ++numGroups;
        }
        int groupDescriptorOffset = this.blockSize + this.superBlock.getS_first_data_block() * this.blockSize;
        reader.setPointerIndex(groupDescriptorOffset);
        monitor.initialize((long)numGroups);
        monitor.setMessage("Reading inode tables");
        Ext4GroupDescriptor[] groupDescriptors = new Ext4GroupDescriptor[numGroups];
        for (int i = 0; i < numGroups; ++i) {
            monitor.checkCancelled();
            groupDescriptors[i] = new Ext4GroupDescriptor(reader, this.superBlock.is64Bit());
            monitor.incrementProgress(1L);
        }
        Ext4Inode[] inodes = this.getInodes(reader, groupDescriptors, monitor);
        Ext4Inode rootDirInode = inodes[2];
        if (!rootDirInode.isDir()) {
            throw new IOException("Unable to find root directory inode");
        }
        int usedInodeCount = this.superBlock.getS_inodes_count() - this.superBlock.getS_free_inodes_count();
        monitor.setMessage("Indexing files");
        monitor.initialize((long)usedInodeCount);
        BitSet processedInodes = new BitSet(inodes.length);
        this.processDirectory(inodes[2], this.fsIndex.getRootDir(), inodes, processedInodes, monitor);
        this.checkUnprocessedInodes(inodes, processedInodes);
    }

    private void checkUnprocessedInodes(Ext4Inode[] inodes, BitSet processedInodes) {
        int count = 0;
        int inodeNum = processedInodes.nextClearBit(this.superBlock.getS_first_ino());
        while (inodeNum < inodes.length) {
            if (!inodes[inodeNum].isUnused()) {
                ++count;
            }
            inodeNum = processedInodes.nextClearBit(inodeNum + 1);
        }
        if (count > 0) {
            Msg.warn((Object)((Object)this), (Object)("Unprocessed inodes: " + count));
        }
    }

    private void processDirectory(Ext4Inode inode, GFile dirFile, Ext4Inode[] inodes, BitSet processedInodes, TaskMonitor monitor) throws IOException, CancelledException {
        try (ByteProvider bp = this.getInodeByteProvider(inode, dirFile.getFSRL(), monitor);){
            this.processDirectoryStream(bp, dirFile, inodes, processedInodes, monitor);
        }
    }

    private void processDirectoryStream(ByteProvider directoryStream, GFile dirGFile, Ext4Inode[] inodes, BitSet processedInodes, TaskMonitor monitor) throws CancelledException, IOException {
        Ext4DirEntry dirEnt;
        boolean isdir2 = this.superBlock.isDirEntry2();
        BinaryReader reader = new BinaryReader(directoryStream, true);
        while ((dirEnt = isdir2 ? Ext4DirEntry2.read(reader) : Ext4DirEntry.read(reader)) != null) {
            monitor.checkCancelled();
            if (dirEnt.isUnused()) continue;
            this.processDirEntry(dirEnt, dirGFile, inodes, processedInodes, monitor);
            monitor.incrementProgress(1L);
        }
    }

    private void processDirEntry(Ext4DirEntry dirEntry, GFile parentDir, Ext4Inode[] inodes, BitSet processedInodes, TaskMonitor monitor) throws IOException, CancelledException {
        int inodeNumber = dirEntry.getInode();
        if (inodeNumber <= 0 || inodeNumber >= inodes.length) {
            Msg.warn((Object)((Object)this), (Object)("Invalid inode number: " + inodeNumber));
            return;
        }
        Ext4Inode inode = inodes[inodeNumber];
        if (inode == null || inode.isUnused()) {
            Msg.warn((Object)((Object)this), (Object)("Reference to bad inode: " + inodeNumber));
            return;
        }
        if (!(inode.isDir() || inode.isFile() || inode.isSymLink())) {
            throw new IOException("Inode " + inode + " has unhandled file type: " + Integer.toHexString(inode.getFileType()));
        }
        String name = dirEntry.getName();
        if (".".equals(name) || "..".equals(name)) {
            return;
        }
        GFile gfile = this.fsIndex.storeFileWithParent(name, parentDir, -1L, inode.isDir(), inode.getSize(), (Object)new Ext4File(name, inode));
        if (processedInodes.get(inodeNumber)) {
            return;
        }
        processedInodes.set(inodeNumber);
        if (inode.isDir()) {
            this.processDirectory(inode, gfile, inodes, processedInodes, monitor);
        }
    }

    public FileAttributes getFileAttributes(GFile file, TaskMonitor monitor) {
        FileAttributes result = new FileAttributes();
        Ext4File ext4File = (Ext4File)this.fsIndex.getMetadata(file);
        if (ext4File != null) {
            Ext4Inode inode = ext4File.getInode();
            result.add(FileAttributeType.NAME_ATTR, (Object)ext4File.getName());
            result.add(FileAttributeType.SIZE_ATTR, (Object)inode.getSize());
            result.add(FileAttributeType.FILE_TYPE_ATTR, (Object)this.inodeToFileType(inode));
            if (inode.isSymLink()) {
                String symLinkDest = "unknown";
                try {
                    symLinkDest = this.readLink(file, ext4File, monitor);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                result.add(FileAttributeType.SYMLINK_DEST_ATTR, (Object)symLinkDest);
            }
            result.add(FileAttributeType.MODIFIED_DATE_ATTR, (Object)new Date(inode.getI_mtime() * 1000));
            result.add(FileAttributeType.UNIX_ACL_ATTR, (Object)(inode.getI_mode() & 0xFFF));
            result.add(FileAttributeType.USER_ID_ATTR, (Object)Short.toUnsignedLong(inode.getI_uid()));
            result.add(FileAttributeType.GROUP_ID_ATTR, (Object)Short.toUnsignedLong(inode.getI_gid()));
            result.add("Link Count", (Object)inode.getI_links_count());
        }
        return result;
    }

    FileType inodeToFileType(Ext4Inode inode) {
        if (inode.isDir()) {
            return FileType.DIRECTORY;
        }
        if (inode.isSymLink()) {
            return FileType.SYMBOLIC_LINK;
        }
        if (inode.isFile()) {
            return FileType.FILE;
        }
        return FileType.UNKNOWN;
    }

    private Ext4Inode resolveSymLink(GFile file, TaskMonitor monitor) throws IOException {
        String symlinkDestPath;
        int lookupCount = 0;
        GFile currentFile = file;
        StringBuilder symlinkDebugPath = new StringBuilder();
        do {
            if (lookupCount++ > 100) {
                throw new IOException("Symlink too long: " + file.getPath() + ", " + symlinkDebugPath);
            }
            Ext4File extFile = (Ext4File)this.fsIndex.getMetadata(currentFile);
            if (extFile == null) {
                throw new IOException("Missing Ext4 metadata for " + currentFile.getPath());
            }
            Ext4Inode inode = extFile.getInode();
            if (!inode.isSymLink()) {
                return inode;
            }
            symlinkDestPath = this.readLink(file, extFile, monitor);
            symlinkDebugPath.append(" -> ").append(symlinkDestPath);
            if (symlinkDestPath.startsWith("/")) continue;
            if (currentFile.getParentFile() == null) {
                throw new IOException("No parent file for " + currentFile);
            }
            symlinkDestPath = FSUtilities.appendPath((String[])new String[]{currentFile.getParentFile().getPath(), symlinkDestPath});
        } while ((currentFile = this.lookup(symlinkDestPath)) != null);
        throw new IOException("Missing symlink dest: " + file.getPath() + ", broken symlink: " + symlinkDebugPath);
    }

    private String readLink(GFile file, Ext4File extFile, TaskMonitor monitor) throws IOException {
        try (ByteProvider bp = this.getInodeByteProvider(extFile.getInode(), file.getFSRL(), monitor);){
            byte[] tmp = bp.readBytes(0L, bp.length());
            String string = new String(tmp, StandardCharsets.UTF_8);
            return string;
        }
    }

    private Ext4Inode getInodeFor(GFile file, TaskMonitor monitor) throws IOException {
        Ext4File extFile = (Ext4File)this.fsIndex.getMetadata(file);
        if (extFile == null) {
            return null;
        }
        Ext4Inode inode = extFile.getInode();
        if (inode == null) {
            return null;
        }
        if (inode.isSymLink()) {
            inode = this.resolveSymLink(file, monitor);
        }
        return inode;
    }

    public ByteProvider getByteProvider(GFile file, TaskMonitor monitor) throws IOException {
        Ext4Inode inode = this.getInodeFor(file, monitor);
        if (inode == null) {
            return null;
        }
        if (inode.isDir()) {
            throw new IOException(file.getName() + " is a directory.");
        }
        return this.getInodeByteProvider(inode, file.getFSRL(), monitor);
    }

    private ByteProvider getInodeByteProvider(Ext4Inode inode, FSRL inodeFSRL, TaskMonitor monitor) throws IOException {
        if (inode.isFlagExtents()) {
            return Ext4ExtentsHelper.getByteProvider(inode.getI_block(), this.provider, inode.getSize(), this.blockSize, inodeFSRL);
        }
        if (inode.isFlagInlineData() || inode.isSymLink()) {
            byte[] data = inode.getInlineDataValue();
            return new ByteArrayProvider(data, inodeFSRL);
        }
        return Ext4BlockMapHelper.getByteProvider(inode.getI_block(), this.provider, inode.getSize(), this.blockSize, inodeFSRL);
    }

    private Ext4Inode[] getInodes(BinaryReader reader, Ext4GroupDescriptor[] groupDescriptors, TaskMonitor monitor) throws IOException, CancelledException {
        int inodeCount = this.superBlock.getS_inodes_count();
        int inodesPerGroup = this.superBlock.getS_inodes_per_group();
        Ext4Inode[] inodes = new Ext4Inode[inodeCount + 1];
        int inodeIndex = 1;
        for (int i = 0; i < groupDescriptors.length; ++i) {
            monitor.checkCancelled();
            long inodeTableBlockOffset = groupDescriptors[i].getBg_inode_table();
            long offset = inodeTableBlockOffset * (long)this.blockSize;
            reader.setPointerIndex(offset);
            monitor.setMessage("Reading inode table " + i + " of " + (groupDescriptors.length - 1) + "...");
            monitor.initialize((long)inodesPerGroup);
            for (int j = 0; j < inodesPerGroup; ++j) {
                monitor.checkCancelled();
                monitor.incrementProgress(1L);
                Ext4Inode inode = new Ext4Inode(reader, this.superBlock.getS_inode_size());
                reader.setPointerIndex(offset += (long)this.superBlock.getS_inode_size());
                inodes[inodeIndex++] = inode;
            }
        }
        return inodes;
    }

    public void close() throws IOException {
        this.refManager.onClose();
        this.provider.close();
        this.provider = null;
        this.fsIndex.clear();
    }

    public String getName() {
        return "%s - %s - %s".formatted(this.fsFSRL.getContainer().getName(), this.volumeName, this.uuid);
    }

    public boolean isClosed() {
        return this.provider == null;
    }
}

