/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.classfinder;

import generic.json.Json;
import ghidra.util.Msg;
import ghidra.util.SystemUtilities;
import ghidra.util.classfinder.ClassDir;
import ghidra.util.classfinder.ClassJar;
import ghidra.util.classfinder.ClassSearcher;
import ghidra.util.classfinder.ExtensionPoint;
import ghidra.util.classfinder.ExtensionPointProperties;
import ghidra.util.exception.CancelledException;
import ghidra.util.extensions.ExtensionDetails;
import ghidra.util.extensions.ExtensionModuleClassLoader;
import ghidra.util.extensions.ExtensionUtils;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utility.module.ModuleUtilities;

public class ClassFinder {
    static final Logger log = LogManager.getLogger(ClassFinder.class);
    private static final boolean IS_USING_RESTRICTED_EXTENSIONS = Boolean.getBoolean("ghidra.extensions.classpath.restricted");
    private static List<Class<?>> FILTER_CLASSES = Collections.unmodifiableList(Arrays.asList(ExtensionPoint.class));
    private Set<ClassDir> classDirs = new HashSet<ClassDir>();
    private Set<ClassJar> classJars = new HashSet<ClassJar>();

    public ClassFinder(List<String> searchPaths, TaskMonitor monitor) throws CancelledException {
        this.initialize(searchPaths, monitor);
    }

    private void initialize(List<String> searchPaths, TaskMonitor monitor) throws CancelledException {
        Msg.trace((Object)this, (Object)("Using restricted extension class loader? " + IS_USING_RESTRICTED_EXTENSIONS));
        LinkedHashSet<String> pathSet = new LinkedHashSet<String>(searchPaths);
        Iterator pathIterator = pathSet.iterator();
        while (pathIterator.hasNext()) {
            monitor.checkCancelled();
            String path = (String)pathIterator.next();
            String lcPath = path.toLowerCase();
            File file = new File(path);
            if ((lcPath.endsWith(".jar") || lcPath.endsWith(".zip")) && file.exists()) {
                if (ClassJar.ignoreJar(lcPath)) {
                    log.trace("Ignoring jar file: {}", (Object)path);
                    continue;
                }
                log.trace("Searching jar file: {}", (Object)path);
                this.classJars.add(new ClassJar(path, monitor));
                continue;
            }
            if (!file.isDirectory()) continue;
            log.trace("Searching classpath directory: {}", (Object)path);
            this.classDirs.add(new ClassDir(path, monitor));
        }
    }

    List<Class<?>> getClasses(TaskMonitor monitor) throws CancelledException {
        HashSet classSet = new HashSet();
        for (ClassDir dir : this.classDirs) {
            monitor.checkCancelled();
            dir.getClasses(classSet, monitor);
        }
        for (ClassJar jar : this.classJars) {
            monitor.checkCancelled();
            jar.getClasses(classSet, monitor);
        }
        ArrayList classList = new ArrayList(classSet);
        Collections.sort(classList, (c1, c2) -> {
            String n2;
            int p2;
            int p1 = ExtensionPointProperties.Util.getPriority(c1);
            if (p1 > (p2 = ExtensionPointProperties.Util.getPriority(c2))) {
                return -1;
            }
            if (p1 < p2) {
                return 1;
            }
            String n1 = c1.getName();
            if (n1.equals(n2 = c2.getName())) {
                return Integer.compare(c1.hashCode(), c2.hashCode());
            }
            return n1.compareTo(n2);
        });
        return classList;
    }

    static Class<?> loadExtensionPoint(String path, String className) {
        if (!ClassSearcher.isExtensionPointName(className)) {
            return null;
        }
        ClassLoader classLoader = ClassFinder.getClassLoader(path);
        try {
            Class<?> c = Class.forName(className, true, classLoader);
            if (ClassFinder.isClassOfInterest(c)) {
                return c;
            }
        }
        catch (Throwable t) {
            ClassFinder.processClassLoadError(path, className, t);
        }
        return null;
    }

    private static ClassLoader getClassLoader(String path) {
        ClassLoader classLoader = ClassSearcher.class.getClassLoader();
        if (!IS_USING_RESTRICTED_EXTENSIONS) {
            return classLoader;
        }
        ExtensionDetails extension = ExtensionUtils.getExtension(path);
        if (extension != null) {
            Msg.trace(ClassFinder.class, (Object)("Installing custom extension class loader for: " + Json.toStringFlat(extension)));
            classLoader = new ExtensionModuleClassLoader(extension);
        }
        return classLoader;
    }

    private static void processClassLoadError(String path, String name, Throwable t) {
        if (t instanceof LinkageError) {
            Msg.trace(ClassFinder.class, (Object)("LinkageError loading class " + name + "; Incompatible class version? "), (Throwable)t);
            return;
        }
        if (!(t instanceof ClassNotFoundException)) {
            Msg.error(ClassFinder.class, (Object)("Error loading class " + name + " - " + t.getMessage()), (Throwable)t);
            return;
        }
        ClassFinder.processClassNotFoundExcepetion(path, name, (ClassNotFoundException)t);
    }

    private static void processClassNotFoundExcepetion(String path, String name, ClassNotFoundException t) {
        if (!ClassFinder.isModuleEntryMissingFromClasspath(path)) {
            Msg.error(ClassFinder.class, (Object)("Error loading class " + name + " - " + t.getMessage()), (Throwable)t);
            return;
        }
        if (SystemUtilities.isInTestingMode()) {
            return;
        }
        Msg.error(ClassFinder.class, (Object)("Module class is missing from the classpath.\n\tUpdate your launcher accordingly.\n\tModule: '" + path + "'\n\tClass: '" + name + "'"));
    }

    private static boolean isModuleEntryMissingFromClasspath(String path) {
        boolean inModule = ModuleUtilities.isInModule((String)path);
        if (!inModule) {
            return false;
        }
        String classPath = System.getProperty("java.class.path");
        boolean inClassPath = classPath.contains(path);
        return !inClassPath;
    }

    public static boolean isClassOfInterest(Class<?> c) {
        if (Modifier.isAbstract(c.getModifiers())) {
            return false;
        }
        if (c.getEnclosingClass() != null && !Modifier.isStatic(c.getModifiers())) {
            return false;
        }
        if (!Modifier.isPublic(c.getModifiers())) {
            return false;
        }
        if (ExtensionPointProperties.Util.isExcluded(c)) {
            return false;
        }
        for (Class<?> filterClasse : FILTER_CLASSES) {
            if (!filterClasse.isAssignableFrom(c)) continue;
            return true;
        }
        return false;
    }
}

