/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.app.util.pdb.pdbapplicator;

import java.util.*;

import ghidra.app.cmd.function.ApplyFunctionSignatureCmd;
import ghidra.app.cmd.function.CallDepthChangeInfo;
import ghidra.app.util.bin.format.pdb2.pdbreader.*;
import ghidra.app.util.bin.format.pdb2.pdbreader.symbol.*;
import ghidra.app.util.bin.format.pdb2.pdbreader.type.AbstractMsType;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.FunctionDefinition;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.*;
import ghidra.program.model.symbol.SourceType;
import ghidra.util.InvalidNameException;
import ghidra.util.exception.*;
import ghidra.util.task.TaskMonitor;

/**
 * Applier for {@link AbstractProcedureStartMsSymbol} and  {@link AbstractThunkMsSymbol} symbols.
 */
public class FunctionSymbolApplier extends MsSymbolApplier
		implements DeferrableFunctionSymbolApplier {

	private static final String BLOCK_INDENT = "   ";

	private AbstractProcedureMsSymbol procedureSymbol;
	private AbstractThunkMsSymbol thunkSymbol;
	private Address specifiedAddress;
	private Address address;
	private boolean isNonReturning;
	private Function function = null;
	private long specifiedFrameSize = 0;
	private long currentFrameSize = 0;
	private BlockCommentsManager comments;

	private int symbolBlockNestingLevel;
	private Address currentBlockAddress;

	// might not need this, but investigating whether it will help us.  TODO remove?
	private int baseParamOffset = 0;

//	private List<RegisterRelativeSymbolApplier> stackVariableAppliers = new ArrayList<>();

	private List<MsSymbolApplier> allAppliers = new ArrayList<>();
	private RegisterChangeCalculator registerChangeCalculator;

	/**
	 * Constructor
	 * @param applicator the {@link DefaultPdbApplicator} for which we are working.
	 * @param iter the Iterator containing the symbol sequence being processed
	 * @throws CancelledException upon user cancellation
	 */
	public FunctionSymbolApplier(DefaultPdbApplicator applicator, MsSymbolIterator iter)
			throws CancelledException {
		super(applicator, iter);
		AbstractMsSymbol abstractSymbol = iter.next();
		symbolBlockNestingLevel = 0;
		comments = new BlockCommentsManager();
		currentBlockAddress = null;

		if (abstractSymbol instanceof AbstractProcedureMsSymbol) {
			procedureSymbol = (AbstractProcedureMsSymbol) abstractSymbol;
			specifiedAddress = applicator.getRawAddress(procedureSymbol);
			address = applicator.getAddress(procedureSymbol);
			isNonReturning =
				((AbstractProcedureStartMsSymbol) procedureSymbol).getFlags().doesNotReturn();
		}
		else if (abstractSymbol instanceof AbstractProcedureStartIa64MsSymbol) {
			procedureSymbol = (AbstractProcedureStartIa64MsSymbol) abstractSymbol;
			specifiedAddress = applicator.getRawAddress(procedureSymbol);
			address = applicator.getAddress(procedureSymbol);
			isNonReturning =
				((AbstractProcedureStartIa64MsSymbol) procedureSymbol).getFlags().doesNotReturn();
		}
		else if (abstractSymbol instanceof AbstractProcedureStartMipsMsSymbol) {
			procedureSymbol = (AbstractProcedureStartMipsMsSymbol) abstractSymbol;
			specifiedAddress = applicator.getRawAddress(procedureSymbol);
			address = applicator.getAddress(procedureSymbol);
			isNonReturning = false; // we do not have ProcedureFlags to check
		}
		else if (abstractSymbol instanceof AbstractThunkMsSymbol) {
			thunkSymbol = (AbstractThunkMsSymbol) abstractSymbol;
			specifiedAddress = applicator.getRawAddress(thunkSymbol);
			address = applicator.getAddress(thunkSymbol);
			// isNonReturning value is not used when thunk; is controlled by thunked function;
		}
		else {
			throw new AssertException(
				"Invalid symbol type: " + abstractSymbol.getClass().getSimpleName());
		}
		manageBlockNesting(this);

		while (notDone()) {
			applicator.checkCancelled();
			MsSymbolApplier applier = applicator.getSymbolApplier(iter);
			allAppliers.add(applier);
			applier.manageBlockNesting(this);
		}
	}

	@Override
	void manageBlockNesting(MsSymbolApplier applierParam) {
		if (applierParam instanceof FunctionSymbolApplier) {
			FunctionSymbolApplier functionSymbolApplier = (FunctionSymbolApplier) applierParam;
			if (procedureSymbol != null) {
				long start = procedureSymbol.getDebugStartOffset();
				long end = procedureSymbol.getDebugEndOffset();
				Address blockAddress = address.add(start);
				long length = end - start;
				functionSymbolApplier.beginBlock(blockAddress, procedureSymbol.getName(), length);
			}
			else if (thunkSymbol != null) {
				functionSymbolApplier.beginBlock(address, thunkSymbol.getName(),
					thunkSymbol.getLength());
			}
		}
	}

	long getLength() {
		if (procedureSymbol != null) {
			return procedureSymbol.getProcedureLength();
		}
		else if (thunkSymbol != null) {
			return thunkSymbol.getLength();
		}
		throw new AssertException("Unexpected Symbol type");
	}

	/**
	 * Returns the {@link Function} for this applier.
	 * @return the Function
	 */
	Function getFunction() {
		return function;
	}

	/**
	 * Returns the current frame size.
	 * @return the current frame size.
	 */
	long getCurrentFrameSize() {
		return currentFrameSize;
	}

	/**
	 * Returns the frame size as specified by the PDB
	 * @return the frame size.
	 */
	long getSpecifiedFrameSize() {
		return specifiedFrameSize;
	}

	/**
	 * Set the specified frame size.
	 * @param specifiedFrameSize the frame size.
	 */
	void setSpecifiedFrameSize(long specifiedFrameSize) {
		this.specifiedFrameSize = specifiedFrameSize;
		currentFrameSize = specifiedFrameSize;
	}

	/**
	 * Get the function name
	 * @return the function name
	 */
	String getName() {
		if (procedureSymbol != null) {
			return procedureSymbol.getName();
		}
		else if (thunkSymbol != null) {
			return thunkSymbol.getName();
		}
		return "";
	}

	@Override
	void applyTo(MsSymbolApplier applyToApplier) {
		// Do nothing.
	}

	@Override
	void apply() throws PdbException, CancelledException {
		boolean result = applyTo(applicator.getCancelOnlyWrappingMonitor());
		if (result == false) {
			throw new PdbException(this.getClass().getSimpleName() + ": failure at " + address +
				" applying " + getName());
		}
	}

	boolean applyTo(TaskMonitor monitor) throws PdbException, CancelledException {
		if (applicator.isInvalidAddress(address, getName())) {
			return false;
		}

		boolean functionSuccess = applyFunction(monitor);
		if (functionSuccess == false) {
			return false;
		}
		registerChangeCalculator = new RegisterChangeCalculator(procedureSymbol, function, monitor);

		baseParamOffset = VariableUtilities.getBaseStackParamOffset(function);

		for (MsSymbolApplier applier : allAppliers) {
			applier.applyTo(this);
		}

		// comments
		long addressDelta = address.subtract(specifiedAddress);
		comments.applyTo(applicator.getProgram(), addressDelta);

		// line numbers
		// TODO: not done yet
//	ApplyLineNumbers applyLineNumbers = new ApplyLineNumbers(pdbParser, xmlParser, program);
//	applyLineNumbers.applyTo(monitor, log);

		return true;
	}

	Integer getRegisterPrologChange(Register register) {
		return registerChangeCalculator.getRegChange(applicator, register);
	}

	int getBaseParamOffset() {
		return baseParamOffset;
	}

	/**
	 * Sets a local variable (address, name, type)
	 * @param varAddress Address of the variable.
	 * @param varName varName of the variable.
	 * @param dataType data type of the variable.
	 */
	void setLocalVariable(Address varAddress, String varName, DataType dataType) {
		if (currentBlockAddress == null) {
			return; // silently return.
		}
		if (varName.isBlank()) {
			return; // silently return.
		}

		String plateAddition = "PDB: static local for function (" + address + "): " + getName();
		// TODO: 20220210... consider adding function name as namespace to varName
		applicator.createSymbol(varAddress, varName, true, plateAddition);
	}

	private boolean applyFunction(TaskMonitor monitor) throws CancelledException, PdbException {
		function = applicator.getExistingOrCreateOneByteFunction(address);
		if (function == null) {
			return false;
		}
		applicator.scheduleDeferredFunctionWork(this);

		boolean succeededSetFunctionSignature = false;
		if (thunkSymbol == null) {
			if (function.getSignatureSource().isLowerPriorityThan(SourceType.IMPORTED)) {
				function.setThunkedFunction(null);
				succeededSetFunctionSignature = setFunctionDefinition(monitor);
				function.setNoReturn(isNonReturning);
			}
		}
		// If signature was set, then override existing primary mangled symbol with
		// the global symbol that provided this signature so that Demangler does not overwrite
		// the richer data type we get with global symbols.
		applicator.createSymbol(address, getName(), succeededSetFunctionSignature);

		currentFrameSize = 0;
		return true;
	}

	/**
	 * Sets function signature
	 * @param monitor monitor
	 * @return true if function signature was set
	 * @throws CancelledException upon user cancellation
	 * @throws PdbException upon processing error
	 */
	private boolean setFunctionDefinition(TaskMonitor monitor)
			throws CancelledException, PdbException {
		if (procedureSymbol == null) {
			// TODO: is there anything we can do with thunkSymbol?
			// long x = thunkSymbol.getParentPointer();
			return false;
		}
		// Rest presumes procedureSymbol.
		RecordNumber typeRecordNumber = procedureSymbol.getTypeRecordNumber();
		MsTypeApplier applier = applicator.getTypeApplier(typeRecordNumber);
		AbstractMsType fType = applicator.getPdb().getTypeRecord(typeRecordNumber);
		if (!(applier instanceof AbstractFunctionTypeApplier)) {
			if (!((applier instanceof PrimitiveTypeApplier prim) && prim.isNoType(fType))) {
				applicator.appendLogMsg("Error: Failed to resolve datatype RecordNumber " +
					typeRecordNumber + " at " + address);
			}
			return false;
		}

		DataType dataType = applicator.getCompletedDataType(typeRecordNumber);
		// Since we know the applier is an AbstractionFunctionTypeApplier, then dataType is either
		//  FunctionDefinition or no type (typedef).
		if (!(dataType instanceof FunctionDefinition)) {
			return false;
		}
		FunctionDefinition def =
			(FunctionDefinition) dataType.copy(applicator.getDataTypeManager());
		try {
			// Must use copy of function definition with preserved function name.
			// While not ideal, this prevents applying an incorrect function name
			// with an IMPORTED source type
			def.setName(function.getName());
		}
		catch (InvalidNameException | DuplicateNameException e) {
			throw new RuntimeException("unexpected exception", e);
		}
		ApplyFunctionSignatureCmd sigCmd =
			new ApplyFunctionSignatureCmd(address, def, SourceType.IMPORTED);
		if (!sigCmd.applyTo(applicator.getProgram(), monitor)) {
			applicator.appendLogMsg(
				"PDB Warning: Failed to apply signature to function at address " + address +
					" due to " + sigCmd.getStatusMsg() + "; dataType: " + def.getName());
			return false;
		}
		return true;
	}

	private boolean notDone() {
		return (symbolBlockNestingLevel > 0) && iter.hasNext();
	}

	int endBlock() {
		if (--symbolBlockNestingLevel < 0) {
			applicator.appendLogMsg(
				"Block Nesting went negative for " + getName() + " at " + address);
		}
		if (symbolBlockNestingLevel == 0) {
			//currentFunctionSymbolApplier = null;
		}
		return symbolBlockNestingLevel;
	}

	void beginBlock(Address startAddress, String name, long length) {

		int nestingLevel = beginBlock(startAddress);
		if (!applicator.getPdbApplicatorOptions().applyCodeScopeBlockComments()) {
			return;
		}
		if (applicator.isInvalidAddress(startAddress, name)) {
			return;
		}

		String indent = getIndent(nestingLevel);

		String baseComment = "level " + nestingLevel + ", length " + length;

		String preComment = indent + "PDB: Block Beg, " + baseComment;
		if (!name.isEmpty()) {
			preComment += " (" + name + ")";
		}
		comments.addPreComment(startAddress, preComment);

		String postComment = indent + "PDB: Block End, " + baseComment;
		Address endAddress = startAddress.add(((length <= 0) ? 0 : length - 1));
		comments.addPostComment(endAddress, postComment);
	}

	private int beginBlock(Address startAddress) {
		currentBlockAddress = startAddress;
		++symbolBlockNestingLevel;
		return symbolBlockNestingLevel;
	}

	private String getIndent(int indentLevel) {
		String indent = "";
		for (int i = 1; i < indentLevel; i++) {
			indent += BLOCK_INDENT;
		}
		return indent;
	}

	// Method copied from ApplyStackVariables (ghidra.app.util.bin.format.pdb package)
	//  on 20191119. TODO: Do we need something like this?
	/**
	 * Get the stack offset after it settles down.
	 * @param monitor TaskMonitor
	 * @return stack offset that stack variables will be relative to.
	 * @throws CancelledException upon user cancellation.
	 */
	private int getFrameBaseOffset(TaskMonitor monitor) throws CancelledException {

		int retAddrSize = function.getProgram().getDefaultPointerSize();

		if (retAddrSize != 8) {
			// don't do this for 32 bit.
			return -retAddrSize;  // 32 bit has a -4 byte offset
		}

		Register frameReg = function.getProgram().getCompilerSpec().getStackPointer();
		Address entryAddr = function.getEntryPoint();
		AddressSet scopeSet = new AddressSet();
		scopeSet.addRange(entryAddr, entryAddr.add(64));
		CallDepthChangeInfo valueChange =
			new CallDepthChangeInfo(function, scopeSet, frameReg, monitor);
		InstructionIterator instructions =
			function.getProgram().getListing().getInstructions(scopeSet, true);
		int max = 0;
		while (instructions.hasNext()) {
			monitor.checkCancelled();
			Instruction next = instructions.next();
			int newValue = valueChange.getDepth(next.getMinAddress());
			if (newValue < -(20 * 1024) || newValue > (20 * 1024)) {
				continue;
			}
			if (Math.abs(newValue) > Math.abs(max)) {
				max = newValue;
			}
		}
		return max;
	}

	private static class RegisterChangeCalculator {

		private Map<Register, Integer> registerChangeByRegisterName = new HashMap<>();
		private CallDepthChangeInfo callDepthChangeInfo;
		private Address debugStart;

		private RegisterChangeCalculator(AbstractProcedureMsSymbol procedureSymbol,
				Function function, TaskMonitor monitor) throws CancelledException {
			callDepthChangeInfo = createCallDepthChangInfo(procedureSymbol, function, monitor);
		}

		private CallDepthChangeInfo createCallDepthChangInfo(
				AbstractProcedureMsSymbol procedureSymbol, Function function, TaskMonitor monitor)
				throws CancelledException {
			if (procedureSymbol == null) {
				return null;
			}
			Register frameReg = function.getProgram().getCompilerSpec().getStackPointer();
			Address entryAddr = function.getEntryPoint();
			debugStart = entryAddr.add(procedureSymbol.getDebugStartOffset());
			AddressSet scopeSet = new AddressSet();
			scopeSet.addRange(entryAddr, debugStart);
			return new CallDepthChangeInfo(function, scopeSet, frameReg, monitor);
		}

		Integer getRegChange(DefaultPdbApplicator applicator, Register register) {
			if (callDepthChangeInfo == null || register == null) {
				return null;
			}
			Integer change = registerChangeByRegisterName.get(register);
			if (change != null) {
				return change;
			}
			change = callDepthChangeInfo.getRegDepth(debugStart, register);
			registerChangeByRegisterName.put(register, change);
			return change;
		}

	}

	@Override
	public void doDeferredProcessing() {
		// TODO:
		// Try to processes parameters, locals, scopes if applicable.
	}

	@Override
	public Address getAddress() {
		return address;
	}

}
