"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CatalogCacheManager = void 0;
var _constants = require("../constants");
var _shared = require("../utils/shared");
var _types = require("../types");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Manages caching for catalog data including data sources and accelerations.
 */
class CatalogCacheManager {
  /**
   * Saves data source cache to local storage.
   * @param {DataSourceCacheData} cacheData - The data source cache data to save.
   */
  static saveDataSourceCache(cacheData) {
    sessionStorage.setItem(this.datasourceCacheKey, JSON.stringify(cacheData));
  }

  /**
   * Retrieves data source cache from local storage.
   * @returns {DataSourceCacheData} The retrieved data source cache.
   */
  static getDataSourceCache() {
    const catalogData = sessionStorage.getItem(this.datasourceCacheKey);
    if (catalogData) {
      return JSON.parse(catalogData);
    } else {
      const defaultCacheObject = {
        version: _constants.CATALOG_CACHE_VERSION,
        dataSources: []
      };
      this.saveDataSourceCache(defaultCacheObject);
      return defaultCacheObject;
    }
  }

  /**
   * Saves accelerations cache to local storage.
   * @param {AccelerationsCacheData} cacheData - The accelerations cache data to save.
   */
  static saveAccelerationsCache(cacheData) {
    sessionStorage.setItem(this.accelerationsCacheKey, JSON.stringify(cacheData));
  }

  /**
   * Retrieves accelerations cache from local storage.
   * @returns {AccelerationsCacheData} The retrieved accelerations cache.
   */
  static getAccelerationsCache() {
    const accelerationCacheData = sessionStorage.getItem(this.accelerationsCacheKey);
    if (accelerationCacheData) {
      return JSON.parse(accelerationCacheData);
    } else {
      const defaultCacheObject = {
        version: _constants.CATALOG_CACHE_VERSION,
        dataSources: []
      };
      this.saveAccelerationsCache(defaultCacheObject);
      return defaultCacheObject;
    }
  }

  /**
   * Adds or updates a data source in the accelerations cache.
   * @param {CachedAccelerationByDataSource} dataSource - The data source to add or update.
   */
  static addOrUpdateAccelerationsByDataSource(dataSource, dataSourceMDSId) {
    let index = -1;
    const accCacheData = this.getAccelerationsCache();
    if (dataSourceMDSId) {
      index = accCacheData.dataSources.findIndex(ds => ds.name === dataSource.name && ds.dataSourceMDSId === dataSourceMDSId);
    } else {
      index = accCacheData.dataSources.findIndex(ds => ds.name === dataSource.name);
    }
    if (index !== -1) {
      accCacheData.dataSources[index] = dataSource;
    } else {
      accCacheData.dataSources.push(dataSource);
    }
    this.saveAccelerationsCache(accCacheData);
  }

  /**
   * Retrieves accelerations cache from local storage by the datasource name.
   * @param {string} dataSourceName - The name of the data source.
   * @returns {CachedAccelerationByDataSource} The retrieved accelerations by datasource in cache.
   * @throws {Error} If the data source is not found.
   */
  static getOrCreateAccelerationsByDataSource(dataSourceName, dataSourceMDSId) {
    const accCacheData = this.getAccelerationsCache();
    let cachedDataSource;
    if (dataSourceMDSId) {
      cachedDataSource = accCacheData.dataSources.find(ds => ds.name === dataSourceName && ds.dataSourceMDSId === dataSourceMDSId);
    } else {
      cachedDataSource = accCacheData.dataSources.find(ds => ds.name === dataSourceName);
    }
    if (cachedDataSource) return cachedDataSource;else {
      let defaultDataSourceObject = {
        name: dataSourceName,
        lastUpdated: '',
        status: _types.CachedDataSourceStatus.Empty,
        accelerations: []
      };
      if (dataSourceMDSId !== '' && dataSourceMDSId !== undefined) {
        defaultDataSourceObject = {
          ...defaultDataSourceObject,
          dataSourceMDSId
        };
      }
      this.addOrUpdateAccelerationsByDataSource(defaultDataSourceObject, dataSourceMDSId);
      return defaultDataSourceObject;
    }
  }

  /**
   * Adds or updates a data source in the cache.
   * @param {CachedDataSource} dataSource - The data source to add or update.
   */
  static addOrUpdateDataSource(dataSource, dataSourceMDSId) {
    const cacheData = this.getDataSourceCache();
    let index;
    if (dataSourceMDSId) {
      index = cacheData.dataSources.findIndex(ds => ds.name === dataSource.name && ds.dataSourceMDSId === dataSourceMDSId);
    }
    index = cacheData.dataSources.findIndex(ds => ds.name === dataSource.name);
    if (index !== -1) {
      cacheData.dataSources[index] = dataSource;
    } else {
      cacheData.dataSources.push(dataSource);
    }
    this.saveDataSourceCache(cacheData);
  }

  /**
   * Retrieves or creates a data source with the specified name.
   * @param {string} dataSourceName - The name of the data source.
   * @returns {CachedDataSource} The retrieved or created data source.
   */
  static getOrCreateDataSource(dataSourceName, dataSourceMDSId) {
    let cachedDataSource;
    if (dataSourceMDSId) {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.dataSourceMDSId === dataSourceMDSId && ds.name === dataSourceName);
    } else {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.name === dataSourceName);
    }
    if (cachedDataSource) {
      return cachedDataSource;
    } else {
      let defaultDataSourceObject = {
        name: dataSourceName,
        lastUpdated: '',
        status: _types.CachedDataSourceStatus.Empty,
        databases: []
      };
      if (dataSourceMDSId !== '' && dataSourceMDSId !== undefined) {
        defaultDataSourceObject = {
          ...defaultDataSourceObject,
          dataSourceMDSId
        };
      }
      this.addOrUpdateDataSource(defaultDataSourceObject, dataSourceMDSId);
      return defaultDataSourceObject;
    }
  }

  /**
   * Retrieves a database from the cache.
   * @param {string} dataSourceName - The name of the data source containing the database.
   * @param {string} databaseName - The name of the database.
   * @returns {CachedDatabase} The retrieved database.
   * @throws {Error} If the data source or database is not found.
   */
  static getDatabase(dataSourceName, databaseName, dataSourceMDSId) {
    let cachedDataSource;
    if (dataSourceMDSId) {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.dataSourceMDSId === dataSourceMDSId && ds.name === dataSourceName);
    } else {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.name === dataSourceName);
    }
    if (!cachedDataSource) {
      throw new Error('DataSource not found exception: ' + dataSourceName);
    }
    const cachedDatabase = cachedDataSource.databases.find(db => db.name === databaseName);
    if (!cachedDatabase) {
      throw new Error('Database not found exception: ' + databaseName);
    }
    return cachedDatabase;
  }

  /**
   * Retrieves a table from the cache.
   * @param {string} dataSourceName - The name of the data source containing the database.
   * @param {string} databaseName - The name of the database.
   * @param {string} tableName - The name of the database.
   * @returns {Cachedtable} The retrieved database.
   * @throws {Error} If the data source, database or table is not found.
   */
  static getTable(dataSourceName, databaseName, tableName, dataSourceMDSId) {
    const cachedDatabase = this.getDatabase(dataSourceName, databaseName, dataSourceMDSId);
    const cachedTable = cachedDatabase.tables.find(table => table.name === tableName);
    if (!cachedTable) {
      throw new Error('Table not found exception: ' + tableName);
    }
    return cachedTable;
  }

  /**
   * Updates a database in the cache.
   * @param {string} dataSourceName - The name of the data source containing the database.
   * @param {CachedDatabase} database - The database to be updated.
   * @throws {Error} If the data source or database is not found.
   */
  static updateDatabase(dataSourceName, database, dataSourceMDSId) {
    let cachedDataSource;
    if (dataSourceMDSId) {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.dataSourceMDSId === dataSourceMDSId && ds.name === dataSourceName);
    } else {
      cachedDataSource = this.getDataSourceCache().dataSources.find(ds => ds.name === dataSourceName);
    }
    if (!cachedDataSource) {
      throw new Error('DataSource not found exception: ' + dataSourceName);
    }
    const index = cachedDataSource.databases.findIndex(db => db.name === database.name);
    if (index !== -1) {
      cachedDataSource.databases[index] = database;
      this.addOrUpdateDataSource(cachedDataSource, dataSourceMDSId);
    } else {
      throw new Error('Database not found exception: ' + database.name);
    }
  }

  /**
   * Clears the data source cache from local storage.
   */
  static clearDataSourceCache() {
    sessionStorage.removeItem(this.datasourceCacheKey);
  }

  /**
   * Clears the accelerations cache from local storage.
   */
  static clearAccelerationsCache() {
    sessionStorage.removeItem(this.accelerationsCacheKey);
  }
}
exports.CatalogCacheManager = CatalogCacheManager;
/**
 * Key for the data source cache in local storage.
 */
_defineProperty(CatalogCacheManager, "datasourceCacheKey", _shared.ASYNC_QUERY_DATASOURCE_CACHE);
/**
 * Key for the accelerations cache in local storage.
 */
_defineProperty(CatalogCacheManager, "accelerationsCacheKey", _shared.ASYNC_QUERY_ACCELERATIONS_CACHE);